/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.configBrowser;

import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.properties.PropertyClass;
import de.bsvrz.sys.funclib.kappich.properties.PropertyName;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * {@code PartialSegmentProperties} ist eine Klasse, die {@link ConfigObjectProperties} um die Informationen für Objekte vom Typ {@code
 * StraßenTeilSegment} erweitert.
 *
 * @author Kappich Systemberatung
 */
@PropertyClass
public class PartialSegmentProperties extends ConfigObjectProperties {

    private static final Object _configurationAccess = new Object();
    private static final Map<SystemObject, List<Object>> _segmentToSegmentMap = new HashMap<>();
    private static boolean _segmentToSegmentMapInitialized;

    /**
     * Der Konstruktor.
     *
     * @param systemObject ein SystemObject
     */
    public PartialSegmentProperties(@NotNull final SystemObject systemObject) {
        super(systemObject);
    }

    /**
     * Diese Methode bestimmt, welche Netz-Eigenschaften für äußere Straßensegmente angezeigt werden.
     *
     * @return ein ConfigObjectProperties-Objekt oder eine Liste davon
     */
    @SuppressWarnings("SynchronizationOnStaticField")
    @PropertyName(name = "Straßensegmente", sortKey = 10)
    public Object getStreetSegmentProperties() {
        synchronized (_configurationAccess) {
            if (!_segmentToSegmentMapInitialized) {
                initializeLookup();
            }
        }
        List<Object> list = _segmentToSegmentMap.get(_systemObject);
        if (null == list) {
            return Collections.EMPTY_LIST;
        } else if (list.size() == 1) {
            return list.get(0);
        } else {
            return list;
        }
    }

    private void initializeLookup() {
        _segmentToSegmentMap.clear();

        DataModel configuration = _systemObject.getDataModel();
        SystemObjectType segmentType = configuration.getType("typ.straßenSegment");
        List<SystemObject> segments = segmentType.getElements();
        AttributeGroup atg = configuration.getAttributeGroup("atg.bestehtAusLinienObjekten");
        for (SystemObject segment : segments) {
            final Data data = segment.getConfigurationData(atg);
            if (null != data) {
                final Data.ReferenceValue[] partialSegmentRefs = data.getReferenceArray("LinienReferenz").getReferenceValues();
                for (Data.ReferenceValue partialSegmentRef : partialSegmentRefs) {
                    SystemObject partialSegment = partialSegmentRef.getSystemObject();
                    if (_segmentToSegmentMap.containsKey(partialSegment)) {
                        _segmentToSegmentMap.get(partialSegment).add(createConfigObjectProperties(segment));
                    } else {
                        List<Object> newList = new ArrayList<>(1);
                        newList.add(createConfigObjectProperties(segment));
                        _segmentToSegmentMap.put(partialSegment, newList);
                    }
                }
            }
        }
        _segmentToSegmentMapInitialized = true;
    }
}
