/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.kappich.pat.gnd.csv;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.nio.charset.Charset;
import java.util.SortedMap;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import javax.swing.JOptionPane;

/**
 * @author Kappich Systemberatung
 */
public class CsvFormat implements Comparable<CsvFormat> {

    private static final SortedMap<String, Charset> AVAILABLE_CHARSETS = Charset.availableCharsets();

    private static final String INFO = "INFO";
    private static final String NAME_COLUMN = "NameColumn";
    private static final String X_COLUMN = "XColumn";
    private static final String Y_COLUMN = "YColumn";
    private static final String LINE_COLUMN = "LineColumn";
    private static final String OFFSET_COLUMN = "OffsetColumn";
    private static final String OBJECT_COLUMN = "ObjectColumn";
    private static final String REF_HIERARCHY = "RefHierarchy";
    private static final String PRIORITY = "Priority";
    private static final String CHARSET = "Charset";
    private static final String SEPARATOR = "Separator";
    private static final String QUOTE = "Quote";

    private String _name;
    private String _info;
    // Spaltennamen:
    private String _nameColumn = "";
    private String _xColumn = "";
    private String _yColumn = "";
    private String _lineColumn = "";
    private String _offsetColumn = "";
    private String _objectColumn = "";
    private String _referenceHierarchy = "";
    private CsvPriority _csvPriority = CsvPriority.getDefaultPrioroty();
    private Charset _charset;
    private CsvSeparator _separator = CsvSeparator.COMMA;
    private CsvQuote _quote = CsvQuote.DOUBLE_QUOTE;

    /**
     * Ein Default-Konstruktor, der z.B. für die Initialisierung mit initializeFromPreferences() benötigt wird.
     */
    public CsvFormat() {
        _name = null;
        _info = null;
    }

    /**
     * Im folgenden Konstruktor werden dem CsvFormat seine ersten Bestandteile mitgegeben.
     *
     * @param name der Name des CsvFormats
     * @param info der Infotext zum CsvFormat
     */
    public CsvFormat(String name, @Nullable String info) {
        _name = name;
        _info = info;
    }

    /**
     * Diese Methode kopiert die Elemente eines CsvFormats in ein anderes, wobei wählber ist, ob auch der Name kopiert wird.
     *
     * @param target
     * @param source
     * @param withName
     */
    public static void copy(CsvFormat target, CsvFormat source, boolean withName) {
        if (withName) {
            target._name = source._name;
        }
        target._xColumn = source._xColumn;
        target._yColumn = source._yColumn;
        target._lineColumn = source._lineColumn;
        target._offsetColumn = source._offsetColumn;
        target._objectColumn = source._objectColumn;
        target._referenceHierarchy = source._referenceHierarchy;
        target._csvPriority = source._csvPriority;
        target._charset = source._charset;
        target._separator = source._separator;
        target._quote = source._quote;
    }

    /**
     * Der Getter für den Namen.
     *
     * @return den Namen
     */
    public String getName() {
        return _name;
    }

    /**
     * Der Setter für den Namen.
     *
     * @param name der Name des CsvFormats
     */
    public void setName(final String name) {
        _name = name;
    }

    /**
     * Der Getter für den Infotext.
     *
     * @return der Infotext des CsvFormats
     */
    public String getInfo() {
        return _info;
    }

    /**
     * Der Setter für den Infotext.
     *
     * @param info der Infotext
     */
    public void setInfo(String info) {
        _info = info;
    }

    /**
     * @return
     */
    public String getNameColumn() {
        return _nameColumn;
    }

    /**
     * @param nameColumn
     */
    public void setNameColumn(final String nameColumn) {
        _nameColumn = nameColumn;
    }

    /**
     * @return
     */
    public String getXColumn() {
        return _xColumn;
    }

    /**
     * @param xColumn
     */
    public void setXColumn(final String xColumn) {
        _xColumn = xColumn;
    }

    /**
     * @return
     */
    public String getYColumn() {
        return _yColumn;
    }

    /**
     * @param yColumn
     */
    public void setYColumn(final String yColumn) {
        _yColumn = yColumn;
    }

    /**
     * @return
     */
    public String getLineColumn() {
        return _lineColumn;
    }

    /**
     * @param lineColumn
     */
    public void setLineColumn(final String lineColumn) {
        _lineColumn = lineColumn;
    }

    /**
     * @return
     */
    public String getOffsetColumn() {
        return _offsetColumn;
    }

    /**
     * @param offsetColumn
     */
    public void setOffsetColumn(final String offsetColumn) {
        _offsetColumn = offsetColumn;
    }

    /**
     * @return
     */
    public String getObjectColumn() {
        return _objectColumn;
    }

    /**
     * @param objectColumn
     */
    public void setObjectColumn(final String objectColumn) {
        _objectColumn = objectColumn;
    }

    /**
     * @return
     */
    public String getReferenceHierarchy() {
        return _referenceHierarchy;
    }

    /**
     * @param referenceHierarchy
     */
    public void setReferenceHierarchy(final String referenceHierarchy) {
        _referenceHierarchy = referenceHierarchy;
    }

    public CsvPriority getCsvPriority() {
        return _csvPriority;
    }

    public void setCsvPriority(CsvPriority priority) {
        _csvPriority = priority;
    }

    public Charset getCharset() {
        return _charset;
    }

    public void setCharset(final Charset charset) {
        _charset = charset;
    }

    public CsvSeparator getSeparator() {
        return _separator;
    }

    public void setSeparator(final CsvSeparator separator) {
        _separator = separator;
    }

    public CsvQuote getQuote() {
        return _quote;
    }

    public void setQuote(final CsvQuote quote) {
        _quote = quote;
    }

    /**
     * Speichert die Präferenzen des CsvFormats unter dem übergebenen Knoten.
     *
     * @param prefs der Knoten, unter dem die Präferenzen gespeichert werden
     */
    public void putPreferences(Preferences prefs) {
        deletePreferences(prefs);
        Preferences objectPrefs = prefs.node(getName());
        if (null == _info) {
            objectPrefs.put(INFO, "");
        } else {
            objectPrefs.put(INFO, _info);
        }
        if (null == getNameColumn()) {
            objectPrefs.put(NAME_COLUMN, "");
        } else {
            objectPrefs.put(NAME_COLUMN, getNameColumn());
        }
        if (null == getXColumn()) {
            objectPrefs.put(X_COLUMN, "");
        } else {
            objectPrefs.put(X_COLUMN, getXColumn());
        }
        if (null == getYColumn()) {
            objectPrefs.put(Y_COLUMN, "");
        } else {
            objectPrefs.put(Y_COLUMN, getYColumn());
        }
        if (null == getLineColumn()) {
            objectPrefs.put(LINE_COLUMN, "");
        } else {
            objectPrefs.put(LINE_COLUMN, getLineColumn());
        }
        if (null == getOffsetColumn()) {
            objectPrefs.put(OFFSET_COLUMN, "");
        } else {
            objectPrefs.put(OFFSET_COLUMN, getOffsetColumn());
        }
        if (null == getObjectColumn()) {
            objectPrefs.put(OBJECT_COLUMN, "");
        } else {
            objectPrefs.put(OBJECT_COLUMN, getObjectColumn());
        }
        if (null == getReferenceHierarchy()) {
            objectPrefs.put(REF_HIERARCHY, "");
        } else {
            objectPrefs.put(REF_HIERARCHY, getReferenceHierarchy());
        }
        objectPrefs.put(PRIORITY, getCsvPriority().getDescription());
        objectPrefs.put(CHARSET, getCharset().name());
        objectPrefs.put(SEPARATOR, String.valueOf(_separator));
        objectPrefs.put(QUOTE, String.valueOf(_quote));
    }

    /**
     * Löscht die Präferenzen des CsvFormats unter dem Knoten.
     *
     * @param prefs der Knoten, unter dem die Präferenzen gelöscht werden
     */
    public void deletePreferences(Preferences prefs) {
        Preferences objectPrefs = prefs.node(getName());
        try {
            objectPrefs.removeNode();
        } catch (BackingStoreException e) {
            JOptionPane.showMessageDialog(null, "Das Löschen des CsvFormats war nicht erfolgreich." + e.toString(), "Fehlermeldung",
                                          JOptionPane.ERROR_MESSAGE);
        }
    }

    /**
     * Initialisiert das CsvFormat aus den Präferenzen unter dem übergebenen Knoten.
     *
     * @param prefs der Knoten, unter dem die Präferenzen gesucht werden
     *
     * @return gibt {@code true} zurück, wenn die Initialisierung erfolgreich war, und {@code false}, falls nicht
     */
    public boolean initializeFromPreferences(Preferences prefs) {
        _name = prefs.name();
        _info = prefs.get(INFO, "");
        _nameColumn = prefs.get(NAME_COLUMN, "");
        _xColumn = prefs.get(X_COLUMN, "");
        _yColumn = prefs.get(Y_COLUMN, "");
        _lineColumn = prefs.get(LINE_COLUMN, "");
        _offsetColumn = prefs.get(OFFSET_COLUMN, "");
        _objectColumn = prefs.get(OBJECT_COLUMN, "");
        _referenceHierarchy = prefs.get(REF_HIERARCHY, "");
        _csvPriority = CsvPriority.getCsvPriority(prefs.get(PRIORITY, ""));

        _charset = AVAILABLE_CHARSETS.get(prefs.get(CHARSET, ""));
        String seperatorString = prefs.get(SEPARATOR, "");
        _separator = CsvSeparator.getSeparator(seperatorString);
        String quoteString = prefs.get(QUOTE, "");
        _quote = CsvQuote.getQuote(quoteString);
        return true;
    }

    /**
     * Diese Methode gibt eine Kopie des CsvFormats zurück.
     *
     * @return die Kopie
     */
    public CsvFormat getCopy() {
        final CsvFormat copy = new CsvFormat(_name, _info);
        copy._nameColumn = _nameColumn;
        copy._xColumn = _xColumn;
        copy._yColumn = _yColumn;
        copy._lineColumn = _lineColumn;
        copy._offsetColumn = _offsetColumn;
        copy._objectColumn = _objectColumn;
        copy._referenceHierarchy = _referenceHierarchy;
        copy._csvPriority = _csvPriority;
        copy._charset = _charset;
        copy._separator = _separator;
        copy._quote = _quote;
        return copy;
    }

    @Override
    public int compareTo(final CsvFormat o) {
        return getName().toLowerCase().compareTo(o.getName().toLowerCase());
    }

    @Override
    public boolean equals(Object o) {
	    if (!(o instanceof CsvFormat other)) {
            return false;
        }
        return getName().toLowerCase().equals(other.getName().toLowerCase());
    }

    @Override
    public int hashCode() {
        return getName().toLowerCase().hashCode();
    }

    @Override
    public String toString() {
        return "CsvFormat{" + "_name='" + _name + '\'' + ", _info='" + _info + '\'' + ", _nameColumn='" + _nameColumn + '\'' + ", _xColumn='" +
               _xColumn + '\'' + ", _yColumn='" + _yColumn + '\'' + ", _lineColumn='" + _lineColumn + '\'' + ", _offsetColumn='" + _offsetColumn +
               '\'' + ", _objectColumn='" + _objectColumn + '\'' + ", _referenceHierarchy='" + _referenceHierarchy + '\'' + ", _csvPriority='" +
               _csvPriority + '\'' + ", _charset='" + _charset + '\'' + ", _separator='" + _separator + '\'' + ", _quote='" + _quote + '\'' + '}';
    }
}
