/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.csv.view;

import de.kappich.pat.gnd.csv.CsvFormat;
import de.kappich.pat.gnd.csv.CsvFormatManager;
import de.kappich.pat.gnd.documentation.GndHelp;
import de.kappich.pat.gnd.utils.SpringUtilities;
import de.kappich.pat.gnd.utils.view.GndFrame;
import de.kappich.pat.gnd.utils.view.GndTable;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Frame;
import java.awt.Window;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.List;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.SpringLayout;
import javax.swing.WindowConstants;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.DefaultTableCellRenderer;

/**
 * @author Kappich Systemberatung
 * @version $Revision$
 */
@SuppressWarnings("OverlyLongMethod")
public class CfmDialog {

    private final GndFrame _frame;

    private CsvFormatDefinitionDialog _csvFormatDefinitionDialog = null;

    /**
     * Konstruiert einen neuen CfmDialog.
     */
    public CfmDialog() {

        _frame = new GndFrame("CfmDialog", "GND: CSV-Format-Verwaltung");

        _frame.setLayout(new BorderLayout());

        final JButton newCsvFormatButton = new JButton("Neues CSV-Format");
        newCsvFormatButton.setActionCommand("Ein neues CSV-Format anlegen");

        final JButton editCsvFormatButton = new JButton("Bearbeiten");
        editCsvFormatButton.setActionCommand("Details des augewählten CSV-Formats bearbeiten");

        final JButton copyCsvFormatButton = new JButton("Kopieren");
        copyCsvFormatButton.setActionCommand("Das ausgewählte CSV-Format kopieren und bearbeiten");

        final JButton deleteCsvFormatButton = new JButton("Löschen");
        deleteCsvFormatButton.setActionCommand("Das ausgewählte CSV-Format löschen");

        final JButton getHelpButton = new JButton("Hilfe");
        getHelpButton.setActionCommand("Die Online-Hilfe öffnen");

        editCsvFormatButton.setEnabled(false);
        copyCsvFormatButton.setEnabled(false);
        deleteCsvFormatButton.setEnabled(false);

        JPanel buttonsPanelEast = new JPanel();
        buttonsPanelEast.setLayout(new BoxLayout(buttonsPanelEast, BoxLayout.Y_AXIS));

        Dimension d = new Dimension(15, 15);
        buttonsPanelEast.add(Box.createRigidArea(d));

        buttonsPanelEast.add(newCsvFormatButton);
        buttonsPanelEast.add(Box.createRigidArea(d));

        buttonsPanelEast.add(editCsvFormatButton);
        buttonsPanelEast.add(Box.createRigidArea(d));

        buttonsPanelEast.add(copyCsvFormatButton);
        buttonsPanelEast.add(Box.createRigidArea(d));

        buttonsPanelEast.add(deleteCsvFormatButton);
        buttonsPanelEast.add(Box.createRigidArea(d));

        buttonsPanelEast.add(getHelpButton);
        buttonsPanelEast.add(Box.createRigidArea(d));

        buttonsPanelEast.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));

        JPanel buttonsPanelSouth = new JPanel();
        buttonsPanelSouth.setLayout(new SpringLayout());
        buttonsPanelSouth.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));

        JButton schliessenButton = new JButton("Dialog schließen");
        buttonsPanelSouth.add(schliessenButton);
        SpringUtilities.makeCompactGrid(buttonsPanelSouth, 1, 20, 5);

        final GndTable table = new GndTable(CsvFormatManager.getInstance());

        table.setFillsViewportHeight(true);
        table.setDefaultRenderer(Object.class, new CsvFormatTableCellRenderer());
        // vordefinierte Darstellungstypen dürfen nicht bearbeitet oder gelöscht werden
        final ListSelectionModel selectionModel = table.getSelectionModel();

        final List<CsvFormat> formatList = CsvFormatManager.getInstance().getCsvFormats();

        ListSelectionListener listSelctionListener = e -> {
            final int selectedRow = table.getSelectedRow();

            if(selectedRow == -1) {
                editCsvFormatButton.setToolTipText("Es ist keine Ansicht ausgewählt worden");
                editCsvFormatButton.setEnabled(false);
                copyCsvFormatButton.setToolTipText("Es ist keine Ansicht ausgewählt worden");
                copyCsvFormatButton.setEnabled(false);
                deleteCsvFormatButton.setToolTipText("Es ist keine Ansicht ausgewählt worden");
                deleteCsvFormatButton.setEnabled(false);
            }
            else {
                final boolean changeable = CsvFormatManager.getInstance().isChangeable(formatList.get(selectedRow));
                if(!changeable) {
                    editCsvFormatButton.setText("Betrachten");
                    editCsvFormatButton.setToolTipText("Details der augewählten CSV-Formats betrachten");
                    editCsvFormatButton.setEnabled(true);
                    copyCsvFormatButton.setToolTipText("Kopie des ausgewählten CSV-Formats erstellen und bearbeiten");
                    copyCsvFormatButton.setEnabled(true);
                    deleteCsvFormatButton.setToolTipText("Das ausgewählte CSV-Formats ist nicht löschbar");
                    deleteCsvFormatButton.setEnabled(false);
                }
                else {
                    editCsvFormatButton.setText("Bearbeiten");
                    editCsvFormatButton.setToolTipText("Details des ausgewählten CSV-Formats bearbeiten");
                    editCsvFormatButton.setEnabled(true);
                    copyCsvFormatButton.setToolTipText("Kopie des ausgewählten CSV-Formats erstellen und bearbeiten");
                    copyCsvFormatButton.setEnabled(true);
                    deleteCsvFormatButton.setToolTipText("Das ausgewählten CSV-Formats löschen");
                    deleteCsvFormatButton.setEnabled(true);
                }
            }
        };
        selectionModel.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        selectionModel.addListSelectionListener(listSelctionListener);

        table.addKeyListener(new KeyAdapter() {
            @Override
            public void keyPressed(KeyEvent e) {
                if(e.getKeyCode() == KeyEvent.VK_DELETE) {
                    final CsvFormat format = formatList.get(table.getSelectedRow());
                    final CsvFormatManager instance = CsvFormatManager.getInstance();
                    if(instance.isChangeable(format)) {
                        instance.removeCsvFormat(format);
                    }
                }
            }
        });

        table.addMouseListener(new MouseAdapter() {
            @Override
            public void mousePressed(final MouseEvent e) {
                if(e.getClickCount() > 1) {
                    editOrInspectAction(table);
                }
            }
        });

        _frame.add(buttonsPanelEast, BorderLayout.EAST);
        _frame.add(buttonsPanelSouth, BorderLayout.SOUTH);
        _frame.add(new JScrollPane(table), BorderLayout.CENTER);

        // Neu, Bearbeiten, Kopieren, Löschen
        ActionListener actionListenerNew = e -> {
            CsvFormat newFormat = new CsvFormat();
            newFormat.setName("");
            newFormat.setInfo("");
            final String title = "GND: neues CSV-Format bearbeiten";
            if(_csvFormatDefinitionDialog == null) {
                _csvFormatDefinitionDialog = new CsvFormatDefinitionDialog(newFormat, true, true, title);
            }
            else {
                _csvFormatDefinitionDialog.setCsvFormat(newFormat, true, true);
                _csvFormatDefinitionDialog.setTitle(title);
                _csvFormatDefinitionDialog.setPositionAndSize();
                _csvFormatDefinitionDialog.setVisible(true);
                _csvFormatDefinitionDialog.toFront();
            }
        };
        newCsvFormatButton.addActionListener(actionListenerNew);


        ActionListener actionListenerLoeschen = e -> {
            if(table.getSelectedRowCount() == 0) {
                JOptionPane.showMessageDialog(
                    _frame.getFrame(),
                    "Bitte wählen Sie mindestens eine Zeile aus der Liste aus!",
                    "Fehler",
                    JOptionPane.ERROR_MESSAGE
                );
                return;
            }
            final int[] selectedRows = table.getSelectedRows();
            boolean allFormatsRemoved = true;
            final CsvFormatManager formatManager = CsvFormatManager.getInstance();
            for(int i = selectedRows.length - 1; i >= 0; i--) {
                final CsvFormat format = formatManager.getCsvFormat(selectedRows[i]);
                if(!formatManager.removeCsvFormat(format)) {
                    allFormatsRemoved = false;
                }
            }
            formatManager.fireTableDataChanged();
            if(!allFormatsRemoved) {
                JOptionPane.showMessageDialog(
                    _frame.getFrame(),
                    "Es konnten nicht alle ausgewählten CSV-Formate gelöscht werden!",
                    "Fehler",
                    JOptionPane.ERROR_MESSAGE
                );
            }
        };
        deleteCsvFormatButton.addActionListener(actionListenerLoeschen);

        ActionListener actionListenerKopieren = e -> {
            if(table.getSelectedRowCount() != 1) {
                JOptionPane.showMessageDialog(
                    _frame.getFrame(),
                    "Bitte wählen Sie genau eine Zeile aus der Liste aus!",
                    "Fehler",
                    JOptionPane.ERROR_MESSAGE
                );
                return;
            }
            final int selectedRow = table.getSelectedRow();
            final CsvFormat format = CsvFormatManager.getInstance().getCsvFormat(selectedRow);
            CsvFormat newFormat = format.getCopy();
            newFormat.setName(format.getName() + " (Kopie)");
            final String title = "GND: ein kopiertes CSV-Format bearbeiten";
            if(_csvFormatDefinitionDialog == null) {
                _csvFormatDefinitionDialog = new CsvFormatDefinitionDialog(newFormat, true, true, title);
            }
            else {
                _csvFormatDefinitionDialog.setCsvFormat(newFormat, true, true);
                _csvFormatDefinitionDialog.setTitle(title);
                _csvFormatDefinitionDialog.setPositionAndSize();
                _csvFormatDefinitionDialog.setVisible(true);
                _csvFormatDefinitionDialog.toFront();
            }
        };
        copyCsvFormatButton.addActionListener(actionListenerKopieren);

        ActionListener actionListenerBearbeiten = e -> editOrInspectAction(table);
        editCsvFormatButton.addActionListener(actionListenerBearbeiten);

        ActionListener actionListenerHelp = e -> GndHelp.openHelp("#theCsvFormatManagerDialog");
        getHelpButton.addActionListener(actionListenerHelp);

        ActionListener actionListenerDialogSchliessen = e -> closeDialog();
        schliessenButton.addActionListener(actionListenerDialogSchliessen);

        class FrameListener extends WindowAdapter {
            @Override
            public void windowClosing(WindowEvent e) {
                /*
                 * wenn nur noch ein einziges Fenster geöffnet ist
                 * beendet sich das Programm beim Schließen des Fensters
                 */
//				final Frame[] frames = Frame.getFrames();
//				int length = frames.length - 1;
//				for(Frame frame : frames) {
//					if(!frame.isVisible()) {
//						length--;
//					}
//				}
//				if(length == 0) {
//					_frame.setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
//				}
                // Die alte Implentation beendete gelegentlich die ganze Anwendung, wenn
                // nur ein Frame geschlossen wurde, etwa während einer Initialisierung einer
                // neuen Ansicht. Statt Frames versuche ich jetzt Windows.
                final Window[] windows = Window.getWindows();
                int length = windows.length - 1;
                for(Window window : windows) {
                    if(!window.isVisible()) {
                        length--;
                    }
                }
                if(length <= 0) {
                    _frame.setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
                }
            }
        }
        _frame.addWindowListener(new FrameListener());

        table.setVisibleRowCount(Math.min(40, table.getModel().getRowCount()));
        _frame.setPositionAndSize(0, 0, 860, 50, true, 0, 0);
        _frame.setVisible(true);
    }

    private void editOrInspectAction(final JTable table) {
        final int selectedRow = table.getSelectedRow();
        if(selectedRow == -1) {
            JOptionPane.showMessageDialog(
                _frame.getFrame(),
                "Bitte wählen Sie mindestens eine Zeile aus der Liste aus!",
                "Fehler",
                JOptionPane.ERROR_MESSAGE
            );
            return;
        }
        final CsvFormat format = CsvFormatManager.getInstance().getCsvFormat(selectedRow);
        final boolean changeable = CsvFormatManager.getInstance().isChangeable(format);
        final String title;
        if(changeable) {
            title = "GND: ein CSV-Format bearbeiten";
        }
        else {
            title = "GND: ein CSV-Format betrachten";
        }
        if(_csvFormatDefinitionDialog == null) {
            _csvFormatDefinitionDialog = new CsvFormatDefinitionDialog(format, changeable, false, title);
        }
        else {
            _csvFormatDefinitionDialog.setCsvFormat(format, changeable, false);
            _csvFormatDefinitionDialog.setTitle(title);
            _csvFormatDefinitionDialog.setPositionAndSize();
            _csvFormatDefinitionDialog.setVisible(true);
            _csvFormatDefinitionDialog.toFront();
        }
    }

    private static class CsvFormatTableCellRenderer extends DefaultTableCellRenderer {

        private static final long serialVersionUID = 1L;

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
            super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);

            if(null != value) { // null passiert auf Macs
	            if (value instanceof String name) {
                    // Text
                    setText(name);
                    if(CsvFormatManager.getInstance().hasCsvFormatToLowerCase(name)) {
                        CsvFormat format = CsvFormatManager.getInstance().getCsvFormat(name);
                        // Font und Farbe
                        if(CsvFormatManager.getInstance().isChangeable(format)) {
                            setForeground(Color.BLACK);
                            Font font = getFont();
                            setFont(new Font(font.getName(), Font.PLAIN, font.getSize()));
                        }
                        else {
                            setForeground(Color.GRAY);
                            Font font = getFont();
                            setFont(new Font(font.getName(), Font.ITALIC, font.getSize()));
                        }
                        // Tooltipp
                        setToolTipText(format.getInfo());
                    }
                    else { // sollte nicht eintreten
                        setForeground(Color.RED);
                        Font font = getFont();
                        setFont(new Font(font.getName(), Font.BOLD, font.getSize()));
                    }
                }
                else { // sollte nicht eintreten
                    setText(value.toString());
                    setForeground(Color.RED);
                    Font font = getFont();
                    setFont(new Font(font.getName(), Font.BOLD, font.getSize()));
                }
            }
            return this;
        }
    }

    /**
     * Macht die Dialog sichtbar.
     */
    public void showDialog(boolean visible) {
        _frame.setState(Frame.NORMAL);
        _frame.setVisible(visible);
    }

    /**
     * Beendet den Dialog.
     */
    public void closeDialog() {
        _frame.setVisible(false);
        _frame.storePreferenceBounds();
        _frame.dispose();
    }

    @Override
    public String toString() {
        return "CfmDialog";
    }


}
