/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.csvPlugin;

import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.coorTransform.UTMCoordinate;
import de.kappich.pat.gnd.csv.CsvFormat;
import de.kappich.pat.gnd.csv.CsvPriority;
import de.kappich.pat.gnd.displayObjectToolkit.DOTCollection;
import de.kappich.pat.gnd.displayObjectToolkit.DisplayObject;
import de.kappich.pat.gnd.displayObjectToolkit.GeoInitializer;
import de.kappich.pat.gnd.elrPlugin.ElrInitializer;
import de.kappich.pat.gnd.extLocRef.ReferenceHierarchy;
import de.kappich.pat.gnd.extLocRef.ReferenceHierarchyManager;
import de.kappich.pat.gnd.gnd.MapPane;
import de.kappich.pat.gnd.layerManagement.Layer;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectPainter;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectTypePlugin;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectsInitializer;
import de.kappich.pat.gnd.utils.PointWithAngle;
import de.kappich.sys.funclib.csv.CsvColumn;
import de.kappich.sys.funclib.csv.CsvParseException;
import de.kappich.sys.funclib.csv.CsvReader;
import de.kappich.sys.funclib.csv.IterableCsvData;
import java.awt.geom.Point2D;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.swing.JProgressBar;

/**
 * Diese Klasse implementiert {@link DisplayObjectsInitializer} für das CSV-Plugin.
 *
 * @author Kappich Systemberatung
 */
public class CsvInitializer implements DisplayObjectsInitializer {

    private static final Debug _debug = Debug.getLogger();

    @SuppressWarnings({"OverlyLongMethod", "OverlyNestedMethod"})
    @Override
    public void initializeDisplayObjects(final DataModel configuration, final Layer layer, final MapPane mapPane, final JProgressBar progressBar,
                                         final List<DisplayObject> returnList) {
        if (!layer.getPlugin().getName().equals("CSV")) {
            return;
        }
        StringBuilder sb = new StringBuilder("Layer: " + layer.getName() + System.lineSeparator());
        File file = layer.getCsvFile();
        CsvFormat format = layer.getCsvFormat();
        if (null == file) {
            sb.append("Keine Csv-Datei zur Initialisierungszeit!").append(System.lineSeparator());
            sb.append("Die Initialisierungs wurde abgebrochen.").append(System.lineSeparator());
            layer.setCsvInitInfo(sb.toString());
            return;
        }
        if (null == format) {
            sb.append("Keine Csv-Format zur Initialisierungszeit!").append(System.lineSeparator());
            sb.append("Die Initialisierungs wurde abgebrochen.").append(System.lineSeparator());
            return;
        }
        sb.append("Csv-Datei: ").append(file.getAbsolutePath()).append(System.lineSeparator());
        sb.append("Csv-Format: ").append(format.getName()).append(System.lineSeparator());
        // Die folgenden Objekte werden durch das Lesen der CSV-Datei mit Inhalt gefüllt.
        String[] headerCells = null;
        List<IterableCsvData.CsvRow> xyList = new ArrayList<>();
        List<IterableCsvData.CsvRow> lineList = new ArrayList<>();
        List<IterableCsvData.CsvRow> objectList = new ArrayList<>();
        CsvColumn<String> nameColumn = null;
        CsvColumn<Double> xColumn = null;
        CsvColumn<Double> yColumn = null;
        CsvColumn<String> lineColumn = null;
        CsvColumn<Double> offsetColumn = null;
        CsvColumn<String> objectColumn = null;

        try (FileInputStream inputStream = new FileInputStream(file)) {
            CsvReader reader =
                new CsvReader(format.getCharset(), inputStream, format.getSeparator().getCharacter(), format.getQuote().getCharacter());
            IterableCsvData csvData = reader.read();
            headerCells = csvData.getHeaderCells();

            try {
                //noinspection ConstantConditions
                if (null != csvData) {
                    // Schritt 1: Überprüfung, ob das CsvFormat passende Spaltennamen spezifiziert.
                    Map<String, Integer> columnMp = csvData.getColumnNameToIndexMap();

                    if (!format.getNameColumn().isEmpty()) {
                        if (!columnMp.containsKey(format.getNameColumn())) {
                            sb.append("Die Spalte '").append(format.getNameColumn()).
                                append("' für die Namen konnte in der Datei nicht gefunden werden.").append(System.lineSeparator());
                            sb.append("Die Initialisierungs wurde abgebrochen.").append(System.lineSeparator());
                            layer.setCsvInitInfo(sb.toString());
                            return;
                        } else {
                            nameColumn = csvData.getColumn(format.getNameColumn());
                        }
                    }
                    if (!format.getXColumn().isEmpty()) {
                        if (!columnMp.containsKey(format.getXColumn())) {
                            sb.append("Die Spalte '").append(format.getXColumn()).
                                append("' für die X-Koordinaten konnte in der Datei nicht gefunden werden.").append(System.lineSeparator());
                            sb.append("Die Initialisierungs wurde abgebrochen.").append(System.lineSeparator());
                            layer.setCsvInitInfo(sb.toString());
                            return;
                        } else {
                            xColumn = csvData.getDoubleColumn(format.getXColumn());
                        }
                    }
                    if (!format.getYColumn().isEmpty()) {
                        if (!columnMp.containsKey(format.getYColumn())) {
                            sb.append("Die Spalte '").append(format.getYColumn()).
                                append("' für die Y-Koordinaten konnte in der Datei nicht gefunden werden.").append(System.lineSeparator());
                            sb.append("Die Initialisierungs wurde abgebrochen.").append(System.lineSeparator());
                            layer.setCsvInitInfo(sb.toString());
                            return;
                        } else {
                            yColumn = csvData.getDoubleColumn(format.getYColumn());
                        }
                    }
                    if (!format.getLineColumn().isEmpty()) {
                        if (!columnMp.containsKey(format.getLineColumn())) {
                            sb.append("Die Spalte '").append(format.getLineColumn()).
                                append("' für die Linien-Referenzen konnte in der Datei nicht gefunden werden.").append(System.lineSeparator());
                            sb.append("Die Initialisierungs wurde abgebrochen.").append(System.lineSeparator());
                            layer.setCsvInitInfo(sb.toString());
                            return;
                        } else {
                            lineColumn = csvData.getColumn(format.getLineColumn());
                        }
                    }
                    if (!format.getOffsetColumn().isEmpty()) {
                        if (!columnMp.containsKey(format.getOffsetColumn())) {
                            sb.append("Die Spalte '").append(format.getOffsetColumn()).
                                append("' für die Linien-Offsets konnte in der Datei nicht gefunden werden.").append(System.lineSeparator());
                            sb.append("Die Initialisierungs wurde abgebrochen.").append(System.lineSeparator());
                            layer.setCsvInitInfo(sb.toString());
                            return;
                        } else {
                            offsetColumn = csvData.getDoubleColumn(format.getOffsetColumn());
                        }
                    }
                    if (!format.getObjectColumn().isEmpty()) {
                        if (!columnMp.containsKey(format.getObjectColumn())) {
                            sb.append("Die Spalte '").append(format.getObjectColumn()).
                                append("' für die Objekt-Referenzen konnte in der Datei nicht gefunden werden.").append(System.lineSeparator());
                            sb.append("Die Initialisierungs wurde abgebrochen.").append(System.lineSeparator());
                            layer.setCsvInitInfo(sb.toString());
                            return;
                        } else {
                            objectColumn = csvData.getColumn(format.getObjectColumn());
                        }
                    }
                    // Schritt 2: Aufteilen der Zeilen auf solche, die Koordinaten, Linien+Offset bzw. eine Objekt-Referenz haben,
                    //          wobei die Priorität beachtet wird.
                    int rowCounter = 0;
                    List<Integer> problematicRowIndices = new ArrayList<>();
                    for (final IterableCsvData.CsvRow row : csvData) {
                        ++rowCounter;
                        if (!initListsForSteps(row, format.getCsvPriority().getFirst(), xyList, xColumn, yColumn, lineList, lineColumn, offsetColumn,
                                               objectList, objectColumn)) {
                            if (!initListsForSteps(row, format.getCsvPriority().getSecond(), xyList, xColumn, yColumn, lineList, lineColumn,
                                                   offsetColumn, objectList, objectColumn)) {
                                if (!initListsForSteps(row, format.getCsvPriority().getThird(), xyList, xColumn, yColumn, lineList, lineColumn,
                                                       offsetColumn, objectList, objectColumn)) {
                                    problematicRowIndices.add(rowCounter + 1);    // +1 wegen der Kopfzeile
                                }
                            }
                        }
                    }
                    sb.append(System.lineSeparator());
                    sb.append("Aufteilung der Zeilen der Csv-Datei nach Ortsreferenzmethode:").append(System.lineSeparator());
                    sb.append("   ").append(xyList.size()).append(" Zeilen mit Koordinaten").append(System.lineSeparator());
                    sb.append("   ").append(lineList.size()).append(" Zeilen mit Linien-Referenzen und Offset").append(System.lineSeparator());
                    sb.append("   ").append(objectList.size()).append(" Zeilen mit Objekt-Referenzen").append(System.lineSeparator());
                    int without = rowCounter - xyList.size() - lineList.size() - objectList.size();
                    sb.append("   ").append(without).append(" Zeilen ohne gültige Ortsreferenzen").append(System.lineSeparator());
                    sb.append(System.lineSeparator());
                    if (!problematicRowIndices.isEmpty()) {
                        sb.append("Zeilen ohne gültige Ortsreferenz: ");
                        int counter = 0;
                        for (Integer i : problematicRowIndices) {
                            sb.append(i).append(" ");
                            ++counter;
                            if (counter % 10 == 0) {
                                sb.append(System.lineSeparator());
                            }
                        }
                        sb.append(System.lineSeparator()).append(System.lineSeparator());
                    }
                }
            } catch (CsvParseException e) {
                sb.append(System.lineSeparator()).append("Fehler beim Parsen der Csv-Datei.");
                sb.append(e.toString()).append(System.lineSeparator());
                sb.append("Die Initialisierungs wurde abgebrochen.").append(System.lineSeparator());
                layer.setCsvInitInfo(sb.toString());
                return;
            }
        } catch (IOException e) {
            sb.append(System.lineSeparator()).append("I/O-Fehler beim Lesen der Csv-Datei.");
            sb.append(e.toString()).append(System.lineSeparator());
            sb.append("Die Initialisierungs wurde abgebrochen.").append(System.lineSeparator());
            layer.setCsvInitInfo(sb.toString());
            return;
        }

        // 3. Schritt: Berechnung der Koordinaten je nach Ortsreferenz
        Map<IterableCsvData.CsvRow, List<Object>> coordinatesMap = new HashMap<>();
        StringBuilder errors = new StringBuilder();
        if (!xyList.isEmpty()) {
            //noinspection ConstantConditions
            int counter = initCoordinatesWithCoordinates(xyList, xColumn, yColumn, coordinatesMap, errors);
            sb.append(counter).append(" Objekte mit Koordinaten ortsreferenziert (").
                append(counter * 100 / xyList.size()).append(" Prozent).").append(System.lineSeparator());

        }
        if (!lineList.isEmpty()) {
            //noinspection ConstantConditions
            int counter = initCoordinatesByLinesAndOffsets(configuration, lineList, lineColumn, offsetColumn, coordinatesMap, errors);
            sb.append(counter).append(" Objekte mit Linien-Referenzen und Offsets ortsreferenziert (").
                append(counter * 100 / lineList.size()).append(" Prozent).").append(System.lineSeparator());
        }
        if (!objectList.isEmpty()) {
            //noinspection ConstantConditions
            int counter = initCoordinatesByObjectReferences(configuration, objectList, objectColumn, ReferenceHierarchyManager.getInstance()
                .getReferenceHierarchy(format.getReferenceHierarchy()), coordinatesMap, errors);
            sb.append(counter).append(" Objekte mit Linien-Referenzen und Objekt-Referenzen ortsreferenziert (").
                append(counter * 100 / objectList.size()).append(" Prozent).").append(System.lineSeparator());
        }

        // 4. Schritt: Erstellung der OnlineDisplayObjects
        // Versuch: Da es sich um Punkt-Objekte handelt, wird so viel wie möglich vom Punkt-Plugin übernommen.
        // D.s. die DOTs, DOTCollection und der Painter. Allerdings verwenden wir hier nicht das OnlineDisplayObject,
        // sondern ein eigenes DisplayObject, also CsvDisplayObject.
        final DOTCollection dotCollection = layer.getDotCollection();
        final Iterator<DisplayObjectType> iterator = dotCollection.values().iterator();
        if (!iterator.hasNext()) {
            return;
        }
        final DisplayObjectType dot = iterator.next();
        final DisplayObjectTypePlugin displayObjectTypePlugin = dot.getDisplayObjectTypePlugin();
        DisplayObjectPainter painter = displayObjectTypePlugin.getPainter();

        int nameParseErrors = 0;
        for (final Map.Entry<IterableCsvData.CsvRow, List<Object>> entry : coordinatesMap.entrySet()) {
            String name = "";
            if (nameColumn != null) {
                try {
                    final String value = entry.getKey().getValue(nameColumn);
                    if (value != null) {
                        name = value;
                    }
                } catch (CsvParseException ignore) {
                    ++nameParseErrors;
                }
            }
            Map<String, Object> csvRowValues = new LinkedHashMap<>();
            if (null != headerCells) {
                for (int i = 0; i < headerCells.length; ++i) {
                    csvRowValues.put(headerCells[i], entry.getKey().getValue(i));
                }
            }
            CsvDisplayObject displayObject = new CsvDisplayObject(name, painter, dotCollection, entry.getValue(), csvRowValues);
            returnList.add(displayObject);
        }

        if (nameParseErrors > 0) {
            errors.append(nameParseErrors).append(" Fehler beim Lesen der Namen.").append(System.lineSeparator());
        }

        String errs = errors.toString();
        if (!errs.isEmpty()) {
            sb.append(System.lineSeparator()).append("ACHTUNG: ").append(System.lineSeparator()).append(errs);
        }
        layer.setCsvInitInfo(sb.toString());
    }

    @SuppressWarnings({"MethodWithTooManyParameters", "VariableNotUsedInsideIf"})
    private boolean initListsForSteps(IterableCsvData.CsvRow row, CsvPriority.GeoType geoType, List<IterableCsvData.CsvRow> xyList,
                                      @Nullable CsvColumn<Double> xColumn, @Nullable CsvColumn<Double> yColumn, List<IterableCsvData.CsvRow> lineList,
                                      @Nullable CsvColumn<String> lineColumn, @Nullable CsvColumn<Double> offsetColumn,
                                      List<IterableCsvData.CsvRow> objectList, @Nullable CsvColumn<String> objectColumn) {
        try {
            switch (geoType) {
                case COORDINATES:
                    if (null != xColumn && null != yColumn) {
                        Double x = row.getValueOptional(xColumn);
                        Double y = row.getValueOptional(yColumn);
                        if (null != x && null != y) {
                            xyList.add(row);
                            return true;
                        }
                    }
                    break;
                case LINE_WITH_OFFSET:
                    if (null != lineColumn && null != offsetColumn) {
                        String line = row.getValueOptional(lineColumn);
                        Double offset = row.getValueOptional(offsetColumn);
                        if (null != line && !line.isEmpty() && null != offset) {
                            lineList.add(row);
                            return true;
                        }
                    }
                    break;
                case OBJECT_REFERENCE:
                    if (null != objectColumn) {
                        String object = row.getValueOptional(objectColumn);
                        if (null != object && !object.isEmpty()) {
                            objectList.add(row);
                            return true;
                        }
                    }
                    break;
            }
        } catch (CsvParseException e) {
            // Ein Fehler hier muss nicht an den Benutzer weitergeben werden.
            _debug.error("Die folgende CsvParseException wurde in CsvInitializer.initListsForSteps gefangen: " + e.getMessage());
        }
        return false;
    }

    private int initCoordinatesWithCoordinates(final List<IterableCsvData.CsvRow> rowList, CsvColumn<Double> xColumn, CsvColumn<Double> yColumn,
                                               Map<IterableCsvData.CsvRow, List<Object>> coordinatesMap, StringBuilder errors) {
        int counter = 0;
        List<Integer> errorIndices = new ArrayList<>();
        for (IterableCsvData.CsvRow row : rowList) {
            Double x;
            Double y;
            try {
                x = row.getValue(xColumn);
                y = row.getValue(yColumn);
            } catch (CsvParseException ignore) {
                errorIndices.add(row.getRowNumber());
                continue;
            }
            if (null != x && null != y && x > 0. && x < 24. && y > 40. && y < 64.) {
                // Um zu vermeiden, dass Fehler in den CSV-Daten zu vollkommen falschen Extremkoordinaten führen,
                // filtern wir hier x und y.
                UTMCoordinate utm = GeoInitializer.wgs84ToUtm(x, y);
                PointWithAngle newPoint = new PointWithAngle(new Point2D.Double(utm.getX(), utm.getY()), null);
                List<Object> newList = new ArrayList<>(1);
                newList.add(newPoint);
                coordinatesMap.put(row, newList);
                ++counter;
            } else {
                errorIndices.add(row.getRowNumber());
            }
        }
        if (!errorIndices.isEmpty()) {
            errors.append(errorIndices.size()).append(" Fehler beim Lesen der Koordinaten.").append(System.lineSeparator());
            errors.append("Zeilennummern: ");
            int counter2 = 0;
            for (Integer index : errorIndices) {
                errors.append(index).append(" ");
                ++counter2;
                if (counter2 % 10 == 0) {
                    errors.append(System.lineSeparator());
                }
            }
        }
        return counter;
    }

    private int initCoordinatesByLinesAndOffsets(DataModel configuration, final List<IterableCsvData.CsvRow> rowList, CsvColumn<String> lineColumn,
                                                 CsvColumn<Double> offsetColumn, Map<IterableCsvData.CsvRow, List<Object>> coordinatesMap,
                                                 StringBuilder errors) {
        int counter = 0;
        int parseErrors = 0;
        for (IterableCsvData.CsvRow row : rowList) {
            try {
                final String value = row.getValue(lineColumn);
                if (value != null) {
                    SystemObject line = configuration.getObject(value);
                    Double offset = row.getValue(offsetColumn);
                    if (null != offset) {
                        PointWithAngle pwa = GeoInitializer.getInstance().getPointWithAngle(line, offset);
                        if (null != pwa) {
                            List<Object> newList = new ArrayList<>(1);
                            newList.add(pwa);
                            coordinatesMap.put(row, newList);
                            ++counter;
                        }
                    } else {
                        ++parseErrors;
                    }
                } else {
                    ++parseErrors;
                }
            } catch (CsvParseException ignore) {
                ++parseErrors;
            }
        }
        if (parseErrors > 0) {
            errors.append(parseErrors).append(" Csv-Fehler beim Lesen der Linien-Referenzen und Offsets.").append(System.lineSeparator());
        }
        return counter;
    }

    private int initCoordinatesByObjectReferences(DataModel configuration, final List<IterableCsvData.CsvRow> rowList, CsvColumn<String> objectColumn,
                                                  ReferenceHierarchy hierarchy, Map<IterableCsvData.CsvRow, List<Object>> coordinatesMap,
                                                  StringBuilder errors) {
        int counter = 0;
        int unknownObjectReferences = 0;
        Map<SystemObject, List<IterableCsvData.CsvRow>> undoneLookup = new HashMap<>();
        for (IterableCsvData.CsvRow row : rowList) {
            SystemObject systemObject;
            try {
                final String value = row.getValue(objectColumn);
                if (value != null) {
                    systemObject = configuration.getObject(value);
                } else {
                    ++unknownObjectReferences;
                    continue;
                }
            } catch (CsvParseException ignore) {
                ++unknownObjectReferences;
                continue;
            }
            if (null != systemObject) {
                List<Object> coordinates = GeoInitializer.getInstance().getPointCoordinates(systemObject);
                if (!coordinates.isEmpty()) {
                    coordinatesMap.put(row, coordinates);
                    ++counter;
                } else {
                    if (undoneLookup.containsKey(systemObject)) {
                        undoneLookup.get(systemObject).add(row);
                    } else {
                        List<IterableCsvData.CsvRow> newList = new ArrayList<>();
                        newList.add(row);
                        undoneLookup.put(systemObject, newList);
                    }
                }
            }
        }
        // Falls eine EOR-Hierarchy bei dem Format angegeben wurde, so wird nun versucht, für die
        // verbleibenden Systemobjekte damit eine Koordinate zu bestimmen.
        if (null != hierarchy) {
            ElrInitializer elrInitializer = new ElrInitializer();
            Map<SystemObject, Set<SystemObject>> objectReferenceMap = elrInitializer.determineObjectReferences(hierarchy, undoneLookup.keySet());
            for (final Map.Entry<SystemObject, Set<SystemObject>> entry : objectReferenceMap.entrySet()) {
                Set<SystemObject> referencedObjects = entry.getValue();
                for (SystemObject referencedObject : referencedObjects) {
                    List<Object> coordinates = GeoInitializer.getInstance().getPointCoordinates(referencedObject);
                    if (!coordinates.isEmpty() && undoneLookup.containsKey(entry.getKey())) {
                        for (IterableCsvData.CsvRow row : undoneLookup.get(entry.getKey())) {
                            coordinatesMap.put(row, coordinates);
                            ++counter;
                        }
                        break;
                    }
                }
            }
        }
        if (unknownObjectReferences > 0) {
            errors.append(unknownObjectReferences).append(" unbekannte Objektreferenzen.").append(System.lineSeparator());
        }
        return counter;
    }
}
