/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.displayObjectToolkit;

import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.layerManagement.Layer;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.properties.Property;
import de.kappich.pat.gnd.utils.Interval;
import de.kappich.pat.gnd.utils.view.PreferencesDeleter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;
import java.util.TreeMap;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import javax.swing.table.AbstractTableModel;

/**
 * Ein Klasse zur Verwaltung der Darstellungstypen eines {@link Layer Layers}.
 * <p>
 * Eine {@code DOTCollection} verkapselt die Darstellungstypen eines Layers. Jeder Darstellungstyp eines Layers hat eine untere und obere
 * Maßstabsgrenze, zwischen denen der Darstellungstyp angewandt werden kann. Die kombinierten Informationen bestehend aus Darstellungstyp und
 * Maßstabsgrenzen werden im Hinblick auf schnellen Zugriff von der DOTCollection auf zwei Arten verwaltet: als Liste und als Map.
 *
 * @author Kappich Systemberatung
 */
@SuppressWarnings("serial")
public class DOTCollection extends AbstractTableModel implements Cloneable {

    private static final String LOWER_BOUND = "LOWER_BOUND";
    private static final String UPPER_BOUND = "UPPER_BOUND";
    private static final String DOT_NAME = "DOT_NAME";
    private static final Debug _debug = Debug.getLogger();
    private static final String[] columnNames = {"Darstellungstyp", "Von 1:", "Bis 1:"};
    private final List<DOTCollectionItem> _dotList;
    private final TreeMap<Interval<Integer>, DisplayObjectType> _dotTreeMap;

    /**
     * Legt ein leeres Objekt an.
     */
    public DOTCollection() {
        _dotList = new ArrayList<>();
        _dotTreeMap = new TreeMap<>();
    }

    /**
     * Fügt den Darstellungstyp für die übergebenen Maßstabsgrenzen hinzu.
     *
     * @param type       der Darstellungstyp
     * @param lowerScale die unter Grenze
     * @param upperScale die obere Grenze
     */
    public void addDisplayObjectType(DisplayObjectType type, int lowerScale, int upperScale) {
        if (type == null) {
            throw new IllegalArgumentException("DOTCollection.addDisplayObjectType: type darf nicht null sein.");
        }
        if (lowerScale < upperScale) {
            throw new IllegalArgumentException("DOTCollection.addDisplayObjectType: lowerScale darf nicht kleiner upperScale sein.");
        }
        DOTCollectionItem collectionItem = new DOTCollectionItem(type, lowerScale, upperScale);
        Interval<Integer> interval = new Interval<>(upperScale, lowerScale);
        if (_dotTreeMap.containsKey(interval)) {
            final DisplayObjectType dType = _dotTreeMap.get(interval);
            DOTCollectionItem dItem = new DOTCollectionItem(dType, lowerScale, upperScale);
            _dotList.remove(dItem);
        }
        _dotList.add(collectionItem);
        _dotTreeMap.put(interval, type);
        fireTableDataChanged();
    }

    /**
     * Entfernt den Darstellungstyp für die übergebenen Maßstabsgrenzen.
     *
     * @param type       der zu entfernende DisplayObjectType
     * @param lowerScale die untere Intervallgrenze
     * @param upperScale die obere Intervallgrenze
     */
    public void removeDisplayObjectType(DisplayObjectType type, int lowerScale, int upperScale) {
        DOTCollectionItem collectionItem = new DOTCollectionItem(type, lowerScale, upperScale);
        if (_dotList.contains(collectionItem)) {
            _dotList.remove(collectionItem);
            Interval<Integer> interval = new Interval<>(upperScale, lowerScale);
            _dotTreeMap.remove(interval);
            fireTableDataChanged();
        }
    }

    /**
     * Leert die DOTCollection vollständig.
     */
    public void clear() {
        _dotList.clear();
        _dotTreeMap.clear();
        fireTableDataChanged();
    }

    /**
     * Gibt {@code true} zurück, wenn die DOTCollection leer ist, {@code false} sonst.
     *
     * @return {@code true} genau dann, wenn die DOTCollection leer ist
     */
    public boolean isEmpty() {
        return _dotList.isEmpty();
    }

    /**
     * Erzeugt eine Kopie des aufrufenden Objekts
     *
     * @return die Kopie
     */
    @SuppressWarnings("MethodDoesntCallSuperMethod")
    @Override
    public Object clone() {
        DOTCollection theClone = new DOTCollection();
        for (DOTCollectionItem item : _dotList) {
            theClone.addDisplayObjectType(item.getDisplayObjectType(), item.getLowerScale(), item.getUpperScale());
        }
        return theClone;
    }

    /**
     * Gibt eine Kopie der DOTCollection zurück.
     *
     * @return eine Kopie
     */
    public DOTCollection getCopy() {
        return (DOTCollection) clone();
    }

    /**
     * Gibt einen Darstellungstypen für den mit scale angebenen Maßstabswert zurück, wenn ein solcher existiert, sonst {@code null}.
     *
     * @param scale ein Maßstabswert
     *
     * @return eine DisplayObjectType zum Maßstabswert oder {@code null}, wenn kein solcher existiert
     */
    @Nullable
    public DisplayObjectType getDisplayObjectType(int scale) {
        Interval<Integer> interval = new Interval<>(scale, scale);
        final Entry<Interval<Integer>, DisplayObjectType> floorEntry = _dotTreeMap.floorEntry(interval);
        if (floorEntry == null || !interval.intersects(floorEntry.getKey())) {
            final Entry<Interval<Integer>, DisplayObjectType> ceilingEntry = _dotTreeMap.ceilingEntry(interval);
            if (ceilingEntry == null || !interval.intersects(ceilingEntry.getKey())) {
                return null;
            } else {
                return ceilingEntry.getValue();
            }
        } else {
            return floorEntry.getValue();
        }
    }

    /**
     * Diese Methode berechnet eine {@code Map}, deren Schlüssel die {@code DisplayObjectTypes} der {@code DOTCollection} sind, und deren Werte die
     * jeweiligen Listen von {@code PrimitiveFormPropertyPair}-Objekten sind. Das Ergebnis wird nicht gecached.
     *
     * @return die oben beschriebene {@code Map}
     */
    public Map<DisplayObjectType, List<PrimitiveFormPropertyPair>> getPrimitiveFormPropertyPairs() {
        Map<DisplayObjectType, List<PrimitiveFormPropertyPair>> map = new HashMap<>();
        for (DisplayObjectType displayObjectType : this.values()) {
            List<PrimitiveFormPropertyPair> list = new ArrayList<>();
            final int length = displayObjectType.getDisplayObjectTypePlugin().getPrimitiveFormTypes().length;
            if (length == 0) {    // bei Linien und Flächen
                final List<Property> dynamicProperties = displayObjectType.getDynamicProperties(null);
                for (Property dynamicProperty : dynamicProperties) {
                    PrimitiveFormPropertyPair pfpPair = new PrimitiveFormPropertyPair(null, dynamicProperty);
                    list.add(pfpPair);
                }
            } else {    // bei Punkten
                final Set<String> primitiveFormNames = displayObjectType.getPrimitiveFormNames();
                for (String primitiveFormName : primitiveFormNames) {
                    final List<Property> dynamicProperties = displayObjectType.getDynamicProperties(primitiveFormName);
                    for (Property dynamicProperty : dynamicProperties) {
                        PrimitiveFormPropertyPair pfpPair = new PrimitiveFormPropertyPair(primitiveFormName, dynamicProperty);
                        list.add(pfpPair);
                    }
                }
            }
            if (!list.isEmpty()) {
                map.put(displayObjectType, list);
            }
        }
        return map;
    }

    /**
     * Speichert die DOTCollection unter dem angebenen Knoten ab.
     *
     * @param prefs der Knoten, unter dem gespeichert werden soll
     */
    public void putPreferences(Preferences prefs) {
        int i = 0;
        for (final Entry<Interval<Integer>, DisplayObjectType> intervalDisplayObjectTypeEntry : _dotTreeMap.entrySet()) {
            Preferences intervalPrefs = prefs.node("interval" + i);
            intervalPrefs.putInt(LOWER_BOUND, intervalDisplayObjectTypeEntry.getKey().getLowerBound());
            intervalPrefs.putInt(UPPER_BOUND, intervalDisplayObjectTypeEntry.getKey().getUpperBound());
            intervalPrefs.put(DOT_NAME, intervalDisplayObjectTypeEntry.getValue().getName());
            i++;
        }
    }

    /**
     * Initialisiert die DOTCollection aus dem angebenen Knoten.
     *
     * @param prefs      der Knoten, unter dem die Initialisierung beginnt
     * @param dotManager die Darstellungstypenverwaltung
     *
     * @return gibt {@code true} zurück, wenn die Initialisierung erfolgreich war, und {@code false}, falls nicht
     */
    public boolean initializeFromPreferences(Preferences prefs, DOTManager dotManager) {
        String[] intervals;
        try {
            intervals = prefs.childrenNames();
        } catch (BackingStoreException e) {
            _debug.error("Ein benutzer-definierter Layer kann nicht initialisiert werden, " + "BackingStoreException: " + e.toString());
            PreferencesDeleter pd = new PreferencesDeleter("Die Darstellungstypen eines Layers können nicht geladen werden.", prefs);
            pd.run();
            return false;
        }
        for (String interval : intervals) {
            if (!interval.startsWith("interval")) {    // im Moment ein Fehler, aber wer weiß was noch kommt
                continue;
            }
            Preferences intervalPrefs = prefs.node(interval);
            Interval<Integer> iv =
                new Interval<>(intervalPrefs.getInt(LOWER_BOUND, Integer.MIN_VALUE), intervalPrefs.getInt(UPPER_BOUND, Integer.MAX_VALUE));
            String dotName = intervalPrefs.get(DOT_NAME, "");
            final DisplayObjectType displayObjectType = dotManager.getDisplayObjectType(dotName);
            if (displayObjectType != null) {
                _dotTreeMap.put(iv, displayObjectType);
            }
        }
        for (final Entry<Interval<Integer>, DisplayObjectType> intervalDisplayObjectTypeEntry : _dotTreeMap.entrySet()) {
            DisplayObjectType displayObjectType = intervalDisplayObjectTypeEntry.getValue();
            DOTCollectionItem collectionItem = new DOTCollectionItem(displayObjectType, intervalDisplayObjectTypeEntry.getKey().getUpperBound(),
                                                                     intervalDisplayObjectTypeEntry.getKey().getLowerBound());
            _dotList.add(collectionItem);
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        for (final Entry<Interval<Integer>, DisplayObjectType> intervalDisplayObjectTypeEntry : _dotTreeMap.entrySet()) {
            sb.append("[").append(intervalDisplayObjectTypeEntry.getKey().toString()).append(", ")
                .append(intervalDisplayObjectTypeEntry.getValue().toString()).append("] ");
        }
        return sb.toString();
    }

    /**
     * Gibt eine Read-Only-Ansicht aller Darstellungstypen der DOTCollection zurück.
     *
     * @return alle auftretenden DisplayObjectTypes
     */
    public Collection<DisplayObjectType> values() {
        return Collections.unmodifiableCollection(_dotTreeMap.values());
    }

    /*
     * Gehört zur Implementation von TableModel.
     */
    @Override
    public int getColumnCount() {
        return columnNames.length;
    }

    /*
     * Gehört zur Implementation von TableModel.
     */
    @Override
    public int getRowCount() {
        return _dotTreeMap.size();
    }

    /*
     * Gehört zur Implementation von TableModel.
     */
    @Override
    public String getColumnName(int columnIndex) {
        return columnNames[columnIndex];
    }

    /*
     * Gehört zur Implementation von TableModel.
     */
    @Override
    @Nullable
    public Object getValueAt(int rowIndex, int columnIndex) {
        if (columnIndex == 0) {
            return _dotList.get(rowIndex).getDisplayObjectType().getName();
        } else if (columnIndex == 1) {
            return _dotList.get(rowIndex).getLowerScale();
        } else if (columnIndex == 2) {
            return _dotList.get(rowIndex).getUpperScale();
        }
        return null;
    }

    /**
     * Gibt eine Menge mit den Namen aller in den Darstellungstypen der DOTCollection verwendeten Farben zurück.
     *
     * @return eine Menge mit den Namen aller in den Darstellungstypen der DOTCollection verwendeten Farben
     */
    public Set<String> getUsedColors() {
        Set<String> usedColors = new HashSet<>();
        for (DOTCollectionItem dotItem : _dotList) {
            usedColors.addAll(dotItem.getUsedColors());
        }
        return usedColors;
    }

    /**
     * Gibt {@code true} zurück, wenn der Darstellungstyp mit dem übergebenen Namen in der DOTCollection auftritt.
     *
     * @param displayObjectTypeName der Name eines DisplayObjectTypes
     *
     * @return {@code true} genau dann, wenn der Darstellungstyp in der DOTCollection auftritt
     */
    public boolean displayObjectTypeIsUsed(final String displayObjectTypeName) {
        if (displayObjectTypeName == null) {
            return false;
        }
        for (DisplayObjectType displayObjectType : _dotTreeMap.values()) {
            if (displayObjectTypeName.equals(displayObjectType.getName())) {
                return true;
            }
        }
        return false;
    }

    /*
     * Ausgabe auf dem Standardausgabekanal.
     */
    @SuppressWarnings({"unused", "UseOfSystemOutOrSystemErr"})
    private void printAll() {
        System.out.println("Größe der Liste: " + _dotList.size());
        System.out.println("Größe der Map: " + _dotTreeMap.size());
        System.out.println("Listeneinträge");
        for (DOTCollectionItem item : _dotList) {
            System.out.println(item.toString());
        }
        System.out.println("Mapeinträge");
        for (final Entry<Interval<Integer>, DisplayObjectType> intervalDisplayObjectTypeEntry : _dotTreeMap.entrySet()) {
            System.out.println("Lower bound: " + intervalDisplayObjectTypeEntry.getKey().getLowerBound() + ", upper bound: " +
                               intervalDisplayObjectTypeEntry.getKey().getUpperBound() + ", DisplayObjectType: " +
                               intervalDisplayObjectTypeEntry.getValue().getName());
        }
    }

    /*
     * Interne Übrerprüfung mit Ausgabe der Ergebnisse auf dem Standardausgabekanal.
     */
    @SuppressWarnings({"unused", "UseOfSystemOutOrSystemErr"})
    private void checkAll() {
        if (_dotList.size() != _dotTreeMap.size()) {
            System.out.println("Fehler: die Liste hat " + _dotList.size() + " Einträge, die Map " + _dotTreeMap.size());
        }
        for (DOTCollectionItem item : _dotList) {
            Interval<Integer> interval = new Interval<>(item.getUpperScale(), item.getLowerScale());
            if (!_dotTreeMap.containsKey(interval)) {
                System.out.println(
                    "Fehler: die Liste hat einen Eintrag für das Interval [" + interval.getLowerBound() + ", " + interval.getUpperBound() +
                    "], die TreeMap aber nicht!");
            } else {
                if (!Objects.equals(item.getDisplayObjectType().getName(), _dotTreeMap.get(interval).getName())) {
                    System.out.println("Fehler: zum Interval [" + interval.getLowerBound() + ", " + interval.getUpperBound() +
                                       "] sind die DisplayObjectTypes verschieden. In der Liste: " + item.getDisplayObjectType().getName() +
                                       ", in der TreeMap: " + _dotTreeMap.get(interval).getName());
                }
            }
        }
        for (final Entry<Interval<Integer>, DisplayObjectType> entry : _dotTreeMap.entrySet()) {
            DOTCollectionItem item = new DOTCollectionItem(entry.getValue(), entry.getKey().getUpperBound(), entry.getKey().getLowerBound());
            if (!_dotList.contains(item)) {
                System.out.println("Fehler: die TreeMap enthält einen Eintrag zu " + item.toString() + ", der in der Liste nicht auftritt!");
            }
        }
    }

    public boolean areIntervalsDisjoint() {
        Set<Interval<Integer>> keySet = _dotTreeMap.keySet();
        Iterator<Interval<Integer>> iterator = keySet.iterator();
        Interval<Integer> interval1;
        if (iterator.hasNext()) {
            interval1 = iterator.next();
        } else {
            return true;
        }
        while (iterator.hasNext()) {
            Interval<Integer> interval2 = iterator.next();
            if (interval1.getUpperBound() > interval2.getLowerBound()) {
                return false;
            }
            interval1 = interval2;
        }
        return true;
    }

    /**
     * Ein {@code DOTCollectionItem} verkapselt die Information der {@code DOTCollection} bestehend aus einem Darstellungstypen und den
     * Maßstabsgrenzen für die Listenverwaltung.
     */
    private static class DOTCollectionItem {
        private final DisplayObjectType _displayObjectType;
        private final int _lowerScale;
        private final int _upperScale;

        /**
         * Der Konstruktor.
         *
         * @param displayObjectType ein DisplayObjectType
         * @param lowerScale        die untere Intervallgrenze
         * @param upperScale        die obere Intervallgrenze
         */
        public DOTCollectionItem(DisplayObjectType displayObjectType, int lowerScale, int upperScale) {
            super();
            if (displayObjectType == null) {
                throw new IllegalArgumentException("Ein DOTCollectionItem kann nicht ohne DisplayObjectType gebildet werden.");
            }
            _displayObjectType = displayObjectType;
            _lowerScale = lowerScale;
            _upperScale = upperScale;
        }

        /**
         * Der Getter für den DisplayObjectType.
         *
         * @return der DisplayObjectType
         */
        public DisplayObjectType getDisplayObjectType() {
            return _displayObjectType;
        }

        /**
         * Der Getter für die untere Intervallgrenze.
         *
         * @return die untere Intervallgrenze
         */
        public int getLowerScale() {
            return _lowerScale;
        }

        /**
         * Der Getter für die obere Intervallgrenze.
         *
         * @return die obere Intervallgrenze
         */
        public int getUpperScale() {
            return _upperScale;
        }

        @Override
        public boolean equals(Object o) {
	        if (!(o instanceof DOTCollectionItem d)) {
                return false;
            }
            return Objects.equals(getDisplayObjectType().getName(), d.getDisplayObjectType().getName()) && getLowerScale() == d.getLowerScale() &&
                   (getUpperScale() == d.getUpperScale());
        }

        @Override
        public int hashCode() {
            return getDisplayObjectType().getName().hashCode() + getLowerScale() + getUpperScale();
        }

        @Override
        public String toString() {
            return "[DisplayObjectType: " + _displayObjectType.getName() + ", lower bound: " + _lowerScale + ", upper bound: " + _upperScale + "]";
        }

        public Set<String> getUsedColors() {
            return _displayObjectType.getUsedColors();
        }

    }
}
