/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */
package de.kappich.pat.gnd.displayObjectToolkit;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.documentation.GndHelp;
import de.kappich.pat.gnd.pluginInterfaces.DOTDefinitionPanel;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectTypePlugin;
import de.kappich.pat.gnd.pointPlugin.DOTPoint;
import de.kappich.pat.gnd.pointPlugin.DOTPoint.PrimitiveForm;
import de.kappich.pat.gnd.pointPlugin.DOTPoint.PrimitiveFormType;
import de.kappich.pat.gnd.pointPlugin.DOTPointPainter;
import de.kappich.pat.gnd.properties.Property;
import de.kappich.pat.gnd.utils.SpringUtilities;
import de.kappich.pat.gnd.utils.view.GndFrame;
import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.geom.Point2D;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ComboBoxModel;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.JTextField;
import javax.swing.SpinnerModel;
import javax.swing.SpinnerNumberModel;
import javax.swing.SpringLayout;
import javax.swing.WindowConstants;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

/**
 * Der äußere Rahmen aller Dialoge zur Darstellungstypendefinition. Dieser Dialog liefert ein Fenster,
 * in dem schon die wesentlichen Teile zur Darstellungstypdefinition enthalten sind. Die
 * plugin-spezifischen Panels werden von den Implementationen von {@link DOTDefinitionPanel} geliefert.
 *
 * @author Kappich Systemberatung
 * @version $Revision$
 */
@SuppressWarnings({"serial", "OverlyLongMethod"})
public class DOTDefinitionDialog extends GndFrame {

    private final DOTManagerDialog _dotManagerDialog;
    private final ClientDavInterface _connection;
    private final DOTManager _dotManager;
    private DisplayObjectType _displayObjectType;
    private DisplayObjectType _scratchDisplayObjectType;
    // Eine tiefe Kopie des _displayObjectType, in der alle für die Dialoge
    // notwendigen Änderungen eingetragen werden, z.B. die Statisch/Dynamisch-
    // Informationen oder die TableModels.
    // Würde man dies im Original machen, so wäre der aktuelle Stand immer
    // auch der nach dem Drücken des (Gesamt-)Speichern-Knopfs.
    // Das (Gesamt-)Speichern wird (noch) nicht auf Basis dieser Kopie,
    // sondern auf Basis der sichtbaren Einträge u.a. in den Sub-Componenten
    // durchgeführt.

    private boolean _editable;
    private boolean _reviseOnly;

    private final JTextField _nameTextField = new JTextField();
    private final JTextField _infoTextField = new JTextField();
    private final JComboBox<Object> _primitiveFormComboBox = new JComboBox<>();
    private final JTextField _primitiveFormTypeTextField = new JTextField();
    private final JTextField _primitiveFormInfoTextField = new JTextField();
    private final JTextField _positionX = new JTextField();
    private final JTextField _positionY = new JTextField();
    private final JButton _editPrimitiveFormButton = new JButton("Bearbeiten");
    private final JButton _copyPrimitiveFormButton = new JButton("Kopieren");
    private final JButton _newPrimitiveFormButton = new JButton("Neu");
    private final JButton _deletePrimitiveFormButton = new JButton("Löschen");
    private final JComboBox<Property> _propertyComboBox = new JComboBox<>();
    private final JCheckBox _staticCheckBox = new JCheckBox("");
    private boolean _somethingChanged = false;
    private final JButton _saveButton = new JButton("Darstellungstyp speichern");
    private DOTDefinitionPanel _dotDefinitionPanel;
    private JPanel _centerPanel = null;
    //private final Dimension _frameSize = new Dimension(950, 600);
    private static final Debug _debug = Debug.getLogger();

    // Und nun alles für die speziellen Informationen:

    private final JPanel _specialInformationDefinitionPanel = new JPanel();
    private JPanel _specialInformationPanel = null;
    private final JPanel _specialInformationRectangle = new JPanel();
    private final JTextField _siHeight = new JTextField();
    private final JTextField _siWidth = new JTextField();
    private final JPanel _specialInformationCircle = new JPanel();
    private final JTextField _siRadius = new JTextField();
    private final JPanel _specialInformationSemicircle = new JPanel();
    private final JTextField _siSemiRadius = new JTextField();
    private final JTextField _siOrientation = new JTextField();
    private final JPanel _specialInformationTextdarstellung = new JPanel();
    private final JPanel _specialInformationPoint = new JPanel();

    /**
     * Konstruktor für ein funktionsfähiges Objekt.
     *
     * @param dotManagerDialog  ein Dialog der Darstellungstypen-Verwaltung
     * @param connection        die Datenverteiler-Verbindung
     * @param dotManager        die Darstellungstypen-Verwaltung
     * @param displayObjectType ein Darstellungstyp
     * @param editable          ist der Darstellungstyp veränderbar
     * @param reviseOnly        ist die Identität des Darstellungstyps unveränderlich
     * @param title             die anzuzeigende Titelzeile
     */
    public DOTDefinitionDialog(
        final DOTManagerDialog dotManagerDialog,
        final ClientDavInterface connection,
        final DOTManager dotManager,
        final DisplayObjectType displayObjectType,
        final boolean editable,
        final boolean reviseOnly,
        final String title) {
        super("DOTDefinitionDialog", title);
        _dotManagerDialog = dotManagerDialog;
        _connection = connection;
        _dotManager = dotManager;
        _displayObjectType = displayObjectType;
        if(_dotManager == null || _displayObjectType == null) {
            if(dotManager == null) {
                _debug.warning("Ein DOTDefinitionDialog kann nicht mit einem Null-DOTManager konstruiert werden.");
            }
            if(displayObjectType == null) {
                _debug.warning("Ein DOTDefinitionDialog kann nicht mit einem Null-Darstellungstyp konstruiert werden.");
            }
            dispose();
        }
        else {
            _scratchDisplayObjectType = _displayObjectType.getCopy(null);

            _editable = editable;
            _reviseOnly = reviseOnly;

            _staticCheckBox.setEnabled(_editable);

            addToolTips();    // Dies muss schon recht frühzeitig geschehen, weil die hier festgelegten Tooltipps auch
            // von dem Dialog, der neue Grundfiguren anlegt (PrimitiveFormDialog) verwendet werden.

            setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
            setLayout(new BorderLayout());

            Dimension labelSize = new Dimension(100, 20);

            final DisplayObjectTypePlugin displayObjectTypePlugin = _displayObjectType.getDisplayObjectTypePlugin();
            _dotDefinitionPanel = displayObjectTypePlugin.getPanel(this);

            // Oberer Teil mit Name, Info und ggbnflls dem AdditionalCgharacteristicsPanel

            JLabel nameLabel = new JLabel("Name: ");
            nameLabel.setPreferredSize(labelSize);
            _nameTextField.setText(_displayObjectType.getName());
            _nameTextField.setEditable(!_reviseOnly && _editable);
            _nameTextField.getDocument().addDocumentListener(new DocumentListener() {
                @Override
                public void changedUpdate(DocumentEvent e) {
                    update();
                }

                @Override
                public void removeUpdate(DocumentEvent e) {
                    update();
                }

                @Override
                public void insertUpdate(DocumentEvent e) {
                    update();
                }

                private void update() {
                    _scratchDisplayObjectType.setName(_nameTextField.getText());
                    _somethingChanged = true;
                }
            });

            JLabel infoLabel = new JLabel("Info: ");
            infoLabel.setPreferredSize(labelSize);
            _infoTextField.setText(_displayObjectType.getInfo());
            _infoTextField.setEditable(_editable);
            _infoTextField.getDocument().addDocumentListener(new DocumentListener() {
                @Override
                public void changedUpdate(DocumentEvent e) {
                    update();
                }

                @Override
                public void removeUpdate(DocumentEvent e) {
                    update();
                }

                @Override
                public void insertUpdate(DocumentEvent e) {
                    update();
                }

                private void update() {
                    _scratchDisplayObjectType.setInfo(_infoTextField.getText());
                    _somethingChanged = true;
                }
            });

            JPanel upperPanelUpperUpperPart = new JPanel();
            upperPanelUpperUpperPart.setLayout(new SpringLayout());
            upperPanelUpperUpperPart.add(nameLabel);
            upperPanelUpperUpperPart.add(_nameTextField);
            upperPanelUpperUpperPart.add(infoLabel);
            upperPanelUpperUpperPart.add(_infoTextField);
            upperPanelUpperUpperPart.setBorder(BorderFactory.createEmptyBorder(5, 10, 0, 10));
            SpringUtilities.makeCompactGrid(upperPanelUpperUpperPart, 2, 5, 5);

            JPanel upperPanelUpperLowerPart = _dotDefinitionPanel.getAdditionalCharacteristicsPanel(
                _displayObjectType);

            JPanel upperPanelLowerUpperPart = getPrimitiveFormPanel();

            JPanel upperPanelLowerLowerPart = getComboBoxPanel();

            JPanel upperPanelUpperPart = new JPanel();
            upperPanelUpperPart.setLayout(new BoxLayout(upperPanelUpperPart, BoxLayout.Y_AXIS));
            upperPanelUpperPart.setBorder(BorderFactory.createTitledBorder("Übergreifend"));
            upperPanelUpperPart.add(upperPanelUpperUpperPart);
            if(upperPanelUpperLowerPart != null) {
                upperPanelUpperPart.add(upperPanelUpperLowerPart);
            }

            JPanel upperPanelLowerPart = new JPanel();
            upperPanelLowerPart.setLayout(new BoxLayout(upperPanelLowerPart, BoxLayout.Y_AXIS));
            if(_displayObjectType.getDisplayObjectTypePlugin().getPrimitiveFormTypes().length > 0) {
                upperPanelLowerPart.add(upperPanelLowerUpperPart);
            }
            upperPanelLowerPart.add(upperPanelLowerLowerPart);

            JPanel upperPanel = new JPanel();
            upperPanel.setLayout(new BoxLayout(upperPanel, BoxLayout.Y_AXIS));
            upperPanel.add(upperPanelUpperPart);
            upperPanel.add(upperPanelLowerPart);

            add(upperPanel, BorderLayout.NORTH);

            // Der Mittelteil

            _centerPanel = _dotDefinitionPanel.getDOTItemDefinitionPanel();
            if(_centerPanel != null) {
                add(_centerPanel, BorderLayout.CENTER);
            }

            // Unterer Teil mit Buttons

            ActionListener saveActionListener = e -> {
                String name = _nameTextField.getText();
                if(name.length() > 40) { // Präferenzen erlauben 80 Zeichen als Node-Name, doch damit der Name gut in die Legende passt, ist hier 40 gewählt.
                    JOptionPane.showMessageDialog(
                        this.getFrame(),
                        "Bitte verwenden Sie einen Namen mit höchstens 40 Zeichen!",
                        "Fehler",
                        JOptionPane.ERROR_MESSAGE
                    );
                    return;
                }
                if(_dotDefinitionPanel.saveDisplayObjectType()) {
                    _nameTextField.setEditable(false);
                    _reviseOnly = true;
                }
            };
            _saveButton.addActionListener(saveActionListener);

            JButton cancelButton = new JButton("Dialog schließen");

            ActionListener cancelActionListener = e -> {
                if(_somethingChanged && _saveButton.isEnabled()) {
                    Object[] options = {"Ja", "Nein"};
                    int n = JOptionPane.showOptionDialog(
                        this.getFrame(),
                        "Wollen Sie den Dialog wirklich ohne zu Speichern beenden?",
                        "Es wurden Änderungen vorgenommen.",
                        JOptionPane.YES_NO_CANCEL_OPTION,
                        JOptionPane.QUESTION_MESSAGE,
                        null,
                        options,
                        options[1]
                    );
                    if((n == 1) || (n == -1)) {
                        return;
                    }
                }
                if(_dotManagerDialog != null) {
                    _dotManagerDialog.dialogDisposed(DOTDefinitionDialog.this);
                }
                storePreferenceBounds();
                dispose();
            };
            cancelButton.addActionListener(cancelActionListener);

            class MyWindowListener extends WindowAdapter {

                private MyWindowListener() {
                    super();
                }

                @Override
                public void windowClosing(WindowEvent e) {
                    if(_somethingChanged && _saveButton.isEnabled()) {
                        Object[] options = {"Ja", "Nein"};
                        int n = JOptionPane.showOptionDialog(
                            DOTDefinitionDialog.this.getFrame(),
                            "Wollen Sie den Dialog wirklich ohne zu Speichern beenden?",
                            "Es wurden Änderungen vorgenommen.",
                            JOptionPane.YES_NO_CANCEL_OPTION,
                            JOptionPane.QUESTION_MESSAGE,
                            null,
                            options,
                            options[1]
                        );
                        if(n == 0) {
                            setVisible(false);
                            dispose();
                        }
                    }
                    else {
                        setVisible(false);
                        dispose();
                    }
                }
            }

            MyWindowListener myWindowListener = new MyWindowListener();
            this.addWindowListener(myWindowListener);

            JButton helpButton = new JButton("Hilfe");
            ActionListener actionListenerHelp = e -> GndHelp.openHelp("#dotDefinitionDialog");
            helpButton.addActionListener(actionListenerHelp);

            JPanel buttonsPanel = new JPanel();
            buttonsPanel.setLayout(new SpringLayout());

            buttonsPanel.add(_saveButton);
            buttonsPanel.add(cancelButton);
            buttonsPanel.add(helpButton);

            buttonsPanel.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));
            SpringUtilities.makeCompactGrid(buttonsPanel, 3, 20, 5);
            add(buttonsPanel, BorderLayout.SOUTH);

            setEditable(_editable);

            if(displayObjectType instanceof DOTPoint) {
                setPositionAndSize(650, 450, 660, 0, true, 0, 0);
            }
            else {
                setPositionAndSize(550, 375, 660, 0, true, 0, 0);
            }
            if(!hasPreferences()) {
                setLocationRelativeTo(_dotManagerDialog.getComponent());
            }

            setVisible(true);
        }
    }

    /**
     * Setzt den Darstellungstypen des Dialogs. Mit der boolschen Variable wird angegeben, ob der
     * Dialog veränderbar ist oder nicht.
     *
     * @param displayObjectType ein Darstellungstyp
     * @param editable          ist der Darstellungstyp veränderbar
     */
    @SuppressWarnings("SameParameterValue")
    public void setDisplayObjectType(DisplayObjectType displayObjectType, boolean editable) {
        if(_dotManagerDialog != null) {
            _dotManagerDialog.dialogChanged(_displayObjectType, displayObjectType);
        }
        _displayObjectType = displayObjectType;
        _scratchDisplayObjectType = _displayObjectType.getCopy(null);
        _editable = editable;
        _nameTextField.setText(_scratchDisplayObjectType.getName());
        _infoTextField.setText(_scratchDisplayObjectType.getInfo());
        _somethingChanged = false;
    }

    /**
     * Gibt den Darstellungstypen, mit dem der Dialog initialisiert wurde oder der mit
     * setDisplayObjectType() zuletzt gesetzt wurde, zurück.
     *
     * @return der Darstellungstyp
     */
    public DisplayObjectType getDisplayObjectType() {
        return _displayObjectType;
    }


    /**
     * Gibt den Darstellungstypen des Dialogs zurück, der auch alle vom Benutzer seit der
     * Initialisierung des Dialogs bzw. dem letzten Aufruf von setDisplayObjectType() gemachten
     * Änderungen enthält.
     *
     * @return der vom Benutzer veränderbare Darstellungstyp
     */
    public DisplayObjectType getScratchDisplayObjectType() {
        return _scratchDisplayObjectType;
    }

    /**
     * Legt fest, ob der angezeigte Darstellungstyp veränderbar ist oder nicht.
     *
     * @param editable der neue Wert
     */
    public final void setEditable(boolean editable) {
        _editable = editable;
        _saveButton.setEnabled(_editable);
    }

    @SuppressWarnings("SameParameterValue")
    private static String getAsString(final Double d, final int precision) {
        boolean negative = (d < 0.);
        final Double v;
        if(negative) {
            v = -d * precision + 0.000000001;
        }
        else {
            v = d * precision + 0.000000001;
        }
        Integer i = v.intValue();
        Integer j = i / precision;
        Integer k = i % precision;
        String s;
        if(negative) {
            s = '-' + j.toString() + "," + k.toString();
        }
        else {
            s = j.toString() + "," + k.toString();
        }
        return s;
    }

    private JPanel getPrimitiveFormPanel() {
        _primitiveFormTypeTextField.setEditable(false);
        _primitiveFormInfoTextField.setEditable(false);
        _positionX.setEditable(false);
        _positionX.setPreferredSize(new Dimension(50, 30));
        _positionY.setEditable(false);
        _positionY.setPreferredSize(new Dimension(50, 30));

        final ItemListener primitiveFormListener = e -> {
            if(e.getStateChange() == ItemEvent.SELECTED) {
                String primitiveFormName = (String) _primitiveFormComboBox.getSelectedItem();
                _primitiveFormTypeTextField.setText(_scratchDisplayObjectType.getPrimitiveFormType(primitiveFormName));
                _primitiveFormInfoTextField.setText(_scratchDisplayObjectType.getPrimitiveFormInfo(primitiveFormName));
                DOTPoint dotPoint;
                try {
                    dotPoint = (DOTPoint) _scratchDisplayObjectType; // Böse! Muss sauberer werden.
                }
                catch(ClassCastException ignore) {
                    _debug.error("DOTDefinitionPanel: falsches Plugin.");
                    return;
                }
                PrimitiveForm primitiveForm = dotPoint.getPrimitiveForm(primitiveFormName);
                final Double x = primitiveForm.getTranslation().getX();
                _positionX.setText(getAsString(x, 10));
                final Double y = primitiveForm.getTranslation().getY();
                _positionY.setText(getAsString(y, 10));
                final DisplayObjectTypePlugin pluginObject = getPluginObject();
                updateSpecialInformationPanel();
                _editPrimitiveFormButton.setEnabled(_editable);
                _copyPrimitiveFormButton.setEnabled(_editable);
                _deletePrimitiveFormButton.setEnabled(_editable);
                _staticCheckBox.setEnabled(_editable);
                updatePropertyComboBox(pluginObject);
                updateStaticCheckBox();
                updateCenterPanel();
            }
        };
        _primitiveFormComboBox.addItemListener(primitiveFormListener);

        updatePrimitiveFormData();

        _primitiveFormComboBox.setBorder(BorderFactory.createTitledBorder("Name"));
        _primitiveFormTypeTextField.setBorder(BorderFactory.createTitledBorder("Typ"));
        _primitiveFormInfoTextField.setBorder(BorderFactory.createTitledBorder("Info"));

        JPanel postionPanel = getPositionPanel();

        // Nun die obere Zeile:
        JPanel theUpperPanel = new JPanel();
        theUpperPanel.setLayout(new SpringLayout());
        theUpperPanel.add(_primitiveFormComboBox);
        theUpperPanel.add(_primitiveFormTypeTextField);
        theUpperPanel.add(_primitiveFormInfoTextField);
        theUpperPanel.add(postionPanel);
        SpringUtilities.makeCompactGrid(theUpperPanel, 4, 0, 5);

        initSpecialInformationPanel();
        updateSpecialInformationPanel();
        JPanel buttonsPanel = getButtonsPanel();

        // Nun die untere Zeile:
        JPanel theLowerPanel = new JPanel();
        theLowerPanel.setLayout(new SpringLayout());
        _specialInformationDefinitionPanel.setBorder(BorderFactory.createTitledBorder("Definition"));
        if(_specialInformationPanel != null) {    // das Panel kann null sein
            _specialInformationDefinitionPanel.add(_specialInformationPanel);
        }
        theLowerPanel.add(_specialInformationDefinitionPanel);
        theLowerPanel.add(buttonsPanel);
        SpringUtilities.makeCompactGrid(theLowerPanel, 2, 0, 5);

        JPanel thePanel = new JPanel();
        thePanel.setLayout(new BoxLayout(thePanel, BoxLayout.Y_AXIS));
        thePanel.setBorder(BorderFactory.createTitledBorder("Grundfigur"));
        thePanel.add(theUpperPanel);
        thePanel.add(theLowerPanel);

        addEditCopyNewDeleteListeners();
        return thePanel;
    }

    private JPanel getPositionPanel() {
        JPanel positionPanel = new JPanel();
        JLabel xLabel = new JLabel("x: ");
        JLabel yLabel = new JLabel("y: ");
        positionPanel.add(xLabel);
        positionPanel.add(_positionX);
        positionPanel.add(yLabel);
        positionPanel.add(_positionY);
        positionPanel.setBorder(BorderFactory.createTitledBorder("Position"));
        return positionPanel;
    }

    private void initSpecialInformationPanel() {
        // Für ein Recteck
        JLabel heightLabel = new JLabel("Höhe: ");
        JLabel widthLabel = new JLabel("Breite: ");

        _specialInformationRectangle.setLayout(new SpringLayout());
        _specialInformationRectangle.add(heightLabel);
        _siHeight.setEditable(false);
        _siHeight.setPreferredSize(new Dimension(60, 30));
        _specialInformationRectangle.add(_siHeight);
        _specialInformationRectangle.add(widthLabel);
        _siWidth.setEditable(false);
        _siWidth.setPreferredSize(new Dimension(60, 30));
        _specialInformationRectangle.add(_siWidth);
        SpringUtilities.makeCompactGrid(_specialInformationRectangle, 4, 5, 5);

        // Für einen Kreis
        JLabel radiusLabel = new JLabel("Radius: ");

        _specialInformationCircle.setLayout(new SpringLayout());
        _specialInformationCircle.add(radiusLabel);
        _siRadius.setEditable(false);
        _siRadius.setPreferredSize(new Dimension(60, 30));
        _specialInformationCircle.add(_siRadius);
        SpringUtilities.makeCompactGrid(_specialInformationCircle, 2, 5, 5);

        // Für einen Halbkreis
        JLabel semiRadiusLabel = new JLabel("Radius: ");
        JLabel orientationLabel = new JLabel("Orientierung: ");

        _specialInformationSemicircle.setLayout(new SpringLayout());
        _specialInformationSemicircle.add(semiRadiusLabel);
        _siSemiRadius.setEditable(false);
        _siSemiRadius.setPreferredSize(new Dimension(60, 30));
        _specialInformationSemicircle.add(_siSemiRadius);
        _specialInformationSemicircle.add(orientationLabel);
        _siOrientation.setEditable(false);
        _siOrientation.setPreferredSize(new Dimension(200, 30));
        _specialInformationSemicircle.add(_siOrientation);
        SpringUtilities.makeCompactGrid(_specialInformationSemicircle, 4, 5, 5);

        // Für einen Text: im Moment nichts

        // Für einen Punkt: im Moment nichts
    }

    private JPanel getButtonsPanel() {
        JPanel buttonsPanel = new JPanel();
        boolean enableAllOperations;
        final DisplayObjectTypePlugin plugin = _scratchDisplayObjectType.getDisplayObjectTypePlugin();
        if(plugin.getName().equals("Punkt")) {
            enableAllOperations = (_primitiveFormComboBox.getModel().getSize() != 0) && _editable;
        }
        else {
            enableAllOperations = _editable;
        }
        _editPrimitiveFormButton.setEnabled(enableAllOperations);
        _copyPrimitiveFormButton.setEnabled(enableAllOperations);
        _newPrimitiveFormButton.setEnabled(_editable);
        _deletePrimitiveFormButton.setEnabled(enableAllOperations);
        _staticCheckBox.setEnabled(enableAllOperations);
        buttonsPanel.add(_editPrimitiveFormButton);
        buttonsPanel.add(_copyPrimitiveFormButton);
        buttonsPanel.add(Box.createRigidArea(new Dimension(20, 20)));
        buttonsPanel.add(_newPrimitiveFormButton);
        buttonsPanel.add(_deletePrimitiveFormButton);
        buttonsPanel.setBorder(BorderFactory.createTitledBorder("Operationen"));
        return buttonsPanel;
    }

    private void addEditCopyNewDeleteListeners() {

        class PrimitiveFormDialog {
            private final Dimension _dialogSize = new Dimension(680, 240);
            private final JDialog _newPrimitiveFormDialog = new JDialog(DOTDefinitionDialog.this.getFrame(), true);
            private final JTextField _newPrimitiveFormNameTextField = new JTextField();
            private final JComboBox<String> _newPrimitiveFormTypeComboBox = new JComboBox<>();
            private final JTextField _infoOfPrimitiveFormNameTextField = new JTextField();
            private final JSpinner _positionXSpinner = new JSpinner(new SpinnerNumberModel(0., -10000.0, 10000., 0.1));
            private final JSpinner _positionYSpinner = new JSpinner(new SpinnerNumberModel(0., -10000.0, 10000., 0.1));
            private final JButton _savePrimitiveFormButton = new JButton("Speichern");
            private final JButton _leaveButton = new JButton("Dialog beenden");

            // Und nun alles für das variable Panel:
            private final JPanel _pfdDefinitionPanel = new JPanel();
            private JPanel _pfdSpecialInformationPanel = new JPanel();
            private final JPanel _pfdSpecialInformationRectangle = new JPanel();
            private final JSpinner _pfdSiHeight = new JSpinner(new SpinnerNumberModel(5., 0.0, 10000., 0.1));
            private final JSpinner _pfdSiWidth = new JSpinner(new SpinnerNumberModel(5., 0.0, 10000., 0.1));
            private final JPanel _pfdSpecialInformationCircle = new JPanel();
            private final JSpinner _pfdSiRadius = new JSpinner(new SpinnerNumberModel(5., 0.0, 10000., 0.1));
            private final JPanel _pfdSpecialInformationSemicircle = new JPanel();
            private final JSpinner _pfdSiSemiRadius = new JSpinner(new SpinnerNumberModel(5., 0.0, 10000., 0.1));
            private final JComboBox<String> _pfdSiOrientation = new JComboBox<>();
            private final JPanel _pfdSpecialInformationText = new JPanel();
            private final JPanel _pfdSpecialInformationPoint = new JPanel();

            public String getName() {
                return _newPrimitiveFormNameTextField.getText();
            }

            public void setName(final String name) {
                _newPrimitiveFormNameTextField.setText(name);
            }

            public void disableNameField() {
                _newPrimitiveFormNameTextField.setEditable(false);
            }

            public String getType() {
                return (String) _newPrimitiveFormTypeComboBox.getSelectedItem();
            }

            public void setType(final String type) {
                _newPrimitiveFormTypeComboBox.setSelectedItem(type);
            }

            public void disableTypeField() {
                _newPrimitiveFormTypeComboBox.setEnabled(false);
            }

            public String getInfo() {
                return _infoOfPrimitiveFormNameTextField.getText();
            }

            public void setInfo(final String info) {
                _infoOfPrimitiveFormNameTextField.setText(info);
            }

            public Double getX() {
                return (Double) _positionXSpinner.getValue();
            }

            public void setX(final Double x) {
                _positionXSpinner.setValue(x);
            }

            public Double getY() {
                return (Double) _positionYSpinner.getValue();
            }

            public void setY(final Double y) {
                _positionYSpinner.setValue(y);
            }

            @SuppressWarnings("StatementWithEmptyBody")
            private Map<String, Object> getSpecificInformation() {
                String type = getType();
                Map<String, Object> specificInformation = new HashMap<>();
                if(type.equals(PrimitiveFormType.RECHTECK.getName())) {
                    specificInformation.put(PrimitiveForm.HEIGHT, _pfdSiHeight.getValue());
                    specificInformation.put(PrimitiveForm.WIDTH, _pfdSiWidth.getValue());
                }
                else if(type.equals(PrimitiveFormType.KREIS.getName())) {
                    specificInformation.put(PrimitiveForm.RADIUS, _pfdSiRadius.getValue());
                }
                else if(type.equals(PrimitiveFormType.HALBKREIS.getName())) {
                    specificInformation.put(PrimitiveForm.RADIUS, _pfdSiSemiRadius.getValue());
                    specificInformation.put(PrimitiveForm.ORIENTATION, _pfdSiOrientation.getSelectedItem());
                }
                else if(type.equals(PrimitiveFormType.PUNKT.getName())) {
                }
                else if(type.equals(PrimitiveFormType.TEXTDARSTELLUNG.getName())) {
                }
                else {
                    _debug.warning("PrimitiveFormDialog: unbekannter Typ '" + type + "'.");
                }
                return specificInformation;
            }

            public void dispose() {
                _newPrimitiveFormDialog.dispose();
            }

            public void setSize() {
                _newPrimitiveFormDialog.setSize(_dialogSize);
            }

            public void setVisible(boolean b) {
                _newPrimitiveFormDialog.setVisible(b);
            }

            public void pack() {
                _newPrimitiveFormDialog.pack();
            }

            public void setTextOfSaveButton(final String text) {
                _savePrimitiveFormButton.setText(text);
            }

            public void setListenerToSaveButton(final ActionListener l) {
                final ActionListener[] actionListeners = _savePrimitiveFormButton.getActionListeners();
                for(ActionListener actionListener : actionListeners) {
                    _savePrimitiveFormButton.removeActionListener(actionListener);
                }
                _savePrimitiveFormButton.addActionListener(l);
            }

            public void storeNewPrimitiveForm() {
                final String newPrimitiveFormName = getName();
                if((newPrimitiveFormName == null) || (newPrimitiveFormName.length() == 0)) {
                    JOptionPane.showMessageDialog(
                        _newPrimitiveFormDialog,
                        "Bitte geben Sie einen Namen für die Grundfigur an!",
                        "Fehler",
                        JOptionPane.ERROR_MESSAGE
                    );
                    return;
                }
                final String newPrimitiveFormType = getType();
                if(newPrimitiveFormType == null) {
                    JOptionPane.showMessageDialog(
                        _newPrimitiveFormDialog,
                        "Bitte wählen Sie einen Typ aus der Liste aus!",
                        "Fehler",
                        JOptionPane.ERROR_MESSAGE
                    );
                    return;
                }
                final String infoOfPrimitiveFormName = getInfo();

                DOTPoint dotPoint;
                try {
                    dotPoint = (DOTPoint) _scratchDisplayObjectType;    // Das ist böse, ganz böse. Aber ich habe keine Zeit!
                }
                catch(ClassCastException ignore) {
                    _debug.error("DOTDefinitionDialog: falsches Plugin.");
                    return;
                }
                PrimitiveForm primitiveForm = new PrimitiveForm(
                    newPrimitiveFormName,
                    PrimitiveFormType.getPrimitiveFormType(newPrimitiveFormType),
                    infoOfPrimitiveFormName,
                    new Point2D.Double(getX(), getY()),
                    getSpecificInformation()
                );

                if(dotPoint.getPrimitiveForm(primitiveForm.getName()) != null) {
                    JOptionPane.showMessageDialog(
                        _newPrimitiveFormDialog,
                        "Eine Grundfigur mit diesem Namen existiert bereits!",
                        "Fehler",
                        JOptionPane.ERROR_MESSAGE
                    );
                    return;
                }
                dotPoint.addPrimitiveForm(primitiveForm);
                updatePrimitiveFormData();
                _primitiveFormComboBox.setSelectedItem(primitiveForm.getName());
                updatePropertyComboBox(_scratchDisplayObjectType.getDisplayObjectTypePlugin());
                updateStaticCheckBox();
                updateCenterPanel();
                dispose();
            }

            private void addToolTips() {
                _newPrimitiveFormNameTextField.setToolTipText("Der Name der Grundfigur");
                _newPrimitiveFormTypeComboBox.setToolTipText(_primitiveFormTypeTextField.getToolTipText());
                _infoOfPrimitiveFormNameTextField.setToolTipText(_primitiveFormInfoTextField.getToolTipText());
                _positionXSpinner.setToolTipText(_positionX.getToolTipText());
                _positionYSpinner.setToolTipText(_positionY.getToolTipText());
                _savePrimitiveFormButton.setToolTipText("Die Grundfigur speichern");
                _leaveButton.setToolTipText("Diesen Dialog ohne zu speichern verlassen.");
                _pfdDefinitionPanel.setToolTipText(_specialInformationDefinitionPanel.getToolTipText());
                _pfdSiHeight.setToolTipText(_siHeight.getToolTipText());
                _pfdSiWidth.setToolTipText(_siWidth.getToolTipText());
                _pfdSiRadius.setToolTipText(_siRadius.getToolTipText());
                _pfdSiSemiRadius.setToolTipText(_siSemiRadius.getToolTipText());
                _pfdSiOrientation.setToolTipText(_siOrientation.getToolTipText());
            }

            public void clearDialog() {
                _newPrimitiveFormNameTextField.setText("");
                if(_newPrimitiveFormTypeComboBox.getModel().getSize() > 0) {
                    _newPrimitiveFormTypeComboBox.setSelectedIndex(0);
                }
                _infoOfPrimitiveFormNameTextField.setText("");
                _positionXSpinner.setValue(0.);
                _positionYSpinner.setValue(0.);
                _savePrimitiveFormButton.setText("Speichern");
                _leaveButton.setText("Dialog beenden");
                _pfdSiHeight.setValue(5.);
                _pfdSiWidth.setValue(5.);
                _pfdSiRadius.setValue(5.);
                _pfdSiSemiRadius.setValue(5.);
            }

            PrimitiveFormDialog(final String title) {
                _newPrimitiveFormNameTextField.setBorder(BorderFactory.createTitledBorder("Name"));
                _newPrimitiveFormNameTextField.setPreferredSize(new Dimension(100, 30));
                _newPrimitiveFormTypeComboBox.setBorder(BorderFactory.createTitledBorder("Typ"));
                _infoOfPrimitiveFormNameTextField.setBorder(BorderFactory.createTitledBorder("Info"));
                _infoOfPrimitiveFormNameTextField.setPreferredSize(new Dimension(100, 30));
                _pfdDefinitionPanel.setBorder(BorderFactory.createTitledBorder("Definition"));
                String[] pfTypes = _scratchDisplayObjectType.getDisplayObjectTypePlugin().getPrimitiveFormTypes();
                ComboBoxModel<String> comboBoxModel = new DefaultComboBoxModel<>(pfTypes);
                _newPrimitiveFormTypeComboBox.setModel(comboBoxModel);
                final JPanel choicePanelUpperHalf = new JPanel();
                choicePanelUpperHalf.setLayout(new SpringLayout());
                choicePanelUpperHalf.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
                choicePanelUpperHalf.add(_newPrimitiveFormNameTextField);
                choicePanelUpperHalf.add(_newPrimitiveFormTypeComboBox);
                choicePanelUpperHalf.add(_infoOfPrimitiveFormNameTextField);
                JPanel positionPanel = getPFDPositionPanel();
                choicePanelUpperHalf.add(positionPanel);
                SpringUtilities.makeCompactGrid(choicePanelUpperHalf, 4, 20, 5);

                final JPanel choicePanelLowerHalf = new JPanel();
                choicePanelLowerHalf.setLayout(new SpringLayout());
                choicePanelLowerHalf.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
                _pfdDefinitionPanel.add(_pfdSpecialInformationPanel);
                initPFDSpecialInformationPanels();
                choicePanelLowerHalf.add(_pfdDefinitionPanel);
                _leaveButton.addActionListener(e -> dispose());
                final JPanel buttonsInChoicePanelsLowerHalfPanel = new JPanel();
                buttonsInChoicePanelsLowerHalfPanel.setLayout(
                    new BoxLayout(buttonsInChoicePanelsLowerHalfPanel, BoxLayout.X_AXIS));
                buttonsInChoicePanelsLowerHalfPanel.setBorder(BorderFactory.createTitledBorder("Operationen"));
                buttonsInChoicePanelsLowerHalfPanel.add(_savePrimitiveFormButton);
                buttonsInChoicePanelsLowerHalfPanel.add(Box.createRigidArea(new Dimension(10, 10)));
                buttonsInChoicePanelsLowerHalfPanel.add(_leaveButton);
                choicePanelLowerHalf.add(buttonsInChoicePanelsLowerHalfPanel);
                SpringUtilities.makeCompactGrid(choicePanelLowerHalf, 2, 20, 5);

                final JPanel allPanel = new JPanel();
                allPanel.setLayout(new BoxLayout(allPanel, BoxLayout.Y_AXIS));
                allPanel.add(choicePanelUpperHalf);
                allPanel.add(choicePanelLowerHalf);
                allPanel.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));

                _newPrimitiveFormDialog.setTitle(title);
                _newPrimitiveFormDialog.add(allPanel);
                _newPrimitiveFormDialog.setLocation(900, 80);
                addToolTips();
            }

            private JPanel getPFDPositionPanel() {
                JPanel pfdPositionPanel = new JPanel();
                pfdPositionPanel.setBorder(BorderFactory.createTitledBorder("Position"));
                pfdPositionPanel.setLayout(new BoxLayout(pfdPositionPanel, BoxLayout.X_AXIS));

                pfdPositionPanel.add(Box.createHorizontalStrut(5));
                JLabel xLabel = new JLabel("x: ");
                pfdPositionPanel.add(xLabel);
                pfdPositionPanel.add(_positionXSpinner);
                pfdPositionPanel.add(Box.createHorizontalStrut(10));
                JLabel yLabel = new JLabel("y: ");
                pfdPositionPanel.add(yLabel);
                pfdPositionPanel.add(_positionYSpinner);
                pfdPositionPanel.add(Box.createHorizontalStrut(5));

                return pfdPositionPanel;
            }

            private void initPFDSpecialInformationPanels() {
                // Für ein Rechteck
                JLabel heightLabel = new JLabel("Höhe: ");
                JLabel widthLabel = new JLabel("Breite: ");
                _pfdSpecialInformationRectangle.setLayout(new SpringLayout());
                _pfdSpecialInformationRectangle.add(heightLabel);
                _pfdSpecialInformationRectangle.add(_pfdSiHeight);
                _pfdSpecialInformationRectangle.add(widthLabel);
                _pfdSpecialInformationRectangle.add(_pfdSiWidth);
                SpringUtilities.makeCompactGrid(_pfdSpecialInformationRectangle, 4, 5, 5);

                // Für einen Kreis
                JLabel radiusLabel = new JLabel("Radius: ");
                _pfdSpecialInformationCircle.setLayout(new SpringLayout());
                _pfdSpecialInformationCircle.add(radiusLabel);
                _pfdSpecialInformationCircle.add(_pfdSiRadius);
                SpringUtilities.makeCompactGrid(_pfdSpecialInformationCircle, 2, 5, 5);

                // Für einen Halbkreis
                JLabel semiRadiusLabel = new JLabel("Radius: ");
                JLabel orientationLabel = new JLabel("Orientierung: ");
                _pfdSiOrientation.setModel(new DefaultComboBoxModel<>(SEMI_CIRCLE_ITEMS));
                _pfdSpecialInformationSemicircle.setLayout(new SpringLayout());
                _pfdSpecialInformationSemicircle.add(semiRadiusLabel);
                _pfdSpecialInformationSemicircle.add(_pfdSiSemiRadius);
                _pfdSpecialInformationSemicircle.add(orientationLabel);
                _pfdSpecialInformationSemicircle.add(_pfdSiOrientation);
                SpringUtilities.makeCompactGrid(_pfdSpecialInformationSemicircle, 4, 5, 5);

                // Für einen Text: derzeit nichts

                // Für einen Punkt: derzeit nichts

                // Und nun noch den Listener, der das Definitions-Panel aktualisiert
                ActionListener pfTypeListener = e -> {
                    String selectedType = (String) _newPrimitiveFormTypeComboBox.getSelectedItem();
                    _pfdDefinitionPanel.removeAll();
                    if(selectedType.equals("Punkt")) {
                        _positionXSpinner.setValue(0.);
                        _positionYSpinner.setValue(0.);
                        _positionXSpinner.setEnabled(false);
                        _positionYSpinner.setEnabled(false);
                        _pfdSpecialInformationPanel = _pfdSpecialInformationPoint;
                    }
                    else {
                        _positionXSpinner.setEnabled(true);
                        _positionYSpinner.setEnabled(true);
                        switch(selectedType) {
                            case "Rechteck":
                                _pfdSpecialInformationPanel = _pfdSpecialInformationRectangle;
                                break;
                            case "Kreis":
                                _pfdSpecialInformationPanel = _pfdSpecialInformationCircle;
                                break;
                            case "Halbkreis":
                                _pfdSpecialInformationPanel = _pfdSpecialInformationSemicircle;
                                break;
                            case "Textdarstellung":
                                _pfdSpecialInformationPanel = _pfdSpecialInformationText;
                                break;
                            default:
                                _pfdSpecialInformationPanel = new JPanel();
                                _debug.warning("PrimitiveFormDialog: unbekannter Typ '" + selectedType + "'.");
                                break;
                        }
                    }
                    _pfdDefinitionPanel.add(_pfdSpecialInformationPanel);
                    pack();        // Diese beiden Anweisungen besorgen das 'ordentliche' Neuzeichnen;
                    setSize();    // jedwede Anwebdung eines repaint blieb erfolglos.
                };
                _newPrimitiveFormTypeComboBox.addActionListener(pfTypeListener);
            }

            private void updatePfdSpecialInformationPanel() {
                if(!_scratchDisplayObjectType.getDisplayObjectTypePlugin().getName().equals("Punkt")) {
                    return;
                }
                DOTPoint dotPoint = (DOTPoint) _scratchDisplayObjectType; // Böse! Muss sauberer werden.
                final PrimitiveForm primitiveForm = dotPoint.getPrimitiveForm((String) _primitiveFormComboBox.getSelectedItem());
                final String type = (String) _newPrimitiveFormTypeComboBox.getSelectedItem();
                _pfdDefinitionPanel.removeAll();
                if(type.equals("Punkt")) {
                    _positionXSpinner.setValue(0.);
                    _positionYSpinner.setValue(0.);
                    _positionXSpinner.setEnabled(false);
                    _positionYSpinner.setEnabled(false);
                    _pfdSpecialInformationPanel = _pfdSpecialInformationPoint;
                }
                else {
                    _positionXSpinner.setEnabled(true);
                    _positionYSpinner.setEnabled(true);
                    if(type.equals(PrimitiveFormType.RECHTECK.getName())) {
                        final Double h = (Double) primitiveForm.getSpecificInformation(PrimitiveForm.HEIGHT);
                        _pfdSiHeight.setValue(h);
                        final Double w = (Double) primitiveForm.getSpecificInformation(PrimitiveForm.WIDTH);
                        _pfdSiWidth.setValue(w);
                        _pfdSpecialInformationPanel = _pfdSpecialInformationRectangle;
                    }
                    else if(type.equals(PrimitiveFormType.KREIS.getName())) {
                        final Double r = (Double) primitiveForm.getSpecificInformation(PrimitiveForm.RADIUS);
                        _pfdSiRadius.setValue(r);
                        _pfdSpecialInformationPanel = _pfdSpecialInformationCircle;
                    }
                    else if(type.equals(PrimitiveFormType.HALBKREIS.getName())) {
                        final Double r = (Double) primitiveForm.getSpecificInformation(PrimitiveForm.RADIUS);
                        _pfdSiSemiRadius.setValue(r);
                        final String o = (String) primitiveForm.getSpecificInformation(PrimitiveForm.ORIENTATION);
                        _pfdSiOrientation.setSelectedItem(o);
                        _pfdSpecialInformationPanel = _pfdSpecialInformationSemicircle;
                    }
                }
                _pfdDefinitionPanel.add(_pfdSpecialInformationPanel);
            }

            public void saveSpecialInformation() {
                if(!_scratchDisplayObjectType.getDisplayObjectTypePlugin().getName().equals("Punkt")) {
                    return;
                }
                DOTPoint dotPoint = (DOTPoint) _scratchDisplayObjectType; // Böse! Muss sauberer werden.
                final String primitiveFormName = _newPrimitiveFormNameTextField.getText();
                final PrimitiveForm primitiveForm = dotPoint.getPrimitiveForm(primitiveFormName);
                final String type = (String) _newPrimitiveFormTypeComboBox.getSelectedItem();
                if(type.equals(PrimitiveFormType.RECHTECK.getName())) {
                    final Object height = _pfdSiHeight.getValue();
                    primitiveForm.setSpecificInformation(PrimitiveForm.HEIGHT, height);
                    _siHeight.setText(getAsString((Double) height, 10));
                    final Object width = _pfdSiWidth.getValue();
                    primitiveForm.setSpecificInformation(PrimitiveForm.WIDTH, width);
                    _siWidth.setText(getAsString((Double) width, 10));
                }
                else if(type.equals(PrimitiveFormType.KREIS.getName())) {
                    final SpinnerModel model = _pfdSiRadius.getModel();
                    final SpinnerNumberModel numberModel = (SpinnerNumberModel) model;
                    final Double correctSpinnerValue = correctSpinnerValue(_pfdSiRadius.getValue(), numberModel.getStepSize());
                    primitiveForm.setSpecificInformation(PrimitiveForm.RADIUS, correctSpinnerValue);
                    _siRadius.setText(correctSpinnerValue.toString().replace('.', ','));
                }
                else if(type.equals(PrimitiveFormType.HALBKREIS.getName())) {
                    final SpinnerModel model = _pfdSiSemiRadius.getModel();
                    final SpinnerNumberModel numberModel = (SpinnerNumberModel) model;
                    final Double correctSpinnerValue = correctSpinnerValue(_pfdSiSemiRadius.getValue(), numberModel.getStepSize());
                    primitiveForm.setSpecificInformation(PrimitiveForm.RADIUS, correctSpinnerValue);
                    _siSemiRadius.setText(correctSpinnerValue.toString().replace('.', ','));
                    final String orientation = (String) _pfdSiOrientation.getSelectedItem();
                    primitiveForm.setSpecificInformation(PrimitiveForm.ORIENTATION, orientation);
                    _siOrientation.setText(orientation);
                }
            }

            private Double correctSpinnerValue(final Object value, final Number stepSize) {
                final Double dValue = (Double) value;
                final double stepSizeDoubleValue = stepSize.doubleValue();
                if(stepSizeDoubleValue > 1.) {
                    return dValue;
                }
                final Double reciprocalStepSize = 1. / stepSizeDoubleValue;
                final long reciprocalStepSizeLong = Math.round(reciprocalStepSize);
                final Double dValueIncreased = dValue * reciprocalStepSizeLong;
                final long iValueIncreased = Math.round(dValueIncreased);
                return ((double) iValueIncreased) / reciprocalStepSizeLong;
            }

            @Override
            public String toString() {
                return "PrimitiveFormDialog{}";
            }
        }

        ActionListener editPrimitiveFormListener = new ActionListener() {
            PrimitiveFormDialog pfDialog = new PrimitiveFormDialog("GND: Bestehende Grundfigur bearbeiten");

            @Override
            public void actionPerformed(ActionEvent e) {
                if(!_scratchDisplayObjectType.getDisplayObjectTypePlugin().getName().equals("Punkt")) {
                    return;
                }
                pfDialog.setTextOfSaveButton("Daten übernehmen");
                final Object selectedItem = _primitiveFormComboBox.getSelectedItem();
                if(selectedItem == null) {
                    JOptionPane.showMessageDialog(
                        DOTDefinitionDialog.this.getFrame(),
                        "Bitte wählen Sie eine Grundfigur aus!",
                        "Fehler",
                        JOptionPane.ERROR_MESSAGE
                    );
                    return;
                }
                final String name = (String) selectedItem;
                if(name.length() == 0) {
                    JOptionPane.showMessageDialog(
                        DOTDefinitionDialog.this.getFrame(),
                        "Bitte wählen Sie eine Grundfigur aus!",
                        "Fehler",
                        JOptionPane.ERROR_MESSAGE
                    );
                    return;
                }
                pfDialog.setName(name);
                pfDialog.disableNameField();
                pfDialog.setType(_primitiveFormTypeTextField.getText());
                pfDialog.disableTypeField();
                pfDialog.setInfo(_primitiveFormInfoTextField.getText());
                Double xAsDouble = 0.;
                try {
                    xAsDouble = Double.parseDouble(_positionX.getText().replace(',', '.'));
                }
                catch(Exception ex) {
                    _debug.warning("editPrimitiveFormListener: Zeichenkette kann nicht in Zahl gewandelt werden", ex);
                }
                pfDialog.setX(xAsDouble);
                Double yAsDouble = 0.;
                try {
                    yAsDouble = Double.parseDouble(_positionY.getText().replace(',', '.'));
                }
                catch(Exception ex) {
                    _debug.warning("editPrimitiveFormListener: Zeichenkette kann nicht in Zahl gewandelt werden", ex);
                }
                pfDialog.setY(yAsDouble);
                pfDialog.updatePfdSpecialInformationPanel();
                ActionListener saveButtonListener = f -> {
                    DOTPoint dotPoint = (DOTPoint) _scratchDisplayObjectType; // Das ist böse!
                    final PrimitiveForm primitiveForm = dotPoint.getPrimitiveForm((String) _primitiveFormComboBox.getSelectedItem());
                    final String info = pfDialog.getInfo();
                    primitiveForm.setInfo(info);
                    _primitiveFormInfoTextField.setText(info);
                    primitiveForm.setInfo(info);
                    final Double x = pfDialog.getX();
                    _positionX.setText(getAsString(x, 10));
                    final Double y = pfDialog.getY();
                    _positionY.setText(getAsString(y, 10));
                    primitiveForm.setTranslation(new Point2D.Double(x, y));
                    pfDialog.saveSpecialInformation();
                    pfDialog.dispose();
                };
                pfDialog.setListenerToSaveButton(saveButtonListener);

                pfDialog.pack();
                pfDialog.setSize();
                pfDialog.setVisible(true);
            }
        };
        _editPrimitiveFormButton.addActionListener(editPrimitiveFormListener);

        ActionListener copyPrimitiveFormListener = new ActionListener() {
            PrimitiveFormDialog pfDialog = new PrimitiveFormDialog("GND: Kopierte Grundfigur bearbeiten");

            @Override
            public void actionPerformed(ActionEvent e) {
                pfDialog.setTextOfSaveButton("Grundfigur anlegen");
                final Object selectedItem = _primitiveFormComboBox.getSelectedItem();
                if(selectedItem == null) {
                    JOptionPane.showMessageDialog(
                        DOTDefinitionDialog.this.getFrame(),
                        "Bitte wählen Sie eine Grundfigur aus!",
                        "Fehler",
                        JOptionPane.ERROR_MESSAGE
                    );
                    return;
                }
                final String name = (String) selectedItem;
                if(name.length() == 0) {
                    JOptionPane.showMessageDialog(
                        DOTDefinitionDialog.this.getFrame(),
                        "Bitte wählen Sie eine Grundfigur aus!",
                        "Fehler",
                        JOptionPane.ERROR_MESSAGE
                    );
                    return;
                }
                pfDialog.setName(name);
                pfDialog.setType(_primitiveFormTypeTextField.getText());
                pfDialog.setInfo(_primitiveFormInfoTextField.getText());
                pfDialog.setX(Double.parseDouble(_positionX.getText().replace(',', '.')));
                pfDialog.setY(Double.parseDouble(_positionY.getText().replace(',', '.')));
                pfDialog.updatePfdSpecialInformationPanel();
                ActionListener okayListener = f -> pfDialog.storeNewPrimitiveForm();
                pfDialog.setListenerToSaveButton(okayListener);

                pfDialog.pack();
                pfDialog.setSize();
                pfDialog.setVisible(true);
            }
        };
        _copyPrimitiveFormButton.addActionListener(copyPrimitiveFormListener);


        ActionListener newPrimitiveFormListener = new ActionListener() {
            PrimitiveFormDialog pfDialog = new PrimitiveFormDialog("GND: Neue Grundfigur anlegen");

            @Override
            public void actionPerformed(ActionEvent e) {
                pfDialog.clearDialog();
                pfDialog.setTextOfSaveButton("Grundfigur anlegen");
                ActionListener okayListener = f -> {
                    JButton button = (JButton) f.getSource();
                    if(button.isValid()) {
                        // Ab dem 2ten Anlegen kam ein 2ter ActionEvent, der sich nur darin
                        // unterschied, dass der Button invalid war. Kann sein, dass das nur
                        // darauf zurückzuführen war, dass bei jeder äußeren Action wieder
                        // ein neuer ActionListener hinzukam.
                        pfDialog.storeNewPrimitiveForm();
                        _editPrimitiveFormButton.setEnabled(_editable);
                        _copyPrimitiveFormButton.setEnabled(_editable);
                        _deletePrimitiveFormButton.setEnabled(_editable);
                        _staticCheckBox.setEnabled(_editable);
                    }
                };
                pfDialog.setListenerToSaveButton(okayListener);

                pfDialog.pack();
                pfDialog.setSize();
                pfDialog.setVisible(true);
            }
        };
        _newPrimitiveFormButton.addActionListener(newPrimitiveFormListener);

        ActionListener deletePrimitiveFormListener = e -> {
            final String primitiveFormName = (String) _primitiveFormComboBox.getSelectedItem();
            if(primitiveFormName != null) {
                _scratchDisplayObjectType.removePrimitiveForm(primitiveFormName);
                updatePrimitiveFormData();
                updatePropertyComboBox(_scratchDisplayObjectType.getDisplayObjectTypePlugin());
                updateStaticCheckBox();
                updateCenterPanel();
            }
        };
        _deletePrimitiveFormButton.addActionListener(deletePrimitiveFormListener);
    }

    private JPanel getComboBoxPanel() {
        // ToDo: Wie bekommt man das Folgende automatisch hin?
        final DisplayObjectTypePlugin displayObjectTypePlugin = _displayObjectType.getDisplayObjectTypePlugin();
        updatePropertyComboBox(displayObjectTypePlugin);
        updateStaticCheckBox();

        ItemListener propertyListener = e -> {
            if(e.getStateChange() == ItemEvent.SELECTED) {
                updateStaticCheckBox();
                updateCenterPanel();
            }
        };
        _propertyComboBox.addItemListener(propertyListener);

        ActionListener staticCheckBoxListener = e -> {
            String primitiveFormName = (String) _primitiveFormComboBox.getSelectedItem();
            Property property = (Property) _propertyComboBox.getSelectedItem();
            if(primitiveFormName == null) {
                _scratchDisplayObjectType.setPropertyStatic(null, property, _staticCheckBox.isSelected());
            }
            else {
                _scratchDisplayObjectType.setPropertyStatic(primitiveFormName, property, _staticCheckBox.isSelected());
            }
            updateCenterPanel();
        };
        _staticCheckBox.addActionListener(staticCheckBoxListener);

        JPanel propertyComboBoxPanel = new JPanel();
        propertyComboBoxPanel.add(_propertyComboBox);
        propertyComboBoxPanel.setBorder(BorderFactory.createTitledBorder("Eigenschaft"));

        JPanel staticCheckBoxPanel = new JPanel();
        staticCheckBoxPanel.add(_staticCheckBox);
        staticCheckBoxPanel.setBorder(BorderFactory.createTitledBorder("Statisch"));

        JPanel thePanel = new JPanel();
        thePanel.add(Box.createRigidArea(new Dimension(5, 20)));
        thePanel.add(propertyComboBoxPanel);
        thePanel.add(Box.createRigidArea(new Dimension(5, 20)));
        thePanel.add(staticCheckBoxPanel);
        thePanel.add(Box.createRigidArea(new Dimension(50, 20)));
        thePanel.setLayout(new BoxLayout(thePanel, BoxLayout.X_AXIS));

        if(!displayObjectTypePlugin.isDynamicsPossible()) {
            _staticCheckBox.setEnabled(false);
            _staticCheckBox.setSelected(true);
        }

        return thePanel;
    }

    @Nullable
    private DisplayObjectTypePlugin getPluginObject() {
        if(_displayObjectType != null) {
            return _displayObjectType.getDisplayObjectTypePlugin();
        }
        return null;
    }

    private void updatePrimitiveFormData() {
        if(!_scratchDisplayObjectType.getDisplayObjectTypePlugin().getName().equals("Punkt")) {
            return;
        }
        final Set<String> primitiveFormNames = _scratchDisplayObjectType.getPrimitiveFormNames();
        ComboBoxModel<Object> comboBoxModel = new DefaultComboBoxModel<>(primitiveFormNames.toArray());
        _primitiveFormComboBox.setModel(comboBoxModel);
        if(comboBoxModel.getSize() > 0) {
            final String primitiveFormName = (String) comboBoxModel.getSelectedItem();
            _primitiveFormTypeTextField.setText(
                _scratchDisplayObjectType.getPrimitiveFormType(primitiveFormName));
            _primitiveFormInfoTextField.setText(_scratchDisplayObjectType.getPrimitiveFormInfo(primitiveFormName));
            DOTPoint dotPoint = (DOTPoint) _scratchDisplayObjectType; // Böse! Muss sauberer werden.
            PrimitiveForm primitiveForm = dotPoint.getPrimitiveForm(primitiveFormName);
            final Double x = primitiveForm.getTranslation().getX();
            _positionX.setText(getAsString(x, 10));
            final Double y = primitiveForm.getTranslation().getY();
            _positionY.setText(getAsString(y, 10));
        }
        else {
            _primitiveFormTypeTextField.setText(null);
            _primitiveFormInfoTextField.setText(null);
            _positionX.setText(null);
            _positionY.setText(null);
            _editPrimitiveFormButton.setEnabled(false);
            _copyPrimitiveFormButton.setEnabled(false);
            _deletePrimitiveFormButton.setEnabled(false);
            _staticCheckBox.setEnabled(false);
        }
        updateSpecialInformationPanel();
    }

    private void updateSpecialInformationPanel() {
        if(!_scratchDisplayObjectType.getDisplayObjectTypePlugin().getName().equals("Punkt")) {
            return;
        }
        DOTPoint dotPoint = (DOTPoint) _scratchDisplayObjectType; // Böse! Muss sauberer werden.
        final PrimitiveForm primitiveForm = dotPoint.getPrimitiveForm((String) _primitiveFormComboBox.getSelectedItem());
        final String type = _primitiveFormTypeTextField.getText();
        _specialInformationDefinitionPanel.removeAll();
        _specialInformationPanel = null;
        if(type.equals("Punkt")) {
            _specialInformationPanel = _specialInformationPoint;
        }
        else if(type.equals(PrimitiveFormType.TEXTDARSTELLUNG.getName())) {
            _specialInformationPanel = _specialInformationTextdarstellung;
        }
        else if(type.equals(PrimitiveFormType.RECHTECK.getName())) {
            final Double h = (Double) primitiveForm.getSpecificInformation(PrimitiveForm.HEIGHT);
            _siHeight.setText(getAsString(h, 10));
            final Double w = (Double) primitiveForm.getSpecificInformation(PrimitiveForm.WIDTH);
            _siWidth.setText(getAsString(w, 10));
            _specialInformationPanel = _specialInformationRectangle;
        }
        else if(type.equals(PrimitiveFormType.KREIS.getName())) {
            final Double r = (Double) primitiveForm.getSpecificInformation(PrimitiveForm.RADIUS);
            _siRadius.setText(r.toString().replace('.', ','));
            _specialInformationPanel = _specialInformationCircle;
        }
        else if(type.equals(PrimitiveFormType.HALBKREIS.getName())) {
            final Double r = (Double) primitiveForm.getSpecificInformation(PrimitiveForm.RADIUS);
            _siSemiRadius.setText(r.toString().replace('.', ','));
            final String o = (String) primitiveForm.getSpecificInformation(PrimitiveForm.ORIENTATION);
            _siOrientation.setText(o);
            _specialInformationPanel = _specialInformationSemicircle;
        }
        if(_specialInformationPanel != null) {
            _specialInformationDefinitionPanel.add(_specialInformationPanel);
        }
    }

    private void updatePropertyComboBox(@Nullable final DisplayObjectTypePlugin displayObjectTypePlugin) {
        String primitiveFormName = _primitiveFormTypeTextField.getText();
        final Property[] properties;
        if(displayObjectTypePlugin != null) {
            if(primitiveFormName.isEmpty()) {
                properties = displayObjectTypePlugin.getProperties(null);
            }
            else {
                properties = displayObjectTypePlugin.getProperties(primitiveFormName);
            }
        }
        else {
            properties = null;
        }
        _propertyComboBox.removeAllItems();
        if(properties != null) {
            for(Property property : properties) {
                _propertyComboBox.addItem(property);
            }
        }
    }

    private void updateStaticCheckBox() {
        final String primitiveFormName = (String) _primitiveFormComboBox.getSelectedItem();
        Property property = (Property) _propertyComboBox.getSelectedItem();
        if(property != null) {
            if((primitiveFormName == null) || (primitiveFormName.isEmpty())) {
                _staticCheckBox.setSelected(_scratchDisplayObjectType.isPropertyStatic(null, property));
            }
            else {
                _staticCheckBox.setSelected(_scratchDisplayObjectType.isPropertyStatic(primitiveFormName, property));
            }
        }
    }

    /**
     * Gibt {@code true} zurück, wenn die Identität des Darstellungstyps, also der Name,
     * nicht verändert werden kann, oder {@code false} andernfalls.
     *
     * @return {@code true} genau dann, wenn die Identität nicht verändert werden kann
     */
    public boolean isReviseOnly() {
        return _reviseOnly;
    }

    /**
     * Legt fest, ob die Identität des Darstellungstyps unverändert bleiben muss ({@code true}),
     * oder aber nicht.
     *
     * @param reviseOnly der neue Wert
     */
    @SuppressWarnings("unused")
    public void setReviseOnly(boolean reviseOnly) {
        _reviseOnly = reviseOnly;
    }

    /**
     * Gibt die Datenverteiler-Verbindung zurück.
     *
     * @return die Datenverteiler-Verbindung
     */
    public ClientDavInterface getConnection() {
        return _connection;
    }

    /**
     * Gibt die Darstellungstypen-Verwaltung zurück. Wäre im Moment entbehrlich, weil der DOTManager als Singleton
     * implementiert ist.
     *
     * @return die Darstellungstypen-Verwaltung
     */
    public DOTManager getDotManager() {
        return _dotManager;
    }

    /**
     * Gibt {@code true} zurück, wenn der übergebene Darstellungstyp veränderbar ist, und {@code false}
     * sonst.
     *
     * @return ist der Darstellungstyp veränderbar
     */
    public boolean isEditable() {
        return _editable;
    }

    /**
     * Gibt den aktuellen Inhalt des Namensfeldes zurück.
     *
     * @return gibt den aktuellen Inhalt des Namensfeldes zurück
     */
    public String getNameText() {
        return _nameTextField.getText();
    }

    /**
     * Gibt den aktuellen Inhalt des Info-Feldes zurück.
     *
     * @return gibt den aktuellen Inhalt des Info-Feldes zurück
     */
    public String getInfoText() {
        return _infoTextField.getText();
    }

    /**
     * Gibt den Namen der in der Auswahlbox selektierten Grundfigur zurück.
     *
     * @return gibt den Namen der in der Auswahlbox selektierten Grundfigur zurück
     */
    public String getSelectedPrimitiveForm() {
        return (String) _primitiveFormComboBox.getSelectedItem();
    }

    /**
     * Gibt die in der Auswahlbox selektierte Eigenschaft zurück.
     *
     * @return gibt die in der Auswahlbox selektierte Eigenschaft zurück
     */
    public Property getSelectedProperty() {
        return (Property) _propertyComboBox.getSelectedItem();
    }

    /**
     * Gibt den Status der Statisch-Checkbox zurück.
     *
     * @return gibt den Status der Statisch-Checkbox zurück
     */
    public boolean getStaticCheckBoxState() {
        return _staticCheckBox.isSelected();
    }

    /**
     * Setzt den Änderungsstatus.
     *
     * @param somethingChanged der neue Wert
     */
    @SuppressWarnings("SameParameterValue")
    public void setSomethingChanged(boolean somethingChanged) {
        _somethingChanged = somethingChanged;
    }

    /**
     * Gibt an, ob die übergebene Eigenschaft statisch ist. Gehört die Eigenschaft zu einer
     * Grundfigur, so muss deren Name übergeben werden, sonst ist das erste Argument {@code null}.
     *
     * @param primitiveFormName der Name einer Grundfigur oder {@code null}
     * @param property          eine Visulaisierungs-Eigenschaft
     * @return statisch oder dynamisch
     */
    public boolean isPropertyStatic(@Nullable String primitiveFormName, Property property) {
        return _scratchDisplayObjectType.isPropertyStatic(primitiveFormName, property);
    }

    /**
     * Gibt den Wert der übergebenen Eigenschaftzurück. Gehört die Eigenschaft zu einer
     * Grundfigur, so muss deren Name übergeben werden, sonst ist das erste Argument {@code null}.
     *
     * @param primitiveFormName der Name einer Grundfigur oder {@code null}
     * @param property          eine Visulaisierungs-Eigenschaft
     * @return der Wert der statischen Eigenschaft oder null, wenn ein solcher nicht existiert
     */
    public Object getValueOfStaticProperty(@Nullable String primitiveFormName, Property property) {
        return _scratchDisplayObjectType.getValueOfStaticProperty(primitiveFormName, property);
    }

    private void updateCenterPanel() {
        if(_dotDefinitionPanel == null) {    // im Konstruktor sind wir noch nicht so weit
            return;
        }
        if(_centerPanel != null) {
            remove(_centerPanel);
        }
        _centerPanel = _dotDefinitionPanel.getDOTItemDefinitionPanel();
        if(_centerPanel != null) {
            add(_centerPanel, BorderLayout.CENTER);
        }
        revalidate();
        pack();
        repaint();
    }

    private void addToolTips() {
        _nameTextField.setToolTipText("Der Name des Darstellungstyps");
        _infoTextField.setToolTipText("Eine Kurzinformation zu diesem Darstellungstyp");
        _primitiveFormComboBox.setToolTipText("Der Name der Grundfigur");
        _primitiveFormTypeTextField.setToolTipText("Der Typ der Grundfigur");
        _primitiveFormInfoTextField.setToolTipText("Eine Kurzinformation zu der Grundfigur");
        _positionX.setToolTipText("Relative x-Koordinate des Mittelpunkts (außer beim Typ Text)");
        _positionY.setToolTipText("Relative y-Koordinate des Mittelpunkts (außer beim Typ Text)");
        _editPrimitiveFormButton.setToolTipText("Diese Grundfigur bearbeiten.");
        _copyPrimitiveFormButton.setToolTipText("Diese Grundfigur kopieren und die Kopie bearbeiten.");
        _newPrimitiveFormButton.setToolTipText("Eine neue Grundfigur anlegen.");
        _deletePrimitiveFormButton.setToolTipText("Diese Grundfigur löschen.");
        _propertyComboBox.setToolTipText("Eine Eigenschaft der Grundfigur.");
        _staticCheckBox.setToolTipText("Die Eigenschaft ist statisch oder dynamisch.");
        _saveButton.setToolTipText("Diesen Darstellungstyp speichern.");
        _specialInformationDefinitionPanel.setToolTipText("Typabhängige Definitionsmerkmale der Grundfigur.");
        _siHeight.setToolTipText("Die Höhe des Rechtecks.");
        _siWidth.setToolTipText("Die Breite des Rechtecks.");
        _siRadius.setToolTipText("Der Radius des Kreises.");
        _siSemiRadius.setToolTipText("Der Radius des Halbkreises.");
        _siOrientation.setToolTipText("Der Orientierung des Halbkreises.");
    }

    private static final String[] SEMI_CIRCLE_ITEMS = {
        DOTPointPainter.LINKER_HALBKREIS,
        DOTPointPainter.OBERER_HALBKREIS, DOTPointPainter.RECHTER_HALBKREIS,
        DOTPointPainter.UNTERER_HALBKREIS
    };    // Böse! S.o.

    @Override
    public String toString() {
        return "DOTDefinitionDialog{}";
    }
}
