/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.displayObjectToolkit;

import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.asbNodePlugin.DOTAsbNode;
import de.kappich.pat.gnd.gnd.PreferencesHandler;
import de.kappich.pat.gnd.kmPlugin.DOTKm;
import de.kappich.pat.gnd.kmPlugin.KmFormat;
import de.kappich.pat.gnd.layerManagement.LayerManager;
import de.kappich.pat.gnd.linePlugin.DOTLine;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType.DisplayObjectTypeItem;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectTypePlugin;
import de.kappich.pat.gnd.pointPlugin.DOTPoint;
import de.kappich.pat.gnd.pointPlugin.DOTPoint.PrimitiveForm;
import de.kappich.pat.gnd.pointPlugin.DOTPoint.PrimitiveFormType;
import de.kappich.pat.gnd.pointPlugin.DOTPointPainter;
import de.kappich.pat.gnd.properties.ColorProperty;
import de.kappich.pat.gnd.properties.DiameterProperty;
import de.kappich.pat.gnd.properties.DistanceProperty;
import de.kappich.pat.gnd.properties.DistanceRasterProperty;
import de.kappich.pat.gnd.properties.DistanceRasterType;
import de.kappich.pat.gnd.properties.DottingProperty;
import de.kappich.pat.gnd.properties.FillingProperty;
import de.kappich.pat.gnd.properties.KmFormatProperty;
import de.kappich.pat.gnd.properties.RnFormatProperty;
import de.kappich.pat.gnd.properties.StatFormatProperty;
import de.kappich.pat.gnd.properties.StrokeWidthProperty;
import de.kappich.pat.gnd.properties.TextProperty;
import de.kappich.pat.gnd.properties.TextSizeProperty;
import de.kappich.pat.gnd.properties.TextStyleProperty;
import de.kappich.pat.gnd.properties.TransparencyProperty;
import de.kappich.pat.gnd.rnPlugin.DOTRn;
import de.kappich.pat.gnd.rnPlugin.RnDisplayObject;
import de.kappich.pat.gnd.rnPlugin.RnFormat;
import de.kappich.pat.gnd.statPlugin.DOTStat;
import de.kappich.pat.gnd.statPlugin.StatFormat;
import de.kappich.pat.gnd.utils.view.PreferencesDeleter;
import java.awt.Color;
import java.awt.Font;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import javax.swing.table.AbstractTableModel;

/**
 * Der DOTManager verwaltet alle Darstellungstypen. Derzeit ist er als Singleton implementiert, um das Problem der Kommunikation verschiedener Manager
 * zu umgehen. Er ist auch ein TableModel, damit seine Inhalte im DOTManagerDialog angezeigt werden können.
 *
 * @author Kappich Systemberatung
 */
@SuppressWarnings({"serial"})
public final class DOTManager extends AbstractTableModel {
    // Ein Singleton, was aber nicht die ultimative Lösung sein muss; werden
    // mehrere benötigt, so müssen Änderungen zwischen diesen mitgeteilt
    // werden, oder alternativ muss sichergestellt werden, dass überall,
    // wo solche Änderungen benötigt werden, derselbe Manager benutzt wird.

    private static final DOTManager _instance = new DOTManager();
    private static final Map<String, String> _fullClassNames = new HashMap<>();
    private static final Debug _debug = Debug.getLogger();
    private static final String[] _columnNames = {"Name des Darstellungstyps"};
	private final List<DisplayObjectType> _dotList = new ArrayList<>() {
        @Override
        public boolean add(DisplayObjectType dot) {
            int index = Collections.binarySearch(this, dot);
            if (index < 0) {
                index = ~index;
            }
            super.add(index, dot);
            return true;
        }
    };
    private final HashMap<String, DisplayObjectType> _dotMap = new HashMap<>();
    private final Set<String> _unchangeables = new HashSet<>();
    private final List<DOTChangeListener> _dotChangeListeners = new ArrayList<>();
    private boolean _initialized;

    private DOTManager() {
        // Die Instanz wird nicht im Konstruktor initialisiert, damit sich DOTs
        // bereits als DOTManager.DOTChangeListener registrieren können.
    }

    /**
     * Die für eine Singleton typische Methode.
     *
     * @return den DOTManager
     */
    public static DOTManager getInstance() {
        // Die Instanz wird nicht im Konstruktor initialisiert, damit sich DOTs
        // bereits als DOTManager.DOTChangeListener registrieren können.
        if (!_instance._initialized) {
            _instance._initialized = true; // Das muss schon hier stehen, denn die Instanz wird in den folgenden Zeilen bereits benutzt!
            DOTManager.readFullClassNames();
            _instance.initProgramDefinedDOTs();
            _instance.initUserDefinedDOTs();
        }
        return _instance;
    }

    /**
     * Mit Hilfe dieser Methode kann man den DOTManager dazu zwingen, sich erneut zu konstruieren, was etwa nach dem Importieren von Präferenzen
     * angezeigt ist.
     */
    public static void refreshInstance() {
        _instance._dotList.clear();
        _instance._dotMap.clear();
        _instance._unchangeables.clear();
        _instance.initProgramDefinedDOTs();
        _instance.initUserDefinedDOTs();
    }

    @Nullable
    private static Class<?> getClass(final String className) {
        String fullPathName;
        switch (className) {
            case "DOTArea":
                fullPathName = "de.kappich.pat.gnd.areaPlugin.DOTArea";
                break;
            case "DOTComplex":
                fullPathName = "de.kappich.pat.gnd.complexPlugin.DOTComplex";
                break;
            case "DOTLine":
                fullPathName = "de.kappich.pat.gnd.linePlugin.DOTLine";
                break;
            case "DOTPoint":
                fullPathName = "de.kappich.pat.gnd.pointPlugin.DOTPoint";
                break;
            default:    // eine andere als die vordefinierten Klassen muss gesucht werden
                fullPathName = _fullClassNames.get(className);
                if (fullPathName == null || fullPathName.isEmpty()) {
                    _debug.error("DOTManager.getClass() wurde für die nicht unterstützte Klasse " + className + " aufgerufen.");
                    return null;
                }
                break;
        }
        Class<?> c;
        try {
            c = Class.forName(fullPathName);
        } catch (ClassNotFoundException ignore) {
            _debug.error("DOTManager.getClass() wurde für die nicht auffindbare Klasse " + className + " aufgerufen.");
            return null;
        }
        return c;
    }

    /**
     * Gibt den Ausgangsknoten zum Abspeichern aller Präferenzen des DOTManagers an.
     *
     * @return gibt den Ausgangsknoten zum Abspeichern aller Präferenzen des DOTManagers zurück
     */
    private static Preferences getPreferenceStartPath() {
        return PreferencesHandler.getInstance().getPreferenceStartPath().node("DOT");
    }

    private static void storeFullClassNameLookup(final DisplayObjectType dot) {
        final Preferences prefs = getPreferenceStartPath();
        final Preferences classesPrefs = prefs.node("Classes");
        final String binaryClassName = dot.getClass().getName();
        final String shortClassName = binaryClassName.substring(binaryClassName.lastIndexOf('.') + 1);
        classesPrefs.put(shortClassName, binaryClassName);
    }

    /*
     * Initialisiert _fullClassNames
     */
    private static void readFullClassNames() {
        final Preferences prefs = getPreferenceStartPath();
        final Preferences classesPrefs = prefs.node("Classes");
        String[] keys;
        try {
            keys = classesPrefs.keys();
        } catch (BackingStoreException e) {
            _debug.error("Die GND-paketfremden Darstellungstypen können nicht initialisiert werden, " + "BackingStoreException: " + e.toString());
            PreferencesDeleter pd = new PreferencesDeleter("Die GND-paketfremden Darstellungstypen können nicht geladen werden.", classesPrefs);
            pd.run();
            return;
        }
        for (String key : keys) {
            final String value = classesPrefs.get(key, "");
            if (value != null && (!value.isEmpty())) {
                _fullClassNames.put(key, value);
            }
        }
    }

    private static void addFullClassNames(final List<String> fullClassNames) {
        if (fullClassNames == null || (fullClassNames.isEmpty())) {
            return;
        }
        for (String fullClassName : fullClassNames) {
            final int lastIndexOf = fullClassName.lastIndexOf('.');
            if ((lastIndexOf == -1) || (fullClassName.length() - 1 == lastIndexOf)) { // kann nicht sein
                continue;
            }
            String className = fullClassName.substring(lastIndexOf + 1);
            _fullClassNames.put(className, fullClassName);
        }
    }

    /**
     * Der PluginManager ruft diese Methode auf, wenn externe Plugins hinzugefügt wurden. Dies wird statt einem Listener-Mechanismus bevorzugt, weil
     * bei einem solchen sichergestellt werden müsste, dass der DOTManager bereits instanziiert ist.
     *
     * @param plugins die Liste der hinzugefügten Plugins
     */
    public static void pluginsAdded(final List<String> plugins) {
        final List<String> dotClasses = new ArrayList<>();
        for (String plugin : plugins) {
            final Class<?> pluginClass;
            try {
                pluginClass = Class.forName(plugin);
            } catch (ClassNotFoundException ignore) {
                continue;
            }
            final Object pluginObject;
            try {
                pluginObject = pluginClass.newInstance();
            } catch (InstantiationException ignore) {
                _debug.error("Fehler im PluginManager: die Klasse '" + plugin + "' kann nicht instanziiert werden.");
                continue;
            } catch (IllegalAccessException ignore) {
                _debug.error("Fehler im PluginManager: auf die Klasse '" + plugin + "' kann nicht zugegriffen werden.");
                continue;
            }
	        if (pluginObject instanceof DisplayObjectTypePlugin dotPlugin) {
                final DisplayObjectType displayObjectType = dotPlugin.getDisplayObjectType();
                dotClasses.add(displayObjectType.getClass().getName());
            }
        }
        DOTManager.addFullClassNames(dotClasses);
    }

    /**
     * Gibt den Darstellungstypen zu dem übergebenen Namen zurück, falls ein solcher existiert, und {@code null} sonst.
     *
     * @param name ein Name
     *
     * @return ein Darstellungstyp mit diesem Namen oder {@code null}
     */
    public DisplayObjectType getDisplayObjectType(String name) {
        return _dotMap.get(name);
    }

    /**
     * Gibt den Darstellungstypen zu dem übergebenen Index zurück, falls ein solcher existiert, und wirft eine {@code IndexOutOfBoundsException}
     * sonst.
     *
     * @param index ein Index zwischen 0 und der Anzahl der Darstellungstypen - 1
     *
     * @return den zugehörigen Darstellungstypen
     */
    public DisplayObjectType getDisplayObjectType(int index) {
        return _dotList.get(index);
    }

    /**
     * Gibt {@code true} zurück, wenn ein Darstellungstyp mit diesem Namen existiert.
     *
     * @param name ein Name
     *
     * @return {@code true} genau dann, wenn es einen Darstellungstyp mit diesem Namen gibt
     */
    public boolean containsDisplayObjectType(String name) {
        return _dotMap.containsKey(name);
    }

    /**
     * Speichert den übergebenen Darstellungstypen.
     *
     * @param dot ein Darstellungstyp
     */
    public void saveDisplayObjectType(DisplayObjectType dot) {
        if (!_unchangeables.contains(dot.getName())) {
            if (containsDisplayObjectType(dot.getName())) {
                put(dot);
                notifyDOTChangeListenersDOTChanged(dot);
            } else {
                put(dot);
                notifyDOTChangeListenersDOTAdded(dot);
            }
            dot.putPreferences(DOTManager.getPreferenceStartPath());
            DOTManager.storeFullClassNameLookup(dot);
            fireTableDataChanged();
        }
    }

    /**
     * Löscht den übergebenen Darstellungstypen.
     *
     * @param dot ein Darstellungstyp
     *
     * @return {@code true}, falls der Darstellungstyp gelöscht wurde, {@code false} falls er nicht vorhanden war
     */
    public boolean deleteDisplayObjectType(DisplayObjectType dot) {
        if (!_unchangeables.contains(dot.getName()) && !LayerManager.getInstance().displayObjectTypeIsUsed(dot.getName())) {
            remove(dot);
            notifyDOTChangeListenersDOTRemoved(dot.getName());
            dot.deletePreferences(DOTManager.getPreferenceStartPath());
            fireTableDataChanged();
            return true;
        }
        return false;
    }

    /**
     * Löscht den Darstellungstypen mit dem übergebenen Namen, und gibt {@code true} zurück, wenn das Löschen erfolgreich war.
     *
     * @param dotName ein Darstellungstypname
     *
     * @return {@code true} genau dann, wenn das Löschen erfolgreich war.
     */
    @SuppressWarnings("unused")
    public boolean deleteDisplayObjectType(String dotName) {
        if (!_unchangeables.contains(dotName)) {
            final DisplayObjectType dot = getDisplayObjectType(dotName);
            if (dot != null) {
                return deleteDisplayObjectType(dot);
            }
        }
        return false;
    }

    /**
     * Löscht alle Darstellungstypen.
     */
    @SuppressWarnings("unused")
    public void clearDisplayObjectTypes() {
        final Object[] array = _dotMap.keySet().toArray();
        for (Object dotName : array) {
            deleteDisplayObjectType(_dotMap.get(dotName));
        }
    }

    /**
     * Initialisiert die benutzerdefinierten Darstellungstypen.
     */
    @SuppressWarnings("OverlyLongMethod")
    private void initUserDefinedDOTs() {
        Preferences classPrefs = getPreferenceStartPath();
        String[] dotSubClasses;
        try {
            dotSubClasses = classPrefs.childrenNames();
        } catch (BackingStoreException e) {
            _debug.error("Die benutzer-definierten Darstellungstypen können nicht initialisiert werden, " + "BackingStoreException: " + e.toString());
            PreferencesDeleter pd = new PreferencesDeleter("Die benutzer-definierten Darstellungstypen können nicht geladen werden.", classPrefs);
            pd.run();
            return;
        }
        for (String subClassName : dotSubClasses) {
            Class<?> c;
            if (subClassName.equals("Classes")) {    // hier steht das Lookup für die vollständigen Klassennamen
                continue;
            }
            c = getClass(subClassName);
            if (c == null) {
                _debug
                    .error("Ein benutzer-definierter Darstellungstyp kann nicht initialisiert werden, " + "ClassNotFoundException: " + subClassName);
                continue;
            }
            Preferences subClassPrefs = classPrefs.node(subClassName);
            String[] dotObjectNames;
            try {
                dotObjectNames = subClassPrefs.childrenNames();
            } catch (BackingStoreException e) {
                _debug.error("Ein benutzer-definierter Darstellungstyp kann nicht initialisiert werden, " + "BackingStoreException: " + e.toString());
                PreferencesDeleter pd = new PreferencesDeleter("Ein benutzer-definierter Darstellungstyp kann nicht geladen werden.", subClassPrefs);
                pd.run();
                continue;
            }
            for (String dotObjectName : dotObjectNames) {
                Preferences objectPrefs = subClassPrefs.node(dotObjectName);
                Object object;
                try {
                    object = c.newInstance();
                } catch (InstantiationException e) {
                    _debug.error(
                        "Ein benutzer-definierter Darstellungstyp kann nicht vollständig initialisiert werden, " + "InstantiationException: " +
                        e.toString());
                    break;
                } catch (IllegalAccessException e) {
                    _debug.error(
                        "Ein benutzer-definierter Darstellungstyp kann nicht vollständig initialisiert werden, " + "IllegalAccessException: " +
                        e.toString());
                    break;
                }
                final DisplayObjectType newDOT = (DisplayObjectType) object;
                newDOT.initializeFromPreferences(objectPrefs);
                put(newDOT);
            }
        }
    }

    /**
     * Gibt die Namen aller Darstellungstypen zurück.
     *
     * @return gibt die Namen aller Darstellungstypen zurück
     */
    public Object[] getDOTNames() {
        return _dotMap.keySet().toArray();
    }

    /**
     * Gibt den Index des ersten Darstellungstypen, dessen Name mit dem übergebenen Zeichen anfängt, zurück. Gibt es einen solchen Darstellungstypen
     * nicht, so wird -1 zurückgegeben.
     *
     * @param c ein Buchstabe
     *
     * @return ein gültiger Index oder -1
     */
    public int getIndexOfFirstDot(final char c) {
        char s = Character.toLowerCase(c);
        for (int index = 0; index < _dotList.size(); ++index) {
            DisplayObjectType type = _dotList.get(index);
            char t = Character.toLowerCase(type.getName().charAt(0));
            if (s == t) {
                return index;
            }
        }
        return -1;
    }

    /*
     * Initialisiert alle Darstellungstypen.
     */
    private void initProgramDefinedDOTs() {
        initProgramDefinedDOTsForLines();
        initProgramDefinedDOTsForPoints();
        initProgramDefinedDOTsForKm();
        initProgramDefinedDOTsForStat();
        initProgramDefinedDOTsForAsbNodes();
        initProgramDefinedDOTsForRn();
    }

    private void initProgramDefinedDOTsForLines() {
        {
            DOTLine dot1 = new DOTLine("Konfigurationslinie schwarz", "Eine einfache schwarze Linie");
            dot1.setPropertyStatic(null, ColorProperty.getInstance(), true);
            dot1.setValueOfStaticProperty(null, ColorProperty.getInstance(), "schwarz");
            dot1.setPropertyStatic(null, DistanceProperty.getInstance(), true);
            dot1.setValueOfStaticProperty(null, DistanceProperty.getInstance(), 0);
            dot1.setPropertyStatic(null, StrokeWidthProperty.getInstance(), true);
            dot1.setValueOfStaticProperty(null, StrokeWidthProperty.getInstance(), 1.);
            put(dot1);
            _unchangeables.add(dot1.getName());
        }
        {
            DOTLine dot2 = new DOTLine("Konfigurationslinie hellgrau", "Eine etwas breitere graue Linie");
            dot2.setPropertyStatic(null, ColorProperty.getInstance(), true);
            dot2.setValueOfStaticProperty(null, ColorProperty.getInstance(), "hellgrau");
            dot2.setPropertyStatic(null, DistanceProperty.getInstance(), true);
            dot2.setValueOfStaticProperty(null, DistanceProperty.getInstance(), 0);
            dot2.setPropertyStatic(null, StrokeWidthProperty.getInstance(), true);
            dot2.setValueOfStaticProperty(null, StrokeWidthProperty.getInstance(), 5.);
            put(dot2);
            _unchangeables.add(dot2.getName());
        }
        {
            DOTLine dot3 = new DOTLine("Störfallzustand OLSIM 1 (grob)", "Zwei Zustände");
            dot3.setPropertyStatic(null, ColorProperty.getInstance(), false);
            DisplayObjectTypeItem dItem1 =
                new DynamicDOTItem("atg.störfallZustand", "asp.störfallVerfahrenOLSIM1", "Situation", "Grün: frei bis dicht", "grün");
            dot3.setValueOfDynamicProperty(null, ColorProperty.getInstance(), dItem1, 2., 4.);
            DisplayObjectTypeItem dItem2 =
                new DynamicDOTItem("atg.störfallZustand", "asp.störfallVerfahrenOLSIM1", "Situation", "Rot: zäh bis Stau", "rot");
            dot3.setValueOfDynamicProperty(null, ColorProperty.getInstance(), dItem2, 5., 7.);
            dot3.setPropertyStatic(null, DistanceProperty.getInstance(), true);
            dot3.setValueOfStaticProperty(null, DistanceProperty.getInstance(), 20);
            dot3.setPropertyStatic(null, StrokeWidthProperty.getInstance(), true);
            dot3.setValueOfStaticProperty(null, StrokeWidthProperty.getInstance(), 5.);
            put(dot3);
            _unchangeables.add(dot3.getName());
        }
        {
            DOTLine dot4 = new DOTLine("Störfallzustand OLSIM 1 (fein)", "Vier Zustände");
            dot4.setPropertyStatic(null, ColorProperty.getInstance(), false);
            DisplayObjectTypeItem dItem1_ =
                new DynamicDOTItem("atg.störfallZustand", "asp.störfallVerfahrenOLSIM1", "Situation", "Grün: frei/lebhaft", "grün");
            dot4.setValueOfDynamicProperty(null, ColorProperty.getInstance(), dItem1_, 2., 3.);
            DisplayObjectTypeItem dItem2_ =
                new DynamicDOTItem("atg.störfallZustand", "asp.störfallVerfahrenOLSIM1", "Situation", "Orange: dicht/zäh", "orange");
            dot4.setValueOfDynamicProperty(null, ColorProperty.getInstance(), dItem2_, 4., 5.);
            DisplayObjectTypeItem dItem3_ =
                new DynamicDOTItem("atg.störfallZustand", "asp.störfallVerfahrenOLSIM1", "Situation", "Rot: stockend/Stau", "rot");
            dot4.setValueOfDynamicProperty(null, ColorProperty.getInstance(), dItem3_, 6., 7.);
            DisplayObjectTypeItem dItem4_ =
                new DynamicDOTItem("atg.störfallZustand", "asp.störfallVerfahrenOLSIM1", "Situation", "Gelb: Störung/unbekannt", "gelb");
            dot4.setValueOfDynamicProperty(null, ColorProperty.getInstance(), dItem4_, 0., 1.);
            dot4.setPropertyStatic(null, DistanceProperty.getInstance(), true);
            dot4.setValueOfStaticProperty(null, DistanceProperty.getInstance(), 20);
            dot4.setPropertyStatic(null, StrokeWidthProperty.getInstance(), true);
            dot4.setValueOfStaticProperty(null, StrokeWidthProperty.getInstance(), 5.);
            put(dot4);
            _unchangeables.add(dot4.getName());
        }
    }

    private void initProgramDefinedDOTsForPoints() {
        addDOTVerkehrsDatenanalyseKurz();
        addAdvancedDOTForDetectionSites();
        addTestDOTForDetectionSites();
    }

    private void addDOTVerkehrsDatenanalyseKurz() {
        DOTPoint newDOT = new DOTPoint("Verkehrsdatenanalyse kurz", "Aspekt asp.agregation1Minute", 0, false);

        // Nur ein einfaches Rechteck
        Map<String, Object> specificInformation1 = new HashMap<>();
        specificInformation1.put("height", 16.);
        specificInformation1.put("width", 16.);
        PrimitiveForm centralRectangle =
            new PrimitiveForm("Rechteck: Verkehrsdatenanalyse kurz", PrimitiveFormType.RECHTECK, "Aspekt asp.agregation1Minute",
                              new Point2D.Double(0, 0), specificInformation1);
        newDOT.addPrimitiveForm(centralRectangle);
        // Properties des zentralen Rechtecks: Füllung (dynamisch, Strichbreite, Tranzparenz (statisch)
        newDOT.setPropertyStatic(centralRectangle.getName(), FillingProperty.getInstance(), false);
        DisplayObjectTypeItem dItem1 =
            new DynamicDOTItem("atg.verkehrsDatenKurzZeitMq", "asp.agregation1Minute", "VKfz.Wert", "Grün: über 80 km/h", "grün");
        newDOT.setValueOfDynamicProperty(centralRectangle.getName(), FillingProperty.getInstance(), dItem1, 80., 255.);
        DisplayObjectTypeItem dItem2 =
            new DynamicDOTItem("atg.verkehrsDatenKurzZeitMq", "asp.agregation1Minute", "VKfz.Wert", "Orange: zwischen 30 und 80 km/h", "orange");
        newDOT.setValueOfDynamicProperty(centralRectangle.getName(), FillingProperty.getInstance(), dItem2, 30., 79.);
        DisplayObjectTypeItem dItem3 =
            new DynamicDOTItem("atg.verkehrsDatenKurzZeitMq", "asp.agregation1Minute", "VKfz.Wert", "Rot: unter 30 km/h", "rot");
        newDOT.setValueOfDynamicProperty(centralRectangle.getName(), FillingProperty.getInstance(), dItem3, 0., 29.);
        newDOT.setPropertyStatic(centralRectangle.getName(), StrokeWidthProperty.getInstance(), true);
        newDOT.setValueOfStaticProperty(centralRectangle.getName(), StrokeWidthProperty.getInstance(), 2.0);
        newDOT.setPropertyStatic(centralRectangle.getName(), TransparencyProperty.getInstance(), true);
        newDOT.setValueOfStaticProperty(centralRectangle.getName(), TransparencyProperty.getInstance(), 20);

        put(newDOT);
        _unchangeables.add(newDOT.getName());
    }

    private void addAdvancedDOTForDetectionSites() {
        DOTPoint newDOT = new DOTPoint("MQ, einfach kombinierte Darstellung", "MQDT", 30., true);

        // Zunächst ein einfaches Rechteck
        Map<String, Object> specificInformation1 = new HashMap<>();
        specificInformation1.put("height", 16.);
        specificInformation1.put("width", 16.);
        PrimitiveForm centralRectangle =
            new PrimitiveForm("Rechteck: Verkehrsdatenanalyse kurz", PrimitiveFormType.RECHTECK, "Aspekt asp.agregation1Minute",
                              new Point2D.Double(0, 0), specificInformation1);
        newDOT.addPrimitiveForm(centralRectangle);
        // Properties: Füllung (dynamisch), Strichbreite, Tranzparenz (statisch)
        newDOT.setPropertyStatic(centralRectangle.getName(), FillingProperty.getInstance(), false);
        DisplayObjectTypeItem dItem1 =
            new DynamicDOTItem("atg.verkehrsDatenKurzZeitMq", "asp.agregation1Minute", "VKfz.Wert", "Grün: über 80 km/h", "grün");
        newDOT.setValueOfDynamicProperty(centralRectangle.getName(), FillingProperty.getInstance(), dItem1, 80., 255.);
        DisplayObjectTypeItem dItem2 =
            new DynamicDOTItem("atg.verkehrsDatenKurzZeitMq", "asp.agregation1Minute", "VKfz.Wert", "Orange: zwischen 30 und 80 km/h", "orange");
        newDOT.setValueOfDynamicProperty(centralRectangle.getName(), FillingProperty.getInstance(), dItem2, 30., 79.);
        DisplayObjectTypeItem dItem3 =
            new DynamicDOTItem("atg.verkehrsDatenKurzZeitMq", "asp.agregation1Minute", "VKfz.Wert", "Rot: unter 30 km/h", "rot");
        newDOT.setValueOfDynamicProperty(centralRectangle.getName(), FillingProperty.getInstance(), dItem3, 0., 29.);
        newDOT.setPropertyStatic(centralRectangle.getName(), StrokeWidthProperty.getInstance(), true);
        newDOT.setValueOfStaticProperty(centralRectangle.getName(), StrokeWidthProperty.getInstance(), 2.0);
        newDOT.setPropertyStatic(centralRectangle.getName(), TransparencyProperty.getInstance(), true);
        newDOT.setValueOfStaticProperty(centralRectangle.getName(), TransparencyProperty.getInstance(), 20);

        // Nun noch ein Halbkreis
        Map<String, Object> specificInformation3 = new HashMap<>();
        specificInformation3.put("radius", 8d);
        specificInformation3.put("orientation", DOTPointPainter.OBERER_HALBKREIS);
        PrimitiveForm semicircle =
            new PrimitiveForm("Halbkreis: Störfallzustand", PrimitiveFormType.HALBKREIS, "Aspekt asp.störfallVerfahrenConstraint",
                              new Point2D.Double(0, 8), specificInformation3);
        newDOT.addPrimitiveForm(semicircle);
        // Properties: Füllung (dynamisch), Strichbreite, Tranzparenz (statisch)
        newDOT.setPropertyStatic(semicircle.getName(), FillingProperty.getInstance(), false);
        DisplayObjectTypeItem dItem1_ =
            new DynamicDOTItem("atg.störfallZustand", "asp.störfallVerfahrenConstraint", "Situation", "Grün: frei", "grün");
        newDOT.setValueOfDynamicProperty(semicircle.getName(), FillingProperty.getInstance(), dItem1_, 2., 4.);
        DisplayObjectTypeItem dItem2_ =
            new DynamicDOTItem("atg.störfallZustand", "asp.störfallVerfahrenConstraint", "Situation", "Rot: gestört", "rot");
        newDOT.setValueOfDynamicProperty(semicircle.getName(), FillingProperty.getInstance(), dItem2_, 5., 7.);
        DisplayObjectTypeItem dItem3_ =
            new DynamicDOTItem("atg.störfallZustand", "asp.störfallVerfahrenConstraint", "Situation", "Grau: unbekannt", "grau");
        newDOT.setValueOfDynamicProperty(semicircle.getName(), FillingProperty.getInstance(), dItem3_, 0., 1.);
        semicircle.setPropertyStatic(StrokeWidthProperty.getInstance(), true);
        semicircle.setValueOfStaticProperty(StrokeWidthProperty.getInstance(), 2.0);
        semicircle.setPropertyStatic(TransparencyProperty.getInstance(), true);
        semicircle.setValueOfStaticProperty(TransparencyProperty.getInstance(), 20);

        put(newDOT);
        _unchangeables.add(newDOT.getName());

    }

    @SuppressWarnings("OverlyLongMethod")
    private void addTestDOTForDetectionSites() {
        DOTPoint newDOT = new DOTPoint("MQ, Testdarstellung 1", "MQDT 1", 70., true);
        // Der originäre Punkt
        Map<String, Object> specificInformation0 = new HashMap<>();
        PrimitiveForm point = new PrimitiveForm("Der Punkt", PrimitiveFormType.PUNKT, "Verortung", new Point2D.Double(0., 0.), specificInformation0);
        // Properties des Punktes: Durchmesser und Farbe, hier beide statisch
        newDOT.addPrimitiveForm(point);
        newDOT.setPropertyStatic(point.getName(), DiameterProperty.getInstance(), true);
        newDOT.setValueOfStaticProperty(point.getName(), DiameterProperty.getInstance(), 5.);
        newDOT.setPropertyStatic(point.getName(), ColorProperty.getInstance(), true);
        newDOT.setValueOfStaticProperty(point.getName(), ColorProperty.getInstance(), Color.DARK_GRAY);

        // Das zentrale Rechteck
        Map<String, Object> specificInformation1 = new HashMap<>();
        specificInformation1.put("height", 30.);
        specificInformation1.put("width", 30.);
        PrimitiveForm centralRectangle =
            new PrimitiveForm("Zentrales Rechteck", PrimitiveFormType.RECHTECK, "LOS-Rechteck", new Point2D.Double(0, 0), specificInformation1);
        newDOT.addPrimitiveForm(centralRectangle);
        // Properties des zentralen Rechtecks: Füllung (dynamisch, Strichbreite, Tranzparenz (statisch)
        newDOT.setPropertyStatic(centralRectangle.getName(), FillingProperty.getInstance(), false);
        DisplayObjectTypeItem dItem1_ = new DynamicDOTItem("atg.störfallZustand", "asp.störfallVerfahrenMARZ", "Situation", "Grün: frei", "grün");
        newDOT.setValueOfDynamicProperty(centralRectangle.getName(), FillingProperty.getInstance(), dItem1_, 2., 4.);
        DisplayObjectTypeItem dItem2_ = new DynamicDOTItem("atg.störfallZustand", "asp.störfallVerfahrenMARZ", "Situation", "Rot: Stau", "ror");
        newDOT.setValueOfDynamicProperty(centralRectangle.getName(), FillingProperty.getInstance(), dItem2_, 5., 7.);
        DisplayObjectTypeItem dItem3_ =
            new DynamicDOTItem("atg.störfallZustand", "asp.störfallVerfahrenMARZ", "Situation", "Grau: Störung/unbekannt", "grau");
        newDOT.setValueOfDynamicProperty(centralRectangle.getName(), FillingProperty.getInstance(), dItem3_, 0., 1.);

        newDOT.setPropertyStatic(centralRectangle.getName(), StrokeWidthProperty.getInstance(), true);
        newDOT.setValueOfStaticProperty(centralRectangle.getName(), StrokeWidthProperty.getInstance(), 2.0);
        newDOT.setPropertyStatic(centralRectangle.getName(), TransparencyProperty.getInstance(), true);
        newDOT.setValueOfStaticProperty(centralRectangle.getName(), TransparencyProperty.getInstance(), 20);

        // Der Kreis neben dem Rechteck
        Map<String, Object> specificInformation2 = new HashMap<>();
        specificInformation2.put("radius", 10d);
        PrimitiveForm circle = new PrimitiveForm("Ein Kreis", PrimitiveFormType.KREIS, "Güte-Kreis", new Point2D.Double(25, 5), specificInformation2);
        newDOT.addPrimitiveForm(circle);
        // Properties des Kreises: Füllung, Strichbreite, Tranzparenz (erstmal statisch)
        newDOT.setPropertyStatic(circle.getName(), FillingProperty.getInstance(), true);
        newDOT.setValueOfStaticProperty(circle.getName(), FillingProperty.getInstance(), Color.GREEN);
        newDOT.setPropertyStatic(circle.getName(), StrokeWidthProperty.getInstance(), true);
        newDOT.setValueOfStaticProperty(circle.getName(), StrokeWidthProperty.getInstance(), 1.0);
        newDOT.setPropertyStatic(circle.getName(), TransparencyProperty.getInstance(), true);
        newDOT.setValueOfStaticProperty(circle.getName(), TransparencyProperty.getInstance(), 60);
        // Der Halbkreis über dem Rechteck
        Map<String, Object> specificInformation3 = new HashMap<>();
        specificInformation3.put("radius", 15d);
        specificInformation3.put("orientation", DOTPointPainter.OBERER_HALBKREIS);
        PrimitiveForm semicircle =
            new PrimitiveForm("Ein Halbkreis", PrimitiveFormType.HALBKREIS, "Spannungshalbkreis", new Point2D.Double(0, 15), specificInformation3);
        newDOT.addPrimitiveForm(semicircle);
        // Properties des Halbkreises: Füllung, Strichbreite, Tranzparenz (erstmal statisch)
        newDOT.setPropertyStatic(semicircle.getName(), FillingProperty.getInstance(), true);
        newDOT.setValueOfStaticProperty(semicircle.getName(), FillingProperty.getInstance(), Color.BLUE);
        newDOT.setPropertyStatic(semicircle.getName(), StrokeWidthProperty.getInstance(), true);
        newDOT.setValueOfStaticProperty(semicircle.getName(), StrokeWidthProperty.getInstance(), 2.0);
        newDOT.setPropertyStatic(semicircle.getName(), TransparencyProperty.getInstance(), true);
        newDOT.setValueOfStaticProperty(semicircle.getName(), TransparencyProperty.getInstance(), 0);
        // Der Text
        Map<String, Object> specificInformation4 = new HashMap<>();
        PrimitiveForm text =
            new PrimitiveForm("Name, PID oder Id", PrimitiveFormType.TEXTDARSTELLUNG, "Name", new Point2D.Double(-15, -35), specificInformation4);
        newDOT.addPrimitiveForm(text);
        // Properties des Textes: Text (dynamisch), Farbe, Größe und Textstil (statisch)
        newDOT.setPropertyStatic(text.getName(), TextProperty.getInstance(), false);
        DisplayObjectTypeItem textItem =
            new DynamicDOTItem("atg.verkehrsDatenKurzZeitMq", "asp.agregation1Minute", "VKfz.Wert", "VekehrsdatenKurzZeitMq",
                               DOTPointPainter.DYNAMIC_ATTRIBUTE_SCALED);
        newDOT.setValueOfDynamicProperty(text.getName(), TextProperty.getInstance(), textItem, 0., 255.);

        newDOT.setPropertyStatic(text.getName(), ColorProperty.getInstance(), true);
        newDOT.setValueOfStaticProperty(text.getName(), ColorProperty.getInstance(), Color.BLUE);
        newDOT.setPropertyStatic(text.getName(), TextSizeProperty.getInstance(), true);
        newDOT.setValueOfStaticProperty(text.getName(), TextSizeProperty.getInstance(), 20);
        newDOT.setPropertyStatic(text.getName(), TextStyleProperty.getInstance(), true);
        newDOT.setValueOfStaticProperty(text.getName(), TextStyleProperty.getInstance(), Font.ITALIC);
        put(newDOT);
        _unchangeables.add(newDOT.getName());
    }

    @SuppressWarnings("OverlyLongMethod")
    private void initProgramDefinedDOTsForKm() {
        {
            DOTKm km10 = new DOTKm("Betriebskilometrierung 10 m", "Alle 10 Meter eingeblendet");
            km10.setPropertyStatic(null, DistanceRasterProperty.getInstance(), true);
            km10.setValueOfStaticProperty(null, DistanceRasterProperty.getInstance(), DistanceRasterType.TEN);
            km10.setTranslationFactor(12.);
            km10.setJoinByLine(true);
            km10.setPropertyStatic(null, ColorProperty.getInstance(), true);
            km10.setValueOfStaticProperty(null, ColorProperty.getInstance(), "schwarz");
            km10.setPropertyStatic(null, TextSizeProperty.getInstance(), true);
            km10.setValueOfStaticProperty(null, TextSizeProperty.getInstance(), 12);
            km10.setPropertyStatic(null, KmFormatProperty.getInstance(), true);
            km10.setValueOfStaticProperty(null, KmFormatProperty.getInstance(), KmFormat.KM_ONLY);
            km10.setPropertyStatic(null, TextStyleProperty.getInstance(), true);
            km10.setValueOfStaticProperty(null, TextStyleProperty.getInstance(), Font.BOLD);
            put(km10);
            _unchangeables.add(km10.getName());
        }
        {
            DOTKm km50 = new DOTKm("Betriebskilometrierung 50 m", "Alle 50 Meter eingeblendet");
            km50.setPropertyStatic(null, DistanceRasterProperty.getInstance(), true);
            km50.setValueOfStaticProperty(null, DistanceRasterProperty.getInstance(), DistanceRasterType.FIFTY);
            km50.setTranslationFactor(18.);
            km50.setJoinByLine(true);
            km50.setPropertyStatic(null, ColorProperty.getInstance(), true);
            km50.setValueOfStaticProperty(null, ColorProperty.getInstance(), "schwarz");
            km50.setPropertyStatic(null, TextSizeProperty.getInstance(), true);
            km50.setValueOfStaticProperty(null, TextSizeProperty.getInstance(), 12);
            km50.setPropertyStatic(null, KmFormatProperty.getInstance(), true);
            km50.setValueOfStaticProperty(null, KmFormatProperty.getInstance(), KmFormat.KM_ONLY);
            km50.setPropertyStatic(null, DottingProperty.getInstance(), true);
            km50.setValueOfStaticProperty(null, DottingProperty.getInstance(), DistanceRasterType.TEN);
            km50.setPropertyStatic(null, TextStyleProperty.getInstance(), true);
            km50.setValueOfStaticProperty(null, TextStyleProperty.getInstance(), Font.BOLD);
            put(km50);
            _unchangeables.add(km50.getName());
        }
        {
            DOTKm km100 = new DOTKm("Betriebskilometrierung 100 m", "Alle 100 Meter eingeblendet");
            km100.setPropertyStatic(null, DistanceRasterProperty.getInstance(), true);
            km100.setValueOfStaticProperty(null, DistanceRasterProperty.getInstance(), DistanceRasterType.ONE_HUNDRED);
            km100.setTranslationFactor(40.);
            km100.setJoinByLine(true);
            km100.setPropertyStatic(null, ColorProperty.getInstance(), true);
            km100.setValueOfStaticProperty(null, ColorProperty.getInstance(), "schwarz");
            km100.setPropertyStatic(null, TextSizeProperty.getInstance(), true);
            km100.setValueOfStaticProperty(null, TextSizeProperty.getInstance(), 12);
            km100.setPropertyStatic(null, KmFormatProperty.getInstance(), true);
            km100.setValueOfStaticProperty(null, KmFormatProperty.getInstance(), KmFormat.KM_ONLY);
            km100.setPropertyStatic(null, DottingProperty.getInstance(), true);
            km100.setValueOfStaticProperty(null, DottingProperty.getInstance(), DistanceRasterType.FIFTY);
            km100.setPropertyStatic(null, TextStyleProperty.getInstance(), true);
            km100.setValueOfStaticProperty(null, TextStyleProperty.getInstance(), Font.BOLD);
            put(km100);
            _unchangeables.add(km100.getName());
        }
        {
            DOTKm km500 = new DOTKm("Betriebskilometrierung 500 m", "Alle 500 Meter eingeblendet");
            km500.setPropertyStatic(null, DistanceRasterProperty.getInstance(), true);
            km500.setValueOfStaticProperty(null, DistanceRasterProperty.getInstance(), DistanceRasterType.FIVE_HUNDRED);
            km500.setTranslationFactor(60.);
            km500.setJoinByLine(true);
            km500.setPropertyStatic(null, ColorProperty.getInstance(), true);
            km500.setValueOfStaticProperty(null, ColorProperty.getInstance(), "schwarz");
            km500.setPropertyStatic(null, TextSizeProperty.getInstance(), true);
            km500.setValueOfStaticProperty(null, TextSizeProperty.getInstance(), 12);
            km500.setPropertyStatic(null, KmFormatProperty.getInstance(), true);
            km500.setValueOfStaticProperty(null, KmFormatProperty.getInstance(), KmFormat.KM_ONLY);
            km500.setPropertyStatic(null, DottingProperty.getInstance(), true);
            km500.setValueOfStaticProperty(null, DottingProperty.getInstance(), DistanceRasterType.ONE_HUNDRED);
            km500.setPropertyStatic(null, TextStyleProperty.getInstance(), true);
            km500.setValueOfStaticProperty(null, TextStyleProperty.getInstance(), Font.BOLD);
            put(km500);
            _unchangeables.add(km500.getName());
        }
        {
            DOTKm km1000 = new DOTKm("Betriebskilometrierung 1 km", "Alle 1000 Meter eingeblendet");
            km1000.setPropertyStatic(null, DistanceRasterProperty.getInstance(), true);
            km1000.setValueOfStaticProperty(null, DistanceRasterProperty.getInstance(), DistanceRasterType.ONE_THOUSAND);
            km1000.setTranslationFactor(80.);
            km1000.setJoinByLine(true);
            km1000.setPropertyStatic(null, ColorProperty.getInstance(), true);
            km1000.setValueOfStaticProperty(null, ColorProperty.getInstance(), "schwarz");
            km1000.setPropertyStatic(null, TextSizeProperty.getInstance(), true);
            km1000.setValueOfStaticProperty(null, TextSizeProperty.getInstance(), 12);
            km1000.setPropertyStatic(null, KmFormatProperty.getInstance(), true);
            km1000.setValueOfStaticProperty(null, KmFormatProperty.getInstance(), KmFormat.KM_ONLY);
            km1000.setPropertyStatic(null, DottingProperty.getInstance(), true);
            km1000.setValueOfStaticProperty(null, DottingProperty.getInstance(), DistanceRasterType.FIVE_HUNDRED);
            km1000.setPropertyStatic(null, TextStyleProperty.getInstance(), true);
            km1000.setValueOfStaticProperty(null, TextStyleProperty.getInstance(), Font.BOLD);
            put(km1000);
            _unchangeables.add(km1000.getName());
        }
        {
            DOTKm km5000 = new DOTKm("Betriebskilometrierung 5 km", "Alle 5000 Meter eingeblendet");
            km5000.setPropertyStatic(null, DistanceRasterProperty.getInstance(), true);
            km5000.setValueOfStaticProperty(null, DistanceRasterProperty.getInstance(), DistanceRasterType.FIVE_THOUSAND);
            km5000.setTranslationFactor(120.);
            km5000.setJoinByLine(true);
            km5000.setPropertyStatic(null, ColorProperty.getInstance(), true);
            km5000.setValueOfStaticProperty(null, ColorProperty.getInstance(), "dunkelgrau");
            km5000.setPropertyStatic(null, TextSizeProperty.getInstance(), true);
            km5000.setValueOfStaticProperty(null, TextSizeProperty.getInstance(), 12);
            km5000.setPropertyStatic(null, KmFormatProperty.getInstance(), true);
            km5000.setValueOfStaticProperty(null, KmFormatProperty.getInstance(), KmFormat.KM_ONLY);
            km5000.setPropertyStatic(null, DottingProperty.getInstance(), true);
            km5000.setValueOfStaticProperty(null, DottingProperty.getInstance(), DistanceRasterType.ONE_THOUSAND);
            km5000.setPropertyStatic(null, TextStyleProperty.getInstance(), true);
            km5000.setValueOfStaticProperty(null, TextStyleProperty.getInstance(), Font.BOLD);
            put(km5000);
            _unchangeables.add(km5000.getName());
        }
        {
            DOTKm km10000 = new DOTKm("Betriebskilometrierung 10 km", "Alle 10000 Meter eingeblendet");
            km10000.setPropertyStatic(null, DistanceRasterProperty.getInstance(), true);
            km10000.setValueOfStaticProperty(null, DistanceRasterProperty.getInstance(), DistanceRasterType.TEN_THOUSAND);
//			km10000.setTranslationFactor(120.);
            km10000.setPropertyStatic(null, ColorProperty.getInstance(), true);
            km10000.setValueOfStaticProperty(null, ColorProperty.getInstance(), "dunkelgrau");
            km10000.setPropertyStatic(null, TextSizeProperty.getInstance(), true);
            km10000.setValueOfStaticProperty(null, TextSizeProperty.getInstance(), 12);
            km10000.setPropertyStatic(null, KmFormatProperty.getInstance(), true);
            km10000.setValueOfStaticProperty(null, KmFormatProperty.getInstance(), KmFormat.KM_ONLY);
            km10000.setPropertyStatic(null, DottingProperty.getInstance(), true);
            km10000.setValueOfStaticProperty(null, DottingProperty.getInstance(), DistanceRasterType.FIVE_THOUSAND);
            km10000.setPropertyStatic(null, TextStyleProperty.getInstance(), true);
            km10000.setValueOfStaticProperty(null, TextStyleProperty.getInstance(), Font.PLAIN);
            put(km10000);
            _unchangeables.add(km10000.getName());
        }
        {
            DOTKm km50000 = new DOTKm("Betriebskilometrierung 50 km", "Alle 50000 Meter eingeblendet");
            km50000.setPropertyStatic(null, DistanceRasterProperty.getInstance(), true);
            km50000.setValueOfStaticProperty(null, DistanceRasterProperty.getInstance(), DistanceRasterType.FIFTY_THOUSAND);
//			km50000.setTranslationFactor(150.);
            km50000.setPropertyStatic(null, ColorProperty.getInstance(), true);
            km50000.setValueOfStaticProperty(null, ColorProperty.getInstance(), "dunkelgrau");
            km50000.setPropertyStatic(null, TextSizeProperty.getInstance(), true);
            km50000.setValueOfStaticProperty(null, TextSizeProperty.getInstance(), 12);
            km50000.setPropertyStatic(null, KmFormatProperty.getInstance(), true);
            km50000.setValueOfStaticProperty(null, KmFormatProperty.getInstance(), KmFormat.RN_AND_KM);
            km50000.setPropertyStatic(null, DottingProperty.getInstance(), true);
            km50000.setValueOfStaticProperty(null, DottingProperty.getInstance(), DistanceRasterType.TEN_THOUSAND);
            km50000.setPropertyStatic(null, TextStyleProperty.getInstance(), true);
            km50000.setValueOfStaticProperty(null, TextStyleProperty.getInstance(), Font.PLAIN);
            put(km50000);
            _unchangeables.add(km50000.getName());
        }
    }

    @SuppressWarnings("OverlyLongMethod")
    private void initProgramDefinedDOTsForStat() {
        {
            DOTStat stat10 = new DOTStat("ASB-Stationierung 10 m", "Alle 10 Meter eingeblendet");
            stat10.setPropertyStatic(null, DistanceRasterProperty.getInstance(), true);
            stat10.setValueOfStaticProperty(null, DistanceRasterProperty.getInstance(), DistanceRasterType.TEN);
            stat10.setTranslationFactor(12.);
            stat10.setJoinByLine(true);
            stat10.setPropertyStatic(null, ColorProperty.getInstance(), true);
            stat10.setValueOfStaticProperty(null, ColorProperty.getInstance(), "blau");
            stat10.setPropertyStatic(null, TextSizeProperty.getInstance(), true);
            stat10.setValueOfStaticProperty(null, TextSizeProperty.getInstance(), 12);
            stat10.setPropertyStatic(null, TextStyleProperty.getInstance(), true);
            stat10.setValueOfStaticProperty(null, TextStyleProperty.getInstance(), Font.BOLD);
            stat10.setPropertyStatic(null, StatFormatProperty.getInstance(), true);
            stat10.setValueOfStaticProperty(null, StatFormatProperty.getInstance(), StatFormat.STAT_ONLY);
            put(stat10);
            _unchangeables.add(stat10.getName());
        }
        {
            DOTStat stat50 = new DOTStat("ASB-Stationierung 50 m", "Alle 50 Meter eingeblendet");
            stat50.setPropertyStatic(null, DistanceRasterProperty.getInstance(), true);
            stat50.setValueOfStaticProperty(null, DistanceRasterProperty.getInstance(), DistanceRasterType.FIFTY);
            stat50.setTranslationFactor(18.);
            stat50.setJoinByLine(true);
            stat50.setPropertyStatic(null, ColorProperty.getInstance(), true);
            stat50.setValueOfStaticProperty(null, ColorProperty.getInstance(), "blau");
            stat50.setPropertyStatic(null, TextSizeProperty.getInstance(), true);
            stat50.setValueOfStaticProperty(null, TextSizeProperty.getInstance(), 12);
            stat50.setPropertyStatic(null, TextStyleProperty.getInstance(), true);
            stat50.setValueOfStaticProperty(null, TextStyleProperty.getInstance(), Font.BOLD);
            stat50.setPropertyStatic(null, StatFormatProperty.getInstance(), true);
            stat50.setValueOfStaticProperty(null, StatFormatProperty.getInstance(), StatFormat.STAT_ONLY);
            stat50.setPropertyStatic(null, DottingProperty.getInstance(), true);
            stat50.setValueOfStaticProperty(null, DottingProperty.getInstance(), DistanceRasterType.TEN);
            put(stat50);
            _unchangeables.add(stat50.getName());
        }
        {
            DOTStat stat100 = new DOTStat("ASB-Stationierung 100 m", "Alle 100 Meter eingeblendet");
            stat100.setPropertyStatic(null, DistanceRasterProperty.getInstance(), true);
            stat100.setValueOfStaticProperty(null, DistanceRasterProperty.getInstance(), DistanceRasterType.ONE_HUNDRED);
            stat100.setTranslationFactor(40.);
            stat100.setJoinByLine(true);
            stat100.setPropertyStatic(null, ColorProperty.getInstance(), true);
            stat100.setValueOfStaticProperty(null, ColorProperty.getInstance(), "blau");
            stat100.setPropertyStatic(null, TextSizeProperty.getInstance(), true);
            stat100.setValueOfStaticProperty(null, TextSizeProperty.getInstance(), 12);
            stat100.setPropertyStatic(null, TextStyleProperty.getInstance(), true);
            stat100.setValueOfStaticProperty(null, TextStyleProperty.getInstance(), Font.BOLD);
            stat100.setPropertyStatic(null, StatFormatProperty.getInstance(), true);
            stat100.setValueOfStaticProperty(null, StatFormatProperty.getInstance(), StatFormat.STAT_ONLY);
            stat100.setPropertyStatic(null, DottingProperty.getInstance(), true);
            stat100.setValueOfStaticProperty(null, DottingProperty.getInstance(), DistanceRasterType.FIFTY);
            put(stat100);
            _unchangeables.add(stat100.getName());
        }
        {
            DOTStat stat500 = new DOTStat("ASB-Stationierung 500 m", "Alle 500 Meter eingeblendet");
            stat500.setPropertyStatic(null, DistanceRasterProperty.getInstance(), true);
            stat500.setValueOfStaticProperty(null, DistanceRasterProperty.getInstance(), DistanceRasterType.FIVE_HUNDRED);
            stat500.setTranslationFactor(60.);
            stat500.setJoinByLine(true);
            stat500.setPropertyStatic(null, ColorProperty.getInstance(), true);
            stat500.setValueOfStaticProperty(null, ColorProperty.getInstance(), "blau");
            stat500.setPropertyStatic(null, TextSizeProperty.getInstance(), true);
            stat500.setValueOfStaticProperty(null, TextSizeProperty.getInstance(), 12);
            stat500.setPropertyStatic(null, TextStyleProperty.getInstance(), true);
            stat500.setValueOfStaticProperty(null, TextStyleProperty.getInstance(), Font.BOLD);
            stat500.setPropertyStatic(null, StatFormatProperty.getInstance(), true);
            stat500.setValueOfStaticProperty(null, StatFormatProperty.getInstance(), StatFormat.STAT_ONLY);
            stat500.setPropertyStatic(null, DottingProperty.getInstance(), true);
            stat500.setValueOfStaticProperty(null, DottingProperty.getInstance(), DistanceRasterType.ONE_HUNDRED);
            put(stat500);
            _unchangeables.add(stat500.getName());
        }
        {
            DOTStat stat1000 = new DOTStat("ASB-Stationierung 1 km", "Alle 1000 Meter eingeblendet");
            stat1000.setPropertyStatic(null, DistanceRasterProperty.getInstance(), true);
            stat1000.setValueOfStaticProperty(null, DistanceRasterProperty.getInstance(), DistanceRasterType.ONE_THOUSAND);
            stat1000.setTranslationFactor(80.);
            stat1000.setJoinByLine(true);
            stat1000.setPropertyStatic(null, ColorProperty.getInstance(), true);
            stat1000.setValueOfStaticProperty(null, ColorProperty.getInstance(), "blau");
            stat1000.setPropertyStatic(null, TextSizeProperty.getInstance(), true);
            stat1000.setValueOfStaticProperty(null, TextSizeProperty.getInstance(), 12);
            stat1000.setPropertyStatic(null, TextStyleProperty.getInstance(), true);
            stat1000.setValueOfStaticProperty(null, TextStyleProperty.getInstance(), Font.BOLD);
            stat1000.setPropertyStatic(null, StatFormatProperty.getInstance(), true);
            stat1000.setValueOfStaticProperty(null, StatFormatProperty.getInstance(), StatFormat.STAT_ONLY);
            stat1000.setPropertyStatic(null, DottingProperty.getInstance(), true);
            stat1000.setValueOfStaticProperty(null, DottingProperty.getInstance(), DistanceRasterType.FIVE_HUNDRED);
            put(stat1000);
            _unchangeables.add(stat1000.getName());
        }
        {
            DOTStat stat5000 = new DOTStat("ASB-Stationierung 5 km", "Alle 5000 Meter eingeblendet");
            stat5000.setPropertyStatic(null, DistanceRasterProperty.getInstance(), true);
            stat5000.setValueOfStaticProperty(null, DistanceRasterProperty.getInstance(), DistanceRasterType.FIVE_THOUSAND);
            stat5000.setTranslationFactor(120.);
            stat5000.setJoinByLine(true);
            stat5000.setPropertyStatic(null, ColorProperty.getInstance(), true);
            stat5000.setValueOfStaticProperty(null, ColorProperty.getInstance(), "blau");
            stat5000.setPropertyStatic(null, TextSizeProperty.getInstance(), true);
            stat5000.setValueOfStaticProperty(null, TextSizeProperty.getInstance(), 12);
            stat5000.setPropertyStatic(null, TextStyleProperty.getInstance(), true);
            stat5000.setValueOfStaticProperty(null, TextStyleProperty.getInstance(), Font.BOLD);
            stat5000.setPropertyStatic(null, StatFormatProperty.getInstance(), true);
            stat5000.setValueOfStaticProperty(null, StatFormatProperty.getInstance(), StatFormat.STAT_ONLY);
            stat5000.setPropertyStatic(null, DottingProperty.getInstance(), true);
            stat5000.setValueOfStaticProperty(null, DottingProperty.getInstance(), DistanceRasterType.ONE_THOUSAND);
            put(stat5000);
            _unchangeables.add(stat5000.getName());
        }
        {
            DOTStat stat10000 = new DOTStat("ASB-Stationierung 10 km", "Alle 10000 Meter eingeblendet");
            stat10000.setPropertyStatic(null, DistanceRasterProperty.getInstance(), true);
            stat10000.setValueOfStaticProperty(null, DistanceRasterProperty.getInstance(), DistanceRasterType.TEN_THOUSAND);
//			stat10000.setTranslationFactor(250.);
//			stat10000.setJoinByLine(true);
            stat10000.setPropertyStatic(null, ColorProperty.getInstance(), true);
            stat10000.setValueOfStaticProperty(null, ColorProperty.getInstance(), "blau");
            stat10000.setPropertyStatic(null, TextSizeProperty.getInstance(), true);
            stat10000.setValueOfStaticProperty(null, TextSizeProperty.getInstance(), 12);
            stat10000.setPropertyStatic(null, TextStyleProperty.getInstance(), true);
            stat10000.setValueOfStaticProperty(null, TextStyleProperty.getInstance(), Font.PLAIN);
            stat10000.setPropertyStatic(null, StatFormatProperty.getInstance(), true);
            stat10000.setValueOfStaticProperty(null, StatFormatProperty.getInstance(), StatFormat.STAT_ONLY);
            stat10000.setPropertyStatic(null, DottingProperty.getInstance(), true);
            stat10000.setValueOfStaticProperty(null, DottingProperty.getInstance(), DistanceRasterType.FIVE_THOUSAND);
            put(stat10000);
            _unchangeables.add(stat10000.getName());
        }
        {
            DOTStat stat50000 = new DOTStat("ASB-Stationierung 50 km", "Alle 50000 Meter eingeblendet");
            stat50000.setPropertyStatic(null, DistanceRasterProperty.getInstance(), true);
            stat50000.setValueOfStaticProperty(null, DistanceRasterProperty.getInstance(), DistanceRasterType.TEN_THOUSAND);
//			stat50000.setTranslationFactor(250.);
//			stat50000.setJoinByLine(true);
            stat50000.setPropertyStatic(null, ColorProperty.getInstance(), true);
            stat50000.setValueOfStaticProperty(null, ColorProperty.getInstance(), "blau");
            stat50000.setPropertyStatic(null, TextSizeProperty.getInstance(), true);
            stat50000.setValueOfStaticProperty(null, TextSizeProperty.getInstance(), 12);
            stat50000.setPropertyStatic(null, TextStyleProperty.getInstance(), true);
            stat50000.setValueOfStaticProperty(null, TextStyleProperty.getInstance(), Font.PLAIN);
            stat50000.setPropertyStatic(null, StatFormatProperty.getInstance(), true);
            stat50000.setValueOfStaticProperty(null, StatFormatProperty.getInstance(), StatFormat.STAT_ONLY);
            stat50000.setPropertyStatic(null, DottingProperty.getInstance(), true);
            stat50000.setValueOfStaticProperty(null, DottingProperty.getInstance(), DistanceRasterType.TEN_THOUSAND);
            put(stat50000);
            _unchangeables.add(stat50000.getName());
        }
    }

    private void initProgramDefinedDOTsForAsbNodes() {
        {
            DOTAsbNode asbNode0 = new DOTAsbNode("ASB-Knotennummern V0", "Zeigt die ASB-Knotennummern");
            asbNode0.setTranslationFactor(200.);
            asbNode0.setJoinByLine(true);
            asbNode0.setPropertyStatic(null, ColorProperty.getInstance(), true);
            asbNode0.setValueOfStaticProperty(null, ColorProperty.getInstance(), "blau");
            asbNode0.setPropertyStatic(null, TextSizeProperty.getInstance(), true);
            asbNode0.setValueOfStaticProperty(null, TextSizeProperty.getInstance(), 12);
            asbNode0.setPropertyStatic(null, TextStyleProperty.getInstance(), true);
            asbNode0.setValueOfStaticProperty(null, TextStyleProperty.getInstance(), Font.ITALIC);
            put(asbNode0);
            _unchangeables.add(asbNode0.getName());
        }
        {
            DOTAsbNode asbNode1 = new DOTAsbNode("ASB-Knotennummern V1", "Zeigt die ASB-Knotennummern");
            asbNode1.setTranslationFactor(400.);
            asbNode1.setJoinByLine(true);
            asbNode1.setPropertyStatic(null, ColorProperty.getInstance(), true);
            asbNode1.setValueOfStaticProperty(null, ColorProperty.getInstance(), "blau");
            asbNode1.setPropertyStatic(null, TextSizeProperty.getInstance(), true);
            asbNode1.setValueOfStaticProperty(null, TextSizeProperty.getInstance(), 12);
            asbNode1.setPropertyStatic(null, TextStyleProperty.getInstance(), true);
            asbNode1.setValueOfStaticProperty(null, TextStyleProperty.getInstance(), Font.ITALIC);
            put(asbNode1);
            _unchangeables.add(asbNode1.getName());
        }
        {
            DOTAsbNode asbNode2 = new DOTAsbNode("ASB-Knotennummern V2", "Zeigt die ASB-Knotennummern");
            asbNode2.setTranslationFactor(400.);
            asbNode2.setJoinByLine(false);
            asbNode2.setPropertyStatic(null, ColorProperty.getInstance(), true);
            asbNode2.setValueOfStaticProperty(null, ColorProperty.getInstance(), "blau");
            asbNode2.setPropertyStatic(null, TextSizeProperty.getInstance(), true);
            asbNode2.setValueOfStaticProperty(null, TextSizeProperty.getInstance(), 12);
            asbNode2.setPropertyStatic(null, TextStyleProperty.getInstance(), true);
            asbNode2.setValueOfStaticProperty(null, TextStyleProperty.getInstance(), Font.ITALIC);
            put(asbNode2);
            _unchangeables.add(asbNode2.getName());
        }
    }

    private void initProgramDefinedDOTsForRn() {
        {
            DOTRn dotRn0 = new DOTRn("Autobahnschilder Kategorie 0", "Zeigt die Autobahnschilder", RnDisplayObject.Category.NULL);
            dotRn0.setPropertyStatic(null, RnFormatProperty.getInstance(), true);
            dotRn0.setValueOfStaticProperty(null, RnFormatProperty.getInstance(), RnFormat.RN_SMALL);
            put(dotRn0);
            _unchangeables.add(dotRn0.getName());
        }
        {
            DOTRn dotRn1 = new DOTRn("Autobahnschilder Kategorie 0 bis 1", "Zeigt die Autobahnschilder", RnDisplayObject.Category.ONE);
            dotRn1.setPropertyStatic(null, RnFormatProperty.getInstance(), true);
            dotRn1.setValueOfStaticProperty(null, RnFormatProperty.getInstance(), RnFormat.RN_SMALL);
            put(dotRn1);
            _unchangeables.add(dotRn1.getName());
        }
        {
            DOTRn dotRn2 = new DOTRn("Autobahnschilder Kategorie 0 bis 2", "Zeigt die Autobahnschilder", RnDisplayObject.Category.TWO);
            dotRn2.setPropertyStatic(null, RnFormatProperty.getInstance(), true);
            dotRn2.setValueOfStaticProperty(null, RnFormatProperty.getInstance(), RnFormat.RN_SMALL);
            put(dotRn2);
            _unchangeables.add(dotRn2.getName());
        }
        {
            DOTRn dotRn3 = new DOTRn("Autobahnschilder Kategorie 0 bis 3", "Zeigt die Autobahnschilder", RnDisplayObject.Category.THREE);
            dotRn3.setPropertyStatic(null, RnFormatProperty.getInstance(), true);
            dotRn3.setValueOfStaticProperty(null, RnFormatProperty.getInstance(), RnFormat.RN_SMALL);
            put(dotRn3);
            _unchangeables.add(dotRn3.getName());
        }
        {
            DOTRn dotRn4 = new DOTRn("Autobahnschilder Kategorie 0 bis 4", "Zeigt die Autobahnschilder", RnDisplayObject.Category.FOUR);
            dotRn4.setPropertyStatic(null, RnFormatProperty.getInstance(), true);
            dotRn4.setValueOfStaticProperty(null, RnFormatProperty.getInstance(), RnFormat.RN_SMALL);
            put(dotRn4);
            _unchangeables.add(dotRn4.getName());
        }
    }

    /**
     * Gibt {@code true} zurück, wenn der Darstellungstyp veränderbar ist, was genau dann der Fall ist, wenn er nicht im Programmcode, sondern von
     * einem Benutzer definiert wurde, und {@code false} sonst.
     *
     * @param dot ein Darstellungstyp
     *
     * @return {@code true} genau dann, wenn der Darstellungstyp veränderbar ist
     */
    public boolean isChangeable(DisplayObjectType dot) {
        return !_unchangeables.contains(dot.getName());
    }

    private void put(DisplayObjectType dot) {
        remove(dot);
        _dotList.add(dot);
        _dotMap.put(dot.getName(), dot);
    }

    private void remove(DisplayObjectType dot) {
        if (_dotMap.remove(dot.getName()) != null) {
            int index = 0;
            for (DisplayObjectType dType : _dotList) {
                if (dType.getName().equals(dot.getName())) {
                    _dotList.remove(index);
                    break;
                }
                index++;
            }
        }
        if (_dotMap.size() != _dotList.size()) {
            _debug.warning("Der DOTManager ist intern durcheinander!");
        }
    }

    /*
     * Gehört zur Implementaion des TableModel und gibt den Spaltennamen der Spalte mit dem übergebenen Index zurück.
     */
    @Override
    public String getColumnName(int columnIndex) {
        return _columnNames[columnIndex];
    }

    /*
     * Gehört zur Implementaion des TableModel und gibt die Anzahl der Spalten an.
     */
    @Override
    public int getColumnCount() {
        return _columnNames.length;
    }

    /*
     * Gehört zur Implementaion des TableModel und gibt die Anzahl der Zeilen an.
     */
    @Override
    public int getRowCount() {
        return _dotList.size();
    }

    /*
     * Gehört zur Implementaion des TableModel und gibt den Wert der durch die Indizes angebenen Zelle zurück.
     */
    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
        return _dotList.get(rowIndex).getName();
    }

    /**
     * Fügt das übergebene Objekt zur Liste aller auf Darstellungstypänderungen angemeldeten Objekte hinzu.
     *
     * @param listener ein DOTChangeListener
     */
    public void addDOTChangeListener(DOTChangeListener listener) {
        _dotChangeListeners.add(listener);
    }

    /**
     * Löscht das übergebene Objekt aus der Liste aller auf Darstellungstypänderungen angemeldeten Objekte.
     *
     * @param listener ein DOTChangeListener
     */
    @SuppressWarnings("unused")
    public void removeDOTChangeListener(DOTChangeListener listener) {
        _dotChangeListeners.remove(listener);
    }

    /**
     * Benachrichtigt alle auf Darstellungstypänderungen angemeldeten Objekte über das Hinzufügen eines Darstellungstypen.
     *
     * @param displayObjectType ein Darstellungstyp
     */
    private void notifyDOTChangeListenersDOTAdded(final DisplayObjectType displayObjectType) {
        for (DOTChangeListener changeListener : _dotChangeListeners) {
            changeListener.displayObjectTypeAdded(displayObjectType);
        }
    }

    /**
     * Benachrichtigt alle auf Darstellungstypänderungen angemeldeten Objekte über das Verändern eines Darstellungstypen.
     *
     * @param displayObjectType ein Darstellungstyp
     */
    private void notifyDOTChangeListenersDOTChanged(final DisplayObjectType displayObjectType) {
        for (DOTChangeListener changeListener : _dotChangeListeners) {
            changeListener.displayObjectTypeChanged(displayObjectType);
        }
    }

    /**
     * Benachrichtigt alle auf Darstellungstypänderungen angemeldeten Objekte über das Entfernen eines Darstellungstypen.
     *
     * @param displayObjectTypeName ein Name eines Darstellungstyps
     */
    private void notifyDOTChangeListenersDOTRemoved(final String displayObjectTypeName) {
        for (DOTChangeListener changeListener : _dotChangeListeners) {
            changeListener.displayObjectTypeRemoved(displayObjectTypeName);
        }
    }

    @Override
    public String toString() {
        return "DOTManager{" + "_dotList=" + _dotList + ", _dotMap=" + _dotMap + ", _unchangeables=" + _unchangeables + ", _dotChangeListeners=" +
               _dotChangeListeners + ", _initialized=" + _initialized + '}';
    }

    /**
     * Ein Interface für Listener, die über Änderungen von Darstellungstypen informiert werden wollen.
     *
     * @author Kappich Systemberatung
     */
    public interface DOTChangeListener {
        /**
         * Diese Methode wird aufgerufen, wenn der übergebene Darstellungstyp hinzugefügt wurde.
         *
         * @param displayObjectType ein Darstellungstyp
         */
        void displayObjectTypeAdded(final DisplayObjectType displayObjectType);

        /**
         * Diese Methode wird aufgerufen, wenn der übergebene Darstellungstyp geändert wurde.
         *
         * @param displayObjectType ein Darstellungstyp
         */
        void displayObjectTypeChanged(final DisplayObjectType displayObjectType);

        /**
         * Diese Methode wird aufgerufen, wenn der genannte Darstellungstyp gelöscht wurde.
         *
         * @param displayObjectTypeName ein Name eines Darstellungstyp
         */
        void displayObjectTypeRemoved(final String displayObjectTypeName);
    }
}
