/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.displayObjectToolkit;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.ReceiveOptions;
import de.bsvrz.dav.daf.main.ReceiverRole;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.areaPlugin.DOTAreaPlugin;
import de.kappich.pat.gnd.complexPlugin.DOTComplexPlugin;
import de.kappich.pat.gnd.gnd.MapPane;
import de.kappich.pat.gnd.gnd.MapPane.MapScaleListener;
import de.kappich.pat.gnd.layerManagement.Layer;
import de.kappich.pat.gnd.linePlugin.DOTLinePlugin;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectsInitializer;
import de.kappich.pat.gnd.pointPlugin.DOTPointPlugin;
import de.kappich.pat.gnd.viewManagement.ViewEntry;
import java.awt.Rectangle;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import javax.swing.JProgressBar;

/**
 * Der DisplayObjectManager ist die oberste Instanz bei der Verwaltung der {@code DisplayObjects}. Er sorgt mit Hilfe des {@code GeoInitializers} für
 * deren Erstellung und führt An- und Abmeldungen durch.
 *
 * @author Kappich Systemberatung
 */
public class DisplayObjectManager {

    private final ClientDavInterface _connection;
    private final DataModel _configuration;
    private final MapPane _mapPane;
    private final Deque<DisplayObject> _unsubscribedDisplayObjects;
    private final Deque<MapScaleListener> _unaddedMapScaleListeners;

    /**
     * Der Konstruktor der OnlineDisplayObject-Verwaltung.
     *
     * @param connection die Datenverteiler-Verbindung
     * @param mapPane    die Kartenansicht
     */
    public DisplayObjectManager(ClientDavInterface connection, MapPane mapPane) {
        _connection = connection;
        _configuration = _connection.getDataModel();
        _mapPane = mapPane;
        _unsubscribedDisplayObjects = new LinkedList<>();
        _unaddedMapScaleListeners = new LinkedList<>();

        // Initialisiere den GeoInitializer:
        GeoInitializer.getInstance(_configuration);
    }

    /**
     * Der DisplayObjectManager initialisiert beim Aufruf dieser Methode alle DisplayObjects zu dem in dem ViewEntry enthaltenen Layer.
     *
     * @param entry       ein ViewEntry
     * @param progressBar ein ProgressBar
     *
     * @return die Liste aller DisplayObjects des Layers des ViewEntrys
     */
    public List<DisplayObject> getDisplayObjects(ViewEntry entry, final JProgressBar progressBar) {
        List<DisplayObject> returnList = new ArrayList<>();
        final Layer layer = entry.getLayer();
        if (null == layer.getPlugin()) {
            computePluginForLayer(layer);
            if (null == layer.getPlugin()) {
                return Collections.emptyList();
            }
        }
        DisplayObjectsInitializer initializer = layer.getPlugin().getInitializer();
        initializer.initializeDisplayObjects(_configuration, layer, _mapPane, progressBar, returnList);
        if (layer.getPlugin().isDynamicsPossible()) {
            if (!returnList.isEmpty()) {
                synchronized (_unsubscribedDisplayObjects) {
                    _unsubscribedDisplayObjects.addAll(returnList);
                }
            }
        }
        if (layer.getPlugin().isMapScaleListeningNecessary()) {
            if (!returnList.isEmpty()) {
                synchronized (_unaddedMapScaleListeners) {
                    _unaddedMapScaleListeners.addAll(returnList);
                }
            }
        }
        return returnList;
    }

    /**
     * Gibt das die SystemObjects umgebende Rechteck zurück. Ist die Liste leer, so wird das Gesamtrechteck zurückgegeben.
     *
     * @param systemObjects eine Liste von Systemobjekten oder {@code null}
     *
     * @return das anzuzeigende Rechteck
     */
    @Nullable
    public Rectangle getDisplayRectangle(List<SystemObject> systemObjects) {
        return GeoInitializer.getInstance().getDisplayRectangle(systemObjects);
    }

    /**
     * Mit dieser Methode werden alle Anmeldungen beim Datenverteiler vorgenommen, die sich auf seit dem letzten Aufruf dieser Methode durch
     * Initialisierungen neuer DisplayObjects ergeben haben.
     */
    @SuppressWarnings("OverlyNestedMethod")
    public void subscribeDisplayObjects() {
        synchronized (_unsubscribedDisplayObjects) {
            for (DisplayObject displayObject : _unsubscribedDisplayObjects) {
	            if (displayObject instanceof OnlineDisplayObject onlineDisplayObject) {
                    final DOTCollection dotCollection = onlineDisplayObject.getDOTCollection();
                    for (DisplayObjectType displayObjectType : dotCollection.values()) {
                        Set<DOTSubscriptionData> allSubscriptionData = displayObjectType.getSubscriptionData();
                        for (DOTSubscriptionData subscriptionData : allSubscriptionData) {
                            final AttributeGroup onlineAtg;
                            onlineAtg = _configuration.getAttributeGroup(subscriptionData.getAttributeGroup());
                            if (onlineAtg == null) {
                                continue;
                            }
                            final Aspect aspect = _configuration.getAspect(subscriptionData.getAspect());
                            if (aspect == null) {
                                continue;
                            }
                            final DataDescription dataDescription = new DataDescription(onlineAtg, aspect);
                            SystemObject[] systemobjects = new SystemObject[1];
                            systemobjects[0] = onlineDisplayObject.getSystemObject();
                            _connection.subscribeReceiver(onlineDisplayObject, systemobjects, dataDescription, ReceiveOptions.normal(),
                                                          ReceiverRole.receiver());
                        }
                    }
                }
            }
            _unsubscribedDisplayObjects.clear();
        }
    }

    /**
     * Mit dieser Methode werden alle Anmeldungen beim Datenverteiler zurückgenommen, die sich vom DisplayObjectManager in der Methode
     * subscribeDisplayObjects() für die übergebenen DisplayObjects gemacht wurden.
     *
     * @param displayObjects eine Menge von DisplayObjects
     */
    public void unsubscribeDisplayObjects(final Collection<DisplayObject> displayObjects) {
        Runnable unsubscriber = () -> {
            for (DisplayObject displayObject : displayObjects) {
	            if (displayObject instanceof OnlineDisplayObject onlineDisplayObject) {
                    final DOTCollection dotCollection = onlineDisplayObject.getDOTCollection();
                    for (DisplayObjectType displayObjectType : dotCollection.values()) {
                        Set<DOTSubscriptionData> allSubscriptionData = displayObjectType.getSubscriptionData();
                        for (DOTSubscriptionData subscriptionData : allSubscriptionData) {
                            final AttributeGroup onlineAtg;
                            onlineAtg = _configuration.getAttributeGroup(subscriptionData.getAttributeGroup());
                            final Aspect aspect = _configuration.getAspect(subscriptionData.getAspect());
                            if ((onlineAtg != null) && (aspect != null)) {
                                final DataDescription dataDescription = new DataDescription(onlineAtg, aspect);
                                SystemObject[] systemobjects = new SystemObject[1];
                                systemobjects[0] = onlineDisplayObject.getSystemObject();
                                _connection.unsubscribeReceiver(onlineDisplayObject, systemobjects, dataDescription);
                            }
                        }
                    }
                }
            }
        };
        Thread unsubscriberThread = new Thread(unsubscriber);
        unsubscriberThread.start();
    }

    /**
     * Mit dieser Methode werden alle DisplayObjects, die als MapScaleListener zu registrieren sind, beim MapPane registriert.
     */
    public void addMapScaleListeners() {
        synchronized (_unaddedMapScaleListeners) {
            _mapPane.addMapScaleListeners(_unaddedMapScaleListeners);
            _unaddedMapScaleListeners.clear();
        }
    }

    @Override
    public String toString() {
        return "DisplayObjectManager{" + "_connection=" + _connection + ", _configuration=" + _configuration + ", _mapPane=" + _mapPane +
               ", _unsubscribedDisplayObjects=" + _unsubscribedDisplayObjects + ", _unaddedMapScaleListeners=" + _unaddedMapScaleListeners + '}';
    }

    // Diese Methode dient der Herstellung der Abwärtskompatibiliät zu alten in den Präferenzen
    // ohne Plugin abgespeicherten Layern. Eine ähnliche Methode musste im NeedleInitializer
    // hinzugefügt werden.
    private void computePluginForLayer(final Layer layer) {
        SystemObjectType systemObjectType = _configuration.getType(layer.getConfigurationObjectType());
        if (systemObjectType == null) {
            return;     // Passiert, wenn die Konfiguration unvollständig ist.
        }
        SystemObjectType pointType = _configuration.getType("typ.punkt");
        SystemObjectType lineType = _configuration.getType("typ.linie");
        SystemObjectType areaType = _configuration.getType("typ.fläche");
        SystemObjectType complexType = _configuration.getType("typ.komplex");

        if (systemObjectType.inheritsFrom(pointType)) {
            layer.setPlugin(new DOTPointPlugin());
        } else if (systemObjectType.inheritsFrom(lineType)) {
            layer.setPlugin(new DOTLinePlugin());
        } else if (systemObjectType.inheritsFrom(areaType)) {
            layer.setPlugin(new DOTAreaPlugin());
        } else if (systemObjectType.inheritsFrom(complexType)) {
            layer.setPlugin(new DOTComplexPlugin());
        }
    }
}
