/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.displayObjectToolkit;

import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType.DisplayObjectTypeItem;
import de.kappich.pat.gnd.properties.PropertyValue;
import de.kappich.pat.gnd.properties.PropertyValuesManager;
import java.awt.Color;
import java.util.Objects;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;

/**
 * Ein DynamicDOTItem ist die kleinste Einheit bei der Verkapselung der Verwaltung der Informationen zu einer veränderlichen Größe. Dazu kennt das
 * Item Attributgruppe, Aspekt und Attribut, die die Dynamik beschreiben, besitzt eine Kurzbeschreibung seiner Information und kennt gegebenenfalls
 * den Wert für die dynamische Eigenschaft (z.B. eine Zahlwert für Strichbreite, eine Farbe oder einen Text). Die Implementation besteht
 * ausschließlich aus Gettern, Settern und einfachen Dienstleistungsmethoden wie dem Abspeichern in den Präferenzen.
 *
 * @author Kappich Systemberatung
 */
public class DynamicDOTItem implements DisplayObjectTypeItem, Comparable<Object> {

    /**
     * Ein Keine-Daten-Objekt dieser Klasse.
     */
    @SuppressWarnings("ConstantConditions")
    public static final DynamicDOTItem NO_DATA_ITEM = new DynamicDOTItem(null, null, null, null, null);
    /**
     * Ein Keine-Quelle-Objekt dieser Klasse.
     */
    @SuppressWarnings("ConstantConditions")
    public static final DynamicDOTItem NO_SOURCE_ITEM = new DynamicDOTItem(null, null, null, null, null);
    /* Die folgenden Strings dienen als Keys in den Preferences. */
    private static final String ATTRIBUTE_GROUP = "ATTRIBUTE_GROUP";
    private static final String ASPECT = "ASPECT";
    private static final String ATTRIBUTE_NAME = "ATTRIBUTE_NAME";
    private static final String DESCRIPTION = "DESCRIPTION";
    private static final String PROPERTY_VALUE = "PROPERTY_VALUE";
    private static final String PROPERTY_VALUE_CLASS = "PROPERTY_VALUE_CLASS";
    private static final String COLOR_RED = "COLOR_RED";
    private static final String COLOR_BLUE = "COLOR_BLUE";
    private static final String COLOR_GREEN = "COLOR_GREEN";
    private static final String COLOR_ALPHA = "COLOR_ALPHA";

    private final String _attributeGroup;
    private final String _aspect;
    private final String _attributeName;
    private final String _description;
    private final Object _propertyValue;
    private final String _propertyValueClass;
    private final boolean _isValid;

    /**
     * Konstruiert ein DynamicDOTItem aus den übergebenen Informationen. Ein DynamicDOTItem ist gültig, wenn der übergebene Wert {@code propertyValue}
     * einer der Java-Typen Integer, Double, String oder Color ist oder aber {@link PropertyValue}. Andernfalls wird eine IllegalArgumentException
     * ausgelöst, da es sich um eine unvollständige Erweiterung des Kodes handelt.
     *
     * @param attributeGroup die Attributgruppe
     * @param aspect         der Aspekt
     * @param attributeName  der Attributname
     * @param description    die Beschreibung
     * @param propertyValue  der Eigenschaftswert
     */
    public DynamicDOTItem(String attributeGroup, String aspect, String attributeName, String description, Object propertyValue) {
        _attributeGroup = attributeGroup;
        _aspect = aspect;
        _attributeName = attributeName;
        _description = description;
        _propertyValue = propertyValue;
        if (_propertyValue == null) {
            _propertyValueClass = null;
        } else if (_propertyValue instanceof Integer) {
            _propertyValueClass = "Integer";
        } else if (_propertyValue instanceof Double) {
            _propertyValueClass = "Double";
        } else if (_propertyValue instanceof String) {
            _propertyValueClass = "String";
        } else if (_propertyValue instanceof Color) {
            _propertyValueClass = "Color";
        } else if (_propertyValue instanceof PropertyValue) {
            _propertyValueClass = _propertyValue.getClass().getName();
        } else {
            _isValid = false;
            throw new IllegalArgumentException("Die Klasse des Property-Werts wird nicht unterstützt.");
        }
        _isValid = true;
    }

    /**
     * Initialisiert das Item aus dem übergebenen Knoten unter zuhilfenahme des Keys.
     *
     * @param prefs       der Knoten, unter dem die Initialisierung beginnt
     * @param propertyKey der Key in den Präferenzen
     */
    @SuppressWarnings("OverlyLongMethod")
    public DynamicDOTItem(Preferences prefs, String propertyKey) throws BackingStoreException {
        _attributeGroup = prefs.get(ATTRIBUTE_GROUP, "");
        _aspect = prefs.get(ASPECT, "");
        _attributeName = prefs.get(ATTRIBUTE_NAME, "");
        _description = prefs.get(DESCRIPTION, "");
        _propertyValueClass = prefs.get(PROPERTY_VALUE_CLASS, "");
        if (_propertyValueClass.isEmpty()) {
            _propertyValue = null;
            _isValid = false;
            throw new BackingStoreException("Initialisierung aus den Preferences fehlgeschlagen!");
        }
        switch (_propertyValueClass) {
            case "Integer":
                if (Objects.equals(propertyKey, "size")) {
                    // alte Textgrößen werden auf 12 gesetzt
                    _propertyValue = 12;
                } else {
                    _propertyValue = prefs.getInt(PROPERTY_VALUE, -1);
                }
                break;
            case "Double":
                _propertyValue = prefs.getDouble(PROPERTY_VALUE, Double.MIN_VALUE);
                break;
            case "String":
                _propertyValue = prefs.get(PROPERTY_VALUE, "");
                break;
            case "Color":
                int red = prefs.getInt(COLOR_RED, -1);
                int blue = prefs.getInt(COLOR_BLUE, -1);
                int green = prefs.getInt(COLOR_GREEN, -1);
                int alpha = prefs.getInt(COLOR_ALPHA, -1);
                if ((red != -1) && (blue != -1) && (green != -1) && (alpha != -1)) {
                    _propertyValue = new Color(red, blue, green, alpha);
                } else {
                    _propertyValue = null;
                    _isValid = false;
                    throw new BackingStoreException("Initialisierung aus den Preferences fehlgeschlagen!");
                }
                break;
            default:
                Class<?> c = null;
                try {
                    c = Class.forName(_propertyValueClass);
                } catch (ClassNotFoundException e) {
                    e.printStackTrace();
                }
                if (null != c) {
                    Class<?>[] interfaces = c.getInterfaces();
                    boolean implementsPropertyValue = false;
                    for (final Class<?> anInterface : interfaces) {
                        if (anInterface.getName().equals("de.kappich.pat.gnd.properties.PropertyValue")) {
                            implementsPropertyValue = true;
                            break;
                        }
                    }
                    if (implementsPropertyValue) {
                        _propertyValue = PropertyValuesManager.getPropertyValue(_propertyValueClass, prefs);
                    } else {
                        _propertyValue = null;
                    }
                } else {
                    _isValid = false;
                    throw new BackingStoreException("Initialisierung aus den Preferences fehlgeschlagen!");
                }
        }
        _isValid = true;
    }

    /**
     * Gibt die Attributgruppe zurück.
     *
     * @return die Attributgruppe
     */
    @Override
    public String getAttributeGroup() {
        return _attributeGroup;
    }

    /**
     * Gibt den Aspekt zurück.
     *
     * @return den Aspekt
     */
    @Override
    public String getAspect() {
        return _aspect;
    }

    /**
     * Gibt den Namen des Attributs zurück.
     *
     * @return den Attributnamen
     */
    @Override
    public String getAttributeName() {
        return _attributeName;
    }

    /**
     * Gibt die Beschreibung zurück.
     *
     * @return die Beschreibung
     */
    @Override
    public String getDescription() {
        return _description;
    }

    /**
     * Gibt den Wert der Eigenschaft zurück.
     *
     * @return den Eigenschaftswert
     */
    @Override
    public Object getPropertyValue() {
        return _propertyValue;
    }

    /**
     * Gibt {@code true} zurück, wenn das Objekt gültig ist, und {@code false} sonst.
     *
     * @return ist das Item gültig?
     */
    public boolean isValid() {
        return _isValid;
    }

    /**
     * Die Implementation vergleicht die 5 Bestandteile der Items mit equals() von String oder Object.
     *
     * @return {@code true} genau dann, wenn Gleichheit vorliegt
     */
    @Override
    public boolean equals(Object o) {
	    if (!(o instanceof DynamicDOTItem d)) {
            return false;
        }
        if (this == o) {
            return true;
        }
        return !(!_attributeGroup.equals(d._attributeGroup) || !_aspect.equals(d._aspect) || !_attributeName.equals(d._attributeName) ||
                 !_description.equals(d._description)) || _propertyValue.equals(d._propertyValue);
    }

    /**
     * Addiert die Hashcodes von Attributgruppenname, Aspektname und Attributname.
     *
     * @return die Summe der Hashcodes
     */
    @Override
    public int hashCode() {
        return _attributeGroup.hashCode() + _aspect.hashCode() + _attributeName.hashCode();
    }

    /**
     * Eine einfache Selbstbeschreibung.
     *
     * @return die Selbstbeschreibung
     */
    @Override
    public String toString() {
        return getClass().getName() + "[ attributeGroup=" + _attributeGroup + ", aspect=" + _aspect + ", attributename=" + _attributeName +
               ", description=" + _description + ", propertyValue=" + _propertyValue + "]";
    }

    @Override
    public int compareTo(@NotNull Object o) {
        DynamicDOTItem otherItem = (DynamicDOTItem) o;
        if (_attributeGroup.hashCode() != otherItem._attributeGroup.hashCode()) {
            return (_attributeGroup.hashCode() < otherItem._attributeGroup.hashCode()) ? -1 : 1;
        } else {
            if (_aspect.hashCode() != otherItem._aspect.hashCode()) {
                return (_aspect.hashCode() < otherItem._aspect.hashCode()) ? -1 : 1;
            } else {
                if (_attributeName.hashCode() != otherItem._attributeName.hashCode()) {
                    return (_attributeName.hashCode() < otherItem._attributeName.hashCode()) ? -1 : 1;
                }
                if (_description.hashCode() != otherItem._description.hashCode()) {
                    return (_description.hashCode() < otherItem._description.hashCode()) ? -1 : 1;
                }
                if (_propertyValue.hashCode() != otherItem._propertyValue.hashCode()) {
                    return (_propertyValue.hashCode() < otherItem._propertyValue.hashCode()) ? -1 : 1;
                } else {
                    return 0;
                }
            }
        }
    }

    /**
     * Speichert das Item unter dem übergebenen Knoten ab.
     *
     * @param prefs der Knoten, unter dem die Speicherung beginnt
     */
    public void putPreferences(Preferences prefs) {
        prefs.put(ATTRIBUTE_GROUP, _attributeGroup);
        prefs.put(ASPECT, _aspect);
        prefs.put(ATTRIBUTE_NAME, _attributeName);
        prefs.put(DESCRIPTION, _description);
        prefs.put(PROPERTY_VALUE_CLASS, _propertyValueClass);
        switch (_propertyValueClass) {
            case "Integer":
                prefs.putInt(PROPERTY_VALUE, (Integer) _propertyValue);
                break;
            case "Double":
                prefs.putDouble(PROPERTY_VALUE, (Double) _propertyValue);
                break;
            case "String":
                prefs.put(PROPERTY_VALUE, (String) _propertyValue);
                break;
            case "Color":
                Color color = (Color) _propertyValue;
                prefs.putInt(COLOR_RED, color.getRed());
                prefs.putInt(COLOR_BLUE, color.getBlue());
                prefs.putInt(COLOR_GREEN, color.getGreen());
                prefs.putInt(COLOR_ALPHA, color.getAlpha());
                break;
            // Die alte nicht-generische Implementation: löschen nach Tests.
//			case "KmFormat":
//				KmFormat format = (KmFormat) _propertyValue;
//				prefs.putInt(PROPERTY_VALUE, format.getOrdinal());
//				break;
//			case "DistanceRasterType":
//				DistanceRasterType type = (DistanceRasterType) _propertyValue;
//				prefs.putLong(PROPERTY_VALUE, type.getValue());
//				break;
//			case "TextStyle":
//				Integer style = ((TextStyleProperty.Styles) _propertyValue).getIntValue();
//				prefs.putInt(PROPERTY_VALUE, style);
//				break;
            default:
	            if (_propertyValue instanceof PropertyValue pv) {
                    pv.putPreferences(prefs);
                } else {
                    String s = "Unbekannte PropertyValueClass in putPreferences: " + _propertyValueClass;
                    throw new IllegalArgumentException("Ein DynamicDOTItem kann nicht in den Preferences gespeichert werden." + s);
                }
        }
    }

    /**
     * Estellt ein neues Item mit denselben Werten.
     *
     * @return die Kopie
     */
    @Override
    public DynamicDOTItem getCopy() {
        return new DynamicDOTItem(getAttributeGroup(), getAspect(), getAttributeName(), getDescription(), getPropertyValue());
    }
}
