/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.displayObjectToolkit;

import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.asbNodePlugin.AsbNodeDisplayObject;
import de.kappich.pat.gnd.coorTransform.GeoTransformation;
import de.kappich.pat.gnd.coorTransform.UTMCoordinate;
import de.kappich.pat.gnd.kmPlugin.KmDisplayObject;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectPainter;
import de.kappich.pat.gnd.rnPlugin.RnDisplayObject;
import de.kappich.pat.gnd.statPlugin.StatDisplayObject;
import de.kappich.pat.gnd.utils.PointWithAngle;
import de.kappich.pat.gnd.viewManagement.ViewEntry;
import java.awt.Point;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.awt.geom.AffineTransform;
import java.awt.geom.Path2D;
import java.awt.geom.PathIterator;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import javax.swing.JProgressBar;

/**
 * {@code GeoInitializer} ist eine Helfer-Klasse bei der Initialisierung der {@code DisplayObjects} der verschiedenen Plugins. {@code GeoInitializer}
 * übernimmt die teilweise synchronisierte Geo-Initialasierung, die aus einem {@code DisplayObjectManager} heraus aungestoßen wird (s. {@link
 * DisplayObjectManager#getDisplayObjects(ViewEntry, JProgressBar)}).
 * <p>
 * Synchronsiert sind alle Zugriffe mit der Methode {@link SystemObject#getConfigurationData(AttributeGroup)}.
 *
 * @author Kappich Systemberatung
 */
@SuppressWarnings("SynchronizationOnStaticField")
public final class GeoInitializer {

    private static final Debug _debug = Debug.getLogger();
    private static final Object _configurationAccess = new Object();
    private static final double _ignoreDistance = 0.3;
    private static final boolean ästeStrikt = true;
    private static GeoInitializer _instance;
    private static double _xMin = Double.POSITIVE_INFINITY;
    private static double _yMin = Double.POSITIVE_INFINITY;
    private static double _xMax = Double.NEGATIVE_INFINITY;
    private static double _yMax = Double.NEGATIVE_INFINITY;
    private final DataModel _configuration;
    private final SystemObjectType _lineComposedOfLinesType;
    private final SystemObjectType _lineWithCoordinatesType;
    private final AttributeGroup _composedOfLinesAttributeGroup;
    private final AttributeGroup _lineCoordinatesAttributeGroup;
    private final SystemObjectType _pointOnLineType;
    private final SystemObjectType _pointWithCoordinatesType;
    private final AttributeGroup _pointOnLineAttributeGroup;
    private final AttributeGroup _pointCoordinateAttributeGroup;
    private final AttributeGroup _areaCoordinatesAttributeGroup;
    private final AttributeGroup _complexCoordinatesAttributeGroup;
    private final AttributeGroup _gesamtStraßeAttributeGroup;
    private final Aspect _eigenschaftenAspect;
    private final SystemObjectType _straßenTeilSegmentType;
    private final SystemObjectType _straßenSegmentType;
    private final SystemObjectType _äußeresStraßenSegment;
    private final SystemObjectType _inneresStraßenSegment;
    private final AttributeGroup _tmcLocationCodeAttributeGroup;
    private final AttributeGroup _straßenSegmentAttributeGroup;
    private final AttributeGroup _straßeAttributeGroup;
    private final AttributeGroup _äußeresStraßenSegmentAttributeGroup;
    private final Map<SystemObject, List<KmPoint>> _kmSTS = new HashMap<>();
    private final List<SystemObject> _statSTS = new ArrayList<>();

    private GeoInitializer(final DataModel configuration) {
        _configuration = configuration;
        _lineComposedOfLinesType = _configuration.getType("typ.bestehtAusLinienObjekten");
        _lineWithCoordinatesType = _configuration.getType("typ.linieXY");
        _composedOfLinesAttributeGroup = _configuration.getAttributeGroup("atg.bestehtAusLinienObjekten");
        _lineCoordinatesAttributeGroup = _configuration.getAttributeGroup("atg.linienKoordinaten");
        _pointOnLineType = _configuration.getType("typ.punktLiegtAufLinienObjekt");
        _pointWithCoordinatesType = _configuration.getType("typ.punktXY");
        _pointOnLineAttributeGroup = _configuration.getAttributeGroup("atg.punktLiegtAufLinienObjekt");
        _pointCoordinateAttributeGroup = _configuration.getAttributeGroup("atg.punktKoordinaten");
        _areaCoordinatesAttributeGroup = _configuration.getAttributeGroup("atg.flächenKoordinaten");
        _complexCoordinatesAttributeGroup = _configuration.getAttributeGroup("atg.komplexKoordinaten");
        _gesamtStraßeAttributeGroup = _configuration.getAttributeGroup("atg.gesamtStraße");
        _eigenschaftenAspect = _configuration.getAspect("asp.eigenschaften");
        _straßenTeilSegmentType = _configuration.getType("typ.straßenTeilSegment");
        _straßenSegmentType = _configuration.getType("typ.straßenSegment");
        _äußeresStraßenSegment = _configuration.getType("typ.äußeresStraßenSegment");
        _inneresStraßenSegment = _configuration.getType("typ.inneresStraßenSegment");
        _tmcLocationCodeAttributeGroup = _configuration.getAttributeGroup("atg.tmcLocationCode");
        _straßenSegmentAttributeGroup = _configuration.getAttributeGroup("atg.straßenSegment");
        _straßeAttributeGroup = _configuration.getAttributeGroup("atg.straße");
        _äußeresStraßenSegmentAttributeGroup = _configuration.getAttributeGroup("atg.äußeresStraßenSegment");
    }

    /**
     * Mit dieser Methode erhält man das {@code GeoInitializer}-Singleton.
     *
     * @param configuration die Konfiguration
     *
     * @return das {@code GeoInitializer}-Singleton
     */
    public static synchronized GeoInitializer getInstance(final DataModel configuration) {
        if (null == _instance) {
            _instance = new GeoInitializer(configuration);
        }
        return _instance;
    }

    /**
     * Dies ist eine Convenience-Methode zum erhalt des {@code GeoInitializer}-Singletons. Sie kann nur dort angewendet werden, wo bekannt ist, dass
     * das Singleton bereits existiert.
     *
     * @return das {@code GeoInitializer}-Singleton
     */
    public static synchronized GeoInitializer getInstance() {
        return _instance;
    }

    /*
    Diese Methode entscheidet anhand der Liste von KmPoints darüber, ob das zugehörige Segment zur Visualisierung
    der Betriebskilometrierung (true) verwendet wird.
     */
    private static boolean hasToBeKm(final List<KmPoint> kmPoints) {
		/*
		Ein Segment, dessen kmPoints mindestens ein Paar aufeinander folgender Punkte mit aufsteigender
		Kilometrierungsrichtung besitzt, wird zur Visualisierung der Kilometrierung verwendet.
		 */
        return hasDirection(kmPoints, true);
    }

    /*
    Diese Methode entscheidet anhand der Liste von KmPoints darüber, ob das zugehörige Segment zur Visualisierung
    der Betriebskilometrierung (true) verwendet wird.
     */
    private static boolean hasToBeStat(final List<KmPoint> kmPoints) {
		/*
		Ein Segment, dessen kmPoints mindestens ein Paar aufeinander folgender Punkte mit absteigender
		Kilometrierungsrichtung besitzt, wird zur Visualisierung der Stationierung verwendet.
		 */
        return hasDirection(kmPoints, false);
    }

    private static boolean hasDirection(final List<KmPoint> kmPoints, boolean positive) {
        if (kmPoints.size() < 2) {
            _debug.warning("Logischer Fehler in hasToBeKm (bzw. determineKmAndStatSTS).");
            return false;
        }
        for (int k = 0; k < kmPoints.size() - 1; ++k) {
            KmPoint p1 = kmPoints.get(k);
            KmPoint p2 = kmPoints.get(k + 1);
            if (!p1.getRouteNumber().equals(p2.getRouteNumber())) {
                throw new IllegalStateException("Logischer Fehler in hasToBeKm (bzw. determineKmAndStatSTS).");
            }
            if (!p1.getBlockNumber().equals(p2.getBlockNumber())) {
                continue;
            }
            if (p1.getOffset() == p2.getOffset()) {
                continue;
            }
            if (positive) {
                if (p1.getValue() < p2.getValue()) {
                    return true;
                }
            } else {
                if (p1.getValue() > p2.getValue()) {
                    return true;
                }
            }
        }
        return false;
    }

    private static boolean isDummy(final Data stat) {
        return stat.getTextValue("AnfangsKnoten").getValueText().isEmpty();
    }

    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    /* Hier beginnt die Implementation für das Kilometrierungs- und das Stationierungs-Plugin. */

    @Nullable
    private static Rectangle increaseRectangle(@Nullable Rectangle rectangle) {
        if (rectangle == null) {
            return null;
        }
        Rectangle returnRectangle;
        if (rectangle.height > 100 && rectangle.width > 100) {
            int newWidth = (int) (1.03 * rectangle.width);
            int newHeight = (int) (1.03 * rectangle.height);

            Point p = new Point((int) (rectangle.getMinX() + rectangle.width - newWidth), (int) (rectangle.getMinY() + rectangle.height - newHeight));
            returnRectangle = new Rectangle(p);
            p = new Point((int) (rectangle.getMaxX() - rectangle.width + newWidth), (int) (rectangle.getMaxY() - rectangle.height + newHeight));
            returnRectangle.add(p);
        } else {
            returnRectangle = rectangle;
            returnRectangle.grow(50, 50);
        }
        return returnRectangle;
    }

    @Nullable
    private static Rectangle getDisplayRectangle() {
        if ((_xMin == Double.POSITIVE_INFINITY) || (_yMin == Double.POSITIVE_INFINITY) || (_xMax == Double.NEGATIVE_INFINITY) ||
            (_yMax == Double.NEGATIVE_INFINITY)) {
            return null;
        }
        Rectangle rectangle = new Rectangle((int) _xMin, (int) _yMin, (int) (_xMax - _xMin), (int) (_yMax - _yMin));
        return increaseRectangle(rectangle);
    }

    private static long getNextDecaMeter(long value1, long value2) {
        if (value1 % 10 == 0) {
            return value1;
        }
        if (value1 < value2) {
            return (value1 / 10 + 1) * 10;
        } else {
            return (value1 / 10) * 10;
        }
    }

    private static List<PointWithAngle> selectPoints(final List<PointWithAngle> allPoints,   // alle wählbaren Punkte
                                                     final double minDistance,                  // der paarweise Minimalabstand
                                                     final List<PointWithAngle> initialPoints) // zu berücksichtigende Punkte (nicht in 
    // Rückgabe-Liste)
    {
        List<PointWithAngle> selectedPoints = new ArrayList<>();
        for (final PointWithAngle point : allPoints) {
            if (null == point) {
                continue;
            }
            Double distance1 = getDistance(point.getPoint(), initialPoints, minDistance);
            if (distance1 > minDistance) {
                Double distance2 = getDistance(point.getPoint(), selectedPoints, minDistance);
                if (distance2 > minDistance) {
                    selectedPoints.add(point);
                }
            }
        }
        return selectedPoints;
    }

    private static Double getDistance(Point2D point, final List<PointWithAngle> pointList, final Double threshold) {
        Double distance = Double.MAX_VALUE;
        for (PointWithAngle other : pointList) {
            Point2D otherPoint = other.getPoint();
            Double toOther = point.distance(otherPoint);
            if (distance > toOther) {
                distance = toOther;
                if (distance < threshold) {
                    return distance;
                }
            }
        }
        return distance;
    }

    private static void createRnDisplayObjects(final DisplayObjectPainter painter, final DOTCollection dotCollection, final Integer rn,
                                               final List<PointWithAngle> points, final RnDisplayObject.Category category,
                                               final List<DisplayObject> resultLst) {
        for (PointWithAngle point : points) {
            resultLst.add(new RnDisplayObject(painter, dotCollection, rn, category, point));
        }
    }

    private static double computeLength(List<Object> lineCoordinates) {
        if (lineCoordinates.size() != 1) {
            return 0.;
        }
        Object o = lineCoordinates.get(0);
	    if (!(o instanceof Path2D.Double polyline)) {
            return 0.;
        }
        final PathIterator pathIterator = polyline.getPathIterator(null);
        double currentLength = 0.;
        double currentX = Double.MAX_VALUE;
        double currentY = Double.MAX_VALUE;
        while (!pathIterator.isDone()) {
            double[] coordinates = new double[6];
            int type = pathIterator.currentSegment(coordinates);
            switch (type) {
                case PathIterator.SEG_MOVETO:
                    currentX = coordinates[0];
                    currentY = coordinates[1];
                    break;
                case PathIterator.SEG_LINETO:
                    double newX = coordinates[0];
                    double newY = coordinates[1];
                    if (currentX != Double.MAX_VALUE) {
                        currentLength += getMeterDistance(currentX, currentY, newX, newY);
                    }
                    currentX = coordinates[0];
                    currentY = coordinates[1];
                    break;
            }
            pathIterator.next();
        }
        return currentLength;
    }

    @Nullable
    private static PointWithAngle determinePointCoordinate(List<Object> lineCoordinates, // bereits in UTM
                                                           final double offset) {
        if (lineCoordinates.size() != 1) {
            return null;
        }
        if (offset < 0.) {
            return null;
        }
        Object o = lineCoordinates.get(0);
	    if (!(o instanceof Path2D.Double polyline)) {
            return null;
        }
        final PathIterator pathIterator = polyline.getPathIterator(null);
        double currentOffset = 0.;
        double currentX = Double.MAX_VALUE;
        double currentY = Double.MAX_VALUE;
        double deltaX = Double.MAX_VALUE;
        double deltaY = Double.MAX_VALUE;
        while (!pathIterator.isDone()) {
            double[] coordinates = new double[6];
            int type = pathIterator.currentSegment(coordinates);
            switch (type) {
                case PathIterator.SEG_MOVETO:
                    currentX = coordinates[0];
                    currentY = coordinates[1];
                    break;
                case PathIterator.SEG_LINETO:
                    double newX = coordinates[0];
                    double newY = coordinates[1];
                    if (currentX != Double.MAX_VALUE) {
                        double meters = getMeterDistance(currentX, currentY, newX, newY);
                        if ((offset >= currentOffset) && (currentOffset + meters >= offset)) {
                            double lambda = (offset - currentOffset) / meters;
                            double xUTM = currentX + lambda * (newX - currentX);
                            double yUTM = currentY + lambda * (newY - currentY);
                            return new PointWithAngle(new Point2D.Double(xUTM, yUTM), orientedAngleWithXAxis(newX - currentX, newY - currentY));

                        } else {
                            currentOffset += meters;
                            //noinspection ConstantConditions
                            if ((offset - currentOffset < 1.) && (currentX != Double.MAX_VALUE) && (currentY != Double.MAX_VALUE)) {
                                deltaX = newX - currentX;
                                deltaY = newY - currentY;
                            }
                        }
                    }
                    currentX = coordinates[0];
                    currentY = coordinates[1];
                    break;
                default:
                    break;
            }
            pathIterator.next();
        }
        // Es gibt Segmente der Länge 0, auf denen MQs bei Offset 0 liegen. Diese werden hier bedient:
        if ((0. == offset) && (currentX != Double.MAX_VALUE) && (currentY != Double.MAX_VALUE)) {
            return new PointWithAngle(new Point2D.Double(currentX, currentY), null);
        } else if ((offset - currentOffset < 1.) && (currentX != Double.MAX_VALUE) && (currentY != Double.MAX_VALUE) &&
                   (deltaX != Double.MAX_VALUE) && (deltaY != Double.MAX_VALUE)) {
            // Und es kann sein, dass durch die Arithmetik ein am Ende eines Segments liegendes Objekt bisher ignoriert wurde:
            return new PointWithAngle(new Point2D.Double(currentX, currentY), orientedAngleWithXAxis(deltaX, deltaY));
        }
        return null;
    }

    private static double getMeterDistance(double x1, double y1, double x2, double y2) {
        double dx = x1 - x2;
        double dy = y1 - y2;
        return Math.sqrt(dx * dx + dy * dy);
    }

    @Nullable
    private static Double orientedAngleWithXAxis(double x, double y) {
        double norm = Math.sqrt(x * x + y * y);
        if (norm == 0.) {
            return null;
        }
        Double cosinus = x / norm;
        if (cosinus.isNaN() || cosinus.isInfinite()) {
            return null;
        }
        Double angle = Math.acos(cosinus);
        if (angle.isNaN() || angle.isInfinite()) {
            return null;
        }
        if (y < 0.) {
            angle = 2 * Math.PI - angle;
        }
        return angle;
    }

    private static void simplifyCoordinates(List<Path2D.Double> rawPolylines, List<Object> polylines) {
        // Die Koordinaten in rawPolylines werden folgendermaßen vereinfacht:
        // 1. Doppelte aufeinanderfolgende Punkte werden eliminiert.
        // 2. Leere Polylines werden ignoriert.
        Double lastX = Double.MIN_VALUE;
        Double lastY = Double.MAX_VALUE;
        // An dieser Stelle wurde auf Basis einer hprof-Analyse eine Optimierung mit Hilfe
        // des Konstruktors Path2D.Double( int winding_rule, int initialCapacity) versucht;
        // das Ergebnis war nicht überzeugend: nach wie vor wurde der double[]-Konstruktor
        // um die 25% der Zeit aufgerufen, nur von 2 Stellen aus.
        Path2D.Double polyline = new Path2D.Double();
        boolean moveTo = true;
        double[] coords = new double[6];
        boolean lastIgnored = false;
        final AffineTransform dummyTransform = new AffineTransform();
        for (Path2D.Double rawPolyline : rawPolylines) {
            PathIterator pi = rawPolyline.getPathIterator(dummyTransform);
            if (pi.isDone()) {    // leere Linie
                continue;
            }
            pi.currentSegment(coords);
            Double xCoor = coords[0];
            Double yCoor = coords[1];
            if ((Math.abs(lastX - xCoor) > _ignoreDistance) || (Math.abs(lastY - yCoor) > _ignoreDistance)) {
                if (!moveTo) {    // Nur im Sinne von: die letzte Polyline ist nicht leer!
                    polylines.add(polyline);
                    polyline = new Path2D.Double();
                }
                polyline.moveTo(xCoor, yCoor);
                moveTo = false;
                lastIgnored = false;
            } else {
                lastIgnored = true;
            }
            lastX = xCoor;
            lastY = yCoor;

            while (!pi.isDone()) {
                pi.currentSegment(coords);
                xCoor = coords[0];
                yCoor = coords[1];
                if ((Math.abs(lastX - xCoor) > _ignoreDistance) || (Math.abs(lastY - yCoor) > _ignoreDistance)) {
                    polyline.lineTo(xCoor, yCoor);
                    lastIgnored = false;
                } else {
                    lastIgnored = true;
                }
                lastX = xCoor;
                lastY = yCoor;
                pi.next();
            }
            lastX = coords[0];
            lastY = coords[1];
        }
        if (!moveTo || lastIgnored) {
            polylines.add(polyline);
        }
    }

    /*
     * Der Sinn dieser Methode besteht darin, die gerechneten x,y-Koordinaten, die zu einem normalen
     * kartesischen System mit positiven Richtungen links (x) und oben(y) gehören, in ein Bildschirm-System
     * mit positiven Richtungen links(x) und unten(y), zu übertragen. Deshalb wird y durch -y ersetzt.
     */
    private static void customiseUTM(UTMCoordinate utm) {
        utm.setY(-utm.getY());
    }

    private static void updateExtremeCoordinates(UTMCoordinate utm) {
        if (_xMin > utm.getX()) {
            _xMin = utm.getX();
        }
        if (_xMax < utm.getX()) {
            _xMax = utm.getX();
        }
        if (_yMin > utm.getY()) {
            _yMin = utm.getY();
        }
        if (_yMax < utm.getY()) {
            _yMax = utm.getY();
        }
    }

    @SuppressWarnings({"unused", "UseOfSystemOutOrSystemErr"})
    private static void dumpPolylines(List<Object> polylines) {
        for (Object o : polylines) {
            Path2D.Double polyline = (Path2D.Double) o;
            final PathIterator pathIterator = polyline.getPathIterator(null);
            double[] coordinates = new double[6];
            System.out.println("Polyline:");
            while (!pathIterator.isDone()) {
                int type = pathIterator.currentSegment(coordinates);
                switch (type) {
                    case PathIterator.SEG_MOVETO:
                        System.out.println("X: " + coordinates[0] + ", Y: " + coordinates[1]);
                        break;
                    case PathIterator.SEG_LINETO:
                        System.out.println("X: " + coordinates[0] + ", Y: " + coordinates[1]);
                        break;
                    default:
                        break;
                }
                pathIterator.next();
            }
        }
    }

    public static UTMCoordinate wgs84ToUtm(double x, double y) {
        UTMCoordinate utm = new UTMCoordinate();
        GeoTransformation.wGS84ToUTM(x, y, utm);
        customiseUTM(utm);
        updateExtremeCoordinates(utm);
        return utm;
    }

    /**
     * Diese Methode gibt eine Liste mit den Punktkoordinaten des {@code SystemObjects} zurück. Der Zugriff auf die Konfiguration ist synchronisiert.
     *
     * @param systemObject das Systemobjekt
     *
     * @return die Koordinatenliste
     */
    public List<Object> getPointCoordinates(SystemObject systemObject) {
        // Für jedes Systemobjekt wird höchstens eine Koordinate berechnet.
        // Existieren sowohl eine Linie+Offset-Information als auch eine
        // Koordinate, so wird die Linie+Offset-Information verwendet.
        // In diesem Fall wird der Winkel der Linie an diesem Punkt näherungs-
        // weise berechnet und auch in die Koordinatenliste gesteckt.
        List<Object> pointCoordinate = new ArrayList<>(1);
        if (systemObject.isOfType(_pointOnLineType)) {
            final Data pointOnLineData;
            synchronized (_configurationAccess) {
                pointOnLineData = systemObject.getConfigurationData(_pointOnLineAttributeGroup);
            }
            if (pointOnLineData != null) {
                final SystemObject line = pointOnLineData.getReferenceValue("LinienReferenz").getSystemObject();
                List<Object> lineCoordinates = getPolylines(line);    // liefert schon UTM
                final Data.NumberValue scaledOffset = pointOnLineData.getScaledValue("Offset");
                if (scaledOffset.isNumber()) {
                    double offset = scaledOffset.doubleValue();
                    double lengthCorrectionFactor = getLengthCorrectionFactor(line);
                    offset *= lengthCorrectionFactor;
                    PointWithAngle newPoint = determinePointCoordinate(lineCoordinates, offset);
                    if (newPoint != null) {
                        pointCoordinate.add(newPoint);
                        return pointCoordinate;
                    } else {
                        String msg = "Fehler beim Berechnen einer Punkt-Koordinate für " + systemObject.getNameOrPidOrId();
                        _debug.error(msg);
                    }
                }
            }
        }
        if (systemObject.isOfType(_pointWithCoordinatesType)) {
            final Data coordinatesData;
            synchronized (_configurationAccess) {
                coordinatesData = systemObject.getConfigurationData(_pointCoordinateAttributeGroup);
            }
            if (coordinatesData != null) {
                final Data.NumberValue x = coordinatesData.getScaledValue("x");
                final Data.NumberValue y = coordinatesData.getScaledValue("y");
                if (x.isNumber() && y.isNumber()) {
                    final UTMCoordinate utm = new UTMCoordinate();
                    GeoTransformation.wGS84ToUTM(x.doubleValue(), y.doubleValue(), utm);
                    customiseUTM(utm);
                    updateExtremeCoordinates(utm);
                    Point2D.Double newPoint = new Point2D.Double(utm.getX(), utm.getY());
                    PointWithAngle newPiontWithDummyAngle = new PointWithAngle(newPoint, Double.NaN);
                    pointCoordinate.add(newPiontWithDummyAngle);
                    return pointCoordinate;
                }
            }
        }
        return pointCoordinate;
    }

    /**
     * Diese Methode gibt die Polylines eines Linienobjekts zurück. Der Zugriff  auf die Konfiguration ist synchronisiert.
     *
     * @param systemObject das Systemobjekt
     *
     * @return die Polyline-Liste
     */
    public List<Object> getPolylines(SystemObject systemObject) {
        List<Path2D.Double> rawPolylines = new ArrayList<>();
        appendCoordinates(systemObject, rawPolylines);
        List<Object> polylines = new ArrayList<>();
        simplifyCoordinates(rawPolylines, polylines);
        return polylines;
    }

    /**
     * Diese Methode lädt für eine {@code Collection} von {@code SytemOjects} deren Koordinaten und Kompositionsinformationen, sowie die Koordinaten
     * und Kompositionsinformationen anderer Linienobjekte, die in Kompositionen auftreten.
     *
     * @param objectCollection eine {@code Collection} von {@code SytemOjects}
     */
    public void preloadLines(final Collection<SystemObject> objectCollection) {
        _configuration.getConfigurationData(objectCollection, _lineCoordinatesAttributeGroup);
        final Data[] configurationData = _configuration.getConfigurationData(objectCollection, _composedOfLinesAttributeGroup);
        final Collection<Long> lines = new ArrayList<>();
        for (Data data : configurationData) {
            if (data != null) {
                final Data.ReferenceValue[] linesReferenceValues = data.getReferenceArray("LinienReferenz").getReferenceValues();
                for (Data.ReferenceValue line : linesReferenceValues) {
                    lines.add(line.getId());
                }
            }
        }
        if (!lines.isEmpty()) {
            final List<SystemObject> objects = new ArrayList<>(getObjects(lines));
            final int chunkSize = 100;
            for (int i = 0; i < objects.size() - chunkSize; i += chunkSize) {
                final List<SystemObject> subList = objects.subList(i, i + chunkSize);
                preloadLines(subList);
            }
        }
    }

    /**
     * Diese Methode gibt eine {@code Collection} von {@code SystemObjects} zurück, deren Ids ihr übergeben wurden.
     *
     * @param ids eine {@code Collection} von Ids
     *
     * @return eine {@code Collection} von {@code SystemObjects}
     */
    public Collection<SystemObject> getObjects(final Collection<Long> ids) {
        final ArrayList<SystemObject> result = new ArrayList<>(ids.size());
        for (Long id : ids) {
            result.add(_configuration.getObject(id));
        }
        return result;
    }

    /**
     * Diese Methode gibt eine Liste mit den Flächenkoordinaten des {@code SystemObjects} zurück. Der Zugriff auf die Konfiguration ist
     * synchronisiert.
     *
     * @param systemObject das Systemobjekt
     *
     * @return die Koordinatenliste
     */
    public List<Object> getAreaCoordinates(SystemObject systemObject) {
        final Data coordinatesData;
        synchronized (_configurationAccess) {
            coordinatesData = systemObject.getConfigurationData(_areaCoordinatesAttributeGroup);
        }
        List<Object> areaCoordinates = new ArrayList<>();
        if (coordinatesData != null) {
            final Data.NumberArray xArray = coordinatesData.getScaledArray("x");
            final Data.NumberArray yArray = coordinatesData.getScaledArray("y");
            int length = Math.min(xArray.getLength(), yArray.getLength());
            Polygon polygon = new Polygon();
            final UTMCoordinate utm = new UTMCoordinate();
            // Im Moment unterscheiden wir hier noch nicht zwischen Punkten, Linien und Flächen.
            for (int i = 0; i < length; i++) {
                final Data.NumberValue xValue = xArray.getValue(i);
                final Data.NumberValue yValue = yArray.getValue(i);
                if (xValue.isNumber() && yValue.isNumber()) {
                    GeoTransformation.wGS84ToUTM(xValue.doubleValue(), yValue.doubleValue(), utm);
                    customiseUTM(utm);
                    updateExtremeCoordinates(utm);
                    polygon.addPoint((int) utm.getX(), (int) utm.getY());
                }
            }
            areaCoordinates.add(polygon);
        }
        return areaCoordinates;
    }

    /**
     * Diese Methode gibt eine Liste mit den Komplexkoordinaten des {@code SystemObjects} zurück. Der Zugriff auf die Konfiguration ist
     * synchronisiert.
     *
     * @param systemObject das Systemobjekt
     *
     * @return die Koordinatenliste
     */
    public List<Object> getComplexCoordinates(SystemObject systemObject) {
        final Data coordinatesData;
        synchronized (_configurationAccess) {
            coordinatesData = systemObject.getConfigurationData(_complexCoordinatesAttributeGroup);
        }
        List<Object> complexCoordinates = new ArrayList<>();
        if (coordinatesData != null) {
            final Data.ReferenceArray pointReferences = coordinatesData.getReferenceArray("PunktReferenz");
            for (int index = 0; index < pointReferences.getLength(); index++) {
                SystemObject pointSystemObject = pointReferences.getSystemObject(index);
                complexCoordinates.addAll(getPointCoordinates(pointSystemObject));
            }
            final Data.ReferenceArray lineReferences = coordinatesData.getReferenceArray("LinienReferenz");
            for (int index = 0; index < lineReferences.getLength(); index++) {
                SystemObject lineSystemObject = lineReferences.getSystemObject(index);
                complexCoordinates.addAll(getPointCoordinates(lineSystemObject));
            }
            final Data.ReferenceArray areaReferences = coordinatesData.getReferenceArray("FlächenReferenz");
            for (int index = 0; index < areaReferences.getLength(); index++) {
                SystemObject areaReference = areaReferences.getSystemObject(index);
                complexCoordinates.addAll(getAreaCoordinates(areaReference));
            }
            final Data.ReferenceArray complexReferences = coordinatesData.getReferenceArray("KomplexReferenz");
            for (int index = 0; index < complexReferences.getLength(); index++) {
                SystemObject complexReference = complexReferences.getSystemObject(index);
                complexCoordinates.addAll(getAreaCoordinates(complexReference));
            }
        }
        return complexCoordinates;
    }

    /**
     * Diese Methode erzeugt zu dem übergebenen {@code SystemObjects} (vom Typ StraßenKnoten) alle {@code AsbNodeDisplayObjects} und fügt sie der
     * übergebenen Liste hinzu. Der Zugriff auf die Konfiguration ist synchronisiert.
     *
     * @param nodes         ein {@code SystemObject}
     * @param painter       ein {@code DisplayObjectPainter}
     * @param dotCollection eine {@code DOTCollection}
     * @param resultList    die Liste, der die neuen Objekte hinzugefügt werden
     */
    public void createAsbNodeDisplayObjects(final List<SystemObject> nodes, final DisplayObjectPainter painter, final DOTCollection dotCollection,
                                            List<DisplayObject> resultList) {
        AttributeGroup lclInfoAtg = _configuration.getAttributeGroup("atg.straßenKnotenLclInfo");
        AttributeGroup locationCodeAtg = _configuration.getAttributeGroup("atg.tmcLocationCode");

        Set<String> nodeNumbers = new HashSet<>();
        for (SystemObject systemObject : nodes) {
            final Data lclInfoData;
            synchronized (_configurationAccess) {
                lclInfoData = systemObject.getConfigurationData(lclInfoAtg);
            }
            if (lclInfoData != null) {
                final Data.Array array = lclInfoData.getArray("TmcPunkt");
                for (int i = 0; i < array.getLength(); ++i) {
                    Data data = array.getItem(i);
                    SystemObject tmcPoint = data.asReferenceValue().getSystemObject();
                    final Data locationCodeData;
                    synchronized (_configurationAccess) {
                        locationCodeData = tmcPoint.getConfigurationData(locationCodeAtg);
                    }
                    if (null == locationCodeData) {
                        continue;
                    }
                    String nodeNumberA = locationCodeData.getTextValue("NetzKnotenNrA").getValueText();
                    String nodeNumberB = locationCodeData.getTextValue("NetzKnotenNrB").getValueText();
                    if (null == nodeNumberA || nodeNumberA.isEmpty() || nodeNumbers.contains(nodeNumberA)) {
                        // ungültig oder bekannt
                        continue;
                    } else if (null != nodeNumberB && !nodeNumberB.isEmpty()) {
                        // gültige NetzKnotenNrB: TmcPunkt ist kein eigener Straßenknoten
                        continue;
                    } else {
                        // gültig, neu und eigener Straßenknoten
                        nodeNumbers.add(nodeNumberA);
                    }
                    double xWGS84 = locationCodeData.getScaledValue("x").doubleValue();
                    double yWGS84 = locationCodeData.getScaledValue("y").doubleValue();
                    UTMCoordinate utm = new UTMCoordinate();
                    GeoTransformation.wGS84ToUTM(xWGS84, yWGS84, utm);
                    customiseUTM(utm);
                    updateExtremeCoordinates(utm);

                    PointWithAngle newPoint = new PointWithAngle(new Point2D.Double(utm.getX(), utm.getY()), 5. * Math.PI / 4.);
                    AsbNodeDisplayObject asbNodeDisplayObject = new AsbNodeDisplayObject(painter, dotCollection, "NK " + nodeNumberA, newPoint);
                    resultList.add(asbNodeDisplayObject);
                }
            }
        }
    }

    /*
     * Diese Methode gibt alle StraßenTeilSegmente, die zur Visualisierung der Betriebskilometrierung
     * herangezogen werden, zurück.
     */
    private Map<SystemObject, List<KmPoint>> getKmSTS() {
        if (_kmSTS.isEmpty()) {
            determineKmAndStatSTS();
            augmentStatSTS();
        }
        return _kmSTS;
    }

    /*
     * Diese Methode gibt alls StraßenTeilSegmente, die zur Visualisierung der ASB-Stationierung
     * herangezogen werden, zurück.
     */
    private List<SystemObject> getStatSTS() {
        if (_statSTS.isEmpty()) {
            determineKmAndStatSTS();
            augmentStatSTS();
        }
        return _statSTS;
    }

    /*
    Diese Methode fügt der Liste der StraßenTeilSegmente, an denen ASB-Stationierung angezeigt werden soll
    weitere Objekte hinzu. Diese sind diejenigen StraßenTeilSegmente, die folgende Bedingungen erfüllen:
        - sie sind Hauptfahrbahn (d.h. alle äußeren Segmente und jene inneren Segmente mit HFB-Attribut)
        - es gibt eine ASB-Stationierung (ein strikter Abschnittsfilter kann mit dem Flag 'ästeStrikt'
            hinzufeschaltet werden)
        - sie sind kein Segment, auf dem Betriebskilometrierung gezeigt wird (das würde die Optik zerstören)
     */
    @SuppressWarnings("OverlyLongMethod")
    private void augmentStatSTS() {
        // Der Algorithmus verläuft in einer linearen Sequenz von Schritten:
        // Schritt 1: Bestimmung aller HFB-StraenSegmente:
        List<SystemObject> mainlineSegments;
        // Schritt 1 A: alle äußeren Segmente
        synchronized (_configurationAccess) {
            mainlineSegments = new CopyOnWriteArrayList<>(_äußeresStraßenSegment.getElements());
        }
        // Schritt 1 B: innere Segmente mit HFB-Attribut
        final SystemObjectType innerSegmentType = _configuration.getType("typ.inneresStraßenSegment");
        List<SystemObject> innerSegments;
        synchronized (_configurationAccess) {
            innerSegments = innerSegmentType.getElements();
        }
        AttributeGroup mainlineAtg = _configuration.getAttributeGroup("atg.inneresStraßenSegmentHauptFahrBeziehung");
        for (SystemObject innerSegment : innerSegments) {
            final Data mainlineData;
            synchronized (_configurationAccess) {
                mainlineData = innerSegment.getConfigurationData(mainlineAtg);
            }
            if (null != mainlineData) {
                String value = mainlineData.getTextValue("durchgehendeHauptFahrBeziehung").getValueText();
                if (value.equals("Ja")) {
                    mainlineSegments.add(innerSegment);
                }
            }
        }
        // Schritt 2: HFB-Teilsegmente holen
        Set<SystemObject> mainLinePartialSegments = new HashSet<>();
        AttributeGroup consistsOfLinesAtg = _configuration.getAttributeGroup("atg.bestehtAusLinienObjekten");
        for (SystemObject mainlineSegment : mainlineSegments) {
            final Data consistsOfLinesData;
            synchronized (_configurationAccess) {
                consistsOfLinesData = mainlineSegment.getConfigurationData(consistsOfLinesAtg);
            }
            if (null != consistsOfLinesData) {
                final SystemObject[] refLines = consistsOfLinesData.getReferenceArray("LinienReferenz").getSystemObjectArray();
                Collections.addAll(mainLinePartialSegments, refLines);
            }
        }
        // Schritt 3: Teilsegmente auf ASB-Netz beschränken
        Set<SystemObject> mainlinePartialSegementsWithStat = new HashSet<>();
        AttributeGroup statAtg = _configuration.getAttributeGroup("atg.asbStationierung");
        for (SystemObject mainlinePartialSegment : mainLinePartialSegments) {
            final Data statData;
            synchronized (_configurationAccess) {
                statData = mainlinePartialSegment.getConfigurationData(statAtg);
            }
            if (null != statData) {
                Data.Array array = statData.getArray("AsbStationierung");
                if (array.getLength() > 0) {
                    mainlinePartialSegementsWithStat.add(mainlinePartialSegment);
                }
            }
        }
        // Schritt 4: entfernen die Kilometrierungs-Teilsegmente
        for (SystemObject systemObject : _kmSTS.keySet()) {
            mainlinePartialSegementsWithStat.remove(systemObject);
        }

        // Schritt 5: bestimme Delta-Liste
        for (SystemObject systemObject : _statSTS) {
            mainlinePartialSegementsWithStat.remove(systemObject);
        }

        // Schritt 6: Äste strikt herausfiltern nach Bedarf
        if (ästeStrikt) {
            Set<SystemObject> tempSet = mainlinePartialSegementsWithStat;
            mainlinePartialSegementsWithStat = new HashSet<>();
            for (SystemObject systemObject : tempSet) {
                final Data statData;
                synchronized (_configurationAccess) {
                    statData = systemObject.getConfigurationData(statAtg);
                }
                Data.Array array = statData.getArray("AsbStationierung");
                for (int i = 0; i < array.getLength(); ++i) {
                    Data data = array.getItem(i);
                    String beginNode = data.getTextValue("AnfangsKnoten").getValueText();
                    if (beginNode.endsWith("O") || beginNode.endsWith("A")) {
                        mainlinePartialSegementsWithStat.add(systemObject);
                    } else {
                        String endNode = data.getTextValue("EndKnoten").getValueText();
                        if (endNode.endsWith("O") || endNode.endsWith("A")) {
                            mainlinePartialSegementsWithStat.add(systemObject);
                        }
                    }
                }
            }
        }

        // Schritt 7: Hinzufügen der Straßenteilsegmente
        _statSTS.addAll(mainlinePartialSegementsWithStat);
    }

    /*
    Diese Methode bestimmt in erster Linie die StraßenTeilSegmente, an denen Kilometrierung
    angezeigt werden soll. Daneben legt sie einen Grundstock für solche StraßenTeilSegmente,
    an denen ASB-Stationierung angezeigt werden soll. Dieser Grundstock besteht aus all den
    StraßenTeilSegmenten, auf denen es eine absteigende Betriebskilometrierung gibt.

    Die Methode augmentStatSTS fügt diesem Grundstock weitere StraßenTeilSegmente hinzu.
     */
    private void determineKmAndStatSTS() {
        _kmSTS.clear();
        _statSTS.clear();

        final String geoReferenceType = "typ.straßenTeilSegment";
        final SystemObjectType systemObjectType = _configuration.getType(geoReferenceType);
        final List<SystemObject> segments;
        synchronized (_configurationAccess) {
            segments = systemObjectType.getElements();
        }
        AttributeGroup kmListAtg = _configuration.getAttributeGroup("atg.betriebsKilometerListe");

        for (SystemObject segment : segments) {
            final Data kmData;
            synchronized (_configurationAccess) {
                kmData = segment.getConfigurationData(kmListAtg);
            }
            if (kmData != null) {
                final Data.Array array = kmData.getArray("BetriebsKilometerStraße");
                for (int i = 0; i < array.getLength(); ++i) {
                    List<KmPoint> kmPoints = new ArrayList<>();
                    Data data = array.getItem(i);
                    String rn = data.getReferenceValue("GesamtStraße").getSystemObject()
                        .getConfigurationData(_gesamtStraßeAttributeGroup, _eigenschaftenAspect).getTextValue("Bezeichnung").getValueText();
                    final Data.Array kms = data.getArray("BetriebsKilometer");
                    int length = kms.getLength();
                    for (int j = 0; j < length; ++j) {
                        Data km = kms.getItem(j);
                        KmPoint kmPoint = new KmPoint(rn, km.getTextValue("BlockNummer").getValueText(), km.getScaledValue("Offset").doubleValue(),
                                                      km.getScaledValue("Wert").longValue());
                        kmPoints.add(kmPoint);
                    }
                    if (kmPoints.size() < 2) {
                        // Hier werden alle StraßenTeilSegmente ignoriert, die keine Kilometrierung haben,
                        // oder nur einen einzelnen KmPoint (was nicht vorkommen sollte).
                        if (kmPoints.size() == 1) {
                            _debug.warning("Das STS " + segment.getNameOrPidOrId() + " hat nur einen KM-Wert.");
                        }
                        continue;
                    }
                    Collections.sort(kmPoints);

                    if (hasToBeKm(kmPoints)) {
                        _kmSTS.put(segment, kmPoints);
                    }
                    if (hasToBeStat(kmPoints)) {
                        _statSTS.add(segment);
                    }
                }
            }
        }
    }

    /**
     * Diese Methode erzeugt {@code KmDisplayObjects} auf dem übergebenen {@code SystemObject}, und fügt sie der übergebenen Liste hinzu. Der Zugriff
     * auf die Konfiguration ist synchronisiert.
     *
     * @param painter       ein {@code DisplayObjectPainter}
     * @param dotCollection eine {@code DOTCollection}
     * @param resultList    die Liste, der die neuen Objekte hinzugefügt werden
     */
    public void createKmDisplayObjects(final DisplayObjectPainter painter, final DOTCollection dotCollection, List<DisplayObject> resultList) {
        Map<SystemObject, List<KmPoint>> kmMap = GeoInitializer.getInstance().getKmSTS();

//		Set<String> filterSet = new HashSet<>();
        // Achtung: Im Moment werden alle StraßenTeilSegmente der kmMap vollständig zur Erzeugung von KmPoints
        // benutzt. Das auskommentierte filterSet stammt von einem Ansatz, bei dem für jeden KmPoint der String
        // StraßenName|Blocknummer|km-Wert abgespeichert wurde, und nur für neue Strings wurden die KmPoints
        // tatsächlich angelegt. Das führt zu komischen Artefakten an den Enden von STS mit Kilometrierungs-
        // stücken verschiedener Richtung. Aktuell sieht man bei diesen Segmenten auf beiden Seiten Kilometrierungs-
        // werte, und das ist dann auch so bei der ASB-Stationierung. Also optish einheitlich.
        for (Map.Entry<SystemObject, List<KmPoint>> entry : kmMap.entrySet()) {
            GeoInitializer.getInstance().createKmDisplayObjects(entry, painter, dotCollection, resultList);
        }
    }

    private void createKmDisplayObjects(final Map.Entry<SystemObject, List<KmPoint>> systemObjectWithKm, final DisplayObjectPainter painter,
                                        final DOTCollection dotCollection, List<DisplayObject> resultList) {

        SystemObject systemObject = systemObjectWithKm.getKey();
        List<KmPoint> kmPoints = systemObjectWithKm.getValue();
        if (kmPoints.size() < 2) {
            return;
        }
        double lengthCorrectionFactor = getLengthCorrectionFactor(systemObject);
        for (int k = 0; k < kmPoints.size() - 1; ++k) {
            createKmDisplayObjects(systemObject, painter, dotCollection, kmPoints.get(k), kmPoints.get(k + 1), lengthCorrectionFactor, resultList);
        }
    }

    /* Die folgende Methode berechnet einen Korrekturfaktor für Längen bzw. Offsets.
    Hintergrund: Die Längen oder Offsets in der Konfiguration beziehen sich auf WGS84-Koordinaten,
    doch der GeoInitializer rechnet fast ausschließlich mit UTM-Koordinaten. Deshalb müssen viele
    Längen bzw. Offsets leicht korrigiert werden. */
    private double getLengthCorrectionFactor(SystemObject systemObject) {
        AttributeGroup atg;
        if (systemObject.getType().equals(_straßenTeilSegmentType)) {
            atg = _configuration.getAttributeGroup("atg.straßenTeilSegment");
        } else if (systemObject.getType().equals(_straßenSegmentType)) {
            atg = _configuration.getAttributeGroup("atg.straßenSegment");
        } else if (systemObject.getType().equals(_äußeresStraßenSegment)) {
            atg = _configuration.getAttributeGroup("atg.straßenSegment");
        } else if (systemObject.getType().equals(_inneresStraßenSegment)) {
            atg = _configuration.getAttributeGroup("atg.straßenSegment");
        } else {
            _debug.error("Unbekannter SystemObjektTyp in getLengthCorrectionFactor: " + systemObject.getType());
            return 1.;
        }
        List<Object> coors = getPolylines(systemObject);
        double coorLength = computeLength(coors);

        final Data lengthData;
        synchronized (_configurationAccess) {
            lengthData = systemObject.getConfigurationData(atg);
        }
        double confLength = lengthData.getScaledValue("Länge").doubleValue();
        if (0 == confLength) {
            if (0 == coorLength) {
                return 1.;
            } else {
                _debug.error("Fehler: der Längenkorrekturfaktor für " + systemObject.getNameOrPidOrId() + " konnte nicht bestimmt werden.");
                return 1.;
            }
        }
        return coorLength / confLength;
    }

    /**
     * Diese Methode erzeugt {@code StatDisplayObjects} auf dem übergebenen {@code SystemObject}, und fügt sie der übergebenen Liste hinzu. Der
     * Zugriff auf die Konfiguration ist synchronisiert.
     *
     * @param painter       ein {@code DisplayObjectPainter}
     * @param dotCollection eine {@code DOTCollection}
     * @param resultList    die Liste, der die neuen Objekte hinzugefügt werden
     */
    public void createStatDisplayObjects(final DisplayObjectPainter painter, final DOTCollection dotCollection, List<DisplayObject> resultList) {
        List<SystemObject> segments = GeoInitializer.getInstance().getStatSTS();

        for (SystemObject segment : segments) {
            GeoInitializer.getInstance().createStatDisplayObjects(segment, painter, dotCollection, resultList);
        }
    }

    private void createStatDisplayObjects(final SystemObject systemObject, final DisplayObjectPainter painter, final DOTCollection dotCollection,
                                          List<DisplayObject> resultList) {
        AttributeGroup asbStatAtg = _configuration.getAttributeGroup("atg.asbStationierung");
        final Data asbData;
        synchronized (_configurationAccess) {
            asbData = systemObject.getConfigurationData(asbStatAtg);
        }
        if (asbData != null) {
            AttributeGroup stsAtg = _configuration.getAttributeGroup("atg.straßenTeilSegment");
            final Data stsData;
            synchronized (_configurationAccess) {
                stsData = systemObject.getConfigurationData(stsAtg);
            }
            Double stsLength = stsData.getScaledValue("Länge").doubleValue();

            final Data.Array array = asbData.getArray("AsbStationierung");
            List<StatPoint> statPoints = new ArrayList<>();
            StatPoint statPoint = null;
            for (int i = 0; i < array.getLength(); ++i) {
                Data stat = array.getItem(i);
                if (null != statPoint) {
                    statPoint.setOffset(stat.getScaledValue("Offset").doubleValue());
                }
                if (!isDummy(stat)) {
                    statPoints.add(new StatPoint(stat.getTextValue("AnfangsKnoten").getValueText(), stat.getTextValue("EndKnoten").getValueText(),
                                                 stat.getScaledValue("Offset").doubleValue(), stat.getScaledValue("Anfang").longValue()));
                }
                statPoint = new StatPoint(stat.getTextValue("AnfangsKnoten").getValueText(), stat.getTextValue("EndKnoten").getValueText(), stsLength,
                                          stat.getScaledValue("Ende").longValue());
                statPoints.add(statPoint);
            }
//			if ( statPoints.size()>2) {
//				System.out.println("Segment: " + systemObject.getNameOrPidOrId() + ", Länge: " + stsLength);
//				for(StatPoint st: statPoints) {
//					System.out.println(st);
//				}
//			}
            Collections.sort(statPoints);
            Set<String> filterSet = new HashSet<>();
            double lengthCorrectionFactor = getLengthCorrectionFactor(systemObject);
            for (int k = 0; k < statPoints.size() - 1; ++k) {
                createStatDisplayObjects(systemObject, painter, dotCollection, statPoints.get(k), statPoints.get(k + 1), lengthCorrectionFactor,
                                         filterSet, resultList);
            }
        }
    }

    private void createStatDisplayObjects(final SystemObject systemObject, final DisplayObjectPainter painter, final DOTCollection dotCollection,
                                          final StatPoint p1, final StatPoint p2, final double lengthCorrectionFactor, final Set<String> filterSet,
                                          final List<DisplayObject> resultList) {
        /* Diese Methode erzeugt zwischen den beiden StatPoints an den durch 10 teilbaren Stellen des Segments ein StatDisplayObject. */

        // Vorbereitungen:
        if (!p1.getBeginNode().equals(p2.getBeginNode())) {
            return;
        }
        if (!p1.getEndNode().equals(p2.getEndNode())) {
            return;
        }
        if (p1.getOffset() == p2.getOffset()) {
//			System.out.println("Spezialfall 1: " + systemObject.getNameOrPidOrId());
            return;
        }
        if (p1.getValue() == p2.getValue()) {
//			System.out.println("Spezialfall 2" + systemObject.getNameOrPidOrId());
            return;
        }

        // Nun ist p1.getOffset() < p2.getOffset() und p1.getValue() != p2.getValue().
        long statValue = getNextDecaMeter(p1.getValue(), p2.getValue());
        boolean ascending = p1.getValue() < p2.getValue();
        while ((ascending && statValue <= p2.getValue()) || (!ascending && statValue >= p2.getValue())) {
            double newOffset = p1.getOffset() + ((statValue - p1.getValue()) * (p2.getOffset() - p1.getOffset())) / (p2.getValue() - p1.getValue());
            newOffset *= lengthCorrectionFactor;

            // Koordinatenbestimmung:
            List<Object> lineCoordinates = getPolylines(systemObject);    // liefert schon (customised) UTM
            PointWithAngle newPoint = determinePointCoordinate(lineCoordinates, newOffset);
            if (newPoint != null) {
                String s = p1.getBeginNode() + "|" + p1.getEndNode() + "|" + statValue;
                if (!filterSet.contains(s)) {
                    filterSet.add(s);
                    DisplayObject displayObject =
                        new StatDisplayObject(painter, dotCollection, p1.getBeginNode(), p1.getEndNode(), statValue, newPoint);
                    resultList.add(displayObject);
                }
            } else {
                String msg = "Fehler beim Bilden eines ASB-Stationierungs-Punkts: (" + p1.getBeginNode() + ", " + p1.getEndNode() + ", " + statValue +
                             ") konnte auf " + systemObject.getNameOrPidOrId() + " nicht angelegt werden.";
                _debug.error(msg);
            }
            if (ascending) {
                statValue += 10;
            } else {
                statValue -= 10;
            }
        }
    }

    /**
     * Gibt das die SystemObjects umgebende Rechteck zurück. Ist die Liste leer, so wird das Gesamtrechteck zurückgegeben.
     *
     * @param systemObjects eine Liste von Systemobjekten oder {@code null}
     *
     * @return das anzuzeigende Rechteck
     */
    @Nullable
    public Rectangle getDisplayRectangle(List<SystemObject> systemObjects) {
        if ((systemObjects == null) || systemObjects.isEmpty()) {
            return getDisplayRectangle();
        }
        Rectangle rect = null;
        for (SystemObject systemObject : systemObjects) {
            SystemObjectType systemObjectType = systemObject.getType();
            SystemObjectType pointType = _configuration.getType("typ.punkt");
            SystemObjectType lineType = _configuration.getType("typ.linie");
            SystemObjectType areaType = _configuration.getType("typ.fläche");
            SystemObjectType complexType = _configuration.getType("typ.komplex");

            if (systemObjectType.inheritsFrom(pointType)) {
                List<Object> coordinatesList = GeoInitializer.getInstance().getPointCoordinates(systemObject);
                if (!coordinatesList.isEmpty()) {
                    final PointWithAngle pointWithAngle = (PointWithAngle) coordinatesList.get(0);
                    Point2D point = pointWithAngle.getPoint();
                    if (rect == null) {
                        rect = new Rectangle(new Point((int) point.getX(), (int) point.getY()));
                    } else {
                        rect.add(point);
                    }
                }
            } else if (systemObjectType.inheritsFrom(lineType)) {
                for (Object o : GeoInitializer.getInstance().getPolylines(systemObject)) {
                    Path2D.Double polyline = (Path2D.Double) o;
                    if (rect == null) {
                        rect = polyline.getBounds();
                    } else {
                        rect.add(polyline.getBounds());
                    }
                }
            } else if (systemObjectType.inheritsFrom(areaType)) {
                for (Object o : getAreaCoordinates(systemObject)) {
                    Polygon polygon = (Polygon) o;
                    if (rect == null) {
                        rect = polygon.getBounds();
                    } else {
                        rect.add(polygon.getBounds());
                    }
                }
            } else if (systemObjectType.inheritsFrom(complexType)) {
                for (Object o : getComplexCoordinates(systemObject)) {
	                if (o instanceof PointWithAngle pointWithAngle) {
                        Point2D point = pointWithAngle.getPoint();
                        if (rect == null) {
                            rect = new Rectangle(new Point((int) point.getX(), (int) point.getY()));
                        } else {
                            rect.add(point);
                        }
	                } else if (o instanceof Path2D.Double polyline) {
                        if (rect == null) {
                            rect = polyline.getBounds();
                        } else {
                            rect.add(polyline.getBounds());
                        }
	                } else if (o instanceof Polygon polygon) {
                        if (rect == null) {
                            rect = polygon.getBounds();
                        } else {
                            rect.add(polygon.getBounds());
                        }
                    }
                }
            }
        }
        return increaseRectangle(rect);
    }

    private void createKmDisplayObjects(final SystemObject systemObject, final DisplayObjectPainter painter, final DOTCollection dotCollection,
                                        final KmPoint p1, final KmPoint p2, final double lengthCorrectionFactor, List<DisplayObject> resultList) {
        /* Diese Methode erzeugt zwischen den beiden KmPoints an den durch 10 teilbaren Stellen des Segments ein KmDisplayObject. */

        // Vorbereitungen:
        if (!p1.getRouteNumber().equals(p2.getRouteNumber())) {
            return; // unterschiedliche Straßen (sollte nie eintreten)
        }
        if (!p1.getBlockNumber().equals(p2.getBlockNumber())) {
            return; // unterschiedliche Blöcke
        }
        if (p1.getOffset() == p2.getOffset()) {  // Spezialfall: solche Artefakte treten auf;
            return;                             // diese Fälle, die aufgrund der Abbildung NWSIB/OSM entstanden sind, werden ignoriert
        }
        if (p1.getValue() == p2.getValue()) {
            return;
        }

        // Nun ist p1.getOffset() < p2.getOffset() und p1.getValue() != p2.getValue().
        long kmValue = getNextDecaMeter(p1.getValue(), p2.getValue());
        boolean ascending = p1.getValue() < p2.getValue();
        double deltaOffset = p2.getOffset() - p1.getOffset();
        double deltaValue = p2.getValue() - p1.getValue();
        double stretchFaktor = deltaOffset / deltaValue;
        List<Object> lineCoordinates = getPolylines(systemObject);    // liefert schon (customised) UTM
        while ((ascending && kmValue <= p2.getValue()) || (!ascending && kmValue >= p2.getValue())) {
            double kmProgress = kmValue - p1.getValue();
            double newOffset = p1.getOffset() + kmProgress * stretchFaktor;
            newOffset *= lengthCorrectionFactor;
            // Koordinatenbestimmung:
            PointWithAngle newPoint = determinePointCoordinate(lineCoordinates, newOffset);
            if (newPoint != null) {
                DisplayObject displayObject =
                    new KmDisplayObject(painter, dotCollection, p1.getRouteNumber(), p1.getBlockNumber(), kmValue, newPoint);
                resultList.add(displayObject);
            } else {
                String msg = "Fehler beim Bilden eines Kilometer-Punkts: (" + p1.getRouteNumber() + ", " + p1.getBlockNumber() + ", " + kmValue +
                             ") konnte auf " + systemObject.getNameOrPidOrId() + " nicht angelegt werden.";
                _debug.error(msg);
            }
            if (ascending) {
                kmValue += 10;
            } else {
                kmValue -= 10;
            }
        }
    }

    public void createRnDisplayObjects(final DisplayObjectPainter painter, final DOTCollection dotCollection, List<DisplayObject> resultList) {
        List<NumberedSegment> segments = GeoInitializer.getInstance().getRnSegments();
        GeoInitializer.getInstance().createRnDisplayObjects(segments, painter, dotCollection, resultList);
    }

    /*
    Diese Methode erzeugt eine Auswahl von äußeren Straßensegmenten, die zur Erzeugung der RnDisplayObjects
    verwendet werden; es handelt sich um die TMC-Positiven Segmente, die zumindest einen LocationCode in
    der deutschen LCL haben.
     */
    private List<NumberedSegment> getRnSegments() {
        List<SystemObject> outerSegments;
        synchronized (_configurationAccess) {
            outerSegments = _äußeresStraßenSegment.getElements();
        }
        AttributeGroup lclInfoAtg = _configuration.getAttributeGroup("atg.straßenSegmentLclInfo");
        List<SystemObject> germanTmcPositiveOuterSegments = new ArrayList<>();
        for (SystemObject outerSegment : outerSegments) {
            Data lclInfo = outerSegment.getConfigurationData(lclInfoAtg);
            if (isTmcPositive(outerSegment) && isGerman(lclInfo)) {
                germanTmcPositiveOuterSegments.add(outerSegment);
            }
        }
        List<NumberedSegment> theRnSegments = new ArrayList<>();
        for (SystemObject germanOuterSegment : germanTmcPositiveOuterSegments) {
            NumberedSegment numberedSegment = getNumberedSegment(germanOuterSegment);
            if (numberedSegment != null) {
                theRnSegments.add(numberedSegment);
            }
        }
//		System.out.println(theRnSegments.size() + " RnSegments");
        return theRnSegments;
    }

    private boolean isTmcPositive(final SystemObject outerSegment) {
        Data segmentInfo = outerSegment.getConfigurationData(_äußeresStraßenSegmentAttributeGroup);
        return segmentInfo.getTextValue("TmcRichtung").getValueText().equals("positiv");
    }

    private boolean isGerman(final Data lclSegmentInfo) {
        if (null == lclSegmentInfo) {
            return true; // Das ist besser als false.
        }
        SystemObject tmcLocationCodeBegin = lclSegmentInfo.getReferenceValue("TmcPunktAnfang").getSystemObject();
        Data dataBegin = tmcLocationCodeBegin.getConfigurationData(_tmcLocationCodeAttributeGroup);
        if (dataBegin.getTextValue("VerwaltungsBereich").getValueText().startsWith("D.")) {
            return true;
        }
        SystemObject tmcLocationCodeEnd = lclSegmentInfo.getReferenceValue("TmcPunktEnde").getSystemObject();
        Data dataEnd = tmcLocationCodeEnd.getConfigurationData(_tmcLocationCodeAttributeGroup);
        return dataEnd.getTextValue("VerwaltungsBereich").getValueText().startsWith("D.");
    }

    @Nullable
    private NumberedSegment getNumberedSegment(final SystemObject germanOuterSegment) {
        Data streetData1 = germanOuterSegment.getConfigurationData(_straßenSegmentAttributeGroup);
        SystemObject streetSegment = streetData1.getReferenceValue("gehörtZuStraße").getSystemObject();
        Data streetData2 = streetSegment.getConfigurationData(_straßeAttributeGroup);
        if (streetData2.getTextValue("Typ").getValueText().equals("Autobahn")) {
            Data.NumberValue numberValue = streetData2.getScaledValue("Nummer");
            Data.NumberValue lengthValue = streetData1.getScaledValue("Länge");
            if (numberValue != null && lengthValue != null) {
                return new NumberedSegment(germanOuterSegment, (int) numberValue.longValue(), lengthValue.doubleValue());
            }
        }
        return null;
    }

    private void createRnDisplayObjects(final List<NumberedSegment> segments, final DisplayObjectPainter painter, final DOTCollection dotCollection,
                                        final List<DisplayObject> resultLst) {
        // Strategie: Zunächst lege ich eine Map<Integer,List<PointWithAngle>> an; die Keys sind die
        // Routennummern und die Listen enthalten die Koordinaten.
        Map<Integer, List<PointWithAngle>> coorMap = new HashMap<>();
        for (NumberedSegment segment : segments) {
            if (!coorMap.containsKey(segment.getNumber())) {
                coorMap.put(segment.getNumber(), new ArrayList<>());
            }
            List<Object> coors = getPolylines(segment.getSystemObject());
            if (segment.getLength() > 1000.) {
                int n = (int) (segment.getLength() / 1000.) + 1;
                double partialLength = segment.getLength() / n;
                for (int i = 1; i <= n; ++i) {
                    PointWithAngle pwa = determinePointCoordinate(coors, i * partialLength);
                    if (null != pwa) {
                        coorMap.get(segment.getNumber()).add(pwa);
                    }
                }
            } else if (segment.getLength() > 200.) {
                PointWithAngle pwa = determinePointCoordinate(coors, segment.getLength() / 2.);
                if (null != pwa) {
                    coorMap.get(segment.getNumber()).add(pwa);
                }
            }
        }

        // Strategie: als nächstes bearbeite ich die Listen nacheinander; dazu wähle ich mir eine Menge von Punkten,
        // die paarweise einen gewissen Mindestabstand haben; diesen Mindestabstand

        for (final Map.Entry<Integer, List<PointWithAngle>> entry : coorMap.entrySet()) {
            List<PointWithAngle> category0 = selectPoints(entry.getValue(), 25000., new ArrayList<>());
            createRnDisplayObjects(painter, dotCollection, entry.getKey(), category0, RnDisplayObject.Category.NULL, resultLst);
//			System.out.println("Cat 0: " + category0.size());
            List<PointWithAngle> category1 = selectPoints(entry.getValue(), 10000., category0);
            createRnDisplayObjects(painter, dotCollection, entry.getKey(), category1, RnDisplayObject.Category.ONE, resultLst);
//			System.out.println("Cat 1: " + category1.size());
            category1.addAll(category0);
            List<PointWithAngle> category2 = selectPoints(entry.getValue(), 5000., category1);
            createRnDisplayObjects(painter, dotCollection, entry.getKey(), category2, RnDisplayObject.Category.TWO, resultLst);
//			System.out.println("Cat 2: " + category2.size());
            category2.addAll(category1);
            List<PointWithAngle> category3 = selectPoints(entry.getValue(), 2000., category2);
            createRnDisplayObjects(painter, dotCollection, entry.getKey(), category3, RnDisplayObject.Category.THREE, resultLst);
//			System.out.println("Cat 3: " + category3.size());
            category3.addAll(category2);
            List<PointWithAngle> category4 = selectPoints(entry.getValue(), 500., category3);
            createRnDisplayObjects(painter, dotCollection, entry.getKey(), category4, RnDisplayObject.Category.FOUR, resultLst);
//			System.out.println("Cat 4: " + category4.size());
        }
    }

    @SuppressWarnings("OverlyNestedMethod")
    private void appendCoordinates(SystemObject systemObject, List<Path2D.Double> polylines) {
        // Für jedes Systemobjekt werden entweder nur Koordinaten aus der Komposition
        // oder aus den Koordinatendaten übernommen. Besitzt eine Linie beiderlei Informationen,
        // so werden die Daten der Komposition gewählt.
        if (systemObject == null) {
            return;
        }
        int numberOfPolylines = polylines.size();
        if (systemObject.isOfType(_lineComposedOfLinesType)) {
            final Data linesData;
            synchronized (_configurationAccess) {
                linesData = systemObject.getConfigurationData(_composedOfLinesAttributeGroup);
            }
            if (linesData != null) {
                final SystemObject[] linesArray = linesData.getReferenceArray("LinienReferenz").getSystemObjectArray();
                for (SystemObject subSystemObject : linesArray) {
                    appendCoordinates(subSystemObject, polylines);
                }
            }
        }
        if (numberOfPolylines == polylines.size()) {
            if (systemObject.isOfType(_lineWithCoordinatesType)) {
                final Data coordinatesData;
                synchronized (_configurationAccess) {
                    coordinatesData = systemObject.getConfigurationData(_lineCoordinatesAttributeGroup);
                }
                if (coordinatesData != null) {
                    final Data.NumberArray xArray = coordinatesData.getScaledArray("x");
                    final Data.NumberArray yArray = coordinatesData.getScaledArray("y");
                    int length = Math.min(xArray.getLength(), yArray.getLength());
                    Path2D.Double polyline = new Path2D.Double();
                    final UTMCoordinate utm = new UTMCoordinate();
                    for (int i = 0; i < length; i++) {
                        final Data.NumberValue xValue = xArray.getValue(i);
                        final Data.NumberValue yValue = yArray.getValue(i);
                        if (xValue.isNumber() && yValue.isNumber()) {
                            GeoTransformation.wGS84ToUTM(xValue.doubleValue(), yValue.doubleValue(), utm);
                            customiseUTM(utm);
                            updateExtremeCoordinates(utm);
                            if (i == 0) {
                                polyline.moveTo(utm.getX(), utm.getY());
                            } else {
                                polyline.lineTo(utm.getX(), utm.getY());
                            }
                        }
                    }
                    polylines.add(polyline);
                }
            }
        }
    }

    @Override
    public String toString() {
        return "GeoInitializer{" + "_configuration=" + _configuration + ", _lineComposedOfLinesType=" + _lineComposedOfLinesType +
               ", _lineWithCoordinatesType=" + _lineWithCoordinatesType + ", _composedOfLinesAttributeGroup=" + _composedOfLinesAttributeGroup +
               ", _lineCoordinatesAttributeGroup=" + _lineCoordinatesAttributeGroup + ", _pointOnLineType=" + _pointOnLineType +
               ", _pointWithCoordinatesType=" + _pointWithCoordinatesType + ", _pointOnLineAttributeGroup=" + _pointOnLineAttributeGroup +
               ", _pointCoordinateAttributeGroup=" + _pointCoordinateAttributeGroup + ", _areaCoordinatesAttributeGroup=" +
               _areaCoordinatesAttributeGroup + ", _complexCoordinatesAttributeGroup=" + _complexCoordinatesAttributeGroup +
               ", _gesamtStraßeAttributeGroup=" + _gesamtStraßeAttributeGroup + ", _eigenschaftenAspect=" + _eigenschaftenAspect + ", _xMin=" +
               _xMin + ", _yMin=" + _yMin + ", _xMax=" + _xMax + ", _yMax=" + _yMax + '}';
    }

    @Nullable
    public PointWithAngle getPointWithAngle(SystemObject line, double offset) {
        List<Object> lineCoordinates = getPolylines(line);
        double lengthCorrectionFactor = getLengthCorrectionFactor(line);
        offset *= lengthCorrectionFactor;
        PointWithAngle newPoint = determinePointCoordinate(lineCoordinates, offset);
        if (newPoint != null) {
            return newPoint;
        }
        return null;
    }

    private static class NumberedSegment {
        private final SystemObject _systemObject;
        private final Integer _number;
        private final Double _length;

        NumberedSegment(SystemObject segment, Integer number, Double length) {
            _systemObject = segment;
            _number = number;
            _length = length;
        }

        public SystemObject getSystemObject() {
            return _systemObject;
        }

        public Integer getNumber() {
            return _number;
        }

        public Double getLength() {
            return _length;
        }
    }

    /* Diese Klasse hilft während der Initialisierung der ASB-Stationierung. */
    @SuppressWarnings({"NonFinalFieldReferenceInEquals", "CompareToUsesNonFinalVariable", "NonFinalFieldReferencedInHashCode"})
    private static class StatPoint implements Comparable<StatPoint> {

        private final String _beginNode;
        private final String _endNode;
        private final long _value;
        private double _offset;

        /**
         * Ein StatPoint verkapselt einen Stationierungswert auf einem StraßenTeilSegment.
         *
         * @param beginNode der Anfangsnetzknoten
         * @param endNode   der Endnetzknoten
         * @param offset    das Offset auf dem Segment
         * @param value     der Stationierungswert
         */
        public StatPoint(String beginNode, String endNode, double offset, long value) {
            _beginNode = beginNode;
            _endNode = endNode;
            _offset = offset;
            _value = value;
        }

        public String getBeginNode() {
            return _beginNode;
        }

        public String getEndNode() {
            return _endNode;
        }

        public double getOffset() {
            return _offset;
        }

        public void setOffset(final double offset) {
            _offset = offset;
        }

        public long getValue() {
            return _value;
        }

        @Override
        public boolean equals(Object other) {
	        if (!(other instanceof StatPoint otherPoint)) {
                return false;
            }
            return this._offset == otherPoint._offset && this._beginNode.equals(otherPoint._beginNode) && this._endNode.equals(otherPoint._endNode);
        }

        @Override
        public String toString() {
            return "StatPoint[" + "BeginNode: " + _beginNode + ", EndNode: " + _endNode + ", Offset:" + _offset + ", Value:" + _value + "]";
        }

        @Override
        public int compareTo(final StatPoint other) {
            if (!this._beginNode.equals(other._beginNode)) {
                return this._beginNode.compareTo(other._beginNode);
            }
            if (!this._endNode.equals(other._endNode)) {
                return this._endNode.compareTo(other._endNode);
            }
            return Double.compare(this._offset, other._offset);
        }

        @Override
        public int hashCode() {
            int result;
            result = _beginNode != null ? _beginNode.hashCode() : 0;
            result = 31 * result + (_endNode != null ? _endNode.hashCode() : 0);
	        result = 31 * result + Double.hashCode(_offset);
	        result = 31 * result + Long.hashCode(_value);
            return result;
        }
    }

    /* Diese Klasse hilf während der Initialisierung der Betriebskilometrierung. */
    private static class KmPoint implements Comparable<KmPoint> {

        private final String _rn;
        private final String _blockNumber;
        private final double _offset;
        private final long _value;

        /**
         * Ein KmPoint verkapselt einen Kilometrierungswert auf einem StraßenTeilSegment.
         *
         * @param rn          der Straßennname
         * @param blockNumber die Blocknummer
         * @param offset      das Offset auf dem Segment
         * @param value       der Km-Wert
         */
        public KmPoint(String rn, String blockNumber, double offset, long value) {
            _rn = rn;
            _blockNumber = blockNumber;
            _offset = offset;
            _value = value;
        }

        /**
         * Gibt den Straßennamen zurück.
         *
         * @return den Namen
         */
        public String getRouteNumber() {
            return _rn;
        }

        /**
         * Gibt die Blocknummer zurück.
         *
         * @return die Blocknummer
         */
        public String getBlockNumber() {
            return _blockNumber;
        }

        /**
         * Gibt das Offset auf dem Segment zurück.
         *
         * @return das Offset
         */
        public double getOffset() {
            return _offset;
        }

        /**
         * Gibt den Km-Wert zurück.
         *
         * @return der Km-Wert
         */
        public long getValue() {
            return _value;
        }

        @Override
        public boolean equals(Object other) {
	        if (!(other instanceof KmPoint otherPoint)) {
                return false;
            }
            return this._offset == otherPoint._offset && this._blockNumber.equals(otherPoint._blockNumber) && this._rn.equals(otherPoint._rn);
        }

        @Override
        public int compareTo(@NotNull KmPoint other) {
            if (!this._rn.equals(other._rn)) {
                return this._rn.compareTo(other._rn);
            }
            if (!this._blockNumber.equals(other._blockNumber)) {
                return this._blockNumber.compareTo(other._blockNumber);
            }
            return Double.compare(this._offset, other._offset);
        }

        @Override
        public int hashCode() {
            int result;
            result = _rn != null ? _rn.hashCode() : 0;
            result = 31 * result + (_blockNumber != null ? _blockNumber.hashCode() : 0);
	        result = 31 * result + Double.hashCode(_offset);
	        result = 31 * result + Long.hashCode(_value);
            return result;
        }

        @Override
        public String toString() {
            return "KmPoint{" + "_rn='" + _rn + '\'' + ", _blockNumber='" + _blockNumber + '\'' + ", _offset=" + _offset + ", _value=" + _value + '}';
        }
    }
}
