/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.elrPlugin;

import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.Attribute;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.AttributeListDefinition;
import de.bsvrz.dav.daf.main.config.ConfigurationObject;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.ObjectSet;
import de.bsvrz.dav.daf.main.config.ReferenceAttributeType;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.kappich.pat.gnd.displayObjectToolkit.DOTCollection;
import de.kappich.pat.gnd.displayObjectToolkit.DisplayObject;
import de.kappich.pat.gnd.displayObjectToolkit.GeoInitializer;
import de.kappich.pat.gnd.displayObjectToolkit.OnlineDisplayObject;
import de.kappich.pat.gnd.displayObjectToolkit.PrimitiveFormPropertyPair;
import de.kappich.pat.gnd.extLocRef.ComposedReference;
import de.kappich.pat.gnd.extLocRef.DirectedReference;
import de.kappich.pat.gnd.extLocRef.ReferenceHierarchy;
import de.kappich.pat.gnd.extLocRef.SimpleAttributeReference;
import de.kappich.pat.gnd.extLocRef.SimpleReference;
import de.kappich.pat.gnd.extLocRef.SimpleSetReference;
import de.kappich.pat.gnd.gnd.MapPane;
import de.kappich.pat.gnd.layerManagement.Layer;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectPainter;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectTypePlugin;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectsInitializer;
import de.kappich.pat.gnd.utils.PointWithAngle;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.swing.JProgressBar;
import javax.swing.SwingUtilities;

/**
 * Diese Klasse implementiert {@link DisplayObjectsInitializer} für das "Erweiterte Ortsreferenzen"-Plugin.
 *
 * @author Kappich Systemberatung
 */
public class ElrInitializer implements DisplayObjectsInitializer {

    private static List<Object> getReferenceCoordinate(final List<Object> coorList) {
        List<Object> resultList = new ArrayList<>();
        double x = 0.;
        double y = 0.;
        int counter = 0;
        for (Object o : coorList) {
	        if (o instanceof Point2D.Double p) {
                x += p.getX();
                y += p.getY();
                ++counter;
	        } else if (o instanceof PointWithAngle pwa) {
                Point2D p = pwa.getPoint();
                x += p.getX();
                y += p.getY();
                ++counter;
            }
        }
        if (counter > 0) {
            Point2D.Double newPoint = new Point2D.Double(x / counter, y / counter);
            resultList.add(new PointWithAngle(newPoint, null));  // der DOTPointPainter erwartet PointWithAngle
        }
        return resultList;
    }

    private static void workForNextComposedReference(final ComposedReference reference, final Collection<SystemObject> systemObjects,
                                                     final Map<SystemObject, Set<SystemObject>> objectReferenceMap) {
        // Diese Methode berechnet das Lookup für eine einzelne EOR. Dazu werden nur (noch) jene SystemObjects
        // des Layers herangezogen, denen nicht schon zuvor durch eine EOR andere SystemObjects zugeordnet
        // wurden.
        Set<SystemObject> filterSet = new HashSet<>();
        for (SystemObject systemObject : systemObjects) {
            if (!objectReferenceMap.containsKey(systemObject)) {
                filterSet.add(systemObject);
            }
        }
        if (filterSet.isEmpty()) {  // ein klein wenig Optimierung
            return;
        }
        // und nun die eigentliche Arbeit
        workForComposedReference(reference, Collections.unmodifiableSet(filterSet), objectReferenceMap);
    }

    private static void workForComposedReference(final ComposedReference reference, final Set<SystemObject> filterSet,
                                                 final Map<SystemObject, Set<SystemObject>> objectReferenceMap) {
        // Die Aufgabe dieser Methode besteht darin, für so viele Objekte wie möglich aus dem filterSet
        // einen Eintrag der objectReferenceMap hinzuzufügen. Solche Einträge sind nur für jene Objekte
        // zu machen, für die für jede DirectedReference eine weitere Zuordnung möglich ist.

        // Die folgende Map enthält die während der Schleife die Zuordnung der SystemObjects aus dem filterSet
        // zu den bislang abgearbeiteten DirectedReferences.
        Map<SystemObject, Set<SystemObject>> interimMap = null;
        Set<SystemObject> currentFilterSet = new HashSet<>(filterSet);
        boolean firstStep = true;
        for (DirectedReference directedReference : reference.getDirectedReferences().getDirectedReferences()) {
            Map<SystemObject, Set<SystemObject>> nextStepMap = workForNextDirectedReference(directedReference, currentFilterSet);
            // Update interimMap
            if (firstStep) {
                // beim ersten Schritt ist alles einfach:
                interimMap = nextStepMap;
                firstStep = false;
            } else {
                // Danach aber ist ein sorgfältiges Update nötig. Für jeden Key in der interimMap werden
                // alle erreichbaren SystemObjects bestimmt, und am Ende wird entweder der Key gelöscht,
                // wenn es nicht weitergeht, oder aber das alte Set durch das neue ersetzt.
                for (Iterator<Map.Entry<SystemObject, Set<SystemObject>>> it = interimMap.entrySet().iterator(); it.hasNext(); ) {
                    Set<SystemObject> newSet = new HashSet<>();
                    Map.Entry<SystemObject, Set<SystemObject>> entry = it.next();
                    for (SystemObject systemObject : entry.getValue()) {
                        if (nextStepMap.containsKey(systemObject)) {
                            newSet.addAll(nextStepMap.get(systemObject));
                        }
                    }
                    if (newSet.isEmpty()) {
                        it.remove();
                    } else {
                        interimMap.put(entry.getKey(), newSet);
                    }
                }
            }
            // Update des currentFilterSet für den nächsten Schritt:
            currentFilterSet.clear();
            for (Set<SystemObject> systemObjects : interimMap.values()) {
                currentFilterSet.addAll(systemObjects);
            }
        }
        // Einarbeiten der Ergebnisse in die objectReferenceMap:
        if (null != interimMap) {
            for (final Map.Entry<SystemObject, Set<SystemObject>> entry : interimMap.entrySet()) {
                objectReferenceMap.put(entry.getKey(), entry.getValue());
            }
        }
    }

    // Die folgende Methode schaut, ob sich die SystemObjects aus dem filterSet mit Hilfe der DirectedReference
    // auf andere SystemObjects abbilden lassen. Das Ergebnis gibt sie in der Map zurück.
    @SuppressWarnings("OverlyNestedMethod")
    private static Map<SystemObject, Set<SystemObject>> workForNextDirectedReference(final DirectedReference directedReference,
                                                                                     Set<SystemObject> filterSet) {
        Map<SystemObject, Set<SystemObject>> resultMap = new HashMap<>();
        if (filterSet.isEmpty()) {
            return resultMap;
        }
        SystemObjectType firstType = directedReference.getSimpleReference().getFirstType(); // hier ohne Richtung
        List<SystemObject> firstTypeElements = firstType.getElements();
        SimpleReference sr = directedReference.getSimpleReference();
	    if (sr instanceof SimpleAttributeReference sar) {
            for (SystemObject object : firstTypeElements) {
                Set<SystemObject> referencedObjects = getSystemObjects(object, sar);
                if (!referencedObjects.isEmpty()) {
                    if (!directedReference.isReversed()) {
                        if (filterSet.contains(object)) {
                            resultMap.put(object, referencedObjects);
                        }
                    } else {
                        for (SystemObject referenced : referencedObjects) {
                            if (filterSet.contains(referenced)) {
                                if (resultMap.containsKey(referenced)) {
                                    resultMap.get(referenced).add(object);
                                } else {
                                    Set<SystemObject> newSet = new HashSet<>();
                                    newSet.add(object);
                                    resultMap.put(referenced, newSet);
                                }
                            }
                        }
                    }
                }
            }
	    } else if (sr instanceof SimpleSetReference ssr) {
            for (SystemObject object : firstTypeElements) {
                Set<SystemObject> referencedObjects = getSystemObjects(object, ssr);
                if (!referencedObjects.isEmpty()) {
                    if (!directedReference.isReversed()) {
                        if (filterSet.contains(object)) {
                            resultMap.put(object, referencedObjects);
                        }
                    } else {
                        for (SystemObject referenced : referencedObjects) {
                            if (filterSet.contains(referenced)) {
                                if (resultMap.containsKey(referenced)) {
                                    resultMap.get(referenced).add(object);
                                } else {
                                    Set<SystemObject> newSet = new HashSet<>();
                                    newSet.add(object);
                                    resultMap.put(referenced, newSet);
                                }
                            }
                        }
                    }
                }
            }
        }
        return resultMap;
    }

    private static Set<SystemObject> getSystemObjects(final SystemObject object, final SimpleAttributeReference sar) {
        Data atgData = object.getConfigurationData(sar.getAttributeGroup());
        return branch(atgData, object, sar.getAttributeGroup(), sar.getAttribute(), sar.getTheListKeys());
    }

    @NotNull
    private static Set<SystemObject> branch(Data atgData, SystemObject object, AttributeGroup atg, Attribute attr, List<String> listKeys) {
        if (atgData != null) {
            if (atgData.isList()) {
                return getDataForList(atgData, object, atg, attr, listKeys);
            } else if (atgData.isArray() && atgData.getAttributeType() instanceof AttributeListDefinition) {
                Data.Array atgDataArray = atgData.asArray();
                Set<SystemObject> set = new HashSet<>();
                for (int i = 0; i < atgDataArray.getLength(); ++i) {
                    set.addAll(getDataForList(atgDataArray.getItem(i), object, atg, attr, listKeys));
                }
                return set;
            } else if (atgData.isArray() && atgData.getAttributeType() instanceof ReferenceAttributeType) {
                Data.Array atgDataArray = atgData.asArray();
                Set<SystemObject> set = new HashSet<>();
                for (int i = 0; i < atgDataArray.getLength(); ++i) {
                    set.addAll(getDataForReference(atgDataArray.getItem(i)));
                }
                return set;
            } else if (atgData.getAttributeType() instanceof ReferenceAttributeType) {
                return getDataForReference(atgData);
            }
        }
        return new HashSet<>();
    }

    // getSystemObjects für AttributGruppen:

    private static Set<SystemObject> getDataForList(Data list, SystemObject object, AttributeGroup atg, Attribute attr, List<String> listKeys) {
        List<String> newListKeys = new ArrayList<>(listKeys);
        String s = newListKeys.remove(0);
        Data atgData = list.getItem(s);
        return branch(atgData, object, atg, attr, newListKeys);
    }

    private static Set<SystemObject> getDataForReference(Data atgData) {
        Set<SystemObject> resultSet = new HashSet<>();
        Data.ReferenceValue referenceValue = atgData.asReferenceValue();
        if (referenceValue != null) {
            SystemObject object = referenceValue.getSystemObject();
            if (null != object) {
                resultSet.add(object);
            }
        }
        return resultSet;
    }

    @NotNull
    private static Set<SystemObject> getSystemObjects(final SystemObject object, final SimpleSetReference ssr) {
	    if (object instanceof ConfigurationObject configurationObject) {
            ObjectSet objectSet = configurationObject.getObjectSet(ssr.getSetName());
            if (null != objectSet) {
                return new HashSet<>(objectSet.getElements());
            }
        }
        return new HashSet<>();
    }

    @SuppressWarnings("OverlyLongMethod")
    @Override
    public void initializeDisplayObjects(final DataModel configuration, final Layer layer, final MapPane mapPane, final JProgressBar progressBar,
                                         final List<DisplayObject> returnList) {
        DisplayObjectTypePlugin elrPlugin = new DOTElrPlugin();
        if (!layer.getPlugin().getName().equals(elrPlugin.getName())) {
            return;
        }
        List<SystemObject> systemObjects = configuration.getType(layer.getConfigurationObjectType()).getElements();
        ReferenceHierarchy hierarchy = layer.getReferenceHierarchy();
        if (null == hierarchy) {
            return;
        }

        SwingUtilities.invokeLater(() -> progressBar.setIndeterminate(true));

        // Nach den Überprüfungen wird nun die Map deklariert und gefüllt, die das Lookup zwischen
        // den SystemObjects des Layers und den die Ortsreferenz vererbenden SystemObjects enthält.
        Map<SystemObject, Set<SystemObject>> objectReferenceMap = determineObjectReferences(hierarchy, systemObjects);

        // Als nächstes werden die Koordinaten der referenzierten SystemObjects bestimmt.
        Map<SystemObject, List<Object>> coordinatesOfReferenced = new HashMap<>();
        GeoInitializer geoInitializer = GeoInitializer.getInstance(configuration);
        for (final Map.Entry<SystemObject, Set<SystemObject>> systemObjectSetEntry : objectReferenceMap.entrySet()) {
            Set<SystemObject> referencedObjects = systemObjectSetEntry.getValue();
            for (SystemObject referencedObject : referencedObjects) {
                if (!coordinatesOfReferenced.containsKey(referencedObject)) {
                    switch (hierarchy.getGeometryType()) {
                        case "Punkt":
                            coordinatesOfReferenced.put(referencedObject, geoInitializer.getPointCoordinates(referencedObject));
                            break;
                        case "Linie":
                            coordinatesOfReferenced.put(referencedObject, geoInitializer.getPolylines(referencedObject));
                            break;
                        case "Fläche":
                            coordinatesOfReferenced.put(referencedObject, geoInitializer.getAreaCoordinates(referencedObject));
                            break;
                        case "Komplex":
                            coordinatesOfReferenced.put(referencedObject, geoInitializer.getComplexCoordinates(referencedObject));
                            break;
                    }
                }
            }
        }
        // Nun werden die Koordinaten der SystemObjects des Layers bestimmt; zunächst werden alle Korrdinaten
        // der referenzierten Objekte aggregiert.
        Map<SystemObject, List<Object>> coordinatesOfLayer = new HashMap<>();
        for (final Map.Entry<SystemObject, Set<SystemObject>> entry : objectReferenceMap.entrySet()) {
            List<Object> coors = new ArrayList<>();
            for (SystemObject referenced : entry.getValue()) {
                coors.addAll(coordinatesOfReferenced.get(referenced));
            }
            if (!coors.isEmpty()) {
                coordinatesOfLayer.put(entry.getKey(), coors);
            }
        }
        // Für Punkte berechnen wir nun eine Referenzkoordinate:
        if (hierarchy.getGeometryType().equals("Punkt")) {
	        coordinatesOfLayer.replaceAll((k, v) -> getReferenceCoordinate(v));
        }

        // Nun können die DisplayObjects initialisiert werden
        final DOTCollection dotCollection = layer.getDotCollection();
        final Iterator<DisplayObjectType> iterator = dotCollection.values().iterator();
        if (!iterator.hasNext()) {
            return;
        }
        final DisplayObjectType dot = iterator.next();
        final DisplayObjectTypePlugin displayObjectTypePlugin = dot.getDisplayObjectTypePlugin();
        DisplayObjectPainter painter = displayObjectTypePlugin.getPainter();
        final Map<DisplayObjectType, List<PrimitiveFormPropertyPair>> pfpPairs = dotCollection.getPrimitiveFormPropertyPairs();
        for (final Map.Entry<SystemObject, List<Object>> entry : coordinatesOfLayer.entrySet()) {
            OnlineDisplayObject displayObject = new OnlineDisplayObject(entry.getKey(), entry.getValue(), painter, dotCollection, pfpPairs, mapPane);
            returnList.add(displayObject);
        }
    }

    // getSystemObjects für Mengen:

    public Map<SystemObject, Set<SystemObject>> determineObjectReferences(final ReferenceHierarchy hierarchy,
                                                                          Collection<SystemObject> systemObjects) {
        Map<SystemObject, Set<SystemObject>> objectReferenceMap = new HashMap<>();
        for (ComposedReference reference : hierarchy.getComposedReferences().getComposedReferences()) {
            // Herunterbrechen auf einzelne EOR:
            workForNextComposedReference(reference, systemObjects, objectReferenceMap);
        }
        return objectReferenceMap;
    }

}
