/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.extLocRef;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.Attribute;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.debug.Debug;
import de.kappich.pat.gnd.gnd.PreferencesHandler;
import de.kappich.pat.gnd.utils.view.PreferencesDeleter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import javax.swing.table.AbstractTableModel;

/**
 * Dieser Manager verwaltet alle {@link ComposedReference ComposedReference-Objekte}, also Definitionen von Erweiterten Ortsreferenzen (EOR).
 *
 * @author Kappich Systemberatung
 */
@SuppressWarnings("serial")
public final class ComposedReferenceManager extends AbstractTableModel {

    private static final Debug _debug = Debug.getLogger();
    private static final Object _initializationAccess = new Object();
    private static ComposedReferenceManager _instance;
    private final String[] _columnNames = {"Name der Erweiterten Ortsreferenz"};
    private final ClientDavInterface _connection;
    private final DataModel _configuration;
	private final List<ComposedReference> _composedReferences = new ArrayList<>() {
        @Override
        public boolean add(ComposedReference composedReference) {
            int index = Collections.binarySearch(this, composedReference);
            if (index < 0) {
                index = ~index;
            }
            super.add(index, composedReference);
            return true;
        }
    };
    private final Map<String, ComposedReference> _composedReferencesMap = new HashMap<>();
    private final List<ComposedReferenceManager.CrChangeListener> _listeners = new CopyOnWriteArrayList<>();
    private final Set<String> _unchangeables = new HashSet<>();

    private ComposedReferenceManager(ClientDavInterface connection) {
        _connection = connection;
        _configuration = connection.getDataModel();
        initDefaultComposedReferences();
        initUserDefinedComposedReferences();
    }

    public static ComposedReferenceManager getInstance(ClientDavInterface connection) {
        //noinspection SynchronizationOnStaticField
        synchronized (_initializationAccess) {
            if (null == _instance) {
                _instance = new ComposedReferenceManager(connection);
            }
            return _instance;
        }
    }

    public static ComposedReferenceManager getInstance() {
        //noinspection SynchronizationOnStaticField
        synchronized (_initializationAccess) {
            return _instance;
        }
    }

    /**
     * Gibt den Ausgangsknoten zum Abspeichern aller Präferenzen des ComposedReferenceManager an.
     *
     * @return gibt den Ausgangsknoten zum Abspeichern aller Präferenzen des ComposedReferenceManager zurück
     */
    private static Preferences getPreferenceStartPath() {
        return PreferencesHandler.getInstance().getPreferenceStartPath().node("EOR");
    }

    /**
     * Gibt {@code true} zurück, wenn die ComposedReference veränderbar ist. Im Moment ist ein ComposedReference genau dann unveränderbar, wenn er im
     * Kode definiert ist.
     *
     * @param cr eine ComposedReference
     *
     * @return {@code true} genau dann, wenn die ComposedReference veränderbar ist
     */
    public boolean isChangeable(ComposedReference cr) {
        return !_unchangeables.contains(cr.getName());
    }

    /**
     * Gibt eine Liste aller EOR zurück.
     *
     * @return die Liste aller EOR
     */
    public List<ComposedReference> getComposedReferences() {
        return Collections.unmodifiableList(_composedReferences);
    }

    /**
     * Gibt eine Liste aller EOR-Namen zurück.
     *
     * @return die Liste aller EOR-Namen
     */
    public Object[] getComposedReferenceNames(final String geometryType) {
        List<String> names = new ArrayList<>();
        for (ComposedReference cr : _composedReferences) {
            if (cr.getGeometryType().equals(geometryType)) {
                names.add(cr.getName());
            }
        }
        return names.toArray();
    }

    /**
     * Gibt den EOR mit dem übergebenen Namen zurück.
     *
     * @param name der Name
     *
     * @return die geforderten ComposedReference
     */
    public ComposedReference getComposedReference(String name) {
        return _composedReferencesMap.get(name);
    }

    public boolean hasComposedReference(String name) {
        return _composedReferencesMap.containsKey(name);
    }

    /**
     * Gibt den EOR an der i-ten Stelle der EOR-Liste zurück, wobei die Zählung mit 0 beginnt.
     *
     * @param i ein Index
     *
     * @return die geforderten ComposedReference
     */
    public ComposedReference getComposedReference(int i) {
        return _composedReferences.get(i);
    }

    public void addComposedReference(final ComposedReference composedReference) {
        if (_composedReferencesMap.containsKey(composedReference.getName())) {
            throw new IllegalArgumentException("Ein EOR mit diesem Namen existiert bereits.");
        }
        // Klein-/Großschreibung nicht signifikant:
        for (ComposedReference cr : _composedReferences) {
            if (composedReference.equals(cr)) {
                throw new IllegalArgumentException(
                    "Es existiert bereits eine EOR, dessen Name sich nur bezüglich Klein-/Großschreibung unterscheidet.");
            }
        }
        _composedReferences.add(composedReference);
        _composedReferencesMap.put(composedReference.getName(), composedReference);
        composedReference.putPreferences(getPreferenceStartPath());
        fireTableDataChanged();
        notifyChangeListenersReferenceChanged(composedReference);
    }

    public void changeComposedReference(final ComposedReference composedReference) {
        final String name = composedReference.getName();
        if (!_composedReferencesMap.containsKey(composedReference.getName())) {
            throw new IllegalArgumentException("Ein EOR mit diesem Namen existiert nicht.");
        }
        final ComposedReference existing = _composedReferencesMap.get(name);
        existing.deletePreferences(getPreferenceStartPath());
        existing.setInfo(composedReference.getInfo());
        existing.setDirectedReferences(composedReference.getDirectedReferences(), true);
        existing.setGeometryType(composedReference.getGeometryType());
        for (int i = 0; i < _composedReferences.size(); ++i) {
            if (_composedReferences.get(i).getName().equals(name)) {
                fireTableRowsUpdated(i, i);
                break;
            }
        }
        composedReference.putPreferences(getPreferenceStartPath());
        notifyChangeListenersReferenceChanged(composedReference);
    }

    public boolean removeComposedReference(ComposedReference reference) {
        if (null == reference) {
            return false;
        }
        if (ReferenceHierarchyManager.getInstance(_connection).isUsed(reference)) {
            return false;
        }
        final String name = reference.getName();
        if (!_unchangeables.contains(name)) {
            final int index = remove(name);
            if (index > -1) {
                reference.deletePreferences(getPreferenceStartPath());
                fireTableRowsDeleted(index, index);
                notifyChangeListenersReferenceRemoved(name);
                return true;
            }
        }
        return false;
    }

    private int remove(final String name) {
        int index = 0;
        for (ComposedReference reference : _composedReferences) {
            if (name.equals(reference.getName())) {
                _composedReferences.remove(index);
                _composedReferencesMap.remove(name);
                return index;
            }
            index++;
        }
        return -1;
    }

    /*
     * Gehört zur Implementation des TableModel.
     */
    @Override
    public int getColumnCount() {
        return _columnNames.length;
    }

    /*
     * Gehört zur Implementation des TableModel.
     */
    @Override
    public int getRowCount() {
        return _composedReferences.size();
    }

    /*
     * Gehört zur Implementation des TableModel.
     */
    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
        return _composedReferences.get(rowIndex).getName();
    }

    /*
     * Gehört zur Implementation des TableModel.
     */
    @Override
    public String getColumnName(int columnIndex) {
        return _columnNames[columnIndex];
    }

    @SuppressWarnings("unused")
    public void addCrChangeListener(final ComposedReferenceManager.CrChangeListener listener) {
        _listeners.add(listener);
    }

    @SuppressWarnings("unused")
    public void removeCrChangeListener(final ComposedReferenceManager.CrChangeListener listener) {
        _listeners.remove(listener);
    }

    @SuppressWarnings("unused")
    public void clearCrChangeListeners() {
        _listeners.clear();
    }

    /**
     * Informiert die auf Änderungen angemeldeten Listener über eine neu hinzugefügte {@link ComposedReference}.
     *
     * @param reference eine ComposedReference
     */
    @SuppressWarnings("unused")
    private void notifyChangeListenerReferenceAdded(final ComposedReference reference) {
        for (ComposedReferenceManager.CrChangeListener listener : _listeners) {
            listener.composedReferenceAdded(reference);
        }
    }

    /**
     * Informiert die auf Änderungen angemeldeten Listener über eine geänderte {@link ComposedReference}.
     *
     * @param reference eine ComposedReference
     */
    private void notifyChangeListenersReferenceChanged(final ComposedReference reference) {
        for (ComposedReferenceManager.CrChangeListener listener : _listeners) {
            listener.composedReferenceChanged(reference);
        }
    }

    /**
     * Informiert die auf Änderungen angemeldeten Listener über eine gelöschten {@link ComposedReference}.
     *
     * @param name eine Name
     */
    private void notifyChangeListenersReferenceRemoved(final String name) {
        for (ComposedReferenceManager.CrChangeListener listener : _listeners) {
            listener.composedReferenceRemoved(name);
        }
    }

    private void initDefaultComposedReferences() {
        initFSinheritsFromMQ();
        initFSinheritsFromDE();
        initFSinheritsFromEAK();
    }

    //++++

    private void initFSinheritsFromMQ() {
        List<DirectedReference> directedReferences = new ArrayList<>();
        SystemObjectType fsType = _configuration.getType("typ.fahrStreifen");
        SystemObjectType mqType = _configuration.getType("typ.messQuerschnitt");
        String setName = "FahrStreifen";

        Set<SimpleSetReference> fsRefsRev = SimpleReferenceManager.getInstance(_connection).getSimpleSetReferences(fsType, true);
        for (SimpleSetReference ref : fsRefsRev) {
            if (ref.getFirstType().equals(mqType) && ref.getSetName().equals(setName)) {
                directedReferences.add(new DirectedReference("FS von MQ", ref, true));
            }
        }
        if (directedReferences.isEmpty()) {
            _debug.error("Achtung: die vordefinierte EOR 'Fahrstreifen -> Messquerschnitt' konnte nicht initialisiert werden.");
            return;
        }
        ComposedReference cr =
            new ComposedReference("Fahrstreifen -> Messquerschnitt", "Fahrstreifen erhalten die EOR von MQs über deren Menge Fahrstreifen", "Punkt",
                                  directedReferences);
        _composedReferences.add(cr);
        _composedReferencesMap.put(cr.getName(), cr);
        _unchangeables.add(cr.getName());
    }

    private void initFSinheritsFromDE() {
        List<DirectedReference> directedReferences = new ArrayList<>();
        SystemObjectType fsType = _configuration.getType("typ.fahrStreifen");
        SystemObjectType deType = _configuration.getType("typ.deLve");
        AttributeGroup atg = _configuration.getAttributeGroup("atg.fahrStreifen");
        Attribute attribute = atg.getAttribute("FahrStreifenQuelle");

        Set<SimpleAttributeReference> fsRefs = SimpleReferenceManager.getInstance(_connection).getSimpleAttributeReferences(fsType, false);
        for (SimpleAttributeReference ref : fsRefs) {
            if (ref.getSecondType().equals(deType) && ref.getAttributeGroup().equals(atg) && ref.getAttribute().equals(attribute)) {
                directedReferences.add(new DirectedReference("FS von DE", ref, false));
            }
        }
        if (directedReferences.isEmpty()) {
            _debug.error("Achtung: die vordefinierte EOR 'Fahrstreifen -> DeLve' konnte nicht initialisiert werden.");
            return;
        }
        ComposedReference cr = new ComposedReference("Fahrstreifen -> DeLve",
                                                     "Fahrstreifen erhalten die EOR von De über Attributgruppe Fahrstreifen " +
                                                     "und Attribute FahrStreifenQuelle", "Punkt", directedReferences);
        _composedReferences.add(cr);
        _composedReferencesMap.put(cr.getName(), cr);
        _unchangeables.add(cr.getName());
    }

    private void initFSinheritsFromEAK() {
        List<DirectedReference> directedReferences = new ArrayList<>();
        SystemObjectType fsType = _configuration.getType("typ.fahrStreifen");
        SystemObjectType deType = _configuration.getType("typ.deLve");
        AttributeGroup atg = _configuration.getAttributeGroup("atg.fahrStreifen");
        Attribute attribute = atg.getAttribute("FahrStreifenQuelle");

        Set<SimpleAttributeReference> fsRefs = SimpleReferenceManager.getInstance(_connection).getSimpleAttributeReferences(fsType, false);
        for (SimpleAttributeReference ref : fsRefs) {
            if (ref.getSecondType().equals(deType) && ref.getAttributeGroup().equals(atg) && ref.getAttribute().equals(attribute)) {
                directedReferences.add(new DirectedReference("FS von DE", ref, false));
            }
        }
        if (directedReferences.size() != 1) {
            _debug.error("Achtung: die vordefinierte EOR 'Fahrstreifen -> Eak' (über DeLve und menge.de) konnte nicht initialisiert werden.");
            return;
        }

        SystemObjectType eakType = _configuration.getType("typ.eak");
        String setName = "De";
        Set<SimpleSetReference> eakRefs = SimpleReferenceManager.getInstance(_connection).getSimpleSetReferences(eakType, false);
        for (SimpleSetReference ref : eakRefs) {
            if (deType.inheritsFrom(ref.getSecondType()) && ref.getSetName().equals(setName)) {
                directedReferences.add(new DirectedReference("DE von EAK", ref, true));
            }
        }
        if (directedReferences.size() != 2) {
            _debug.error("Achtung: die vordefinierte EOR 'Fahrstreifen -> Eak' konnte nicht initialisiert werden.");
            return;
        }

        ComposedReference cr =
            new ComposedReference("Fahrstreifen -> Eak (über DeLve und menge.de)", "Fahrstreifen erhalten die EOR von Eak über De", "Punkt",
                                  directedReferences);
        _composedReferences.add(cr);
        _composedReferencesMap.put(cr.getName(), cr);
        _unchangeables.add(cr.getName());
    }

    private void initUserDefinedComposedReferences() {
        Preferences classPrefs = getPreferenceStartPath();
        String[] composedReferenceNames;
        try {
            composedReferenceNames = classPrefs.childrenNames();
        } catch (BackingStoreException | IllegalStateException ignored) {
            PreferencesDeleter pd =
                new PreferencesDeleter("Die benutzer-definierten Erweiterten Ortsreferenzen (EOR) können nicht geladen werden.", classPrefs);
            pd.run();
            return;
        }
        for (String name : composedReferenceNames) {
            Preferences layerPrefs = classPrefs.node(name);
            ComposedReference reference = new ComposedReference();
            if (reference.initializeFromPreferences(layerPrefs, _configuration)) {
                _composedReferences.add(reference);
                _composedReferencesMap.put(reference.getName(), reference);
            }
        }
    }

    @Override
    public String toString() {
        return "ComposedReferenceManager{}";
    }

    /**
     * Ein Interface für Listener, die über das Hinzufügen, Löschen und Ändern von ComposedReference informiert werden wollen.
     *
     * @author Kappich Systemberatung
     */
    public interface CrChangeListener {
        /**
         * Diese Methode wird aufgerufen, wenn die EOR hinzugefügt wurde.
         *
         * @param composedReference eine ComposedReference
         */
        void composedReferenceAdded(final ComposedReference composedReference);

        /**
         * Diese Methode wird aufgerufen, wenn die EOR geändert wurde.
         *
         * @param composedReference eine ComposedReference
         */
        void composedReferenceChanged(final ComposedReference composedReference);

        /**
         * Diese Methode wird aufgerufen, wenn die EOR gelöscht wurde.
         *
         * @param name eine Name
         */
        void composedReferenceRemoved(final String name);
    }
}
