/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.extLocRef;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import javax.swing.table.AbstractTableModel;

/**
 * Diese Klasse verkapselt eine Liste von {@link DirectedReference DirectedReferences} und ist zugleich ein TableModel.
 *
 * @author Kappich Systemberatung
 */
public class DRCollection extends AbstractTableModel {

    private static final String[] _columnNames = {"Beschreibung", "Objekttyp 1", "Objekttyp 2", "Richtung"};
    private final List<DirectedReference> _directedReferences;

    public DRCollection() {
        _directedReferences = new ArrayList<>();
    }

    public DRCollection(final List<DirectedReference> composedReferences) {
        _directedReferences = new ArrayList<>(composedReferences);
    }

    public boolean add(final DirectedReference directedReference) {
        _directedReferences.add(directedReference);
        fireTableDataChanged();
        return true;
    }

    public boolean remove(final DirectedReference directedReference) {
        for (int i = 0; i < _directedReferences.size(); ++i) {
            if (directedReference.equals(_directedReferences.get(i))) {
                _directedReferences.remove(i);
                fireTableDataChanged();
                return true;
            }
        }
        return false;
    }

    public boolean remove(int i) {
        DirectedReference dr = _directedReferences.remove(i);
        if (dr != null) {
            fireTableDataChanged();
            return true;
        } else {
            return false;
        }
    }

    /**
     * Diese Methode ändert die Reihenfolge. Alle DirectedReferences, deren Indizes übergeben werden, werden in der Reihenfolge um eins naoch oben
     * verschoben. Dazu darf der Index 0 nicht in dem Array enthalten sein. Die Indizes müssen nicht sortiert sein: das macht die Methode selber.
     *
     * @param indices Indizes
     *
     * @return
     */
    public boolean moveUpwards(final int[] indices) {
        if (indices.length == 0) {
            return true;
        }
        Arrays.sort(indices);
        if (0 == indices[0]) {
            return false;
        }
        for (final int index : indices) {
            Collections.swap(_directedReferences, index - 1, index);
        }
        fireTableDataChanged();
        return true;
    }

    public DirectedReference get(final int index) {
        return _directedReferences.get(index);
    }

    public int size() {
        return _directedReferences.size();
    }

    public boolean isEmpty() {
        return _directedReferences.isEmpty();
    }

    public void clear() {
        _directedReferences.clear();
    }

    // Absicht!
    @SuppressWarnings("ReturnOfCollectionOrArrayField")
    public List<DirectedReference> getDirectedReferences() {
        return _directedReferences;
    }

    /* Implementation des TableModel */
    @Override
    public int getRowCount() {
        return _directedReferences.size();
    }

    @Override
    public int getColumnCount() {
        return _columnNames.length;
    }

    @Override
    public String getColumnName(int columnIndex) {
        return _columnNames[columnIndex];
    }

    @Nullable
    @Override
    public Object getValueAt(final int rowIndex, final int columnIndex) {
        if (0 <= rowIndex && rowIndex < _directedReferences.size()) {
            DirectedReference directedReference = _directedReferences.get(rowIndex);
            if (0 == columnIndex) {
                return directedReference.getDescription();
            } else if (1 == columnIndex) {
                return directedReference.getFirstType();
            } else if (2 == columnIndex) {
                return directedReference.getSecondType();
            } else if (3 == columnIndex) {
                if (directedReference.isReversed()) {
                    return "rückwärts";
                } else {
                    return "vorwärts";
                }
            }
        }
        return null;
    }
    /* Ende der Implementation des TableModel */

    @Override
    public String toString() {
        return "DRCollection{" + "_directedReferences=" + _directedReferences + '}';
    }
}
