/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.kappich.pat.gnd.extLocRef;

import java.util.ArrayList;
import java.util.List;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import javax.swing.JOptionPane;

/**
 * Diese Klasse verkapselt eine Hierarchie von Erweiterte Ortsreferenzen (EOR, {@link ComposedReference}). Dabei müssen alle EOR mit demselben
 * SystemObjektType als Startpunkt beginnen. Ein Objekt dieser Klasse dient dazu, die Geo-Initialisierung eines Layers für EORs zu ermöglichen.
 *
 * @author Kappich Systemberatung
 */
public class ReferenceHierarchy implements Comparable<ReferenceHierarchy> {

    private static final String INFO = "info";
    private static final String GEOMETRY_TYPE = "geometryType";
    private static final String COMPOSED_REFERENCE = "cr";
    private static final String CR_NAME = "crName";

    private String _name;
    private String _info;
    private String _geometryType;
    private CRCollection _composedReferences;

    public ReferenceHierarchy() {
        _name = "";
        _info = "";
        _geometryType = "";
        _composedReferences = new CRCollection();
    }

    public ReferenceHierarchy(String name, String info, String geometryType) {
        _name = name;
        _info = info;
        _geometryType = geometryType;
        _composedReferences = new CRCollection();
    }

    public ReferenceHierarchy(String name, String info, String geometryType, final List<ComposedReference> composedReferences) {
        _name = name;
        _info = info;
        _geometryType = geometryType;
        _composedReferences = new CRCollection(composedReferences);
    }

    public String getName() {
        return _name;
    }

    public void setName(String name) {
        _name = name;
    }

    public String getInfo() {
        return _info;
    }

    public void setInfo(String info) {
        _info = info;
    }

    public String getGeometryType() {
        return _geometryType;
    }

    public CRCollection getComposedReferences() {
        return _composedReferences;
    }

    public void setComposedReferences(List<ComposedReference> composedReferences, boolean copy) {
        if (copy) {
            _composedReferences = new CRCollection(new ArrayList<>(composedReferences.size()));
            for (ComposedReference cr : composedReferences) {
                _composedReferences.add(cr.getCopy());
            }
        } else {
            _composedReferences = new CRCollection(composedReferences);
        }
    }

    public boolean addComposedReference(final ComposedReference reference) {
        for (int i = 0; i < _composedReferences.size(); ++i) {
            if (reference.getName().equals(_composedReferences.get(i).getName())) {
                return false;
            }
        }
        _composedReferences.add(reference);
        return true;
    }

    public ReferenceHierarchy getCopy() {
        List<ComposedReference> composedReferences = new ArrayList<>(_composedReferences.getComposedReferences());
        return new ReferenceHierarchy(_name, _info, _geometryType, composedReferences);
    }

    public void deletePreferences(final Preferences prefs) {
        Preferences objectPrefs = prefs.node(getName());
        try {
            objectPrefs.removeNode();
        } catch (BackingStoreException e) {
            JOptionPane.showMessageDialog(null, "Das Löschen der EOR-Hierarchie war nicht erfolgreich." + e.toString(), "Fehlermeldung",
                                          JOptionPane.ERROR_MESSAGE);
        }
    }

    public void putPrefernces(final Preferences prefs) {
        deletePreferences(prefs);
        Preferences objectPrefs = prefs.node(getName());
        if (_info == null) {
            objectPrefs.put(INFO, "");
        } else {
            objectPrefs.put(INFO, _info);
        }
        if (_geometryType == null) {
            objectPrefs.put(GEOMETRY_TYPE, "");
        } else {
            objectPrefs.put(GEOMETRY_TYPE, _geometryType);
        }
        for (int i = 0; i < _composedReferences.size(); ++i) {
            Preferences crPrefs = objectPrefs.node(COMPOSED_REFERENCE + i);
            crPrefs.put(CR_NAME, _composedReferences.get(i).getName());
        }
    }

    public boolean initializeFromPreferences(final Preferences prefs) {
        _name = prefs.name();
        _info = prefs.get(INFO, "");
        _geometryType = prefs.get(GEOMETRY_TYPE, "");
        int i = 0;
        String pathName = COMPOSED_REFERENCE + i;
        try {
            while (prefs.nodeExists(pathName)) {
                Preferences crPrefs = prefs.node(pathName);
                String crName = crPrefs.get(CR_NAME, null);
                if (null != crName) {
                    ComposedReference reference = ComposedReferenceManager.getInstance().getComposedReference(crName);
                    if (null != reference) {
                        _composedReferences.add(reference);
                    } else {
                        return false;
                    }
                } else {
                    return false;
                }
                ++i;
                pathName = COMPOSED_REFERENCE + i;
            }
        } catch (BackingStoreException ignore) {
            return false;
        }
        return true;
    }

    @Override
    public int compareTo(final ReferenceHierarchy o) {
        return getName().toLowerCase().compareTo(o.getName().toLowerCase());
    }

    @Override
    public boolean equals(Object o) {
	    if (!(o instanceof ReferenceHierarchy other)) {
            return false;
        }
        return getName().toLowerCase().equals(other.getName().toLowerCase());
    }

    @Override
    public int hashCode() {
        return getName().toLowerCase().hashCode();
    }

    @Override
    public String toString() {
        return "ReferenceHierarchy{" + "_name='" + _name + '\'' + ", _info='" + _info + '\'' + ", _geometryType='" + '\'' + ", _composedReferences=" +
               _composedReferences + '}';
    }
}
