/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.extLocRef;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.Attribute;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.AttributeGroupUsage;
import de.bsvrz.dav.daf.main.config.AttributeListDefinition;
import de.bsvrz.dav.daf.main.config.AttributeType;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.ObjectSetType;
import de.bsvrz.dav.daf.main.config.ObjectSetUse;
import de.bsvrz.dav.daf.main.config.ReferenceAttributeType;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Der {@code SimpleReferenceManager} ist ein Singleton, der die verschiedenen Instanzen von {@link SimpleReference SimpleReferences} zur Verfügung
 * stellt.
 *
 * @author Kappich Systemberatung
 */
public final class SimpleReferenceManager {

    private static final Set<String> _configAreaBlackList;
    private static SimpleReferenceManager _instance;

    static {
        _configAreaBlackList = new HashSet<>();
        _configAreaBlackList.add("kb.tmSystemKalenderGlobal");
        _configAreaBlackList.add("kb.tmVewBetriebGlobal");
        _configAreaBlackList.add("kb.fachModellGlobal");
        _configAreaBlackList.add("kb.systemModellGlobal");
        _configAreaBlackList.add("kb.tmDatenAbgabeUeberwachung");
        _configAreaBlackList.add("kb.tmKExEmailFaxGlobal");
        _configAreaBlackList.add("kb.tmVewBetriebGlobal");
        _configAreaBlackList.add("kb.tmNbaEreignisVerwaltung");
        _configAreaBlackList.add("kb.tmGanglinienGlobal");
        _configAreaBlackList.add("kb.tmBAStBandExportImport");
        _configAreaBlackList.add("kb.modellOsi3Umleitung");
        _configAreaBlackList.add("kb.tmDambachErweiterungen");
        _configAreaBlackList.add("kb.tmAnzeigenGlobalZwischenschichtTls");
        _configAreaBlackList.add("kb.tmVewProtokolleGlobal");
        _configAreaBlackList.add("kb.tmNbaAuswertungen");
        _configAreaBlackList.add("kb.tmUsv");
        _configAreaBlackList.add("kb.metaModellGlobal");
        _configAreaBlackList.add("kb.tmBuVGlobal");
    }

    private final DataModel _configuration;
    private final Map<SystemObjectType, Set<SimpleSetReference>> _simpleSetReferences = new HashMap<>();
    private final Map<SystemObjectType, Set<SimpleSetReference>> _simpleSetReferencesReversed = new HashMap<>();
    private final Map<SystemObjectType, Set<SimpleAttributeReference>> _simpleAttributeReferences = new HashMap<>();
    private final Map<SystemObjectType, Set<SimpleAttributeReference>> _simpleAttributeReferencesReversed = new HashMap<>();
    private final List<SystemObjectType> _systemObjectTypes = new ArrayList<>();

    private SimpleReferenceManager(final ClientDavInterface connection) {
        _configuration = connection.getDataModel();
        initSimpleSetReferences();
        initSimpleAttributeReferences();
//		statistics();
        initSystemObjectTypes();
//		dump(false,true, true);
    }

    public static synchronized SimpleReferenceManager getInstance(final ClientDavInterface connection) {
        if (null == _instance) {
            _instance = new SimpleReferenceManager(connection);
        }
        return _instance;
    }

    private static boolean isConfigurating(AttributeGroup atg) {
        Collection<AttributeGroupUsage> atgUsages = atg.getAttributeGroupUsages();
        for (AttributeGroupUsage usage : atgUsages) {
            if (usage.isConfigurating()) {
                return true;
            }
        }
        return false;
    }

    private static boolean hasEigenschaftenAspekt(AttributeGroup atg) {
        Collection<Aspect> aspects = atg.getAspects();
        for (Aspect aspect : aspects) {
            if (aspect.getPid().equals("asp.eigenschaften")) {
                return true;
            }
        }
        return false;
    }

    private static void applyBlackList(Set<SystemObjectType> set) {
        set.removeIf(type -> _configAreaBlackList.contains(type.getConfigurationArea().getPid()));
    }

    private static void addSubTypes(SystemObjectType type, Set<SystemObjectType> list) {
        list.add(type);
        for (SystemObjectType subType : type.getSubTypes()) {
            addSubTypes(subType, list);
        }
    }

    @SuppressWarnings("unused")
    public List<SimpleReference> getSimpleReferences(final SystemObjectType type, boolean reversed) {
        List<SimpleReference> simpleReferences = new ArrayList<>();
        if (!reversed) {
            {
                Set<SimpleAttributeReference> sarList = _simpleAttributeReferences.get(type);
                if (null != sarList) {
                    simpleReferences.addAll(sarList);
                }
            }
            {
                Set<SimpleSetReference> ssrList = _simpleSetReferences.get(type);
                if (null != ssrList) {
                    simpleReferences.addAll(ssrList);
                }
            }
        } else {
            {
                Set<SimpleAttributeReference> sarList = _simpleAttributeReferencesReversed.get(type);
                if (null != sarList) {
                    simpleReferences.addAll(sarList);
                }
            }
            {
                Set<SimpleSetReference> ssrList = _simpleSetReferencesReversed.get(type);
                if (null != ssrList) {
                    simpleReferences.addAll(ssrList);
                }
            }
        }
        return simpleReferences;
    }

    @SuppressWarnings("SameParameterValue")
    public Set<SimpleAttributeReference> getSimpleAttributeReferences(final SystemObjectType type, boolean reversed) {
        if (!reversed) {
            if (_simpleAttributeReferences.containsKey(type)) {
                return Collections.unmodifiableSet(_simpleAttributeReferences.get(type));
            }
        } else {
            if (_simpleAttributeReferences.containsKey(type)) {
                return Collections.unmodifiableSet(_simpleAttributeReferencesReversed.get(type));
            }
        }
        return new HashSet<>();
    }

    public Set<SimpleSetReference> getSimpleSetReferences(final SystemObjectType type, boolean reversed) {
        if (!reversed) {
            if (_simpleSetReferences.containsKey(type)) {
                return Collections.unmodifiableSet(_simpleSetReferences.get(type));
            }
        } else {
            if (_simpleSetReferencesReversed.containsKey(type)) {
                return Collections.unmodifiableSet(_simpleSetReferencesReversed.get(type));
            }
        }
        return new HashSet<>();
    }

    /**
     * Diese Methode gibt eine Liste aller {@link SystemObjectType SystemObjectTypes} zurück, die an irgendeiner {@link SimpleReference} beteiligt
     * sind.
     *
     * @return die beschriebene Liste
     */
    @SuppressWarnings("unused")
    public List<SystemObjectType> getSystemObjectTypes() {
        return Collections.unmodifiableList(_systemObjectTypes);
    }

    private void initSimpleSetReferences() {
        // Initialisiert _simpleSetReferences und _simpleSetReferencesReversed.
        SystemObjectType configObjectType = _configuration.getType("typ.konfigurationsObjekt");
        Set<SystemObjectType> allConfigTypes = new HashSet<>(); // Set wegen Mehrfachvererbung!
        addSubTypes(configObjectType, allConfigTypes);

        // Initialiesiert eine Map für SystemObjectType -> Pids seiner Mengen
        Map<SystemObjectType, List<ObjectSetUse>> setMap = new HashMap<>();
        for (SystemObjectType type : allConfigTypes) {
            List<ObjectSetUse> objectSetUses = type.getObjectSetUses();
            if (objectSetUses.isEmpty()) {
                continue;
            }
            setMap.put(type, objectSetUses);
        }

        // Bilde die SimpleSetReferences für _simpleSetReferences:
        for (final Map.Entry<SystemObjectType, List<ObjectSetUse>> entry : setMap.entrySet()) {
            SystemObjectType firstType = entry.getKey();
            Set<SimpleSetReference> simpleSetReferences = new HashSet<>();
            for (ObjectSetUse use : entry.getValue()) {
                ObjectSetType objectSetType = use.getObjectSetType();
                if (objectSetType != null) {
                    List<SystemObjectType> secondTypes = objectSetType.getObjectTypes();

                    for (SystemObjectType secondType : secondTypes) {
                        SimpleSetReference simpleSetReference = new SimpleSetReference(firstType, secondType, use.getObjectSetName());
                        simpleSetReferences.add(simpleSetReference);
                    }
                }
            }
            _simpleSetReferences.put(firstType, simpleSetReferences);
        }

        // Bilde die SimpleSetReferences für _simpleSetReferencesReversed:
        for (final Map.Entry<SystemObjectType, Set<SimpleSetReference>> systemObjectTypeListEntry : _simpleSetReferences.entrySet()) {
            Set<SimpleSetReference> list = systemObjectTypeListEntry.getValue();
            for (SimpleSetReference simpleSetReference : list) {
                SystemObjectType secondType = simpleSetReference.getSecondType();
                if (_simpleSetReferencesReversed.containsKey(secondType)) {
                    _simpleSetReferencesReversed.get(secondType).add(simpleSetReference);
                } else {
                    Set<SimpleSetReference> simpleSetReferences = new HashSet<>();
                    simpleSetReferences.add(simpleSetReference);
                    _simpleSetReferencesReversed.put(secondType, simpleSetReferences);
                }
            }
        }
    }

    private void initSimpleAttributeReferences() {
        SystemObjectType configObjectType = _configuration.getType("typ.konfigurationsObjekt");
        Set<SystemObjectType> allConfigTypes = new HashSet<>(); // Set, wegen Mehrfachvererbung!
        addSubTypes(configObjectType, allConfigTypes);
        applyBlackList(allConfigTypes);

        // Initialisiere zunächst _simpleAttributeReferences:
        for (SystemObjectType objectType : allConfigTypes) {
            List<AttributeGroup> atgs = objectType.getAttributeGroups();
            for (AttributeGroup atg : atgs) {
                for (Attribute attr : atg.getAttributes()) {
                    List<String> listKeys = new ArrayList<>(1);
                    initForAttribute(objectType, atg, attr, listKeys);
                }
            }
        }

        // Fülle _simpleAttributeReferencesReversed:
        for (final Map.Entry<SystemObjectType, Set<SimpleAttributeReference>> systemObjectTypeListEntry : _simpleAttributeReferences.entrySet()) {
            Set<SimpleAttributeReference> set = systemObjectTypeListEntry.getValue();
            for (SimpleAttributeReference simpleAttributeReference : set) {
                SystemObjectType secondType = simpleAttributeReference.getSecondType();
                if (_simpleAttributeReferencesReversed.containsKey(secondType)) {
                    _simpleAttributeReferencesReversed.get(secondType).add(simpleAttributeReference);
                } else {
                    Set<SimpleAttributeReference> simpleAttributeReferences = new HashSet<>();
                    simpleAttributeReferences.add(simpleAttributeReference);
                    _simpleAttributeReferencesReversed.put(secondType, simpleAttributeReferences);
                }
            }
        }
    }

    private void initForAttribute(SystemObjectType objectType, AttributeGroup atg, Attribute attr, List<String> listKeys) {
        if (!isConfigurating(atg) || !hasEigenschaftenAspekt(atg)) {
            return;
        }

        listKeys.add(attr.getName()); // dies war eine schwer zu findende Zeile; verstanden ist das nicht!

        AttributeType attrType = attr.getAttributeType();
	    if (attrType instanceof ReferenceAttributeType refAttrType) {
            if (refAttrType.getReferencedObjectType() != null) {
                // Glück gehabt!
                SimpleAttributeReference simpleReference =
                    new SimpleAttributeReference(objectType, refAttrType.getReferencedObjectType(), atg, attr, listKeys);
                if (_simpleAttributeReferences.containsKey(objectType)) {
                    _simpleAttributeReferences.get(objectType).add(simpleReference);
                } else {
                    Set<SimpleAttributeReference> newSet = new HashSet<>();
                    newSet.add(simpleReference);
                    _simpleAttributeReferences.put(objectType, newSet);
                }
            } else {
                // Pech! Nun muss man über alle SystemObjects die Information selbst berechnen.
                initByInspectingAllSystemObjects(objectType, atg, attr, listKeys);
            }
	    } else if (attrType instanceof AttributeListDefinition ald) {
            List<Attribute> attributes = ald.getAttributes();
            for (Attribute attribute : attributes) {
                initForAttribute(objectType, atg, attribute, listKeys);
            }
        }
    }

    private void initByInspectingAllSystemObjects(SystemObjectType objectType, AttributeGroup atg, Attribute attr, List<String> listKeys) {
        List<SystemObject> objects = objectType.getElements();
        _configuration.getConfigurationData(objects, atg);  // Holt die Daten in den Cache.
        for (SystemObject object : objects) {
            initByInspectingSystemObject(object, objectType, atg, attr, listKeys);
        }
    }

    private void initByInspectingSystemObject(SystemObject object, SystemObjectType objectType, AttributeGroup atg, Attribute attr,
                                              List<String> listKeys) {
        Data atgData = object.getConfigurationData(atg);
        branch(atgData, object, objectType, atg, attr, listKeys);
    }

    private void branch(Data atgData, SystemObject object, SystemObjectType objectType, AttributeGroup atg, Attribute attr, List<String> listKeys) {
        if (atgData != null) {
            if (atgData.isList()) {
                initForList(atgData, object, objectType, atg, attr, listKeys);
            } else if (atgData.isArray() && atgData.getAttributeType() instanceof AttributeListDefinition) {
                Data.Array atgDataArray = atgData.asArray();
                for (int i = 0; i < atgDataArray.getLength(); ++i) {
                    initForList(atgDataArray.getItem(i), object, objectType, atg, attr, listKeys);
                }
            } else if (atgData.isArray() && atgData.getAttributeType() instanceof ReferenceAttributeType) {
                Data.Array atgDataArray = atgData.asArray();
                for (int i = 0; i < atgDataArray.getLength(); ++i) {
                    initForReferenceValue(atgDataArray.getItem(i), objectType, atg, attr, listKeys);
                }
            } else if (atgData.getAttributeType() instanceof ReferenceAttributeType) {
                initForReferenceValue(atgData, objectType, atg, attr, listKeys);
            }
        }
    }

    private void initForList(Data list, SystemObject object, SystemObjectType objectType, AttributeGroup atg, Attribute attr, List<String> listKeys) {
        List<String> newListKeys = new ArrayList<>(listKeys);
        String s = newListKeys.remove(0);
        Data atgData = list.getItem(s);
        branch(atgData, object, objectType, atg, attr, newListKeys);
    }

    private void initForReferenceValue(Data atgData, SystemObjectType objectType, AttributeGroup atg, Attribute attr, List<String> listKeys) {

        Data.ReferenceValue referenceValue = atgData.asReferenceValue();

        Set<SystemObjectType> otherTypes = new HashSet<>();
        if (referenceValue != null) {
            SystemObject otherObject = referenceValue.getSystemObject();
            if (null != otherObject) {
                SystemObjectType otherType = otherObject.getType();
                otherTypes.add(otherType);
            }
        }
        for (SystemObjectType otherType : otherTypes) {

            SimpleAttributeReference simpleReference = new SimpleAttributeReference(objectType, otherType, atg, attr, listKeys);
            if (_simpleAttributeReferences.containsKey(objectType)) {
                _simpleAttributeReferences.get(objectType).add(simpleReference);
            } else {
                Set<SimpleAttributeReference> newSet = new HashSet<>();
                newSet.add(simpleReference);
                _simpleAttributeReferences.put(objectType, newSet);
            }
        }
    }

    private void initSystemObjectTypes() {
        Set<SystemObjectType> set = new HashSet<>();
        set.addAll(_simpleAttributeReferences.keySet());
        set.addAll(_simpleAttributeReferencesReversed.keySet());
        set.addAll(_simpleSetReferences.keySet());
        set.addAll(_simpleSetReferencesReversed.keySet());

        _systemObjectTypes.clear();
        _systemObjectTypes.addAll(set);
        _systemObjectTypes.sort(Comparator.comparing(SystemObject::getNameOrPidOrId));
    }

    @SuppressWarnings("UseOfSystemOutOrSystemErr")
    private void statistics() {
        int counter = 0;
        for (final Map.Entry<SystemObjectType, Set<SimpleSetReference>> systemObjectTypeListEntry : _simpleSetReferences.entrySet()) {
            counter += systemObjectTypeListEntry.getValue().size();
        }
        System.out.println("MengenReferenzen (normal): " + _simpleSetReferences.size() + " SystemObjectTypes, " + counter + " Einträge");

        counter = 0;
        for (final Map.Entry<SystemObjectType, Set<SimpleSetReference>> systemObjectTypeListEntry : _simpleSetReferencesReversed.entrySet()) {
            counter += systemObjectTypeListEntry.getValue().size();
        }
        System.out.println("MengenReferenzen (reversed): " + _simpleSetReferencesReversed.size() + " SystemObjectTypes, " + counter + " Einträge");

        counter = 0;
        for (final Map.Entry<SystemObjectType, Set<SimpleAttributeReference>> systemObjectTypeListEntry : _simpleAttributeReferences.entrySet()) {
            counter += systemObjectTypeListEntry.getValue().size();
        }
        System.out
            .println("AttributGruppenReferenzen (normal): " + _simpleAttributeReferences.size() + " SystemObjectTypes, " + counter + " Einträge");

        counter = 0;
        for (final Map.Entry<SystemObjectType, Set<SimpleAttributeReference>> systemObjectTypeListEntry : _simpleAttributeReferencesReversed
            .entrySet()) {
            counter += systemObjectTypeListEntry.getValue().size();
        }
        System.out.println(
            "AttributGruppenReferenzen (reversed): " + _simpleAttributeReferencesReversed.size() + " SystemObjectTypes, " + counter + " Einträge");
    }

    @SuppressWarnings({"unused", "UseOfSystemOutOrSystemErr"})
    private void dump(final boolean sets, final boolean atg, final boolean keysOnly) {
        if (!keysOnly) {
            if (sets) {
                System.out.println("MengenReferenzen (normal): ");
                for (final Map.Entry<SystemObjectType, Set<SimpleSetReference>> entry : _simpleSetReferences.entrySet()) {
                    for (SimpleSetReference ssr : entry.getValue()) {
                        System.out.println("Key: " + entry.getKey().getNameOrPidOrId() + "| Value: " + ssr);
                    }
                }
                System.out.println("MengenReferenzen (reversed): ");
                for (final Map.Entry<SystemObjectType, Set<SimpleSetReference>> entry : _simpleSetReferencesReversed.entrySet()) {
                    for (SimpleSetReference ssr : entry.getValue()) {
                        System.out.println("Key: " + entry.getKey().getNameOrPidOrId() + "| Value: " + ssr);
                    }
                }
            }
            if (atg) {
                System.out.println("AttributGruppenReferenzen (normal): ");
                for (final Map.Entry<SystemObjectType, Set<SimpleAttributeReference>> entry : _simpleAttributeReferences.entrySet()) {
                    for (SimpleAttributeReference ssr : entry.getValue()) {
                        System.out.println("Key: " + entry.getKey().getNameOrPidOrId() + "| Value: " + ssr);
                    }
                }
                System.out.println("AttributGruppenReferenzen (reversed): ");
                for (final Map.Entry<SystemObjectType, Set<SimpleAttributeReference>> entry : _simpleAttributeReferencesReversed.entrySet()) {
                    for (SimpleAttributeReference ssr : entry.getValue()) {
                        System.out.println("Key: " + entry.getKey().getNameOrPidOrId() + "| Value: " + ssr);
                    }
                }
            }
        } else {
            if (sets) {
                System.out.println("Keys der MengenReferenzen (normal): ");
                for (final SystemObjectType type : _simpleSetReferences.keySet()) {
                    System.out.println(type.getNameOrPidOrId());
                }
                System.out.println("Keys der MengenReferenzen (reversed): ");
                for (final SystemObjectType type : _simpleSetReferencesReversed.keySet()) {
                    System.out.println(type.getNameOrPidOrId());
                }
            }
            if (atg) {
                System.out.println("Keys der AttributGruppenReferenzen (normal): ");
                for (final SystemObjectType type : _simpleAttributeReferences.keySet()) {
                    System.out.println(type.getNameOrPidOrId());
                }
                System.out.println("Keys der AttributGruppenReferenzen (reversed): ");
                for (final SystemObjectType type : _simpleAttributeReferencesReversed.keySet()) {
                    System.out.println(type.getNameOrPidOrId());
                }
            }
        }
    }

	/*
	Hier beginnen die tatsächlich benutzten Methoden (siehe DRSelectionDialog). Grundsätzlich könnte man hier die
	vier Container (_simpleSetReferences, etc) refaktorieren, so dass diese Methoden effizienter ablaufen. Allerdings
	ist die Performanz auch so okay.
	 */

    // Zunächst für Attributgruppen:
    public Object[] getSystemObjectTypesForAttributes() {
        List<SystemObjectType> list = new ArrayList<>(_simpleAttributeReferences.keySet());
        list.sort(Comparator.comparing(SystemObject::getNameOrPidOrId));
        return list.toArray();
    }

    public Object[] getAttributeGroups(final SystemObjectType type) {
        if (null != type && _simpleAttributeReferences.containsKey(type)) {
            Set<SimpleAttributeReference> set = _simpleAttributeReferences.get(type);
            Set<AttributeGroup> atgSet = new HashSet<>();
            for (SimpleAttributeReference reference : set) {
                atgSet.add(reference.getAttributeGroup());
            }
            List<AttributeGroup> atgList = new ArrayList<>(atgSet);
            atgList.sort(Comparator.comparing(SystemObject::getNameOrPidOrId));
            return atgList.toArray();
        } else {
            return new Object[0];
        }
    }

    public Object[] getAttributes(final SystemObjectType type, final AttributeGroup atg) {
        if (null != type && _simpleAttributeReferences.containsKey(type) && null != atg) {
            Set<SimpleAttributeReference> set = _simpleAttributeReferences.get(type);
            Set<Attribute> attrSet = new HashSet<>();
            for (SimpleAttributeReference reference : set) {
                if (reference.getAttributeGroup().equals(atg)) {
                    attrSet.add(reference.getAttribute());
                }
            }
            List<Attribute> attrList = new ArrayList<>(attrSet);
            return attrList.toArray();
        } else {
            return new Object[0];
        }
    }

    @Nullable
    public SystemObjectType getSystemObjectType(final SystemObjectType type, final AttributeGroup atg, final Attribute attr) {
        if (null != type && _simpleAttributeReferences.containsKey(type) && null != atg && null != attr) {
            for (SimpleAttributeReference reference : _simpleAttributeReferences.get(type)) {
                if (reference.getAttributeGroup().equals(atg) && reference.getAttribute().equals(attr)) {
                    return reference.getSecondType();
                }
            }
            return null;
        } else {
            return null;
        }
    }

    @Nullable
    public SimpleReference checkForSimpleReference(final SystemObjectType type1, final AttributeGroup atg, final Attribute attr,
                                                   final SystemObjectType type2) {
        if (null == type1 || null == atg || null == attr || null == type2) {
            return null;
        }
        Set<SimpleAttributeReference> set = _simpleAttributeReferences.get(type1);
        for (SimpleAttributeReference reference : set) {
            if (reference.getAttributeGroup().equals(atg) && reference.getAttribute().equals(attr) && reference.getSecondType().equals(type2)) {
                return reference;
            }
        }
        return null;
    }

    @Nullable
    public SimpleReference checkForSimpleReference(final SystemObjectType type1, final String setName, final SystemObjectType type2) {
        if (null == type1 || null == setName || null == type2) {
            return null;
        }
        Set<SimpleSetReference> set = _simpleSetReferences.get(type1);
        for (SimpleSetReference reference : set) {
            if (reference.getSetName().equals(setName) && reference.getSecondType().equals(type2)) {
                return reference;
            }
        }
        return null;
    }

    // Nun für Mengen:
    public Object[] getSystemObjectTypesForSets() {
        List<SystemObjectType> list = new ArrayList<>(_simpleSetReferences.keySet());
        list.sort(Comparator.comparing(SystemObject::getNameOrPidOrId));
        return list.toArray();
    }

    public Object[] getSets(final SystemObjectType type) {
        if (null != type && _simpleSetReferences.containsKey(type)) {
            Set<SimpleSetReference> set = _simpleSetReferences.get(type);
            Set<String> setNamesSet = new HashSet<>();
            for (SimpleSetReference reference : set) {
                setNamesSet.add(reference.getSetName());
            }
            List<String> setList = new ArrayList<>(setNamesSet);
            setList.sort(String::compareTo);
            return setList.toArray();
        } else {
            return new Object[0];
        }
    }

    @Nullable
    public SystemObjectType getSystemObjectType(final SystemObjectType type, final String setName) {
        if (null != type && _simpleSetReferences.containsKey(type) && null != setName) {
            for (SimpleSetReference reference : _simpleSetReferences.get(type)) {
                if (reference.getSetName().equals(setName)) {
                    return reference.getSecondType();
                }
            }
            return null;
        } else {
            return null;
        }
    }

    @Override
    public String toString() {
        return "SimpleReferenceManager{}";
    }
}
