/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.extLocRef.view;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.sys.funclib.debug.Debug;
import de.kappich.pat.gnd.documentation.GndHelp;
import de.kappich.pat.gnd.extLocRef.ComposedReference;
import de.kappich.pat.gnd.extLocRef.ComposedReferenceManager;
import de.kappich.pat.gnd.extLocRef.DRCollection;
import de.kappich.pat.gnd.extLocRef.DirectedReference;
import de.kappich.pat.gnd.gnd.PluginManager;
import de.kappich.pat.gnd.utils.SpringUtilities;
import de.kappich.pat.gnd.utils.view.GndFrame;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SpringLayout;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelListener;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableColumn;

/**
 * Der Dialog zur Definition und Bearbeitung Erweiterter Ortsreferenzen, kurz der EOR-Editor.
 *
 * @author Kappich Systemberatung
 */
public class CrDefinitionDialog extends GndFrame {

    private static final Debug _debug = Debug.getLogger();
    private static final int _lowerHeightBound = 310;
    private static final int _upperHeightBound = Toolkit.getDefaultToolkit().getScreenSize().height;
    private final ClientDavInterface _connection;
    private final JTextField _nameTextField = new JTextField();
    private final JTextField _infoTextField = new JTextField();
    private final JComboBox<String> _geometryPlugins = new JComboBox<>(PluginManager.getGeometryPluginNames());
    private final JTable _crTable = new JTable();
    private final List<EditableListenButton> _listeningButtons = new ArrayList<>();
    private final EditableListenButton _newButton = new EditableListenButton("Neue Zeile");
    private final JButton _deleteButton = new JButton("Löschen");
    private final JButton _upwardButton = new JButton("Aufwärts");
    private boolean _editable;
    private boolean _nameChangeable;
    private boolean _somethingChanged;
    private ComposedReference _scratchComposedReference;
    private ComposedReference _unchangeableOriginalComposedReference;

    /**
     * Konstruktor zum Anlegen eines neuen EOR-Editors.
     *
     * @param connection        die Datenverteiler-Verbindung
     * @param composedReference eine EOR
     * @param editable          ist die EOR veränderbar?
     * @param nameChangeable    ist der Name und damit die Identität änderbar?
     * @param title             der Titel des Fensters
     */
    @SuppressWarnings("OverlyLongMethod")
    public CrDefinitionDialog(final ClientDavInterface connection, final ComposedReference composedReference, final boolean editable,
                              final boolean nameChangeable, final String title) {
        super("CrDefinitionDialog", title);
        _connection = connection;
        if (composedReference != null) {
            _scratchComposedReference = composedReference.getCopy();
            _unchangeableOriginalComposedReference = _scratchComposedReference.getCopy();
        } else {
            throw new RuntimeException("Ein CrDefinitionDialog kann nicht ohne EOR-Objekt konstruiert werden.");
        }
        _editable = editable;
        _nameChangeable = nameChangeable;

        setLayout(new BorderLayout());

        Dimension labelSize = new Dimension(100, 20);

        // Oberer Teil mit Name, Info und Geometrie-Klasse

        JLabel nameLabel = new JLabel("Name: ");
        nameLabel.setPreferredSize(labelSize);
        _nameTextField.setText(_scratchComposedReference.getName());
        _nameTextField.setEditable(nameChangeable);

        JLabel infoLabel = new JLabel("Info: ");
        infoLabel.setPreferredSize(labelSize);
        _infoTextField.setText(_scratchComposedReference.getInfo());

        JLabel geometryLabel = new JLabel("Geometrie-Typ: ");
        geometryLabel.setPreferredSize(labelSize);
        _geometryPlugins.setPreferredSize(new Dimension(200, 25));
        _geometryPlugins.setSelectedItem(_scratchComposedReference.getGeometryType());

        JPanel upperPanel = new JPanel();
        upperPanel.setLayout(new SpringLayout());
        upperPanel.add(nameLabel);
        upperPanel.add(_nameTextField);
        upperPanel.add(infoLabel);
        upperPanel.add(_infoTextField);
        upperPanel.add(geometryLabel);
        upperPanel.add(_geometryPlugins);
        upperPanel.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));
        SpringUtilities.makeCompactGrid(upperPanel, 2, 5, 5);
        add(upperPanel, BorderLayout.NORTH);

        // Mittelteil für die EOR

        setTableProperties(_scratchComposedReference.getDirectedReferences());
        addListSelectionListener();

        addButtonListener();
        JPanel buttonsPanel = new JPanel();
        buttonsPanel.setLayout(new SpringLayout());

        buttonsPanel.add(_newButton);
        buttonsPanel.add(_deleteButton);
        buttonsPanel.add(_upwardButton);

        buttonsPanel.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));
        SpringUtilities.makeCompactGrid(buttonsPanel, 1, 5, 20);

        JPanel centerPanel = new JPanel();
        centerPanel.setLayout(new SpringLayout());
        centerPanel.setBorder(BorderFactory.createMatteBorder(2, 2, 2, 2, Color.BLACK));
        centerPanel.add(new JScrollPane(_crTable));
        centerPanel.add(buttonsPanel);
        SpringUtilities.makeCompactGrid(centerPanel, 2, 20, 5);

        add(centerPanel, BorderLayout.CENTER);

        // Unterer Teil mit Buttons

        EditableListenButton saveButton = new EditableListenButton("Speichern");
        JButton cancelButton = new JButton("Dialog schließen");
        JButton helpButton = new JButton("Hilfe");
        addButtonListener(saveButton, cancelButton, helpButton);

        JPanel crButtonsPanel = new JPanel();
        crButtonsPanel.setLayout(new SpringLayout());

        crButtonsPanel.add(saveButton);
        crButtonsPanel.add(cancelButton);
        crButtonsPanel.add(helpButton);

        crButtonsPanel.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));
        SpringUtilities.makeCompactGrid(crButtonsPanel, 3, 20, 5);
        add(crButtonsPanel, BorderLayout.SOUTH);

        addChangeListeners();
        addFrameListener();
        addRenderer();

        setEditable(_editable, _nameChangeable);
        setPositionAndSize();
        setVisible(true);
        toFront();
    }

    private static boolean saveChangesWanted() {
        Object[] options = {"Änderungen speichern", "Nicht speichern"};
        int n = JOptionPane
            .showOptionDialog(new JFrame(), "Änderungen speichern?", "Es wurden Änderungen an der EOR vorgenommen.", JOptionPane.YES_NO_CANCEL_OPTION,
                              JOptionPane.QUESTION_MESSAGE, null, options, options[1]);
        return n == 0;
    }

    public final void setPositionAndSize() {
        Dimension d = _crTable.getPreferredSize();
        setPositionAndSize(690, 202 + (int) d.getHeight(), 660, 50, false, 690,
                           Math.min(Math.max(202 + (int) d.getHeight(), _lowerHeightBound), _upperHeightBound));
    }

    private void addChangeListeners() {
        {
            _nameTextField.getDocument().addDocumentListener(new DocumentListener() {
                @Override
                public void changedUpdate(DocumentEvent e) {
                    _somethingChanged = true;
                }

                @Override
                public void removeUpdate(DocumentEvent e) {
                    _somethingChanged = true;
                }

                @Override
                public void insertUpdate(DocumentEvent e) {
                    _somethingChanged = true;
                }
            });
        }

        {
            _infoTextField.getDocument().addDocumentListener(new DocumentListener() {
                @Override
                public void changedUpdate(DocumentEvent e) {
                    _somethingChanged = true;
                }

                @Override
                public void removeUpdate(DocumentEvent e) {
                    _somethingChanged = true;
                }

                @Override
                public void insertUpdate(DocumentEvent e) {
                    _somethingChanged = true;
                }
            });
        }

        TableModelListener tableModelListener = e -> _somethingChanged = true;
        _crTable.getModel().addTableModelListener(tableModelListener);
    }

    private void setTableProperties(DRCollection drCollection) {
        _crTable.setModel(drCollection);
        _crTable.setRowHeight(25);
        int vColIndex = 0;
        TableColumn col = _crTable.getColumnModel().getColumn(vColIndex);
        col.setPreferredWidth(327);
        _deleteButton.setEnabled(false);
        _upwardButton.setEnabled(_crTable.getSelectedColumnCount() > 0);
    }

    /**
     * Setzt die Felder des EOR-Editors mit den Informationen der übergebenen EOR und aktiviert die Veränderbarkeit gemäß der zwei boolschen Werte.
     *
     * @param reference      eine EOR
     * @param editable       ist die EOR veränderbar?
     * @param nameChangeable ist der Name und damit die Identität der EOR änderbar?
     */
    public void setComposedReference(ComposedReference reference, boolean editable, boolean nameChangeable) {
        _scratchComposedReference = reference;
        _unchangeableOriginalComposedReference = _scratchComposedReference.getCopy();
        _editable = editable;
        _nameTextField.setText(_scratchComposedReference.getName());
        _infoTextField.setText(_scratchComposedReference.getInfo());
        setTableProperties(_scratchComposedReference.getDirectedReferences());
        setEditable(editable, nameChangeable);
        for (EditableListenButton elButton : _listeningButtons) {
            elButton.rhEditingStateChanged(_editable);
        }
        _somethingChanged = false;
    }

    /**
     * Setzt den Wert der internen Variable, die darüber entscheidet, ob die Informationen der angezeigten EOR veränderbar sind, und macht Textfelder
     * veränderbar oder nicht, aktiviert bzw. deaktiviert Knöpfe usw.
     *
     * @param editable       ist die EOR veränderbar?
     * @param nameChangeable ist der Name und damit die Identität der EOR änderbar?
     */
    public final void setEditable(boolean editable, boolean nameChangeable) {
        _editable = editable;
        _nameChangeable = nameChangeable;
        for (EditableListenButton elButton : _listeningButtons) {
            elButton.rhEditingStateChanged(_editable);
        }
        _nameTextField.setEditable(_nameChangeable);
        _infoTextField.setEditable(_editable);
        _crTable.setRowSelectionAllowed(_editable);
        _newButton.setEnabled(_editable);
    }

    /**
     * Mit dieser Methode kann man der angezeigten {@link ComposedReference} eine {@link DirectedReference} hinzufügen.
     *
     * @param reference wird hinzugefügt
     *
     * @return {@code true} im Erfolgsfall
     */
    public boolean addDirectedReference(final DirectedReference reference) {
        return _scratchComposedReference.getDirectedReferences().add(reference);
    }

    @SuppressWarnings("OverlyLongMethod")
    private void addButtonListener() {

        ActionListener actionListenerNew = e -> {
            DRSelectionDialog drSelectionDialog = new DRSelectionDialog(_connection, this, true);
            drSelectionDialog.runDialog();
        };
        _newButton.addActionListener(actionListenerNew);

        ActionListener actionListenerDelete = e -> {
            final int[] selectedRows = _crTable.getSelectedRows();
            if (selectedRows.length == 0) {
                JOptionPane.showMessageDialog(new JFrame(), "Bitte wählen Sie mindestens eine Zeile aus der Liste aus!", "Fehler",
                                              JOptionPane.ERROR_MESSAGE);
                return;
            }
            for (int i = selectedRows.length - 1; i >= 0; i--) {
                _scratchComposedReference.getDirectedReferences().remove(selectedRows[i]);
            }
        };
        _deleteButton.addActionListener(actionListenerDelete);

        ActionListener actionListenerUpward = e -> {
            final int[] selectedRows = _crTable.getSelectedRows();
            if (selectedRows.length == 0) {
                JOptionPane.showMessageDialog(new JFrame(), "Bitte wählen Sie mindestens eine Zeile aus der Liste aus!", "Fehler",
                                              JOptionPane.ERROR_MESSAGE);
                return;
            }
            Arrays.sort(selectedRows);
            if (0 == selectedRows[0]) {
                JOptionPane.showMessageDialog(new JFrame(), "Die erste Zeile darf nicht ausgewählt sein!", "Fehler", JOptionPane.ERROR_MESSAGE);
                return;
            }
            if (_scratchComposedReference.getDirectedReferences().moveUpwards(selectedRows)) {
                for (int index : selectedRows) {
                    _crTable.addRowSelectionInterval(index - 1, index - 1);
                }
            }
        };
        _upwardButton.addActionListener(actionListenerUpward);
    }

    private void addButtonListener(final JButton saveButton, final JButton cancelButton, final JButton helpButton) {

        ActionListener actionListenerSave = e -> saveComposedReference();
        saveButton.addActionListener(actionListenerSave);

        ActionListener actionListenerCancel = e -> {
            if (_editable && _somethingChanged) {
                if (!saveChangesWanted()) {      // Es wird nicht gespeichert: Änderungen rückgängig machen! Dieses Verhalten wurde
                    // vom LayerDefinitionDialog übernommen.
                    setComposedReference(_unchangeableOriginalComposedReference.getCopy(), _editable, _nameChangeable);
                }
            }
            setVisible(false);
            storePreferenceBounds();
            dispose();
        };
        cancelButton.addActionListener(actionListenerCancel);

        ActionListener actionListenerHelp = e -> GndHelp.openHelp("#eorDefinitionDialog");
        helpButton.addActionListener(actionListenerHelp);
    }

    private void saveComposedReference() {
        if (!_editable) {    // Sollte nie passieren, da der Button disabled ist.
            JOptionPane.showMessageDialog(this.getFrame(), "Diese EOR ist nicht veränderbar!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return;
        }
        String crName = _nameTextField.getText();
        if (crName.isEmpty()) {
            JOptionPane.showMessageDialog(this.getFrame(), "Bitte geben Sie einen Namen ein!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return;
        } else if (crName.length() > 40) {    // Präferenzen erlauben 80 Zeichen als Node-Name, doch hier wird 40 gewählt.
            JOptionPane
                .showMessageDialog(this.getFrame(), "Bitte verwenden Sie einen Namen mit höchstens 40 Zeichen!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return;
        }
        String infoText = _infoTextField.getText();
        if (infoText == null) {
            infoText = "";
        }
        Object selected = _geometryPlugins.getSelectedItem();
        if (null == selected) {
            JOptionPane.showMessageDialog(this.getFrame(), "Bitte wählen Sie einen Geometrie-Typ aus!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return;
        }
        String geometryType = (String) selected;
        final DRCollection drCollection = (DRCollection) _crTable.getModel();
        if (drCollection.isEmpty()) {
            JOptionPane.showMessageDialog(this.getFrame(), "Bitte geben Sie mindestens eine Zeile an!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return;
        }
        final ComposedReference existingReference = ComposedReferenceManager.getInstance().getComposedReference(crName);
        if (existingReference != null) {
            if (ComposedReferenceManager.getInstance().isChangeable(existingReference)) {
                Object[] options = {"EOR überschreiben", "Speichern abbrechen"};
                int n = JOptionPane
                    .showOptionDialog(this.getFrame(), "Soll die bestehende EOR mit diesem Namen wirklich überschrieben werden?", "EOR speichern",
                                      JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE, null, options, options[1]);
                if (n != 0) {
                    return;
                }
            } else {
                JOptionPane
                    .showMessageDialog(this.getFrame(), "Die bestehende EOR darf nicht überschrieben werden!", "Fehler", JOptionPane.ERROR_MESSAGE);
                return;
            }
        } else if (ComposedReferenceManager.getInstance().hasComposedReference(crName)) {
            JOptionPane.showMessageDialog(this.getFrame(), "Es existiert bereits eine EOR mit diesem Namen!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return;
        }

        ComposedReference newReference = new ComposedReference(crName, infoText, geometryType);
        newReference.setDirectedReferences(drCollection.getDirectedReferences(), false);

        //noinspection VariableNotUsedInsideIf
        if (existingReference != null) {
            ComposedReferenceManager.getInstance().changeComposedReference(newReference);
        } else {
            ComposedReferenceManager.getInstance().addComposedReference(newReference);
        }
        _scratchComposedReference = newReference;
        _unchangeableOriginalComposedReference = _scratchComposedReference.getCopy();
        _somethingChanged = false;
        // Damit der Benutzer dasselbe Objekt nicht unter anderem Namen speichern kann:
        _nameChangeable = false;
        _nameTextField.setEditable(false);
    }

    private void addFrameListener() {
        class FrameListener extends WindowAdapter {
            @Override
            public void windowClosing(WindowEvent e) {
                if (_somethingChanged && _editable) {
                    if (saveChangesWanted()) {
                        saveComposedReference();
                        setVisible(false);
                        storePreferenceBounds();
                        dispose();
                    }
                }
            }
        }
        addWindowListener(new FrameListener());
    }

    private void addRenderer() {
//		_crTable.setDefaultRenderer(ComposedReference.class, new ComposedReferenceRenderer()); // funktioniert nicht. warum?
        _crTable.getColumnModel().getColumn(0).setCellRenderer(new ComposedReferenceRenderer());
    }

    private void addListSelectionListener() {
        final ListSelectionModel selectionModel = _crTable.getSelectionModel();
        // vordefinierte Darstellungsobjekttypen dürfen nicht bearbeitet oder gelöscht werden
        ListSelectionListener listSelectionListener = e -> {
            final int selectedRow = _crTable.getSelectedRow();
            if (selectedRow == -1) {
                _deleteButton.setEnabled(false);
                _upwardButton.setEnabled(false);
            } else {
                _deleteButton.setEnabled(_editable);
                if (selectedRow == 0) {
                    _upwardButton.setEnabled(false);
                } else {
                    _upwardButton.setEnabled(_editable);
                }
            }
        };
        selectionModel.addListSelectionListener(listSelectionListener);
    }

    @Override
    public String toString() {
        return "CrDefinitionDialog{}";
    }

    private interface RhEditingStateListener {
        @SuppressWarnings("unused")
        void rhEditingStateChanged(boolean editable);
    }

    private static class ComposedReferenceRenderer extends DefaultTableCellRenderer {
        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
            super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
            if (null != value) { // nur zu Sicherheit bei künftigen Änderungen: Macs rufen mit null auf.
                if (value instanceof ComposedReference) {
                    setText(((ComposedReference) value).getName());
                }
            }
            return this;
        }
    }

    private class EditableListenButton extends JButton implements RhEditingStateListener {

        EditableListenButton(String text) {
            super(text);
            _listeningButtons.add(this);

        }

        @Override
        public void rhEditingStateChanged(boolean editable) {
            setEnabled(editable);
        }
    }

}
