/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.extLocRef.view;

import de.bsvrz.sys.funclib.debug.Debug;
import de.kappich.pat.gnd.documentation.GndHelp;
import de.kappich.pat.gnd.extLocRef.CRCollection;
import de.kappich.pat.gnd.extLocRef.ComposedReference;
import de.kappich.pat.gnd.extLocRef.ComposedReferenceManager;
import de.kappich.pat.gnd.extLocRef.ReferenceHierarchy;
import de.kappich.pat.gnd.extLocRef.ReferenceHierarchyManager;
import de.kappich.pat.gnd.gnd.PluginManager;
import de.kappich.pat.gnd.utils.SpringUtilities;
import de.kappich.pat.gnd.utils.view.GndDialog;
import de.kappich.pat.gnd.utils.view.GndFrame;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SpringLayout;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelListener;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableModel;

/**
 * Der Dialog zur Definition und Bearbeitung von {@link ReferenceHierarchy ReferenceHierarchies}.
 *
 * @author Kappich Systemberatung
 */
@SuppressWarnings("OverlyLongMethod")
public class RhDefinitionDialog extends GndFrame {

    private static final Debug _debug = Debug.getLogger();
    private static final int _lowerHeightBound = 310;
    private static final int _upperHeightBound = Toolkit.getDefaultToolkit().getScreenSize().height;
    private final JTextField _nameTextField = new JTextField();
    private final JTextField _infoTextField = new JTextField();
    private final JComboBox<String> _geometryPlugins = new JComboBox<>(PluginManager.getGeometryPluginNames());
    private final JTable _crTable = new JTable();
    private final List<EditableListenButton> _listeningButtons = new ArrayList<>();
    private final EditableListenButton _newButton = new EditableListenButton("Neue Zeile");
    private final JButton _deleteButton = new JButton("Löschen");
    private final JButton _upwardButton = new JButton("Aufwärts");
    private ReferenceHierarchy _scratchReferenceHierarchy;
    private ReferenceHierarchy _unchangeableOriginalReferenceHierarchy;
    private boolean _editable;
    private boolean _nameChangeable;
    private boolean _somethingChanged;

    /**
     * Konstruktor zum Anlegen eines neuen EOR-Hierarchie-Editors.
     *
     * @param hierarchy      eine ReferenceHierarchy
     * @param editable       ist die ReferenceHierarchy veränderbar?
     * @param nameChangeable ist der Name und damit die Identität änderbar?
     * @param title          der Titel des Fensters
     */
    public RhDefinitionDialog(final ReferenceHierarchy hierarchy, final boolean editable, final boolean nameChangeable, final String title) {
        super("RhDefinitionDialog", title);
        if (hierarchy != null) {
            _scratchReferenceHierarchy = hierarchy.getCopy();
            _unchangeableOriginalReferenceHierarchy = _scratchReferenceHierarchy.getCopy();
        } else {
            _debug.error("Der Konstruktor des EOR-Hierarchie-Editors wurde ohne EOR-Hierarchie aufgerufen.");
            throw new IllegalArgumentException("Der Konstruktor des EOR-Hierarchie-Editors wurde ohne EOR-Hierarchie aufgerufen.");
        }
        _editable = editable;
        _nameChangeable = nameChangeable;

        setLayout(new BorderLayout());

        Dimension labelSize = new Dimension(100, 20);

        // Oberer Teil mit Name, Info und Geometrie-Typ

        JLabel nameLabel = new JLabel("Name: ");
        nameLabel.setPreferredSize(labelSize);
        _nameTextField.setText(_scratchReferenceHierarchy.getName());
        _nameTextField.setEditable(nameChangeable);

        JLabel infoLabel = new JLabel("Info: ");
        infoLabel.setPreferredSize(labelSize);
        _infoTextField.setText(_scratchReferenceHierarchy.getInfo());

        JLabel geometryLabel = new JLabel("Geometrie-Typ:");
        geometryLabel.setPreferredSize(labelSize);
        _geometryPlugins.setPreferredSize(new Dimension(200, 25));
        _geometryPlugins.setSelectedItem(_scratchReferenceHierarchy.getGeometryType());

        JPanel upperPanel = new JPanel();
        upperPanel.setLayout(new SpringLayout());
        upperPanel.add(nameLabel);
        upperPanel.add(_nameTextField);
        upperPanel.add(infoLabel);
        upperPanel.add(_infoTextField);
        upperPanel.add(geometryLabel);
        upperPanel.add(_geometryPlugins);

        upperPanel.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));
        SpringUtilities.makeCompactGrid(upperPanel, 2, 5, 5);
        add(upperPanel, BorderLayout.NORTH);

        // Mittelteil für die EOR

        setTableProperties(_scratchReferenceHierarchy.getComposedReferences());
        addListSelectionListener();

        addButtonListener();

        JPanel buttonsPanel = new JPanel();
        buttonsPanel.setLayout(new SpringLayout());

        buttonsPanel.add(_newButton);
        buttonsPanel.add(_deleteButton);
        buttonsPanel.add(_upwardButton);

        buttonsPanel.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));
        SpringUtilities.makeCompactGrid(buttonsPanel, 1, 5, 20);

        JPanel centerPanel = new JPanel();
        centerPanel.setLayout(new SpringLayout());
        centerPanel.setBorder(BorderFactory.createMatteBorder(2, 2, 2, 2, Color.BLACK));
        centerPanel.add(new JScrollPane(_crTable));
        centerPanel.add(buttonsPanel);
        SpringUtilities.makeCompactGrid(centerPanel, 2, 20, 5);

        add(centerPanel, BorderLayout.CENTER);

        // Unterer Teil mit Buttons

        EditableListenButton saveButton = new EditableListenButton("Speichern");
        JButton cancelButton = new JButton("Dialog schließen");
        JButton helpButton = new JButton("Hilfe");
        addButtonListener(saveButton, cancelButton, helpButton);

        JPanel crButtonsPanel = new JPanel();
        crButtonsPanel.setLayout(new SpringLayout());

        crButtonsPanel.add(saveButton);
        crButtonsPanel.add(cancelButton);
        crButtonsPanel.add(helpButton);

        crButtonsPanel.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));
        SpringUtilities.makeCompactGrid(crButtonsPanel, 3, 20, 5);
        add(crButtonsPanel, BorderLayout.SOUTH);

        addChangeListeners();
        addFrameListener();
        addRenderer();

        setEditable(_editable, _nameChangeable);
        setPositionAndSize();
        setVisible(true);
        toFront();
    }

    private static boolean saveChangesWanted() {
        Object[] options = {"Änderungen speichern", "Nicht speichern"};
        int n = JOptionPane.showOptionDialog(new JFrame(), "Änderungen speichern?", "Es wurden Änderungen an der EOR-Hierarchie vorgenommen.",
                                             JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE, null, options, options[1]);
        return n == 0;
    }

    private void setPositionAndSize() {
        Dimension d = _crTable.getPreferredSize();
        setPositionAndSize(690, 202 + (int) d.getHeight(), 660, 0, false, 690,
                           Math.min(Math.max(202 + (int) d.getHeight(), _lowerHeightBound), _upperHeightBound));
    }

    private void addChangeListeners() {
        {
            _nameTextField.getDocument().addDocumentListener(new DocumentListener() {
                @Override
                public void changedUpdate(DocumentEvent e) {
                    _somethingChanged = true;
                }

                @Override
                public void removeUpdate(DocumentEvent e) {
                    _somethingChanged = true;
                }

                @Override
                public void insertUpdate(DocumentEvent e) {
                    _somethingChanged = true;
                }
            });
        }

        {
            _infoTextField.getDocument().addDocumentListener(new DocumentListener() {
                @Override
                public void changedUpdate(DocumentEvent e) {
                    _somethingChanged = true;
                }

                @Override
                public void removeUpdate(DocumentEvent e) {
                    _somethingChanged = true;
                }

                @Override
                public void insertUpdate(DocumentEvent e) {
                    _somethingChanged = true;
                }
            });
        }

        TableModelListener tableModelListener = e -> _somethingChanged = true;
        _crTable.getModel().addTableModelListener(tableModelListener);
    }

    private void setTableProperties(CRCollection crCollection) {
        _crTable.setModel(crCollection);
        _crTable.setRowHeight(25);
        int vColIndex = 0;
        TableColumn col = _crTable.getColumnModel().getColumn(vColIndex);
        col.setPreferredWidth(327);
        _deleteButton.setEnabled(false);
        _upwardButton.setEnabled(_crTable.getSelectedColumnCount() > 0);
    }

    /**
     * Setzt die Felder des EOR-Hierarchie-Editors mit den Informationen der übergebenen EOR-Hierarchie und aktiviert die Veränderbarkeit gemäß der
     * zwei boolschen Werte.
     *
     * @param hierarchy      eine EOR-Hierarchie
     * @param editable       ist die EOR-Hierarchie veränderbar?
     * @param nameChangeable ist der Name und damit die Identität der EOR-Hierarchie änderbar?
     */
    private void setHierarchy(ReferenceHierarchy hierarchy, boolean editable, boolean nameChangeable) {
        _scratchReferenceHierarchy = hierarchy;
        _unchangeableOriginalReferenceHierarchy = _scratchReferenceHierarchy.getCopy();
        _editable = editable;
        _nameTextField.setText(_scratchReferenceHierarchy.getName());
        _infoTextField.setText(_scratchReferenceHierarchy.getInfo());
        setTableProperties(_scratchReferenceHierarchy.getComposedReferences());
        setEditable(editable, nameChangeable);
        for (EditableListenButton elButton : _listeningButtons) {
            elButton.rhEditingStateChanged(_editable);
        }
        _somethingChanged = false;
    }

    /**
     * Setzt den Wert der internen Variable, die darüber entscheidet, ob die Informationen der angezeigten EOR-Hierachie veränderbar sind, und macht
     * Textfelder veränderbar oder nicht, aktiviert bzw. deaktiviert Knöpfe usw.
     *
     * @param editable       ist die EOR-Hierachie veränderbar?
     * @param nameChangeable ist der Name und damit die Identität der EOR-Hierachie änderbar?
     */
    public final void setEditable(boolean editable, boolean nameChangeable) {
        _editable = editable;
        _nameChangeable = nameChangeable;
        for (EditableListenButton elButton : _listeningButtons) {
            elButton.rhEditingStateChanged(_editable);
        }
        _nameTextField.setEditable(_nameChangeable);
        _infoTextField.setEditable(_editable);
        _crTable.setRowSelectionAllowed(_editable);
        _newButton.setEnabled(_editable);
    }

    @SuppressWarnings("OverlyLongMethod")
    private void addButtonListener() {

        final class CRDialog extends GndDialog {

            private final JComboBox<Object> _comboBox;

            private CRDialog(final String geometryType) {
                super("CRDialog", RhDefinitionDialog.this.getFrame(), true);

                JLabel crLabel = new JLabel("EOR: ");
                Object[] array = ComposedReferenceManager.getInstance().getComposedReferenceNames(geometryType);

                Arrays.sort(array, (first, second) -> {
                    String f = (String) first;
                    String s = (String) second;
                    return f.toLowerCase().compareTo(s.toLowerCase());
                });
                _comboBox = new JComboBox<>(array);

                int selectedRow = _crTable.getSelectedRow();
                if (selectedRow == -1) {
                    if (_crTable.getModel().getRowCount() > 0) {
                        selectedRow = 0;
                    }
                }
                if (selectedRow > -1) {
                    final CRCollection crCollection = _scratchReferenceHierarchy.getComposedReferences();
                    _comboBox.setSelectedItem(crCollection.getValueAt(selectedRow, 0));
                }

                JPanel panel = new JPanel();
                panel.setLayout(new SpringLayout());
                panel.add(crLabel);
                panel.add(_comboBox);
                SpringUtilities.makeCompactGrid(panel, 2, 20, 5);

                EditableListenButton saveButton = new EditableListenButton("Übernehmen");
                JButton cancelButton = new JButton("Dialog schließen");

                JPanel buttonsPanel = new JPanel();
                buttonsPanel.setLayout(new SpringLayout());

                buttonsPanel.add(saveButton);
                buttonsPanel.add(cancelButton);

                buttonsPanel.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));
                SpringUtilities.makeCompactGrid(buttonsPanel, 2, 20, 5);
                addButtonListeners(saveButton, cancelButton);
                addDialogListener();

                setTitle("GND: Darstellungsfestlegung einer EOR-Hierarchie");
                setLayout(new BorderLayout());
                add(panel, BorderLayout.NORTH);
                add(buttonsPanel, BorderLayout.SOUTH);

                final Rectangle bounds = getBounds();
                setPositionAndSize(480, 160, (int) (bounds.getHeight() / 1.5 + 720), (int) (bounds.getWidth() / 1.5), true, 0, 0);
            }

            private void runDialog() {
                setVisible(true);
            }

            private void addButtonListeners(JButton saveButton, JButton cancelButton) {
                ActionListener actionListenerSave = e -> {
                    final Object item = _comboBox.getSelectedItem();
                    if (item != null && (item instanceof String)) {
                        ComposedReference cr = ComposedReferenceManager.getInstance().getComposedReference((String) item);
                        if (!_scratchReferenceHierarchy.addComposedReference(cr)) {
                            JOptionPane.showMessageDialog(this.getDialog(), "Die EOR-Hierarchie enthält diese EOR bereits!", "Fehler",
                                                          JOptionPane.ERROR_MESSAGE);
                        }
                    }
                };
                saveButton.addActionListener(actionListenerSave);

                ActionListener actionListenerCancel = e -> {
                    storePreferenceBounds();
                    dispose();
                };
                cancelButton.addActionListener(actionListenerCancel);
            }

            private void addDialogListener() {
                class DialogListener extends WindowAdapter {
                    @Override
                    public void windowClosing(WindowEvent e) {
                        storePreferenceBounds();
                    }
                }
                addWindowListener(new DialogListener());
            }

            @Override
            public String toString() {
                return "CRDialog{}";
            }
        }

        ActionListener actionListenerNew = e -> {
            Object selected = _geometryPlugins.getSelectedItem();
            if (null == selected) {
                JOptionPane.showMessageDialog(new JFrame(), "Bitte wählen Sie einen Geometrie-Typ aus!", "Fehler", JOptionPane.ERROR_MESSAGE);
                return;
            }
            CRDialog crDialog = new CRDialog((String) selected);
            crDialog.runDialog();
        };
        _newButton.addActionListener(actionListenerNew);

        ActionListener actionListenerDelete = e -> {
            final int[] selectedRows = _crTable.getSelectedRows();
            if (selectedRows.length == 0) {
                JOptionPane.showMessageDialog(new JFrame(), "Bitte wählen Sie mindestens eine Zeile aus der Liste aus!", "Fehler",
                                              JOptionPane.ERROR_MESSAGE);
                return;
            }
            for (int i = selectedRows.length - 1; i >= 0; i--) {
                final int selectedRow = selectedRows[i];
                final TableModel model = _crTable.getModel();
                CRCollection crCollection = (CRCollection) model;
                final ComposedReference cr = (ComposedReference) crCollection.getValueAt(selectedRow, 0);
                if (null != cr) {
                    _scratchReferenceHierarchy.getComposedReferences().remove(cr);
                    crCollection.remove(cr);
                }
            }
        };
        _deleteButton.addActionListener(actionListenerDelete);

        ActionListener actionListenerUpward = e -> {
            final int[] selectedRows = _crTable.getSelectedRows();
            if (selectedRows.length == 0) {
                JOptionPane.showMessageDialog(new JFrame(), "Bitte wählen Sie mindestens eine Zeile aus der Liste aus!", "Fehler",
                                              JOptionPane.ERROR_MESSAGE);
                return;
            }
            Arrays.sort(selectedRows);
            if (0 == selectedRows[0]) {
                JOptionPane.showMessageDialog(new JFrame(), "Die erste Zeile darf nicht ausgewählt sein!", "Fehler", JOptionPane.ERROR_MESSAGE);
                return;
            }
            if (_scratchReferenceHierarchy.getComposedReferences().moveUpwards(selectedRows)) {
                for (int index : selectedRows) {
                    _crTable.addRowSelectionInterval(index - 1, index - 1);
                }
            }
        };
        _upwardButton.addActionListener(actionListenerUpward);
    }

    private void addButtonListener(final JButton saveButton, final JButton cancelButton, final JButton helpButton) {

        ActionListener actionListenerSave = e -> saveReferenceHierarchie();
        saveButton.addActionListener(actionListenerSave);

        ActionListener actionListenerCancel = e -> {
            if (_editable && _somethingChanged) {
                if (!saveChangesWanted()) {      // Es wird nicht gespeichert: Änderungen rückgängig machen! Dieses Verhalten
                    // wurde vom LayerDefinitionDialog übrnommen.
                    setHierarchy(_unchangeableOriginalReferenceHierarchy.getCopy(), _editable, _nameChangeable);
                }
            }
            setVisible(false);
            storePreferenceBounds();
            dispose();
        };
        cancelButton.addActionListener(actionListenerCancel);

        ActionListener actionListenerHelp = e -> GndHelp.openHelp("#eorHierarchyDefinitionDialog");
        helpButton.addActionListener(actionListenerHelp);
    }

    private void saveReferenceHierarchie() {
        if (!_editable) {    // Sollte nie passieren, da der Button disabled ist.
            JOptionPane.showMessageDialog(this.getFrame(), "Diese EOR-Hierarchie ist nicht veränderbar!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return;
        }
        String rhName = _nameTextField.getText();
        if (rhName.isEmpty()) {
            JOptionPane.showMessageDialog(this.getFrame(), "Bitte geben Sie einen Namen ein!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return;
        } else if (rhName.length() > 40) {    // Präferenzen erlauben 80 Zeichen als Node-Name, doch hier wird 40 gewählt.
            JOptionPane
                .showMessageDialog(this.getFrame(), "Bitte verwenden Sie einen Namen mit höchstens 40 Zeichen!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return;
        }
        String infoText = _infoTextField.getText();
        if (infoText == null) {
            infoText = "";
        }
        Object selected = _geometryPlugins.getSelectedItem();
        if (null == selected) {
            JOptionPane.showMessageDialog(this.getFrame(), "Bitte wählen Sie einen Geometrie-Typ aus!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return;
        }
        String geometryType = (String) selected;

        final CRCollection crCollection = (CRCollection) _crTable.getModel();

        if (crCollection.isEmpty()) {
            JOptionPane.showMessageDialog(this.getFrame(), "Bitte geben Sie mindestens eine EOR an!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return;
        }

        for (ComposedReference cr : crCollection.getComposedReferences()) {
            ComposedReference crOrig = ComposedReferenceManager.getInstance().getComposedReference(cr.getName());
            if (!crOrig.getGeometryType().equals(geometryType)) {
                JOptionPane.showMessageDialog(this.getFrame(),
                                              "Das Speichern ist unmöglich: die EOR '" + cr.getName() + "' hat einen abweichenden Geometrie-Typ!",
                                              "Fehler", JOptionPane.ERROR_MESSAGE);
                return;
            }
        }

        final ReferenceHierarchy existingHierarchy = ReferenceHierarchyManager.getInstance().getReferenceHierarchy(rhName);
        if (existingHierarchy != null) {
            if (ReferenceHierarchyManager.getInstance().isChangeable(existingHierarchy)) {
                Object[] options = {"EOR-Hierarchie überschreiben", "Speichern abbrechen"};
                int n = JOptionPane
                    .showOptionDialog(this.getFrame(), "Soll die bestehende EOR-Hierarchie mit diesem Namen wirklich überschrieben werden?",
                                      "EOR-Hierarchie speichern", JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE, null, options,
                                      options[1]);
                if (n != 0) {
                    return;
                }
            } else {
                JOptionPane.showMessageDialog(this.getFrame(), "Die bestehende EOR-Hierarchie darf nicht überschrieben werden!", "Fehler",
                                              JOptionPane.ERROR_MESSAGE);
                return;
            }
        } else if (ReferenceHierarchyManager.getInstance().hasReferenceHierarchie(rhName)) {
            JOptionPane.showMessageDialog(this.getFrame(), "Es existiert bereits eine EOR-Hierarchie mit diesem Namen!", "Fehler",
                                          JOptionPane.ERROR_MESSAGE);
            return;
        }

        ReferenceHierarchy newHierarchie = new ReferenceHierarchy(rhName, infoText, geometryType);
        newHierarchie.setComposedReferences(crCollection.getComposedReferences(), false);

        //noinspection VariableNotUsedInsideIf
        if (existingHierarchy != null) {
            ReferenceHierarchyManager.getInstance().changeReferenceHierarchy(newHierarchie);
        } else {
            ReferenceHierarchyManager.getInstance().addReferenceHierarchy(newHierarchie);
        }
        _scratchReferenceHierarchy = newHierarchie;
        _unchangeableOriginalReferenceHierarchy = _scratchReferenceHierarchy.getCopy();
        _somethingChanged = false;
        // Damit der Benutzer dasselbe Objekt nicht unter anderem Namen speichern kann:
        _nameChangeable = false;
        _nameTextField.setEditable(false);
    }

    private void addFrameListener() {
        class FrameListener extends WindowAdapter {
            @Override
            public void windowClosing(WindowEvent e) {
                if (_somethingChanged && _editable) {
                    if (saveChangesWanted()) {
                        saveReferenceHierarchie();
                        setVisible(false);
                        storePreferenceBounds();
                        dispose();
                    }
                }
            }
        }
        addWindowListener(new FrameListener());
    }

    private void addRenderer() {
//		_crTable.setDefaultRenderer(ComposedReference.class, new ComposedReferenceRenderer()); // funktioniert nicht. warum?
        _crTable.getColumnModel().getColumn(0).setCellRenderer(new ComposedReferenceRenderer());
    }

    private void addListSelectionListener() {
        final ListSelectionModel selectionModel = _crTable.getSelectionModel();
        // vordefinierte Darstellungsobjekttypen dürfen nicht bearbeitet oder gelöscht werden
        ListSelectionListener listSelectionListener = e -> {
            final int selectedRow = _crTable.getSelectedRow();
            if (selectedRow == -1) {
                _deleteButton.setEnabled(false);
                _upwardButton.setEnabled(false);
            } else {
                _deleteButton.setEnabled(_editable);
                if (selectedRow == 0) {
                    _upwardButton.setEnabled(false);
                } else {
                    _upwardButton.setEnabled(_editable);
                }
            }
        };
        selectionModel.addListSelectionListener(listSelectionListener);
    }

    @Override
    public String toString() {
        return "RhDefinitionDialog{}";
    }

    private interface RhEditingStateListener {
        @SuppressWarnings("unused")
        void rhEditingStateChanged(boolean editable);
    }

    private static class ComposedReferenceRenderer extends DefaultTableCellRenderer {
        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
            super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
            if (null != value) { // nur zur Sicherheit bei künftigen Änderungen: Macs rufen mit null auf.
                if (value instanceof ComposedReference) {
                    setText(((ComposedReference) value).getName());
                }
            }
            return this;
        }
    }

    private class EditableListenButton extends JButton implements RhEditingStateListener {

        EditableListenButton(String text) {
            super(text);
            _listeningButtons.add(this);
        }

        @Override
        public void rhEditingStateChanged(boolean editable) {
            setEnabled(editable);
        }
    }
}
