/*
 * Copyright 2010-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.gnd;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.pat.sysbed.help.GtmHelp;
import de.bsvrz.pat.sysbed.main.ApplicationInterface;
import de.bsvrz.pat.sysbed.main.GenericTestMonitorApplication;
import de.bsvrz.pat.sysbed.plugins.api.ButtonBar;
import de.bsvrz.pat.sysbed.plugins.api.DataIdentificationChoice;
import de.bsvrz.pat.sysbed.plugins.api.DialogInterface;
import de.bsvrz.pat.sysbed.plugins.api.ExternalModuleAdapter;
import de.bsvrz.pat.sysbed.plugins.api.settings.SettingsData;
import de.bsvrz.pat.sysbed.preselection.lists.PreselectionLists;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.colorManagement.ColorManager;
import de.kappich.pat.gnd.extLocRef.ReferenceHierarchyManager;
import de.kappich.pat.gnd.viewManagement.View;
import de.kappich.pat.gnd.viewManagement.ViewManager;
import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Window;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;

/**
 * Eine Klasse, um die GND als GTM-Plugin zur Verfügung zu stellen.
 *
 * @author Kappich Systemberatung
 */
public class GNDPlugin extends ExternalModuleAdapter {

    private static final Debug _debug = Debug.getLogger();
    /** speichert den Dialog des GND-Plugins */
    private static GNDPluginDialog _dialog;
    private String _tooltipText;
    /** Kennzeichnet, ob eine neue Instanz des GND geöffnet werden soll. */
    private boolean _newInstanceOfGnd;
    /** Ist gekoppelt mit: */
    private JCheckBox _checkInstance;

    @Override
    public void change(SettingsData settingsData) {
        _dialog = new GNDPluginDialog(getApplication().getParent());
        _dialog.setSettings(settingsData);
    }

    @Override
    public boolean isPreselectionValid(SettingsData settingsData) {
        SystemObjectType type = getApplication().getConnection().getDataModel().getType("typ.geoReferenzObjekt");

        final List<SystemObject> objects = settingsData.getObjects();

        if (objects != null) {
            for (SystemObject systemObject : objects) {
                if (!systemObject.getType().inheritsFrom(type)) {
                    _tooltipText = "Es muss ein Georeferenzobjekt ausgewählt werden.";
                    return false;
                }
            }
        }
        _tooltipText = "Auswahl übernehmen";
        return true;
    }

    @Override
    public String getButtonText() {
        return "GND starten";
    }

    @Override
    public String getModuleName() {
        return "GND-Plugin";
    }

    @Override
    public String getTooltipText() {
        return _tooltipText;
    }

    @Override
    public void startModule(SettingsData settingsData) {
        _dialog = new GNDPluginDialog(getApplication().getParent());
        _dialog.setDataIdentification(settingsData);
    }

    @Override
    public void startSettings(SettingsData settingsData) {
        _dialog = new GNDPluginDialog(getApplication().getParent());
        _dialog.startSettings(settingsData);
    }

    @SuppressWarnings("ConstantConditions")
    public void setSelectedObjects(Collection<SystemObject> systemObjects) {
        ArrayList<SystemObject> arrayListSystemObjects = new ArrayList<>(systemObjects);
        ApplicationInterface application = getApplication();
	    if (application instanceof GenericTestMonitorApplication genericTestMonitorApplication) {
            PreselectionLists preselectionLists = genericTestMonitorApplication.getPreselectionLists();
            preselectionLists.setPreselectedObjects(arrayListSystemObjects);
            preselectionLists.setPreselectedAspects(null);
            preselectionLists.setPreselectedAttributeGroups(null);
            preselectionLists.setPreselectedObjectTypes(null);
        }
    }

    @Override
    public String toString() {
        return "GNDPlugin{}";
    }

    @SuppressWarnings("serial")
    static class StopInitializationException extends RuntimeException {
    }

    private class GNDPluginDialog implements DialogInterface {

        private final Window _parent;
        ViewManager _viewManager;
        /** speichert den Dialog */
        private JDialog _settingsDialog;
        /** speichert eine Instanz der Datenidentifikationsauswahl */
        private DataIdentificationChoice _dataIdentificationChoice;
        private JComboBox<String> _displayViewsComboBox;
        private JComboBox<GenericNetDisplay> _displayGenericNetDisplays;

        GNDPluginDialog(@Nullable final Window parent) {
            _parent = parent;
            // Der ReferenceHierarchyManager wird schon im LayerManager benötigt, wo allerdings kein ClientDavInterface
            // zur Verfügung steht; deshalb Initialiesierung hier, insbesondere vor dem ViewManager.
            ReferenceHierarchyManager.getInstance(getApplication().getConnection());
            // Der ColorManager wird schon hier initialisiert, weil er ansonsten erst bei paintDisplayObject initialisiert
            // wird, und wenn es da zu Problemen käme, gibt es eine NPE.
            //noinspection ResultOfMethodCallIgnored
            ColorManager.getInstance();
            _viewManager = ViewManager.getInstance();
            _settingsDialog = null;
        }

        /**
         * Getter für die Ansicht.
         *
         * @return der Ansichtsname
         */
        public String getView() {
            return (String) _displayViewsComboBox.getSelectedItem();
        }

        /**
         * Setter für die Ansicht.
         *
         * @param view der Ansichtsname
         */
        public void setView(String view) {
            _displayViewsComboBox.setSelectedItem(view);
        }

        /**
         * Getter für das GenericNetDisplay.
         *
         * @return die GenericNetDisplay
         */
        @Nullable
        GenericNetDisplay getGenericNetDisplay() {
            if (_displayGenericNetDisplays == null) {
                return null;
            }
            return (GenericNetDisplay) _displayGenericNetDisplays.getSelectedItem();
        }

        /**
         * Setter für das GenericNetDisplay in der Combo-Box.
         *
         * @param genericNetDisplay GenericNetDisplay
         */
        @SuppressWarnings("unused")
        public void setGenericNetDisplay(final GenericNetDisplay genericNetDisplay) {
            if (_displayGenericNetDisplays != null) {
                _displayGenericNetDisplays.setSelectedItem(genericNetDisplay);
            }
        }

        /** diese Methode schließt den Dialog */
        @Override
        public void doCancel() {
            _settingsDialog.setVisible(false);
            _settingsDialog.dispose();
        }

        /** Durch betätigen des "OK"-Buttons wird die GND gestartet und dieser Dialog wird geschlossen. Die Parameter werden gespeichert. */
        @Override
        public void doOK() {
            SettingsData settingsData = getSettings("");
            startSettings(settingsData);
            doCancel();
            saveSettings(settingsData);
        }

        /**
         * diese Methode speichert die Parameter
         *
         * @param title Titel dieser Konfiguration
         */
        @Override
        public void doSave(String title) {
            SettingsData settingsData = getSettings(title);
            saveSettings(settingsData);
        }

        /**
         * Durch Betätigen des "Hilfe"-Buttons wird die Kontexthilfe geöffnet.
         */
        @Override
        public void openHelp() {
            ApplicationInterface application = getApplication();
            if (application instanceof GenericTestMonitorApplication) {
                GtmHelp.openHelp("#GND");
            }
        }

        /**
         * Erstellt die Einstellungsdaten.
         *
         * @param title der Name für die Einstellungen
         *
         * @return die Einstellungsdaten
         */
        private SettingsData getSettings(String title) {
            Class<GNDPlugin> moduleClass = GNDPlugin.class;
            List<SystemObjectType> objectTypes = _dataIdentificationChoice.getObjectTypes();
            AttributeGroup atg = _dataIdentificationChoice.getAttributeGroup();
            Aspect asp = _dataIdentificationChoice.getAspect();
            List<SystemObject> objects = _dataIdentificationChoice.getObjects();

            SettingsData settingsData = new SettingsData(getModuleName(), moduleClass, objectTypes, atg, asp, objects);
            settingsData.setTitle(title);
            settingsData.setSimulationVariant(_dataIdentificationChoice.getSimulationVariant());
            settingsData.setTreePath(_dataIdentificationChoice.getTreePath());
            settingsData.setSettingsMap(getSettingsMap());

            return settingsData;
        }

        /** Erstellt den Dialog. Bestandteil ist die Datenidentifikation und die Anmeldeoption, bestehend aus der Rolle und der Anmeldeart. */
        @SuppressWarnings("OverlyLongMethod")
        private void createDialog() {
            _settingsDialog = new JDialog(_parent);
            _settingsDialog.setTitle(getButtonText());
            _settingsDialog.setResizable(false);

            Container pane = _settingsDialog.getContentPane();
            pane.setLayout(new BoxLayout(pane, BoxLayout.Y_AXIS));

            // Datenidentifikationsauswahl-Panel
            final List<SystemObjectType> types = new LinkedList<>();
            DataModel configuration = getConnection().getDataModel();
            types.add(configuration.getType("typ.konfigurationsObjekt"));
            types.add(configuration.getType("typ.dynamischesObjekt"));
            //noinspection ConstantConditions
            _dataIdentificationChoice = new DataIdentificationChoice(null, types);
            pane.add(_dataIdentificationChoice);

            // Darstellungsoptionen
            JPanel displayOptionsPanel = new JPanel();
            displayOptionsPanel.setLayout(new BoxLayout(displayOptionsPanel, BoxLayout.X_AXIS));
            displayOptionsPanel.setBorder(BorderFactory.createTitledBorder("Ansichtseinstellung"));
            JLabel displayOptionsLabel = new JLabel("Ansicht: ");

            Set<String> viewNames = _viewManager.getViewNames();
            String[] viewArray = viewNames.toArray(new String[0]);

            _displayViewsComboBox = new JComboBox<>(viewArray);
            final String startViewName = GenericNetDisplay.getStartViewNameFromPreferences(getConnection());
            if ((startViewName != null) && _viewManager.hasView(startViewName)) {
                _displayViewsComboBox.setSelectedItem(startViewName);
            } else if (_viewManager.hasView("Vordefinierte Ansicht 1")) {
                _displayViewsComboBox.setSelectedItem(GenericNetDisplay.INTERNAL_START_VIEW_NAME);
            } else if (viewArray.length > 0) {
                _displayViewsComboBox.setSelectedIndex(0);
            }
            displayOptionsLabel.setLabelFor(_displayViewsComboBox);
            displayOptionsPanel.add(displayOptionsLabel);
            displayOptionsPanel.add(Box.createHorizontalStrut(5));
            displayOptionsPanel.add(_displayViewsComboBox);

            pane.add(displayOptionsPanel);

            // Instanzoptionen
            JPanel gndOptionsPanel = new JPanel();

            //gndOptionsPanel.setLayout(new BoxLayout(gndOptionsPanel, BoxLayout.X_AXIS));
            gndOptionsPanel.setLayout(new BorderLayout());
            gndOptionsPanel.setBorder(BorderFactory.createTitledBorder("Fensteroptionen"));
            _checkInstance = new JCheckBox("GND im neuen Fenster öffnen.");

            _newInstanceOfGnd = _checkInstance.isSelected();

            Collection<GenericNetDisplay> instances = GenericNetDisplay.getInstances();
            _checkInstance.setSelected(instances.isEmpty());
            _newInstanceOfGnd = _checkInstance.isSelected();
            _checkInstance.addActionListener(e -> _newInstanceOfGnd = _checkInstance.isSelected());

            gndOptionsPanel.add(Box.createHorizontalStrut(5));
            gndOptionsPanel.add(_checkInstance, BorderLayout.CENTER);

            pane.add(gndOptionsPanel);

            // GND-Auswahl
            JPanel gndSelectionPanel = new JPanel();
            gndSelectionPanel.setLayout(new BoxLayout(gndSelectionPanel, BoxLayout.X_AXIS));
            JLabel gndLabel = new JLabel("GND: ");
            Collection<GenericNetDisplay> gnds = GenericNetDisplay.getInstances();
            GenericNetDisplay[] gndArray = gnds.toArray(new GenericNetDisplay[0]);
            Arrays.sort(gndArray);
            _displayGenericNetDisplays = new JComboBox<>(gndArray);
            gndSelectionPanel.setBorder(BorderFactory.createTitledBorder("GND-Auswahl"));

            gndLabel.setLabelFor(_displayGenericNetDisplays);
            gndSelectionPanel.add(gndLabel);
            gndSelectionPanel.add(Box.createHorizontalStrut(5));
            gndSelectionPanel.add(_displayGenericNetDisplays);

            pane.add(gndSelectionPanel);

            // untere Buttonleiste
            final ButtonBar buttonBar = new ButtonBar(this);
            _settingsDialog.getRootPane().setDefaultButton(buttonBar.getAcceptButton());
            pane.add(buttonBar);
        }

        /**
         * Diese Methode zeigt den Dialog an und trägt die Einstellungsdaten in die entsprechenden Felder ein.
         *
         * @param data die Einstellungsdaten
         */
        public void setSettings(final SettingsData data) {
            if (_settingsDialog == null) {
                createDialog();
            }
            _dataIdentificationChoice.setDataIdentification(data.getObjectTypes(), data.getAttributeGroup(), data.getAspect(), data.getObjects(),
                                                            data.getSimulationVariant());
            _dataIdentificationChoice.showTree(getApplication().getTreeNodes(), getApplication().getConnection(), data.getTreePath());

            Map<String, String> settingsMap = data.getSettingsMap();
            if (settingsMap.containsKey("view")) {
                setView(settingsMap.get("view"));
            }
            if (settingsMap.containsKey("instance")) {
	            _checkInstance.setSelected(Boolean.parseBoolean(settingsMap.get("instance")));
            }
            showDialog();
        }

        /** Durch diese Methode wird der Dialog angezeigt. */
        private void showDialog() {
            _settingsDialog.setLocation(70, 70);
            _settingsDialog.pack();
            _settingsDialog.setVisible(true);
        }

        /**
         * Startet die GND anhand der Einstellungsdaten.
         *
         * @param settingsData die Einstellungsdaten
         */
        public void startSettings(SettingsData settingsData) {
            final Map<String, String> settingsMap = settingsData.getSettingsMap();
            final List<SystemObject> objects = settingsData.getObjects();

            Thread gndThread = new Thread(() -> {
                try {
                    View view = new View();
                    String viewName = null;
                    if (!settingsMap.isEmpty()) {
                        if (settingsMap.containsKey("view")) {
                            viewName = settingsMap.get("view");
                            view = _viewManager.getView(viewName);
                        } else if (settingsMap.containsKey("instance")) {
	                        _newInstanceOfGnd = Boolean.parseBoolean(settingsMap.get("instance"));
                        }
                    } else {
                        viewName = (String) _displayViewsComboBox.getSelectedItem();
                        view = _viewManager.getView(viewName);
                    }
                    if (view == null) {
                        if (viewName == null) {
                            viewName = "???";
                        }
                        _debug.warning("GNDPlugin: eine Ansicht namens '" + viewName + "' kann nicht gefunden werden.");
                        return;
                    }

                    final ClientDavInterface connection = getApplication().getConnection();

                    GenericNetDisplay genericNetDisplay = getGenericNetDisplay();
                    if (_newInstanceOfGnd || genericNetDisplay == null) { // ein neues GenericNetDisplay wird geöffnet
                        GenericNetDisplay gnd = new GenericNetDisplay(view, connection, false, GNDPlugin.this);
                        gnd.setSystemObjects(objects);
                        gnd.setVisible(true);
                    } else { // das ausgewählte GenericNetDisplay wird auf den Stand gebracht
                        String nameOfActualView = genericNetDisplay.getView().getName();
                        if (!nameOfActualView.equals(view.getName())) {
                            genericNetDisplay.setSplitPaneFromView(view);
                        }
                        genericNetDisplay.setSystemObjects(objects);
                        genericNetDisplay.setVisible(true);
                        genericNetDisplay.toFront();
                    }
                } catch (StopInitializationException ignore) {
                }
            }

            );
            gndThread.setUncaughtExceptionHandler((t, e) -> _debug.error("Fehler: ", e));
            gndThread.start();
        }

        /**
         * Sammelt alle Parameter des Dialogs.
         *
         * @return Liste aller Parameter des Dialogs
         */
        private Map<String, String> getSettingsMap() {
            Map<String, String> map = new HashMap<>();
            map.put("view", getView());
            map.put("instance", String.valueOf(_checkInstance.isSelected()));
            return map;
        }

        /**
         * Mit dieser Methode können die Datenidentifikationsdaten übergeben werden. Der Dialog wird mit Default-Werten dargestellt.
         *
         * @param settingsData enthält die ausgewählte Datenidentifikation
         */
        public void setDataIdentification(SettingsData settingsData) {
            if (_settingsDialog == null) {
                createDialog();
            }
            _dataIdentificationChoice.setDataIdentification(settingsData.getObjectTypes(), settingsData.getAttributeGroup(), settingsData.getAspect(),
                                                            settingsData.getObjects(), settingsData.getSimulationVariant());
            _dataIdentificationChoice.showTree(getApplication().getTreeNodes(), getApplication().getConnection(), settingsData.getTreePath());
            showDialog();
        }
    }
}
