/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2022 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.kappich.pat.gnd.gnd;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.filechooser.AwtFileChooser;
import de.bsvrz.sys.funclib.kappich.onlinehelp.HelpMenu;
import de.bsvrz.sys.funclib.kappich.onlinehelp.HelpPage;
import de.kappich.pat.gnd.choose.ChooseByDialog;
import de.kappich.pat.gnd.choose.ChooseFromDialog;
import de.kappich.pat.gnd.colorManagement.ColorDialog;
import de.kappich.pat.gnd.colorManagement.ColorManager;
import de.kappich.pat.gnd.configBrowser.BrowserTree;
import de.kappich.pat.gnd.coorTransform.GeoTransformation;
import de.kappich.pat.gnd.csv.view.CfmDialog;
import de.kappich.pat.gnd.displayObjectToolkit.DOTManager;
import de.kappich.pat.gnd.displayObjectToolkit.DOTManagerDialog;
import de.kappich.pat.gnd.displayObjectToolkit.DisplayObject;
import de.kappich.pat.gnd.displayObjectToolkit.OnlineDisplayObject;
import de.kappich.pat.gnd.documentation.GndHelp;
import de.kappich.pat.gnd.extLocRef.SimpleReferenceManager;
import de.kappich.pat.gnd.extLocRef.view.CrmDialog;
import de.kappich.pat.gnd.extLocRef.view.RhmDialog;
import de.kappich.pat.gnd.gnd.MapPane.MapScaleListener;
import de.kappich.pat.gnd.layerManagement.Layer;
import de.kappich.pat.gnd.layerManagement.LayerManager;
import de.kappich.pat.gnd.layerManagement.LayerManagerDialog;
import de.kappich.pat.gnd.notice.NoticeManager;
import de.kappich.pat.gnd.utils.view.GndDialog;
import de.kappich.pat.gnd.utils.view.GndFrame;
import de.kappich.pat.gnd.utils.view.PreferencesDeleter;
import de.kappich.pat.gnd.viewManagement.*;

import javax.imageio.ImageIO;
import javax.print.attribute.HashPrintRequestAttributeSet;
import javax.print.attribute.PrintRequestAttributeSet;
import javax.print.attribute.ResolutionSyntax;
import javax.print.attribute.standard.PrinterResolution;
import javax.swing.*;
import javax.swing.border.TitledBorder;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.MenuEvent;
import javax.swing.event.MenuListener;
import javax.swing.filechooser.FileNameExtensionFilter;
import java.awt.*;
import java.awt.event.*;
import java.awt.image.BufferedImage;
import java.awt.print.PageFormat;
import java.awt.print.PrinterJob;
import java.io.*;
import java.util.List;
import java.util.*;
import java.util.prefs.BackingStoreException;
import java.util.prefs.InvalidPreferencesFormatException;
import java.util.prefs.Preferences;

/**
 * Ein GenericNetDisplay-Objekt ist ein Top-Level-Fenster, das eine Menüzeile besitzt und die Kartendarstellung zeigt. Diese Klasse wird wahlweise aus
 * RunGND heraus instanziiert (Stand-Alone-Anwendung) oder aus dem GNDPlugin (Plugin in einer anderen Anwendung, z.B. dem GTM).
 * <p>
 * Die Initialisierung eines Objekts erfolgt mit Hilfe einer {@link View Ansicht}, die eine geordnete Reihenfolge von Layern enthält. Diese Layer
 * werden in der Reihenfolge von oben nach unten in der Legende angeben und in der Kartenansicht gezeichnet. Um die Konfigurationsdaten und eventuell
 * dynamische Daten erhalten zu können, bekommt ein GenericNetDisplay-Objekt eine Datenverteilerverbindung übergeben. Eine Liste von Systemobjekten
 * beeinflußt den Kartenausschnitt, mit dem die Kartenansicht anfänglich gezeigt wird: ist die Liste leer, so wird die ganze Karte gezeichnet,
 * andernfalls wird aus den Koordinaten der übergebenen Systemobjekte ein diese Objekte umfassendes Rechteck berechnet und angezeigt.
 *
 * @author Kappich Systemberatung
 */
@SuppressWarnings("serial")
public final class GenericNetDisplay extends GndFrame implements View.ViewChangeListener, SelectionListener, Comparable<GenericNetDisplay> {

    static final String INTERNAL_START_VIEW_NAME = "Vordefinierte Ansicht 1";
    private static final String DIVIDER_LOCATION = "DividerLocation";
    private static final String RESOLUTION = "Resolution";
    private static final String START_VIEW_NAME = "StartViewName";
    private static final String DISPLAY_NAMES_FOR_SELECTION = "DisplayNamesForSelection";
    private static final Debug _debug = Debug.getLogger();
    /*
    Instanzenverwaltung: die nicht-standAlone-Instanzen müssen verwaltet werden.
     */
    private static final Set<GenericNetDisplay> _instances = Collections.newSetFromMap(new WeakHashMap<>());
    private static int _maxGndNumber;
    private final boolean _isDoubleBuffered;
    private final boolean _isAntiAliasingOn;
    private final boolean _isMapsTooltipOn;
    private final List<ResolutionListener> _resolutionListeners = new ArrayList<>();
    private final int _gndNumber;
    private final JMenu _csvMenu = new JMenu("CSV");
    private View _view;
    private ClientDavInterface _connection;
    private List<SystemObject> _systemObjects;
    private boolean _standAlone;
    private GNDPlugin _gndPlugin;
    private JSplitPane _splitPane;
    private LegendPane _legendPane;
    private MapPane _mapPane;
    private Icon _logo;
    private ViewManagerDialog _viewManagerDialog;
    private LayerManagerDialog _layerManagerDialog;
    private DOTManagerDialog _dotManagerDialog;
    private RhmDialog _rhmDialog;
    private CrmDialog _crmDialog;
    private CfmDialog _cfmDialog;
    private JFrame _resolutionFrame;
    private String _startViewName;
    private Double _screenResolution;
    private JList<Object> _noticeObjectList;
    private JTextArea _noticeTextArea;
    private NoticeManager _noticeManager;
    private DynamicListModel _selectedObjects;
    private boolean _systemEdit;
    private boolean _displayNamesForSelection;

    /**
     * Konstruiert ein Objekt aus den übergebenen Informationen. Der Stand-Alone-Wert gibt der Netzdarstellung die Information, ob sie sich als
     * eigenständige Anwendung betrachten kann oder nicht. Im Falle des Schließens des Fensters wird sie im Stand-Alone-Fall komplett beendet.
     *
     * @param view       eine Ansicht
     * @param connection eine Datenverteiler-Verbindung
     * @param standAlone {@code true} genau dann, wenn sie sich als eigenständige Anwendung betrachten soll
     */
    @SuppressWarnings("OverlyLongMethod")
    public GenericNetDisplay(final View view, final ClientDavInterface connection, final boolean standAlone, final GNDPlugin gndPlugin) {

        super("GenericNetDisplay");

        _standAlone = standAlone;
        _gndPlugin = gndPlugin;
        _connection = connection;
        _view = view;
        _systemObjects = new ArrayList<>();

        _noticeManager = new NoticeManager(getPreferenceStartPath(_connection).node("notices"), _connection.getDataModel());

//		long t0 = System.currentTimeMillis();
        getPreferences();

        final String property = System.getProperty("de.kappich.pat.gnd.ZentralMeridian");
        if (property == null) {
            GeoTransformation.setCentralMeridianFromOutside(9.0);
        } else {
            final double centralMeriadian = Double.parseDouble(property);
            if ((centralMeriadian < -9.) && (centralMeriadian > 27.)) {
                GeoTransformation.setCentralMeridianFromOutside(9.0);
            } else {
                GeoTransformation.setCentralMeridianFromOutside(centralMeriadian);
            }
        }

        if (_standAlone && (_startViewName != null) && ViewManager.getInstance().hasView(_startViewName)) {
            _view = ViewManager.getInstance().getView(_startViewName);
        }

        Container contentPane = getContentPane();
        contentPane.setLayout(new BorderLayout());

        if (_standAlone) {
            setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        } else {
            setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
        }
        _isDoubleBuffered = true;
        _isAntiAliasingOn = true;
        _isMapsTooltipOn = false;

        createMenu();

        try {
            _logo = new ImageIcon(HelpPage.class.getResource("dtv-logo.png"));
        } catch (Exception ignore) {
            _logo = null;
            _debug.info("Generische Netzdarstellung: das Kappich-Logo konnte nicht geladen werden.");
        }

        // _gndNumber is needed in setSplitPaneFromView().
        if (!standAlone) {
            ++_maxGndNumber;
            _gndNumber = _maxGndNumber;
        } else {
            _gndNumber = 0;
        }
        setSplitPaneFromView(_view);

//		long dt = System.currentTimeMillis() - t0;
//		System.out.println("dt = " + dt);
        if (!standAlone) {
            _instances.add(this);
        }

        SimpleReferenceManager.getInstance(_connection);
    }

    /**
     * Diese Methode gibt alle nicht-standAlone Instanzen zurück.
     *
     * @return alle nicht-standAlone Instanzen
     */
    public static Collection<GenericNetDisplay> getInstances() {
        return Collections.unmodifiableCollection(_instances);
    }

    /**
     * Gibt den Ausgangknoten zum Abspeichern der Benutzer-Präferenzen zurück.
     *
     * @param connection Datenverteilerverbindung zum ermitteln des zugehörigen Einstellungs-Knotens (KV-Abhängig)
     *
     * @return den Ausgangknoten zum Abspeichern der Benutzer-Präferenzen
     */
    private static Preferences getPreferenceStartPath(final ClientDavInterface connection) {
//		final String kvPid = GenericNetDisplay.getConnection().getLocalConfigurationAuthority().getPid();
//		return Preferences.userRoot().node("de/kappich/pat/gnd/").node(kvPid).node("GND");
        PreferencesHandler.setKvPid(connection.getLocalConfigurationAuthority().getPid());
        //if(instance == null) instance = new PreferencesHandler(_connection);
        return PreferencesHandler.getInstance().getPreferenceStartPath().node("GND");
    }

    /**
     * Holt die Bildschirmauflösung aus den Präferenzen, wenn sie dort hinterlegt ist, oder berechnet sie andernfalls.
     *
     * @param connection Datenverteilerverbindung zum ermitteln des zugehörigen Einstellungs-Knotens (KV-Abhängig)
     *
     * @return die Bildschirmauflösung
     */
    private static Double getScreenResolutionFromPreferences(final ClientDavInterface connection) {
        final Preferences gndPrefs = getPreferenceStartPath(connection);
        Preferences resolutionPrefs = gndPrefs.node("resolution");
        return resolutionPrefs.getDouble(RESOLUTION, (double) Toolkit.getDefaultToolkit().getScreenResolution());
    }

    /**
     * Holt den Namen der Startansicht aus den Präferenzen, wenn er dort hinterlegt ist.
     *
     * @param connection Datenverteilerverbindung zum ermitteln des zugehörigen Einstellungs-Knotens (KV-Abhängig)
     *
     * @return der Name der Startansicht
     */
    static String getStartViewNameFromPreferences(final ClientDavInterface connection) {
        final Preferences gndPrefs = getPreferenceStartPath(connection);
        Preferences startViewPrefs = gndPrefs.node("startViewName");
        return startViewPrefs.get(START_VIEW_NAME, null);
    }

    private static boolean getDisplayNamesForSelection(final ClientDavInterface connection) {
        final Preferences gndPrefs = getPreferenceStartPath(connection);
        Preferences displayNamesPrefs = gndPrefs.node("displayNamesForSelection");
        return displayNamesPrefs.getBoolean(DISPLAY_NAMES_FOR_SELECTION, true);
    }

    /**
     * Diese Methode macht die externen Plugins bekannt, indem die vollständigen Namen der Klassen, die DisplayObjectTypePlugin implementieren,
     * übergeben werden. Sie muss vor dem ersten Zugriff auf Teile dieser Plugins aufgerufen werden; der beste Moment dafür ist, bevor der erste
     * Konstruktor von GenericNetDisplay aufgerufen wird, denn sonst könnte schon die Initialisierung aus den Präferenzen scheitern. Die Arbeit wird
     * an den PluginManager delegiert. Durch das Anbieten dieser Methode muss der Benutzer (also z.B. GTM oder RunGND) der GND nur mit
     * GenericNetDisplay arbeiten.
     *
     * @param plugins die neuen externen Plugins
     */
    static void addPlugins(final List<String> plugins) {

        if (plugins == null || (plugins.isEmpty())) {
            return;
        }
        PluginManager.addPlugins(plugins);
    }

    private void setPositionAndSize() {
        if (hasPreferences()) {
            /* Hier wird sichergestellt, dass diese Instanz im Screen sichtbar wird. */
            final Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
            int x = Math.max(getPreferencesX(), 0);
            int y = Math.max(getPreferencesY(), 0);
            if (x > screenSize.width - 100) {  // zu weit rechts
                x = 50;
            }
            if (y > screenSize.height - 100) { // zu weit unten
                y = 50;
            }
            setLocation(x, y);
            setPreferredSize(new Dimension(getPreferencesWidth(), getPreferencesHeight()));
        } else {
            final Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
            if ((screenSize.getWidth() >= 1000) && (screenSize.getHeight() >= 780)) {
                setPreferredSize(new Dimension(800, 700));
                setLocation(50, 50);
            } else {
                setPreferredSize(new Dimension((int) screenSize.getWidth() - 50, (int) screenSize.getHeight() - 50));
                setLocation(0, 0);
            }
        }
    }

    /**
     * Gibt {@code true} zurück, wenn das GenericNetDisplay-Objekt sich als eigenständige Anwendung betrachtet.
     *
     * @return gibt {@code true} zurück, wenn das sich this als eigenständige Anwendung betrachtet
     */
    public boolean isStandAlone() {
        return _standAlone;
    }

    /**
     * Gibt die aktuelle Ansicht zurück.
     *
     * @return aktuelle Ansicht
     */
    public View getView() {
        return _view;
    }

    /**
     * Gibt das {@link MapPane} zurück.
     *
     * @return ein {@code MapPane}
     */
    public MapPane getMapPane() {
        return _mapPane;
    }

    /**
     * Diese Methode zeigt die übergebene Ansicht in der Kartendarstellung des GenericNetDisplay-Objekts, d.h. in seiner Legende und der
     * Kartenansicht.
     *
     * @param view die neue Ansicht
     */
    @SuppressWarnings("OverlyLongMethod")
    public void setSplitPaneFromView(View view) {

        if (_splitPane != null) {
            final Component[] components = _splitPane.getComponents();
            for (Component component : components) {
	            if (component instanceof MapPane oldMapPane) {
                    oldMapPane.clearEverything();
                }
            }
            remove(_splitPane);
        }
        _splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, true);
        _splitPane.setOneTouchExpandable(true);
        Object obj = getUserdefinedPreference(DIVIDER_LOCATION);
	    if (obj instanceof Integer value) {
            if (value > 50 && value < 1000) {
                _splitPane.setDividerLocation(value);
            }
        } else {
            _splitPane.setDividerLocation(250);
        }
        _splitPane.setBackground(Color.WHITE);
        _splitPane.addPropertyChangeListener(JSplitPane.DIVIDER_LOCATION_PROPERTY,
                                             pce -> putUserDefinedPreference(DIVIDER_LOCATION, _splitPane.getDividerLocation()));

        _view = view;
        _view.addChangeListener(this);
        JPanel scaleAndLegendPanel = new JPanel();
        scaleAndLegendPanel.setLayout(new BorderLayout());

        final class ScaleTextField extends JTextField implements MapScaleListener {

            private ScaleTextField() {
                setEditable(false);
            }

            @Override
            public void mapScaleChanged(double scale) {
                setText("1:" + (int) scale);
            }
        }

        if (_logo != null) {
            JLabel logoLabel = new JLabel(_logo);
            logoLabel.setBorder(BorderFactory.createEmptyBorder(1, 1, 1, 1));
            JPanel borderPanel = new JPanel(new BorderLayout());
            borderPanel.add(logoLabel, BorderLayout.CENTER);
            borderPanel.setBackground(Color.WHITE);
            scaleAndLegendPanel.add(borderPanel, BorderLayout.NORTH);
        }

        // Kartenansicht zurerst
        _mapPane = new MapPane(this, _view);
        _mapPane.addSelectionListener(this);
        _mapPane.setTooltip(_isMapsTooltipOn);
        _splitPane.setRightComponent(_mapPane);

        // Legende erst jetzt, weil sie den Maßstab benötigt
        ScaleTextField scaleTextField = new ScaleTextField();
        scaleTextField.setBorder(new TitledBorder("Maßstab"));
        scaleAndLegendPanel.add(scaleTextField, BorderLayout.SOUTH);
        _legendPane = new LegendPane(_connection, this, _view);
        scaleAndLegendPanel.add(getTabbedPane(), BorderLayout.CENTER);
        _splitPane.setLeftComponent(scaleAndLegendPanel);

        add(_splitPane);
        setPositionAndSize();
        pack();

        setVisible(false);    // Sonst kann man beim Wechseln der Ansicht ein repaint auslösen ...
        _mapPane.init();
        _legendPane.init(_mapPane.getMapScale());

        //noinspection VariableNotUsedInsideIf
        if (_logo == null) {
            if (_gndNumber > 0) {
                setTitle("Kappich Systemberatung - Generische Netzdarstellung " + _gndNumber + " - " + view.getName());
            } else {
                setTitle("Kappich Systemberatung - Generische Netzdarstellung - " + view.getName());
            }
        } else {
            if (_gndNumber > 0) {
                setTitle("Generische Netzdarstellung " + _gndNumber + " - " + view.getName());
            } else {
                setTitle("Generische Netzdarstellung - " + view.getName());
            }
        }
        _splitPane.revalidate();

        _legendPane.setMapScale(_mapPane.getMapScale());
        Collection<MapScaleListener> mapScaleListeners = new ArrayList<>();
        mapScaleListeners.add(scaleTextField);
        mapScaleListeners.add(_legendPane);
        _mapPane.addMapScaleListeners(mapScaleListeners);

        scaleTextField.setText("1:" + _mapPane.getMapScale().intValue());

        setVisible(true);

        ViewDialog.closeAll();
    }

    private JComponent getTabbedPane() {
        JTabbedPane tabbedPane = new JTabbedPane();
        // erstes Tab: Legende
        tabbedPane.addTab("Legende", new JScrollPane(_legendPane));

        // zweites Tab: Notices
        _noticeObjectList = new JList<>();
        _selectedObjects = new DynamicListModel();
        _noticeObjectList.setModel(_selectedObjects);

        /*
         *  Hack: Eigentlich sollte man den {@link ListSelectionListener} benutzen, aber der reagiert auch, wenn durch
         *  Zoomen o.ä. der dargestellte Inhalt geändert wird.
         *  Daher wird hier manuell bei Tastatur- oder Mauseingaben das ausgewählte Element geändert.
         */
        ObjectListListener l = new ObjectListListener();
        _noticeObjectList.addMouseListener(l);
        _noticeObjectList.addKeyListener(l);
        _noticeObjectList.addMouseListener(new MouseAdapter() {

            @Override
            public void mousePressed(final MouseEvent e) {
                final int selectedIndex = _noticeObjectList.getSelectedIndex();
                if (selectedIndex != -1) {
                    if (e.getClickCount() == 2) {
                        _mapPane.centerObject(_selectedObjects.getObject(selectedIndex), false);
                    }
                }
            }
        });

//		_noticeObjectList.addListSelectionListener(new ListSelectionListener() {
//			@Override
//			public void valueChanged(final ListSelectionEvent e) {
//				if(_noticeObjectListFlag) {
//					final int selectedIndex = _noticeObjectList.getSelectedIndex();
//					_mapPane.selectObject(_selectedObjects.getObject(selectedIndex), false);
//				}
//			}
//		});

        _noticeObjectList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        _noticeTextArea = new JTextArea();
        _noticeTextArea.getDocument().addDocumentListener(new DocumentListener() {
            @Override
            public void insertUpdate(final DocumentEvent e) {
                saveNotice();
            }

            @Override
            public void removeUpdate(final DocumentEvent e) {
                saveNotice();
            }

            @Override
            public void changedUpdate(final DocumentEvent e) {
                saveNotice();
            }

            private void saveNotice() {
                final String text = _noticeTextArea.getText();
                if (_noticeTextArea.isEditable() && !_systemEdit) {
                    for (final SystemObject object : _mapPane.getSelectedSystemObjects()) {
                        _noticeManager.setNotice(object, text);
                    }
                }
                SwingUtilities.invokeLater(() -> _mapPane.redraw());
            }
        });
        _noticeTextArea.setWrapStyleWord(true);
        _noticeTextArea.setLineWrap(true);
        selectionChanged();
        JPanel noticeObjectPanel = new JPanel();
        noticeObjectPanel.setLayout(new BorderLayout());
        JLabel noticeObjectLabel = new JLabel("Objekte mit Notizen:");
        noticeObjectPanel.add(noticeObjectLabel, BorderLayout.NORTH);
        noticeObjectPanel.add(new JScrollPane(_noticeObjectList), BorderLayout.CENTER);
        JSplitPane jSplitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT, noticeObjectPanel, new JScrollPane(_noticeTextArea));
        jSplitPane.setOneTouchExpandable(true);
        jSplitPane.setResizeWeight(0.5);
        tabbedPane.addTab("Notizen", new JScrollPane(jSplitPane));

        // drittes Tab: InfoFenster
        BrowserTree browserTree = new BrowserTree();
        tabbedPane.addTab("Infofenster", new JScrollPane(browserTree));
        _mapPane.addSelectionListener(browserTree);

        return tabbedPane;
    }

    /**
     * Gibt die Datenverteilerverbindung zurück.
     *
     * @return die Datenverteilerverbindung
     */
    public ClientDavInterface getConnection() {

        return _connection;
    }

    private void createMenu() {
        JMenuBar menuBar = new JMenuBar();
        addFileMenu(menuBar);
        addViewMenu(menuBar);
        addLayerMenu(menuBar);
        addDotMenu(menuBar);
        addEorMenu(menuBar);
        updateCsvMenu();
        menuBar.add(_csvMenu);
        addNoticesMenu(menuBar);
        addGotoMenu(menuBar);
        addExtraMenu(menuBar);
        addHelpMenu(menuBar);
        setJMenuBar(menuBar);

    }

    private void addFileMenu(final JMenuBar menuBar) {
        // Datei-Menü
        JMenu fileMenu = new JMenu("Datei");
        fileMenu.setMnemonic(KeyEvent.VK_D);
        fileMenu.getAccessibleContext().setAccessibleDescription("Das Datei-Menue");

        JMenuItem menuItem;

        menuItem = new JMenuItem("Speichern");
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        menuItem.getAccessibleContext().setAccessibleDescription("Speichert die Kartendarstellung.");
        fileMenu.add(menuItem);
        menuItem.addActionListener(e -> {
            try {
                saveGraphic();
            } catch (IOException ignore) {
                JOptionPane.showMessageDialog(GenericNetDisplay.this.getFrame(), "Fehler beim Speichern", "Fehlermeldung", JOptionPane.ERROR_MESSAGE);
            }
        });

        menuItem = new JMenuItem("Drucken");
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_P, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        menuItem.getAccessibleContext().setAccessibleDescription("Druckt die Kartendarstellung.");
        fileMenu.add(menuItem);
        menuItem.addActionListener(e -> printNetDisplay());

        fileMenu.addSeparator();

        if (_standAlone) {
            menuItem = new JMenuItem("Beenden");
            menuItem.getAccessibleContext().setAccessibleDescription("Beendet das Programm.");
        } else {
            menuItem = new JMenuItem("GND schließen");
            menuItem.getAccessibleContext().setAccessibleDescription("Schließt nur dieses GND-Fenster");
        }
        fileMenu.add(menuItem);
        menuItem.addActionListener(e -> {
            GndFrame.storePreferenceBounds(Frame.getFrames());
            if (_viewManagerDialog != null) {
                _viewManagerDialog.closeDialog();

            }
            if (_standAlone) {
                System.exit(0);
            } else {
                dispose();
            }
        });
        menuBar.add(fileMenu);
    }

    private void addViewMenu(final JMenuBar menuBar) {
        // Ansicht-Menue
        JMenu viewMenu = new JMenu("Ansichten");
        viewMenu.setMnemonic(KeyEvent.VK_A);
        viewMenu.getAccessibleContext().setAccessibleDescription("Das Ansicht-Menue");

        JMenuItem menuItem = new JMenuItem("Ansichtsverwaltung");
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_N, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        menuItem.getAccessibleContext().setAccessibleDescription("Startet den Ansichtsverwaltungs-Dialog.");
        viewMenu.add(menuItem);
        menuItem.addActionListener(e -> {
            if (_viewManagerDialog == null) {
                _viewManagerDialog = new ViewManagerDialog(GenericNetDisplay.this, false);
            }
            _viewManagerDialog.showDialog();
        });

        viewMenu.add(new JSeparator());

        menuItem = new JMenuItem("Aktuelle Ansicht");
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_K, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        menuItem.getAccessibleContext().setAccessibleDescription("Startet den Ansicht-Verwalten-Dialog.");
        viewMenu.add(menuItem);
        menuItem.addActionListener(e -> {
            final String title = "GND: aktuelle Ansicht bearbeiten";
            ViewDialog.runDialog(ViewManager.getInstance(), _view, true, ViewManager.getInstance().isChangeable(_view), false, title);
        });
        menuBar.add(viewMenu);
    }

    private void addLayerMenu(final JMenuBar menuBar) {
        // Layer-Menue
        JMenu layerMenu = new JMenu("Layer");
        layerMenu.setMnemonic(KeyEvent.VK_L);
        layerMenu.getAccessibleContext().setAccessibleDescription("Das Layer-Menue");

        JMenuItem menuItem = new JMenuItem("Layer-Verwaltung");
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_L, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        menuItem.getAccessibleContext().setAccessibleDescription("Startet den Layer-Verwalten-Dialog.");
        layerMenu.add(menuItem);
        menuItem.addActionListener(e -> {
            if (_layerManagerDialog == null) {
                _layerManagerDialog = new LayerManagerDialog(_connection);
            } else {
                _layerManagerDialog.showDialog(true);
            }
        });
        menuBar.add(layerMenu);
    }

    private void addDotMenu(final JMenuBar menuBar) {
        // DOT-Menue
        JMenu dotMenu = new JMenu("Darstellungstypen");
        dotMenu.setMnemonic(KeyEvent.VK_T);
        dotMenu.getAccessibleContext().setAccessibleDescription("Das Darstellungstypen-Menue");

        JMenuItem menuItem = new JMenuItem("Darstellungstypenverwaltung");
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_T, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        menuItem.getAccessibleContext().setAccessibleDescription("Startet den Darstellungstypen-Verwalten-Dialog.");
        dotMenu.add(menuItem);
        menuItem.addActionListener(e -> {
            if (_dotManagerDialog == null) {
                _dotManagerDialog = new DOTManagerDialog(DOTManager.getInstance(), _connection);
            } else {
                _dotManagerDialog.showDialog(true);
            }
        });
        menuBar.add(dotMenu);
    }

    private void addEorMenu(final JMenuBar menuBar) {
        // EOR-Menue
        JMenu eorMenu = new JMenu("EOR");
        eorMenu.setMnemonic(KeyEvent.VK_E);
        eorMenu.getAccessibleContext().setAccessibleDescription("Das Menü der Erweiterten Ortsreferenzen");

        JMenuItem menuItem = new JMenuItem("EOR-Verwaltung");
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_E, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        menuItem.getAccessibleContext().setAccessibleDescription("Startet den EOR-Verwalten-Dialog.");
        eorMenu.add(menuItem);
        menuItem.addActionListener(e -> {
            if (null == _crmDialog) {
                _crmDialog = new CrmDialog(_connection);
            } else {
                _crmDialog.showDialog(true);
            }
        });

        menuItem = new JMenuItem("EOR-Hierarchien-Verwaltung");
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_H, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        menuItem.getAccessibleContext().setAccessibleDescription("Startet den EOR-Hierarchien-Verwalten-Dialog.");
        eorMenu.add(menuItem);
        menuItem.addActionListener(e -> {
            if (null == _rhmDialog) {
                _rhmDialog = new RhmDialog(_connection);
            } else {
                _rhmDialog.showDialog(true);
            }
        });

        menuBar.add(eorMenu);
    }

    public void updateCsvMenu() {
        // CSV-Menü
        _csvMenu.removeAll();
        _csvMenu.setMnemonic(KeyEvent.VK_C);
        _csvMenu.getAccessibleContext().setAccessibleDescription("Das CSV-Menü");

        JMenuItem menuItem = new JMenuItem("CSV-Formate-Verwaltung");
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_V, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        menuItem.getAccessibleContext().setAccessibleDescription("Startet den CSV-Formate-Verwalten-Dialog.");
        _csvMenu.add(menuItem);
        menuItem.addActionListener(e -> {
            if (null == _cfmDialog) {
                _cfmDialog = new CfmDialog();
            } else {
                _cfmDialog.showDialog(true);
            }
        });

        JMenu initInfoMenu = new JMenu("Intialisierungs-Infos");
        boolean addInitInfoMenu = false;
        for (ViewEntry entry : _view.getViewEntriesWithoutNoticeEntries()) {
            Layer layer = entry.getLayer();
            if (layer.getCsvFile() != null) {
                addInitInfoMenu = true;
                JMenuItem initMenuItem = new JMenuItem(layer.getName());
                initInfoMenu.add(initMenuItem);
                initMenuItem.addActionListener(e -> {
                    JDialog infoDialog = new JDialog(this.getFrame());
                    infoDialog.setTitle("Initialisierungsinformationen für Layer " + layer.getName());
                    JPanel panel = new JPanel();
                    panel.setLayout(new BoxLayout(panel, BoxLayout.Y_AXIS));    // BoxLayout, damit das JscrollPane funktiniert
                    panel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
                    String infoText = layer.getCsvInitInfo();
                    JTextArea textArea = new JTextArea(30, 50);
                    textArea.setEditable(false);
                    if (infoText != null && !infoText.isEmpty()) {
                        textArea.setText(infoText);
                    } else {
                        textArea.setText("Es liegen keine Informationen vor");
                    }
                    panel.add(new JScrollPane(textArea));
                    infoDialog.add(panel);
                    infoDialog.pack();
                    infoDialog.setLocationRelativeTo(this.getFrame());
                    infoDialog.setVisible(true);
                });
            }
        }
        if (addInitInfoMenu) {
            _csvMenu.addSeparator();
            _csvMenu.add(initInfoMenu);
        }
    }

    @SuppressWarnings("OverlyLongMethod")
    private void addNoticesMenu(final JMenuBar menuBar) {
        // Notizen-Menü
        JMenu noticeMenu = new JMenu("Notizen");
        noticeMenu.setMnemonic(KeyEvent.VK_N);
        noticeMenu.getAccessibleContext().setAccessibleDescription("Das Notizen-Menue");

        JMenuItem menuItem = new JMenuItem("Notizen exportieren");
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_E, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        menuItem.getAccessibleContext().setAccessibleDescription("Exportiert alle Notizen.");
        menuItem.setEnabled(true);
        noticeMenu.add(menuItem);
        menuItem.addActionListener(e -> {
            final JFileChooser fileChooser;
            if (System.getProperty("os.name").toLowerCase().startsWith("mac")) {
                fileChooser = AwtFileChooser.createFileChooser();
            } else {
                fileChooser = new JFileChooser();
            }
            fileChooser.setFileFilter(new FileNameExtensionFilter("CSV-Datei", "csv"));
            fileChooser.setDialogTitle("GND: Notizen exportieren");
            fileChooser.setApproveButtonText("Exportieren");

            File csvFile;
            while (true) {
                int showSaveDialog = fileChooser.showSaveDialog(GenericNetDisplay.this.getFrame());
                if (!(showSaveDialog == JFileChooser.CANCEL_OPTION)) {
                    File selectedFile = fileChooser.getSelectedFile();
                    String path = selectedFile.getPath();

                    if (!path.toLowerCase().endsWith(".csv")) {
                        //noinspection StringConcatenationInLoop
                        path += ".csv";
                    }
                    csvFile = new File(path);

                    if (csvFile.exists()) {
                        int n = JOptionPane
                            .showConfirmDialog(fileChooser, "Die Datei '" + csvFile.getName() + "' existiert bereits.\nDatei überschreiben?",
                                               "Warnung", JOptionPane.YES_NO_OPTION);
                        if (n == JOptionPane.YES_OPTION) {
                            break;
                        }
                    } else {
                        break;
                    }
                } else {
                    return;
                }
            }

            try {
                _noticeManager.exportToFile(csvFile);
            } catch (IOException e1) {
                JOptionPane.showMessageDialog(GenericNetDisplay.this.getFrame(), "Fehler beim Exportieren der Notizen: " + e1.getMessage());
                _debug.warning("Kann Notizen nicht exportieren", e1);
            }
        });

        menuItem = new JMenuItem("Notizen löschen");
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_D, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        menuItem.getAccessibleContext().setAccessibleDescription("Löscht alle Notizen.");
        if (_standAlone) {
            menuItem.setEnabled(true);
        } else {
            menuItem.setEnabled(false);
        }
        noticeMenu.add(menuItem);
        menuItem.addActionListener(e -> {
            Object[] options = {"Löschen", "Abbrechen"};
            int n = JOptionPane.showOptionDialog(GenericNetDisplay.this.getFrame(), "Alle Notizen werden unwiderruflich gelöscht.", "Notizen löschen",
                                                 JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE, null, options, options[1]);
            if (n == 0) {
                _noticeManager.clear();
            }
        });
        menuBar.add(noticeMenu);
    }

    private void addGotoMenu(final JMenuBar menuBar) {
        // Gehe-zu-Menü
        JMenu gotoMenu = new JMenu("Gehe zu");
        gotoMenu.setMnemonic(KeyEvent.VK_G);
        gotoMenu.getAccessibleContext().setAccessibleDescription("Das Gehe-zu-Menue");

        //noinspection OverlyLongMethod
        gotoMenu.addMenuListener(new MenuListener() {
            @Override
            public void menuSelected(final MenuEvent e) {
                gotoMenu.removeAll();
                JMenuItem namePidIdSelectionItem = new JMenuItem("Namen, Pids oder IDs eingeben");
                namePidIdSelectionItem.getAccessibleContext().setAccessibleDescription("Objekte mit Namen, Pids oder IDs finden");
                namePidIdSelectionItem.setToolTipText("Objekte mit Namen, Pids oder IDs finden");
                namePidIdSelectionItem.addActionListener(e1 -> {
                    Collection<Collection<DisplayObject>> displayObjectCollections = new ArrayList<>();
                    Component[] components = _mapPane.getComponents();
                    for (Component c : components) {
	                    if (c instanceof MapPane.LayerPanel layerPanel) {
                            Layer layer = layerPanel.getLayer();
                            if (layer instanceof NoticeLayer) {
                                continue;
                            }
                            final ViewEntry viewEntry = _view.getViewEntry(layer);
                            if (null == viewEntry || !viewEntry.isVisible() || !viewEntry.isSelectable()) {
                                continue;
                            }
                            displayObjectCollections.add(layerPanel.getDisplayObjects());
                        }
                    }
                    ChooseByDialog chooseByDialog = new ChooseByDialog(GenericNetDisplay.this, displayObjectCollections);
                    if (!chooseByDialog.hasBoundsFromPrefs()) {
                        chooseByDialog.pack();
                        chooseByDialog.setLocationRelativeTo(GenericNetDisplay.this.getFrame());
                    }
                    chooseByDialog.setVisible(true);
                });
                gotoMenu.add(namePidIdSelectionItem);
                gotoMenu.addSeparator();
                Component[] components = _mapPane.getComponents();
                for (Component c : components) {
	                if (c instanceof MapPane.LayerPanel layerPanel) {
                        Layer layer = layerPanel.getLayer();
                        if (layer instanceof NoticeLayer) {
                            continue;
                        }
                        final ViewEntry viewEntry = _view.getViewEntry(layer);
                        if (null == viewEntry || !viewEntry.isVisible() || !viewEntry.isSelectable()) {
                            continue;
                        }
                        String layerDescription;
                        String confObjType = layer.getConfigurationObjectType();
                        if (null != confObjType && !confObjType.isEmpty()) {
                            layerDescription = layer.getName() + " (" + confObjType + ")";
                        } else {
                            layerDescription = layer.getName();
                        }
                        JMenuItem layerMenuItem = new JMenuItem(layerDescription);
                        String s = "Alle Objekte des Layers " + layer.getName() + " zur Auswahl anbieten";
                        layerMenuItem.getAccessibleContext().setAccessibleDescription(s);
                        layerMenuItem.setToolTipText(s);
                        gotoMenu.add(layerMenuItem);
                        layerMenuItem.addActionListener(e2 -> {
                            Collection<DisplayObject> displayObjectsWithCoor = new ArrayList<>();
                            for (DisplayObject displayObject : layerPanel.getDisplayObjects()) {
                                if (displayObject.getBoundingRectangle() != null) {
                                    displayObjectsWithCoor.add(displayObject);
                                }
                            }
                            ChooseFromDialog chooseFromDialog = new ChooseFromDialog(GenericNetDisplay.this, displayObjectsWithCoor);
                            if (!chooseFromDialog.hasBoundsFromPrefs()) {
                                chooseFromDialog.pack();
                                chooseFromDialog.setLocationRelativeTo(GenericNetDisplay.this.getFrame());
                            }
                            chooseFromDialog.setVisible(true);
                            chooseFromDialog.giveFocusAway();
                        });
                    }
                }
            }

            @Override
            public void menuDeselected(final MenuEvent e) {
            }

            @Override
            public void menuCanceled(final MenuEvent e) {
            }
        });
        menuBar.add(gotoMenu);
    }

    @SuppressWarnings("OverlyLongMethod")
    private void addExtraMenu(final JMenuBar menuBar) {
        // Extras-Menü
        JMenu extrasMenu = new JMenu("Extras");
        extrasMenu.setMnemonic(KeyEvent.VK_X);
        extrasMenu.getAccessibleContext().setAccessibleDescription("Das Extras-Menue");

        JMenuItem menuItem = new JMenuItem("Farbenverwaltung");
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        menuItem.getAccessibleContext().setAccessibleDescription("Öffnet den Farbdefinitionsdialog.");
        menuItem.setEnabled(true);
        extrasMenu.add(menuItem);
        menuItem.addActionListener(e -> {
            ColorDialog colorDialog = new ColorDialog();
            colorDialog.setVisible(true);
        });

        extrasMenu.add(new JSeparator());

        menuItem = new JMenuItem("Präferenzen exportieren");
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_X, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        menuItem.getAccessibleContext().setAccessibleDescription("Exportiert alle Benutzer-Einstellungen.");
        menuItem.setEnabled(true);
        extrasMenu.add(menuItem);
        menuItem.addActionListener(e -> {
            final JFileChooser fileChooser;
            if (System.getProperty("os.name").toLowerCase().startsWith("mac")) {
                fileChooser = AwtFileChooser.createFileChooser();
            } else {
                fileChooser = new JFileChooser();
            }
            fileChooser.setFileFilter(new FileNameExtensionFilter("XML-Datei", "xml"));
            fileChooser.setDialogTitle("GND: Präferenzen exportieren");
            fileChooser.setApproveButtonText("Exportieren");

            File xmlFile;
            while (true) {
                int showSaveDialog = fileChooser.showSaveDialog(GenericNetDisplay.this.getFrame());
                if (!(showSaveDialog == JFileChooser.CANCEL_OPTION)) {
                    File selectedFile = fileChooser.getSelectedFile();
                    String path = selectedFile.getPath();

                    if (!path.toLowerCase().endsWith(".xml")) {
                        //noinspection StringConcatenationInLoop
                        path += ".xml";
                    }
                    xmlFile = new File(path);

                    if (xmlFile.exists()) {
                        int n = JOptionPane
                            .showConfirmDialog(fileChooser, "Die Datei '" + xmlFile.getName() + "' existiert bereits.\nDatei überschreiben?",
                                               "Warnung", JOptionPane.YES_NO_OPTION);
                        if (n == JOptionPane.YES_OPTION) {
                            break;
                        }
                    } else {
                        break;
                    }
                } else {
                    return;
                }
            }
            Preferences gndPrefs = PreferencesHandler.getInstance().getPreferenceStartPath();
            try {
                gndPrefs.exportSubtree(new FileOutputStream(xmlFile));
            } catch (IOException | BackingStoreException e1) {
                JOptionPane.showMessageDialog(GenericNetDisplay.this.getFrame(), "Die Präferenzen konnten nicht exportiert werden. " + e1.toString(),
                                              "Fehlermeldung", JOptionPane.ERROR_MESSAGE);
            }
        });

        menuItem = new JMenuItem("Präferenzen importieren");
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_I, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        menuItem.getAccessibleContext().setAccessibleDescription("Importiert alle Benutzer-Einstellungen.");
        menuItem.setEnabled(true);
        extrasMenu.add(menuItem);
        menuItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                Object[] options = {"Weiter", "Abbrechen", "Hilfe"};
                int n = JOptionPane
                    .showOptionDialog(GenericNetDisplay.this.getFrame(), "Nach dem Import ist eine erneute Initialisierung oder ein Neustart nötig.",
                                      "Präferenzen importieren", JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE, null, options,
                                      options[1]);
                if (n == 2) {
                    GndHelp.openHelp("#theExtrasMenu");
                }
                if (n != 0) {
                    return;
                }
                n = JOptionPane.showOptionDialog(GenericNetDisplay.this.getFrame(),
                                                 "Ihre alten Präferrenzen werden komplett gelöscht. Noch können Sie sie sichern.",
                                                 "Präferenzen importieren", JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE, null,
                                                 options, options[1]);
                if (n != 0) {
                    return;
                }

                final JFileChooser fileChooser;
                if (System.getProperty("os.name").toLowerCase().startsWith("mac")) {
                    fileChooser = AwtFileChooser.createFileChooser();
                } else {
                    fileChooser = new JFileChooser();
                }
                fileChooser.setFileFilter(new FileNameExtensionFilter("XML-Datei", "xml"));
                fileChooser.setDialogTitle("GND: Präferenzen importieren");
                fileChooser.setApproveButtonText("Importieren");

                int returnVal = fileChooser.showOpenDialog(GenericNetDisplay.this.getFrame());

                if (returnVal == JFileChooser.APPROVE_OPTION) {
                    File file = fileChooser.getSelectedFile();
                    try {
                        clearPreferences();
                        Preferences.importPreferences(new FileInputStream(file));
                    } catch (FileNotFoundException e1) {

                        throw new UnsupportedOperationException("Catch-Block nicht implementiert - FileNotFoundException", e1);
                    } catch (IOException e1) {

                        throw new UnsupportedOperationException("Catch-Block nicht implementiert - IOException", e1);
                    } catch (InvalidPreferencesFormatException e1) {

                        throw new UnsupportedOperationException("Catch-Block nicht implementiert - InvalidPreferencesFormatException", e1);
                    }
                    getPreferences();
                    ColorManager.refreshInstance();
                    DOTManager.refreshInstance();
                    if (_dotManagerDialog != null) {
                        _dotManagerDialog.closeDialog();
                        _dotManagerDialog = new DOTManagerDialog(DOTManager.getInstance(), _connection);
                        _dotManagerDialog.showDialog(false);
                    }
                    LayerManager.refreshInstance();
                    if (_layerManagerDialog != null) {
                        _layerManagerDialog.closeDialog();
                        _layerManagerDialog = new LayerManagerDialog(_connection);
                        _layerManagerDialog.showDialog(false);
                    }
                    ViewManager.refreshInstance();
                    if (_viewManagerDialog != null) {
                        _viewManagerDialog.closeDialog();
                    }
                    _viewManagerDialog = new ViewManagerDialog(GenericNetDisplay.this, true);
                    _viewManagerDialog.showDialog();
                }
            }

            void clearPreferences() {

                Preferences gndPrefs = PreferencesHandler.getInstance().getPreferenceStartPath();
                try {
                    gndPrefs.clear();
                } catch (BackingStoreException e) {
                    JOptionPane.showMessageDialog(GenericNetDisplay.this.getFrame(),
                                                  "Beim Löschen der Präferenzen ist ein Fehler aufgetreten. " + e.toString(), "Fehlermeldung",
                                                  JOptionPane.ERROR_MESSAGE);
                    return;
                }
                clearChildren(gndPrefs);
            }

            private void clearChildren(Preferences prefs) {

                String[] childrenNames;
                try {
                    childrenNames = prefs.childrenNames();
                } catch (BackingStoreException e) {
                    JOptionPane.showMessageDialog(GenericNetDisplay.this.getFrame(),
                                                  "Beim Löschen der Präferenzen ist ein Fehler aufgetreten. " + e.toString(), "Fehlermeldung",
                                                  JOptionPane.ERROR_MESSAGE);
                    return;
                }
                for (String childName : childrenNames) {
                    Preferences childPrefs = prefs.node(childName);
                    try {
                        childPrefs.clear();
                    } catch (BackingStoreException e) {
                        JOptionPane.showMessageDialog(GenericNetDisplay.this.getFrame(),
                                                      "Beim Löschen der Präferenzen ist ein Fehler aufgetreten. " + e.toString(), "Fehlermeldung",
                                                      JOptionPane.ERROR_MESSAGE);
                        return;
                    }
                    clearChildren(childPrefs);
                    try {
                        childPrefs.removeNode();
                    } catch (BackingStoreException e) {
                        JOptionPane.showMessageDialog(GenericNetDisplay.this.getFrame(),
                                                      "Beim Löschen der Präferenzen ist ein Fehler aufgetreten. " + e.toString(), "Fehlermeldung",
                                                      JOptionPane.ERROR_MESSAGE);
                        return;
                    }
                }
            }
        });

        JMenu subMenu = new JMenu("Präferenzen löschen");
        menuItem = new JMenuItem("Alle Präferenzen löschen");
        menuItem.getAccessibleContext().setAccessibleDescription("Löscht alle Benutzer-Einstellungen.");

        subMenu.add(menuItem);
        menuItem.addActionListener(e -> {
            Object[] options = {"Löschen und beenden", "Nicht löschen", "Hilfe"};
            int n = JOptionPane.showOptionDialog(GenericNetDisplay.this.getFrame(),
                                                 "Sollen alle Benutzereinstellungen unwiderruflich gelöscht und das Programm beendet werden?",
                                                 "Präferenzen löschen", JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE, null, options,
                                                 options[1]);
            if (n == 0) {
                int m = JOptionPane
                    .showOptionDialog(GenericNetDisplay.this.getFrame(), "Wichtig: Bitte beenden Sie vor dem Löschen alle anderen GND-Instanzen!",
                                      "Präferenzen löschen", JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.WARNING_MESSAGE, null, options,
                                      options[1]);
                if (m == 0) {
                    Preferences gndPrefs = PreferencesHandler.getInstance().getPreferenceStartPath();
                    try {
                        gndPrefs.removeNode();
                        if (_standAlone) {
                            System.exit(0);
                        } else {
                            setVisible(false);
                            dispose();
                        }
                    } catch (BackingStoreException ex2) {
                        JOptionPane.showMessageDialog(GenericNetDisplay.this.getFrame(),
                                                      "Beim Löschen der Präferenzen ist ein Fehler aufgetreten. " + ex2.toString(), "Fehlermeldung",
                                                      JOptionPane.ERROR_MESSAGE);
                    }
                } else if (m == 2) {
                    GndHelp.openHelp("#theExtrasMenu");
                }
            } else if (n == 2) {
                GndHelp.openHelp("#theExtrasMenu");
            }
        });

        menuItem = new JMenuItem("Fensterpositionen und -größen löschen");
        menuItem.getAccessibleContext().setAccessibleDescription("Löscht nur die Fensterpositionen und -größen.");
        subMenu.add(menuItem);
        menuItem.addActionListener(e -> {
            Object[] options = {"Löschen und beenden", "Nicht löschen", "Hilfe"};
            int n = JOptionPane.showOptionDialog(GenericNetDisplay.this.getFrame(), "Sollen alle Fensterpositionen und -größen gelöscht werden?",
                                                 "Fensterpositionen und -größen löschen", JOptionPane.YES_NO_CANCEL_OPTION,
                                                 JOptionPane.QUESTION_MESSAGE, null, options, options[1]);
            if (n == 0) {
                GndDialog.removePreferenceBounds();
                GndFrame.removePreferenceBounds();
                System.exit(0);
            } else if (n == 2) {
                GndHelp.openHelp("#theExtrasMenu");
            }
        });
        extrasMenu.add(subMenu);

        extrasMenu.add(new JSeparator());

        menuItem = new JMenuItem("Ausgewählte Objekte an den GTM übergeben");
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_G, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        menuItem.getAccessibleContext().setAccessibleDescription("Übergibt dem GTM die ausgewählten Objekte.");
        if (_standAlone) {
            menuItem.setEnabled(false);
        } else {
            menuItem.setEnabled(true);
        }
        extrasMenu.add(menuItem);
        menuItem.addActionListener(e -> _gndPlugin.setSelectedObjects(_mapPane.getSelectedSystemObjects()));

        extrasMenu.add(new JSeparator());

        JCheckBoxMenuItem checkBoxMenuItem = new JCheckBoxMenuItem("Doppelpufferung");
        checkBoxMenuItem.getAccessibleContext().setAccessibleDescription("Schaltet die Doppelpufferung ein oder aus.");
        checkBoxMenuItem.setEnabled(true);
        checkBoxMenuItem.setSelected(_isDoubleBuffered);
        extrasMenu.add(checkBoxMenuItem);
        checkBoxMenuItem.addItemListener(e -> {

            if (e.getStateChange() == ItemEvent.SELECTED) {
                _mapPane.setDoubleBuffered(true);
            } else {
                _mapPane.setDoubleBuffered(false);
            }
        });

        checkBoxMenuItem = new JCheckBoxMenuItem("Antialiasing");
        checkBoxMenuItem.getAccessibleContext().setAccessibleDescription("Schaltet das Antialising ein oder aus.");
        checkBoxMenuItem.setEnabled(true);
        checkBoxMenuItem.setSelected(_isAntiAliasingOn);
        extrasMenu.add(checkBoxMenuItem);
        checkBoxMenuItem.addItemListener(e -> {

            if (e.getStateChange() == ItemEvent.SELECTED) {
                _mapPane.setAntialising(true);
                _mapPane.repaint();
            } else {
                _mapPane.setAntialising(false);
                _mapPane.repaint();
            }
        });

        extrasMenu.add(new JSeparator());

        checkBoxMenuItem = new JCheckBoxMenuItem("Namen/Pids für selektierte Objekte anzeigen");
        checkBoxMenuItem.getAccessibleContext().setAccessibleDescription("Schaltet die Anzeige der Namen/Pids für selektierte Objekte ein oder aus.");
        checkBoxMenuItem.setSelected(_displayNamesForSelection);
        extrasMenu.add(checkBoxMenuItem);
        checkBoxMenuItem.addItemListener(e -> {
            _displayNamesForSelection = e.getStateChange() == ItemEvent.SELECTED;
            if (null != _mapPane) {
                _mapPane.repaint();
            }
        });

        extrasMenu.add(new JSeparator());

        checkBoxMenuItem = new JCheckBoxMenuItem("Tooltipp auf der Karte");
        checkBoxMenuItem.getAccessibleContext().setAccessibleDescription("Schaltet den Tooltipp auf der Karte ein oder aus.");
        checkBoxMenuItem.setEnabled(true);
        checkBoxMenuItem.setSelected(_isMapsTooltipOn);
        extrasMenu.add(checkBoxMenuItem);
        checkBoxMenuItem.addItemListener(e -> {

            if (e.getStateChange() == ItemEvent.SELECTED) {
                _mapPane.setTooltip(true);
            } else {
                _mapPane.setTooltip(false);
            }
        });

        extrasMenu.add(new JSeparator());

        menuItem = new JMenuItem("Bildschirmauflösung einstellen");
        menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_R, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        menuItem.getAccessibleContext().setAccessibleDescription("Bildschirmauflösung in Dots-per-Inch");
        menuItem.setEnabled(true);
        extrasMenu.add(menuItem);
        menuItem.addActionListener(outerE -> {
            if (_resolutionFrame == null) {
                _resolutionFrame = new JFrame("Bildschirmauflösung einstellen");

                final JLabel textLabel = new JLabel("Geben sie den Wert für die Bildschirmauflösung in Dots per Inch (dpi) an: ");
                Double dpi = getScreenResolution();
                if (dpi == null) {
                    dpi = (double) Toolkit.getDefaultToolkit().getScreenResolution();
                }
                if (dpi.isNaN() || dpi.isInfinite()) {
                    dpi = 72.;
                }
                final Double currentResolution = dpi;    // Wir brauchen diese Variable als 'final' in der nächsten Zeile!
                final SpinnerNumberModel numberModel = new SpinnerNumberModel((double) currentResolution, 1., 1000., 0.1);
                final JSpinner resolutionSpinner = new JSpinner(numberModel);

                final JPanel resolutionPanel = new JPanel();
                resolutionPanel.setLayout(new BoxLayout(resolutionPanel, BoxLayout.X_AXIS));
                resolutionPanel.setBorder(BorderFactory.createEmptyBorder(20, 20, 20, 20));
                resolutionPanel.add(textLabel);
                resolutionPanel.add(Box.createRigidArea(new Dimension(10, 10)));
                resolutionPanel.add(resolutionSpinner);

                final JPanel acceptPanel = new JPanel();
                final JButton acceptButton = new JButton("Übernehmen");
                final JButton clearButton = new JButton("Zurücksetzen");
                acceptButton.setToolTipText("Wert zur Maßstabsberechnung übernehmen");
                clearButton.setToolTipText("Benutzer-Präferenz löschen und berechneten Wert anzeigen");
                acceptPanel.add(acceptButton);
                acceptPanel.add(Box.createRigidArea(new Dimension(10, 10)));
                acceptPanel.add(clearButton);

                final ActionListener acceptActionListener = innerE -> {
                    final Double newScreenResolution = (Double) resolutionSpinner.getValue();
                    if (newScreenResolution < 10.) {
                        JOptionPane.showMessageDialog(_resolutionFrame, "Bitte geben Sie einen Wert größer oder gleich 10 an.", "Fehlermeldung",
                                                      JOptionPane.ERROR_MESSAGE);
                        return;
                    }
                    if (newScreenResolution > 10000.) {
                        JOptionPane.showMessageDialog(_resolutionFrame, "Bitte geben Sie einen Wert kleiner oder gleich 10000 an.", "Fehlermeldung",
                                                      JOptionPane.ERROR_MESSAGE);
                        return;
                    }
                    final Double oldScreenResolution = getScreenResolution();
                    _screenResolution = newScreenResolution;
                    writeResolutionPreference(newScreenResolution);
                    notifyResolutionListenersResolutionChanged(newScreenResolution, oldScreenResolution);
                    _resolutionFrame.dispose();
                    _resolutionFrame = null;
                };
                acceptButton.addActionListener(acceptActionListener);

                final ActionListener clearActionListener = InnerE -> {

                    final Double olddpi = getScreenResolution();
                    clearResolutionPreference();
                    Double newdpi = (double) Toolkit.getDefaultToolkit().getScreenResolution();
                    if (newdpi.isNaN() || newdpi.isInfinite()) {
                        newdpi = 72.;
                    }
                    numberModel.setValue(newdpi);
                    _screenResolution = newdpi;
                    notifyResolutionListenersResolutionChanged(newdpi, olddpi);
                };
                clearButton.addActionListener(clearActionListener);

                final JPanel dummyPanel = new JPanel();
                dummyPanel.setLayout(new BoxLayout(dummyPanel, BoxLayout.Y_AXIS));
                dummyPanel.add(resolutionPanel);
                dummyPanel.add(acceptPanel);
                dummyPanel.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));

                _resolutionFrame.add(dummyPanel);
                _resolutionFrame.pack();
                _resolutionFrame.setLocationRelativeTo(GenericNetDisplay.this.getFrame());
                _resolutionFrame.setVisible(true);
            } else {
                _resolutionFrame.setState(Frame.NORMAL);
                _resolutionFrame.setVisible(true);
            }
        });
        menuBar.add(extrasMenu);
    }

    @SuppressWarnings("OverlyLongMethod")
    private void addHelpMenu(final JMenuBar menuBar) {
        JMenu helpMenu = HelpMenu.createHelpMenu(new GndHelp(), "documentation.html", GenericNetDisplay.this.getFrame());
        menuBar.add(helpMenu);
    }

    /* Methode zum Drucken der Kartenansicht. */
    private void printNetDisplay() {
		/*
		Es scheint so zu sein, als ob die hier gewählte Implementation keine Möglichkeit bietet,
		den Dialog, der sich beim Aufruf von PrinterJob#pageDialog öffnet, an eine bestimmte Stelle
		des Bildschirms zu schieben oder relativ zu einer Component zu plazieren.
		 */

        PrinterJob printerJob = PrinterJob.getPrinterJob();

        // Default Seitenformat festlegen
        PageFormat pageFormat = printerJob.pageDialog(printerJob.defaultPage());
        // Grafik mit eingestelltem Seitenformat drucken
        printerJob.setPrintable(_mapPane, pageFormat);

        PrintRequestAttributeSet printAttributes = new HashPrintRequestAttributeSet();
        printAttributes.add(new PrinterResolution(600, 600, ResolutionSyntax.DPI));

        // Drucken
        if (printerJob.printDialog()) {
            final class PrintThread extends Thread {

                private final PrinterJob _printerJob;

                private final PrintRequestAttributeSet _printAttributes;

                private PrintThread(final PrinterJob job, final PrintRequestAttributeSet attributes) {

                    _printerJob = job;
                    _printAttributes = attributes;
                }

                @Override
                public void run() {

                    try {
                        _printerJob.print(_printAttributes);
                    } catch (Exception ignore) {
                        JOptionPane.showMessageDialog(null, "Fehler beim Drucken", "Fehlermeldung", JOptionPane.ERROR_MESSAGE);
                    }
                }
            }
            PrintThread printThread = new PrintThread(printerJob, printAttributes);
            printThread.run();
        }
    }

    /**
     * Methode zum Abspeichern  der Grafik.
     *
     * @throws IOException wird geworfen, wenn etwas beim Speichern fehlschlägt
     */
    private void saveGraphic() throws IOException {

        final JFileChooser fileChooser;
        if (System.getProperty("os.name").toLowerCase().startsWith("mac")) {
            fileChooser = AwtFileChooser.createFileChooser();
        } else {
            fileChooser = new JFileChooser();
        }
        fileChooser.setFileFilter(new FileNameExtensionFilter("PNG Images", "png"));
        fileChooser.setDialogTitle("Netzdarstellung speichern");
        fileChooser.setVisible(true);

        // Dialog zum Speichern einer Datei
        fileChooser.showSaveDialog(GenericNetDisplay.this.getFrame());

        // Höhe und Breite mithilfe des Rechteckes aus der Grafik werden festgelegt
        Rectangle imageRectangle = _mapPane.getBounds();
        int width = imageRectangle.width;
        int height = imageRectangle.height;

        Dimension dimension = new Dimension(width, height);

        GraphicsConfiguration graphicsConfiguration =
            GraphicsEnvironment.getLocalGraphicsEnvironment().getDefaultScreenDevice().getDefaultConfiguration();
        BufferedImage image = graphicsConfiguration.createCompatibleImage((int) dimension.getWidth(), (int) dimension.getHeight());

        // Grafik wird erstellt
        // Grafik wird abgespeichert
        final Graphics2D g2d = image.createGraphics();
        g2d.setColor(Color.WHITE);
        Rectangle rect = new Rectangle(0, 0, width, height);
        g2d.fill(rect);
        _mapPane.paintComponent(g2d);
        g2d.drawImage(image, width, height, Color.WHITE, fileChooser);
        //Grafik wird im .png Format abgespeichert
        final File selectedFile = fileChooser.getSelectedFile();
        if (selectedFile == null) {
            return;
        }
        if (selectedFile.getName().endsWith("png")) {
            ImageIO.write(image, "png", selectedFile);
        } else {
            ImageIO.write(image, "png", new File(selectedFile + ".png"));
        }
    }

    /**
     * Gibt die Systemobjekte, die im Konstruktor angegeben oder später gesetzt wurden, zurück.
     *
     * @return die Systemobjekte, die im Konstruktor angegeben oder später gesetzt wurden
     */
    public List<SystemObject> getSystemObjects() {
        return _systemObjects;
    }

    /**
     * Mit dieser Methode kann man eine Menge von {@link SystemObject SystemObjekten} setzen. Dies hat zur Folge, dass der Kartenausschnitt auf diese
     * Menge fokussiert wird. Bei einer leeren Menge wird wieder die Gesamtkarte angezeigt.
     *
     * @param systemObjects Systemobjekte
     */
    public void setSystemObjects(final List<SystemObject> systemObjects) {
        _systemObjects = systemObjects;
        Collection<DisplayObject> displayObjects = _mapPane.setSelectedSystemObjects(systemObjects, false);
        if (null != displayObjects && !displayObjects.isEmpty()) {
            _mapPane.focusOnObjects(displayObjects);
        }
    }

    /**
     * Gibt {@code true} zurück, wenn die Kartenansicht mit Doppelpufferung ( double buffering) betrieben wird.
     *
     * @return {@code true} genau dann, wenn die Doppelpufferung aktiv ist
     */
    boolean isDoubleBuffered() {
        return _isDoubleBuffered;
    }

    /**
     * Gibt {@code true} zurück, wenn die Kartenansicht mit Anti-Aliasing betrieben wird.
     *
     * @return {@code true} genau dann, wenn die Anti-Aliasing aktiv ist
     */
    boolean isAntiAliasingOn() {
        return _isAntiAliasingOn;
    }

    /**
     * Gibt {@code true} zurück, wenn die Kartenansicht mit Tooltipp betrieben wird.
     *
     * @return {@code true} genau dann, wenn der Tooltipp der Kartenansicht aktiv ist
     */
    boolean isMapsTooltipOn() {
        return _isMapsTooltipOn;
    }

    /**
     * Gibt die Bildschirmauflösung zurück, mit deren Hilfe das GenericNetDisplay den Maßstab zu bestimmen versucht.
     *
     * @return die aktuelle Bildschirmauflösung
     */
    Double getScreenResolution() {
        return _screenResolution;
    }

    /**
     * Holt den Namen der Startansicht aus den Präferenzen, wenn er dort hinterlegt ist, und gibt andernfalls den vordefinierten Namen der
     * Startansicht zurück.
     *
     * @return der Name der Startansicht
     */
    public String getStartViewName() {
        String s = getStartViewNameFromPreferences(_connection);
        if (null == s) {
            return INTERNAL_START_VIEW_NAME;
        } else {
            return s;
        }
    }

    /*
     * Liest die Benutzer-Präferenzen zu Bildschirmauflösung und Start-Ansicht.
     */
    private void getPreferences() {
        _screenResolution = getScreenResolutionFromPreferences(_connection);
        _startViewName = getStartViewNameFromPreferences(_connection);
        _displayNamesForSelection = getDisplayNamesForSelection(_connection);
    }

    /*
     * Speichert die Benutzer-Präferenzen zur Bildschirmauflösung.
     */
    private void writeResolutionPreference(final Double screenResolution) {

        Preferences gndPrefs = getPreferenceStartPath(_connection);
        Preferences resolutionPrefs = gndPrefs.node("resolution");
        resolutionPrefs.putDouble(RESOLUTION, screenResolution);
    }

    /*
     * Löscht die Benutzer-Präferenzen zur Bildschirmauflösung.
     */
    private void clearResolutionPreference() {

        Preferences prefs = getPreferenceStartPath(_connection);
        Preferences resolutionPrefs = prefs.node("resolution");
        try {
            resolutionPrefs.removeNode();
        } catch (BackingStoreException ignored) {
            PreferencesDeleter pd = new PreferencesDeleter("Die Einstellungen zur Bildschirmauflösung können nicht geladen werden.", resolutionPrefs);
            pd.run();
        }
    }

    /**
     * Speichert den Namen der Startansicht in den Benutzer-Präferenzen.
     *
     * @param startViewName der Name der Startansicht
     */
    public void writeStartViewNamePreference(final String startViewName) {

        Preferences gndPrefs = getPreferenceStartPath(_connection);
        Preferences startViewPrefs = gndPrefs.node("startViewName");
        startViewPrefs.put(START_VIEW_NAME, startViewName);
    }

    void setVisibleObjects(final Set<SystemObject> displayObjects) {

        Collection<SystemObject> objectsWithNotice = _noticeManager.getObjectsWithNotice();
        List<SystemObject> visibleNoticeObjects = new ArrayList<>();
        for (SystemObject object : objectsWithNotice) {
            if (displayObjects.contains(object)) {
                visibleNoticeObjects.add(object);
            }
        }
        _selectedObjects.setElements(visibleNoticeObjects);

        // Bewirkt, dass das ausgewählte Objekt wieder in der Liste selektiert wird
        final Collection<SystemObject> selectedSystemObjects = _mapPane.getSelectedSystemObjects();
        _noticeObjectList.setSelectedIndices(_selectedObjects.getElementIndizes(selectedSystemObjects));
    }

    private void selectionChanged() {
        String text = "Ein Objekt auswählen um eine Notiz einzugeben";
        final Collection<SystemObject> selectedSystemObjects = _mapPane.getSelectedSystemObjects();
        final boolean validObjectSelected = selectedSystemObjects.size() == 1;
        if (validObjectSelected) {
            for (final SystemObject systemObject : selectedSystemObjects) {
                text = _noticeManager.getNotice(systemObject).getMessage();
            }
        }
        _noticeTextArea.setEditable(validObjectSelected);
        if (!validObjectSelected) {
            _noticeTextArea.setForeground(Color.gray);
            _noticeTextArea.setBackground(Color.white);
            _noticeTextArea.setFont(_noticeTextArea.getFont().deriveFont(Font.ITALIC));
        } else {
            _noticeTextArea.setForeground(Color.black);
            _noticeTextArea.setBackground(Color.white);
            _noticeTextArea.setFont(_noticeTextArea.getFont().deriveFont(Font.PLAIN));
        }
        _systemEdit = true;
        _noticeTextArea.setText(text);
        _systemEdit = false;
        _noticeObjectList.setSelectedIndices(_selectedObjects.getElementIndizes(selectedSystemObjects));
    }

    public NoticeManager getNoticeManager() {
        return _noticeManager;
    }

    @Override
    public void selectionChanged(final Collection<DisplayObject> objects) {
        Collection<SystemObject> systemObjects = new HashSet<>();
        for (DisplayObject displayObject : objects) {
            if (displayObject instanceof OnlineDisplayObject) {
                systemObjects.add(((OnlineDisplayObject) displayObject).getSystemObject());
            }
        }
        String text = "Ein Objekt auswählen um eine Notiz einzugeben";
        final boolean validObjectSelected = systemObjects.size() == 1;
        if (validObjectSelected) {
            for (final SystemObject systemObject : systemObjects) {
                text = _noticeManager.getNotice(systemObject).getMessage();
            }
        }
        _noticeTextArea.setEditable(validObjectSelected);
        if (!validObjectSelected) {
            _noticeTextArea.setForeground(Color.gray);
            _noticeTextArea.setBackground(Color.white);
            _noticeTextArea.setFont(_noticeTextArea.getFont().deriveFont(Font.ITALIC));
        } else {
            _noticeTextArea.setForeground(Color.black);
            _noticeTextArea.setBackground(Color.white);
            _noticeTextArea.setFont(_noticeTextArea.getFont().deriveFont(Font.PLAIN));
        }
        _systemEdit = true;
        _noticeTextArea.setText(text);
        _systemEdit = false;
        _noticeObjectList.setSelectedIndices(_selectedObjects.getElementIndizes(systemObjects));
    }

    // Diese Implementierung von compareTo funktioniert, weil ein Stand-Alone-GenericNetDisplay die
    // _gndNumber 0 hat, und die anderen gemäß ihrer Konstruktion durchgezählt werden.
    @Override
    public int compareTo(@NotNull final GenericNetDisplay other) {
        return Integer.compare(_gndNumber, other._gndNumber);
    }

    // equals und hashCode folgen derselben Logik.
    @Override
    public boolean equals(Object other) {
	    if (!(other instanceof GenericNetDisplay otherGnd)) {
            return false;
        }
        return _gndNumber == otherGnd._gndNumber;
    }

    @Override
    public int hashCode() {
        return Integer.hashCode(_gndNumber);
    }

    /*
    Implementation des Interfaces View.ViewChangeListener:
     */
    @Override
    public void viewEntryInserted(final View view, final int newIndex) {
        updateCsvMenu();
    }

    @Override
    public void viewEntryDefinitionChanged(final View view, final int i) {
    }

    @Override
    public void viewEntryPropertyChanged(final View view, final int i) {
    }

    @Override
    public void viewEntryRemoved(final View view, final int i) {
    }

    @Override
    public void viewEntriesSwitched(final View view, final int i, final int j) {
    }

    /**
     * Fügt das übergebene Objekt der Liste der auf Änderungen der Bildschirmauflösung angemeldeten Objekte hinzu.
     *
     * @param listener der hinzuzufügende Listener
     */
    void addResolutionListener(final ResolutionListener listener) {

        _resolutionListeners.add(listener);
    }

    /**
     * Entfernt das übergebene Objekt aus der Liste der auf Änderungen der Bildschirmauflösung angemeldeten Objekte und gibt {@code true} zurück, wenn
     * dies erfolgreich war, und {@code false} sonst.
     *
     * @param listener der zu entfernende Listener
     *
     * @return {@code true} genau dann, wenn der Listener entfernt wurde
     */
    @SuppressWarnings("unused")
    public boolean removeResolutionListener(final ResolutionListener listener) {

        return _resolutionListeners.remove(listener);
    }

    /**
     * Diese Methode gibt an, ob Namen bzw. Pids für selektierte Objekte angezeigt werden müssen. Dies ist eine vom Benutzer wählbare Einstellung.
     *
     * @return {@code true} genau dann, wenn dies nötig ist
     */
    boolean areNamesToBeDisplayedForSelection() {
        return _displayNamesForSelection;
    }

    /**
     * Benachrichtigt alle Objekte, die auf Änderungen der Bildschirmauflösung angemeldet sind.
     *
     * @param newResolution die neue Bildschirmauflösung
     * @param oldResulotion die alte Bildschirmauflösung
     */
    private void notifyResolutionListenersResolutionChanged(final Double newResolution, final Double oldResulotion) {

        for (ResolutionListener listener : _resolutionListeners) {
            listener.resolutionChanged(newResolution, oldResulotion);
        }
    }

    @Override
    public String toString() {
        if (_gndNumber > 0) {
            return "Generische Netzdarstellung " + _gndNumber;
        } else {
            return "Generische Netzdarstellung";
        }
    }

    /**
     * Eine Listener-Interface für Objekte, die sich auf Änderungen der Bildschirmauflösung anmelden wollen.
     *
     * @author Kappich Systemberatung
     */
    interface ResolutionListener {

        /**
         * Diese Methode wird aufgerufen, wenn die Auflösung geändert wird. Aufgrund der Übergabe des alten und neuen Wertes können auch relative
         * Änderungen vollzogen werden.
         *
         * @param newValue die neue Bildschirmauflösung
         * @param oldValue die alte Bildschirmauflösung
         */
        void resolutionChanged(final Double newValue, final Double oldValue);
    }

    private class ObjectListListener extends MouseAdapter implements KeyListener {

        @Override
        public void mousePressed(final MouseEvent e) {
            valueChanged();
        }

        public void valueChanged() {
            final int selectedIndex = _noticeObjectList.getSelectedIndex();
            _mapPane.selectObject(_selectedObjects.getObject(selectedIndex), false);
        }

        @Override
        public void keyTyped(final KeyEvent e) {
        }

        @Override
        public void keyPressed(final KeyEvent e) {
            final int selectedIndex = _noticeObjectList.getSelectedIndex();
            _mapPane.selectObject(_selectedObjects.getObject(selectedIndex), false);
        }

        @Override
        public void keyReleased(final KeyEvent e) {
            final int selectedIndex = _noticeObjectList.getSelectedIndex();
            _mapPane.selectObject(_selectedObjects.getObject(selectedIndex), false);
        }
    }
}
