/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.kappich.pat.gnd.gnd;

import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.csvPlugin.CsvDisplayObject;
import de.kappich.pat.gnd.displayObjectToolkit.DOTManager;
import de.kappich.pat.gnd.displayObjectToolkit.DisplayObject;
import de.kappich.pat.gnd.displayObjectToolkit.DisplayObjectManager;
import de.kappich.pat.gnd.displayObjectToolkit.OnlineDisplayObject;
import de.kappich.pat.gnd.extLocRef.ReferenceHierarchy;
import de.kappich.pat.gnd.extLocRef.ReferenceHierarchyManager;
import de.kappich.pat.gnd.layerManagement.Layer;
import de.kappich.pat.gnd.needlePlugin.DOTNeedlePlugin;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.selection.SelectionPanel;
import de.kappich.pat.gnd.utils.PointWithAngle;
import de.kappich.pat.gnd.viewManagement.NoticeLayer;
import de.kappich.pat.gnd.viewManagement.View;
import de.kappich.pat.gnd.viewManagement.ViewEntry;
import java.awt.BasicStroke;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.awt.geom.AffineTransform;
import java.awt.geom.Line2D;
import java.awt.geom.NoninvertibleTransformException;
import java.awt.geom.Path2D;
import java.awt.geom.PathIterator;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.print.PageFormat;
import java.awt.print.Printable;
import java.awt.print.PrinterException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JLayeredPane;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JProgressBar;
import javax.swing.RepaintManager;
import javax.swing.SwingUtilities;
import javax.swing.ToolTipManager;

/**
 * Die Kartenansicht der Kartendarstellung.
 * <p>
 * Ein MapPane steht für die Kartenansicht der GND. Um die einzelnen Layer darzustellen, ist MapPane von JLayeredPane abgeleitet. Jeder nicht-leere
 * Layer des JLayeredPane enthält genau eine Komponente der Klasse {@link MapPane.LayerPanel}, das die Objekte eines GND-Layers darstellt.
 *
 * @author Kappich Systemberatung
 */
@SuppressWarnings({"serial", "OverlyNestedMethod"})
public class MapPane extends JLayeredPane
    implements View.ViewChangeListener, Printable, GenericNetDisplay.ResolutionListener, DOTManager.DOTChangeListener,
               ReferenceHierarchyManager.RhmChangeListener {

    private static final boolean _isMac = System.getProperty("os.name").toLowerCase().startsWith("mac");
    private static final Double INCREASE_VALUE = 0.1;
    private final GenericNetDisplay _gnd;
    private final View _view;
    private final DisplayObjectManager _displayObjectManager;
    private final Set<DisplayObject> _selectedDisplayObjects = new HashSet<>();
    private final Set<DisplayObject> _tempSelectedDisplayObjects = new HashSet<>();
    private final Set<DisplayObject> _tempToggleDisplayObjects = new HashSet<>();
    private final List<MapScaleListener> _mapScaleListeners = new CopyOnWriteArrayList<>();
    private final List<SelectionListener> _selectionListeners = new ArrayList<>();
    /* Die folgenden zwei Parameter beschreiben die Situation zu Programmbeginn oder bei einer
    Änderung, die durch eine "Gehe-zu"-Aktion (aus dem Menü oder dem GTM) verursacht werden.
    Mehr Erläuterungen bei paintComponent. */
    private AffineTransform _mapTransform;
    private Double _mapScale = 0.;
    /* Die folgenden drei Parameter beschreiben die Transformation der Kartendarstellung relativ
    zur Ausgangssituation: anfänglich sind die Werte _zoomScale=1 und _zoomTranslateX = _zoomTranslateY = 0.
    Durch Mausoperationen (dragging, wheel) werden die Zoomstufe verändert und die Karte verschoben,
    was in diesen Parametern akkumuliert wird. Mehr Erläuterungen bei paintComponent. */
    private double _zoomTranslateX;
    private double _zoomTranslateY;
    private double _zoomScale;
    private boolean _antialising;
    private boolean _isTooltipOn;
    private boolean _showNothing;
    private SelectionPanel _selectionPanel;
    private Point _startPoint;

    /**
     * Konstruiert eine neue Kartenansicht für das übergebene GenericNetDisplay mit der übergebenen Ansicht. Das Objekt wird zunächst aber nur
     * konstruiert, die eigentliche Initialisierung muss mit {@link #init} noch ausgeführt werden.
     *
     * @param gnd  die Netzdarstellung
     * @param view die aktuelle Ansicht
     */

    public MapPane(GenericNetDisplay gnd, View view) {
        super();
        _gnd = gnd;
        _view = view;
        _displayObjectManager = new DisplayObjectManager(_gnd.getConnection(), this);
        setFocusable(true);
        setRequestFocusEnabled(true);
    }

    /* Das Rechteck wird in Höhe und Breite um den Faktor 2*INCREASE_VALUE vergrößert, aber mindestens auf die Breite
    minWidth und die Höhe minHeight. Das alte Recteck ist zentriert im neuen oder anders ausgedrückt: die Vergrößerung
    fügt links und rechts bzw. unten und oben dieselben Größen hinzu. */
    @SuppressWarnings("SameParameterValue")
    @Nullable
    private static Rectangle2D increaseRectangle(@Nullable Rectangle2D rectangle, double minWidth, double minHeight) {
        if (rectangle == null) {
            return null;
        }
        Rectangle2D returnRectangle;
        double newWidth = Math.max((1. + 2 * INCREASE_VALUE) * rectangle.getWidth(), minWidth);
        double newHeight = Math.max((1. + 2 * INCREASE_VALUE) * rectangle.getHeight(), minHeight);
        double newMinX = rectangle.getMinX() - (newWidth - rectangle.getWidth()) / 2;
        double newMinY = rectangle.getMinY() - (newHeight - rectangle.getHeight()) / 2;

        returnRectangle = new Rectangle2D.Double(newMinX, newMinY, newWidth, newHeight);
        return returnRectangle;
    }

    /**
     * Methode zur besseren Auflösung beim Drucken
     *
     * @param c eine Component
     */
    private static void disableDoubleBuffering(Component c) {
        RepaintManager currentManager = RepaintManager.currentManager(c);
        currentManager.setDoubleBufferingEnabled(false);
    }

    /**
     * Methode zum Zurücksetzen der Auflösung für die Ausgabe in der Oberfläche
     *
     * @param c eine Component
     */
    private static void enableDoubleBuffering(Component c) {
        RepaintManager currentManager = RepaintManager.currentManager(c);
        currentManager.setDoubleBufferingEnabled(true);
    }

    /**
     * Der Konstruktor dient der Klasses GenericNetDisplay dazu, das MapPane schon anordnen zu können. In der folgenden init-Methode und ihren
     * Initialisierungen wird JComponent.getBounds() aufgerufen, was erst sinnvoll ist, wenn das MapPane schon im GenericNetDisplay mit pack() gepackt
     * wurde.
     */
    public void init() {
        setMinimumSize(new Dimension(300, 300));
        initTheLayerPanels();
        _view.addChangeListener(this);
        addListeners();
        _zoomTranslateX = 0;
        _zoomTranslateY = 0;
        _zoomScale = 1.;

        setDoubleBuffered(_gnd.isDoubleBuffered());
        setAntialising(_gnd.isAntiAliasingOn());

        _showNothing = false;

        ToolTipManager.sharedInstance().setInitialDelay(200);
        ToolTipManager.sharedInstance().registerComponent(
            this);    // Registrieren ist notwendig, unregister bewirkt nichts, solange getTooltiptext einen nicht-leeren String zurückliefert.
        setTooltip(_gnd.isMapsTooltipOn());

        // Vor der Anmeldung sollte man mal den Maßstab berechnen, wozu auch eine Initialisierung der AT gehört.
        initAffineMapTransform();
        determineCurrentScale();

        new Thread(_displayObjectManager::subscribeDisplayObjects).start();
        _displayObjectManager.addMapScaleListeners();
        _gnd.addResolutionListener(this);
        DOTManager.getInstance().addDOTChangeListener(this);
        ReferenceHierarchyManager.getInstance().addRhmChangeListener(this);
    }

    @SuppressWarnings("unused")
    public Point2D getCenterPoint() {
        final Point2D.Double input = new Point2D.Double(getWidth() / 2, getHeight() / 2);
        final AffineTransform affineTransform = new AffineTransform();
        modifyAffineTransform(affineTransform);
        try {
            return affineTransform.createInverse().transform(input, new Point2D.Double());
        } catch (NoninvertibleTransformException ignored) {
            return new Point2D.Double();
        }
    }

    public GenericNetDisplay getGnd() {
        return _gnd;
    }

    public void redraw() {
        repaint();
        visibleObjectsChanged();
    }

    /**
     * Gibt das UTM-Rechteck, das dem aktuellen Ausschnitt entspricht zurück
     *
     * @return das UTM-Rechteck
     */
    public Rectangle getUTMBounds() {
        final Rectangle bounds = getBounds();
        Point p = new Point(0, 0); // Warum? Funktioniert, während (x,y) nicht funktioniert!
        Point utmPoint = new Point();
        getUTMPoint(p, utmPoint);
        Rectangle utmRectangle = new Rectangle(utmPoint);
        p = new Point((int) bounds.getWidth(), (int) bounds.getHeight()); // Warum? Funktioniert, (x+w,y+h) nicht funktioniert!
        getUTMPoint(p, utmPoint);
        utmRectangle.add(utmPoint);
        return utmRectangle;
    }

    private LayerPanel initALayerPanel(ViewEntry entry, int i, final JProgressBar progressBar) {
        LayerPanel layerPanel = new LayerPanel(entry.getLayer(), _displayObjectManager.getDisplayObjects(entry, progressBar));
        setLayer(layerPanel, i);    // setLayer before add according to documentation
        add(layerPanel);
        entry.setComponent(layerPanel);
        return layerPanel;
    }

    private void initTheLayerPanels() {
        final JDialog progressDialog = new JDialog();
        progressDialog.setTitle("Die GND wird initialisiert");
        progressDialog.setLayout(new BorderLayout());
        final JLabel textLabel = new JLabel("Layer-Initialisierung.");
        textLabel.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
        progressDialog.add(textLabel, BorderLayout.NORTH);
        final JLabel counterLabel = new JLabel();
        counterLabel.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
        progressDialog.add(counterLabel, BorderLayout.WEST);
        final JProgressBar progressBar = new JProgressBar();
        progressBar.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
        progressBar.setIndeterminate(true);
        progressDialog.add(progressBar, BorderLayout.CENTER);
        final JButton cancelButton = new JButton("Abbrechen");
        ActionListener cancelButtonListener = e -> {
            progressDialog.dispose();
            if (_gnd.isStandAlone()) {
                System.exit(0);
            } else {
                throw new GNDPlugin.StopInitializationException();
                // Das funktioniert so nicht: die Idee war wohl, dass der Thread beendet wird doch wird diese Exception auf dem
                // EventDispatchThread geworfen, während die GND nach wie vor in dem anderen Thread initialisiert wird.
                // Deshalb wurde der Cancal-Button nun in diesem Fall deaktiviert (s.u.).
            }
        };
        cancelButton.addActionListener(cancelButtonListener);
        cancelButton.setEnabled(_gnd.isStandAlone());
        final JPanel cancelPanel = new JPanel();
        cancelPanel.add(cancelButton);
        progressDialog.add(cancelPanel, BorderLayout.SOUTH);
        progressDialog.setPreferredSize(new Dimension(400, 150));
        progressDialog.pack();
        progressDialog.setLocation(_gnd.getLocation());
        progressDialog.setVisible(true);
        final List<ViewEntry> viewEntries = _view.getAllViewEntries();
        int index = viewEntries.size();
        for (int i = 0; i < viewEntries.size(); ++i) {
            ViewEntry entry = viewEntries.get(i);
            if (entry.getLayer() instanceof NoticeLayer) {
                textLabel.setText("Ein Notiz-Layer wird initialisiert.");
            } else {
                textLabel.setText("Der Layer '" + entry.getLayer().getName() + "' wird initialisiert.");
            }
            counterLabel.setText("Layer " + (i + 1) + " von " + viewEntries.size());
            progressDialog.pack();
            initALayerPanel(entry, --index, progressBar);
        }
        _selectionPanel = new SelectionPanel(this);
        setLayer(_selectionPanel, JLayeredPane.MODAL_LAYER - 1);    // setLayer before add according to documentation
        add(_selectionPanel);
        addComponentListener(_selectionPanel);
        progressDialog.dispose();
    }

    private void initAffineMapTransform() {
        final List<SystemObject> systemObjects = _gnd.getSystemObjects();

        // Selektion setzen:
        _selectedDisplayObjects.clear();
        _tempSelectedDisplayObjects.clear();
        _tempToggleDisplayObjects.clear();
        for (SystemObject systemObject : systemObjects) {
            _selectedDisplayObjects.addAll(getDisplayObjectsForSystemObject(systemObject, false));
        }
        // Rechteck setzen
        final Rectangle displayRectangle = _displayObjectManager.getDisplayRectangle(systemObjects);
        if (displayRectangle == null) {
            _showNothing = true;
        } else {
            _showNothing = false;
            setDisplayRectangle(displayRectangle);
        }
    }

    // Diese Methode setzt das Rechteck, das den Kartenausschnitt bestimmt. Diese Information wird
    // intern in der Variablen _mapTransform gespeichert, die die Ausgangslage bei Programmstart
    // oder nach eine "Gehe-zu"-Aktion darstellt. Da durch diese Setzung die benutzer-verursachten
    // Veränderungen des Kartenausschnitts ungültig werden, werden die entsprechenden Variablen
    // _zoom* auf neutrale gesetzt.
    private void setDisplayRectangle(@Nullable final Rectangle2D displayRectangle) {
        if (null == displayRectangle) {
            return;
        }
        final Rectangle bounds = getBounds();
        double scaleX = bounds.getWidth() / displayRectangle.getWidth();
        double scaleY = bounds.getHeight() / displayRectangle.getHeight();
        double scale = Math.min(scaleX, scaleY);
        _mapTransform = new AffineTransform();
        _mapTransform.scale(scale, scale);      // Skalierung macht eine Achse passend.
        _mapTransform.translate(-displayRectangle.getMinX(), -displayRectangle.getMinY());
        if (scaleX < scaleY) {
            // x kleiner => x-Achse korrekt. Verschiebe Ausschnitt entlang der y-Achse in die Mitte.
            _mapTransform.translate(0., -(1. - scaleY / scaleX) * displayRectangle.getHeight() / 2.);
        } else {
            // y kleiner => y-Achse korrekt. Verschiebe Ausschnitt entlang der x-Achse in die Mitte.
            _mapTransform.translate(-(1. - scaleX / scaleY) * displayRectangle.getWidth() / 2., 0.);
        }
        _zoomScale = 1.0;
        _zoomTranslateX = 0.0;
        _zoomTranslateY = 0.0;
    }

    /*
     * Gehört zur Implementation des View.ChangeListeners.
     */
    @Override
    public void viewEntriesSwitched(View view, int i, int j) {
        if (i == j) {
            return;
        }
//		int h = highestLayer(); // ALT
        final int h = _view.getNumberOfViewEntries() - 1;
//		System.out.println("h=" + h + ", i=" + i + ", j=" + j);
        Component[] iComponents = getComponentsInLayer(h - i);
        Component[] jComponents = getComponentsInLayer(h - j);
        for (Component component : iComponents) {
            setLayer(component, h - j);
        }
        for (Component component : jComponents) {
            setLayer(component, h - i);
        }
        visibleObjectsChanged();
        repaint();
    }

    private void visibleObjectsChanged() {
        _gnd.setVisibleObjects(getVisibleObjects());
    }

    private Set<SystemObject> getVisibleObjects() {

        final Set<SystemObject> result = new HashSet<>();
        final Rectangle filterRectangle = getUTMBounds();

        for (Component component : getComponents()) {
	        if (component instanceof LayerPanel layerPanel) {
                if (layerPanel.isVisible()) {
                    for (final DisplayObject displayObject : layerPanel.getDisplayObjects()) {
                        if (displayObject == null) {
                            System.err.println("displayObject ist null");
                            Thread.dumpStack();
                            continue;
                        }
                        if (displayObject instanceof OnlineDisplayObject) {
                            if (displayObject.getBoundingRectangle() != null && displayObject.getBoundingRectangle().intersects(filterRectangle)) {
                                result.add(((OnlineDisplayObject) displayObject).getSystemObject());
                            }
                        }
                    }
                }
            }
        }
        return result;
    }

    /**
     * Selektiert alle {@code DisplayObjects} zu dem übergebenen {@code SystemObject}. Macht nichts, wenn dieses {@code null} ist.
     *
     * @param systemObject ein SystemObject
     */
    void selectObject(@Nullable final SystemObject systemObject, @SuppressWarnings("SameParameterValue") final boolean withNeedleLayers) {
        if (null != systemObject) {
            final Set<DisplayObject> displayObjects = getDisplayObjectsForSystemObject(systemObject, withNeedleLayers);
            setSelection(displayObjects);
        }
    }

    /**
     * @param systemObjects    Systemobjekte
     * @param withNeedleLayers mit Notiz-Layer
     *
     * @return die Menge alle selektierten DisplayObjects
     */
    Collection<DisplayObject> setSelectedSystemObjects(final Collection<SystemObject> systemObjects,
                                                       @SuppressWarnings("SameParameterValue") final boolean withNeedleLayers) {
        Collection<DisplayObject> allDisplayObjects = new ArrayList<>();
        for (SystemObject systemObject : systemObjects) {
            final Set<DisplayObject> displayObjects = getDisplayObjectsForSystemObject(systemObject, withNeedleLayers);
            allDisplayObjects.addAll(displayObjects);
        }
        setSelectedObjects(allDisplayObjects);
        return allDisplayObjects;
    }

    /**
     * Selektiert alle übergebenen Objekte. Leert zuvor die Selektion, die temporäre Selektion und löscht ein eventuell vorhandenes
     * Selektions-Rechteck.
     *
     * @param displayObjects DisplayObjects
     */
    public void setSelectedObjects(final Collection<DisplayObject> displayObjects) {
        clearSelection();
        clearTempSelection();
        _selectionPanel.setRectangle(null, _zoomScale, _zoomTranslateX, _zoomTranslateY);
        _selectedDisplayObjects.addAll(displayObjects);
        repaint();
    }

    /**
     * Berechnet das die DisplayObjects umgebende Rechteck und wählt dieses (oder ein nach Höhe und Breite ähnliches) als neuen Kartenausschnitt.
     *
     * @param displayObjects eine Collection von DisplayObjects
     */
    public void focusOnObjects(final Collection<DisplayObject> displayObjects) {
        if (null == displayObjects || displayObjects.isEmpty()) {
            return;
        }
        Rectangle2D rect = null;
        for (DisplayObject displayObject : displayObjects) {
            if (rect == null) {
                rect = displayObject.getBoundingRectangle();
                if (rect != null) {
                    rect = (Rectangle2D) rect.clone();
                }
            } else {
                if (displayObject != null && displayObject.getBoundingRectangle() != null) {
                    rect.add(displayObject.getBoundingRectangle());
                }
            }
        }
        if (null == rect) {
            JOptionPane
                .showMessageDialog(_gnd.getFrame(), "Es konnte kein Kartenausschnitt bestimmt werden.", "Problem", JOptionPane.WARNING_MESSAGE);
            return;
        }
        rect = increaseRectangle(rect, 1000., 1000.);
        setDisplayRectangle(rect);
        determineCurrentScale();
        repaint();
        visibleObjectsChanged();
    }

    /**
     * Diese Methode zentriert die Karte auf das übergebene SystemObject.
     *
     * @param systemObject das SystemObject
     */
    void centerObject(@Nullable final SystemObject systemObject, @SuppressWarnings("SameParameterValue") final boolean withNeedleLayers) {
        if (null == systemObject) {
            return;
        }
        final Set<DisplayObject> displayObjects = getDisplayObjectsForSystemObject(systemObject, withNeedleLayers);
        centerOnObjects(displayObjects);
    }

    private void centerOnObjects(Collection<DisplayObject> displayObjects) {
        if (null == displayObjects) {
            return;
        }
        Rectangle2D rect = null;
        for (DisplayObject displayObject : displayObjects) {
            if (rect == null) {
                rect = displayObject.getBoundingRectangle();
                if (rect != null) {
                    rect = (Rectangle2D) rect.clone();
                }
            } else {
                rect.add(displayObject.getBoundingRectangle());
            }
        }
        centerOnRectangle(rect);
    }

    private void centerOnRectangle(@Nullable final Rectangle2D rectangle) {
        if (rectangle == null) {
            return;
        }
        Point center = new Point((int) rectangle.getCenterX(), (int) rectangle.getCenterY());
        Point utmPoint = new Point();
        AffineTransform affineTransform = new AffineTransform();
        modifyAffineTransform(affineTransform);
        affineTransform.transform(center, utmPoint);
        _zoomTranslateX -= utmPoint.getX() - getWidth() / 2;
        _zoomTranslateY -= utmPoint.getY() - getHeight() / 2;
        repaint();
        visibleObjectsChanged();
    }

    private Set<DisplayObject> getDisplayObjectsForSystemObject(final SystemObject systemObject, final boolean withNeedleLayers) {
        final Set<DisplayObject> displayObjects = new HashSet<>();
        DOTNeedlePlugin needlePlugin = new DOTNeedlePlugin();
        for (Component component : getComponents()) {
	        if (component instanceof LayerPanel layerPanel) {
                if (!withNeedleLayers && layerPanel.getLayer().getPlugin().getName().equals(needlePlugin.getName())) {
                    continue;
                }
                DisplayObject object = layerPanel.getDisplayObject(systemObject);
                if (object != null) {
                    displayObjects.add(object);
                }
            }
        }
        return displayObjects;
    }

    private void initALayer(ViewEntry viewEntry, final int index) {
        LayerPanel layerPanel = initALayerPanel(viewEntry, index, new JProgressBar());
        layerPanel.setVisible(viewEntry.isVisible(getMapScale().intValue()));
        _displayObjectManager.subscribeDisplayObjects();
        _displayObjectManager.addMapScaleListeners();
    }

    /*
     * Gehört zur Implementation des View.ChangeListeners.
     */
    @Override
    public void viewEntryInserted(View view, final int newIndex) {
        // Zuerst werden alle Components oberhalb des newIndex um einen Layer
        // im JLayeredPane höher geschoben (jeder Layer enthält genau ein LayerPanel):
        int max = view.getAllViewEntries().size() - 2;
        for (int i = max; i > max - newIndex; i--) {
            for (Component component : getComponentsInLayer(i)) {
                setLayer(component, i + 1);
            }
        }
        // Nun wird das neue LayerPanel in den freien Layer des JLayeredPane gesteckt:
        // Alte Implementation:
//		ViewEntry entry = view.getAllViewEntries().get(newIndex);
//		initALayerPanel(entry, max - newIndex + 1, new JProgressBar());
//		_displayObjectManager.subscribeDisplayObjects();
//		_displayObjectManager.addMapScaleListeners();
        initALayer(view.getAllViewEntries().get(newIndex), max - newIndex + 1);
        if (_showNothing) {
            initAffineMapTransform();
            determineCurrentScale();
        }
        visibleObjectsChanged();
        repaint();
    }

    /*
     * Gehört zur Implementation des View.ChangeListeners.
     */
    @Override
    public void viewEntryDefinitionChanged(View view, int i) {
        final int j = view.getAllViewEntries().size() - 1 - i;
        for (Component component : getComponentsInLayer(j)) {
//			component.setVisible(view.getAllViewEntries().get(i).isVisible(getMapScale().intValue()));    // ALTE IMPLEMENTATION
            // Gegenüber der alten Implementation (s.o.) muss spätestens seit Einführung der EORs der Layer
            // komplett neu initialisiert werden.
            // Altes wegräumen:
	        if (component instanceof LayerPanel layerPanel) {
                // Abmelden mit Optimierung
                Collection<DisplayObject> displayObjects = null;
                if (layerPanel.getLayer().getPlugin().isDynamicsPossible()) {
                    displayObjects = layerPanel.getDisplayObjects();
                    _displayObjectManager.unsubscribeDisplayObjects(displayObjects);
                }
                if (layerPanel.getLayer().getPlugin().isMapScaleListeningNecessary()) {
                    if (null == displayObjects) {
                        displayObjects = layerPanel.getDisplayObjects();
                    }
                    final Collection<MapScaleListener> mapScaleListeners = new ArrayList<>();
                    mapScaleListeners.addAll(displayObjects);
                    MapPane.this.removeMapScaleListeners(mapScaleListeners);
                }
            }
            remove(component);

            // Neues einfügen:
            // Alte Implementation
//			ViewEntry viewEntry = view.getAllViewEntries().get(i);
//			LayerPanel layerPanel = initALayerPanel(viewEntry, j, new JProgressBar());
//			layerPanel.setVisible(viewEntry.isVisible(getMapScale().intValue()));
//			_displayObjectManager.subscribeDisplayObjects();
//			_displayObjectManager.addMapScaleListeners();
            initALayer(view.getAllViewEntries().get(i), j);
        }
        visibleObjectsChanged();
        clearSelection();
        _selectionPanel.setRectangle(null, _zoomScale, _zoomTranslateX, _zoomTranslateY);
        repaint();
    }

    /*
     * Gehört zur Implementation des View.ChangeListeners.
     */
    @Override
    public void viewEntryPropertyChanged(View view, int i) {
        final int j = view.getAllViewEntries().size() - 1 - i;
        for (Component component : getComponentsInLayer(j)) {
            component.setVisible(view.getAllViewEntries().get(i).isVisible(getMapScale().intValue()));
        }
        visibleObjectsChanged();
        clearSelection();
        _selectionPanel.setRectangle(null, _zoomScale, _zoomTranslateX, _zoomTranslateY);
        repaint();
    }

    /*
     * Gehört zur Implementation des View.ChangeListeners.
     */
    @Override
    public void viewEntryRemoved(View view, int i) {
        final int j = view.getAllViewEntries().size() - i;
        for (Component component : getComponentsInLayer(j)) {
	        if (component instanceof LayerPanel layerPanel) {
                // Abmelden mit Optimierung
                Collection<DisplayObject> displayObjects = null;
                if (layerPanel.getLayer().getPlugin().isDynamicsPossible()) {
                    displayObjects = layerPanel.getDisplayObjects();
                    _displayObjectManager.unsubscribeDisplayObjects(displayObjects);
                }
                if (layerPanel.getLayer().getPlugin().isMapScaleListeningNecessary()) {
                    if (null == displayObjects) {
                        displayObjects = layerPanel.getDisplayObjects();
                    }
                    final Collection<MapScaleListener> mapScaleListeners = new ArrayList<>();
                    mapScaleListeners.addAll(displayObjects);
                    MapPane.this.removeMapScaleListeners(mapScaleListeners);
                }
            }
            remove(component);
        }
        for (int k = j + 1; k <= highestLayer(); k++) {
            for (Component component : getComponentsInLayer(k)) {
                setLayer(component, k - 1);
            }
        }
        visibleObjectsChanged();
        clearSelection();
        _selectionPanel.setRectangle(null, _zoomScale, _zoomTranslateX, _zoomTranslateY);
        repaint();
    }

    /*
     * print-Methode zum Drucken schreiben.
     */
    @Override
    public int print(Graphics g, PageFormat pageFormat, int pageIndex) throws PrinterException {

        if (pageIndex >= 1) {
            return NO_SUCH_PAGE;
        }

        Graphics2D g2d = (Graphics2D) g;
        // Seitenformat anpassen
        final Rectangle bounds = getBounds();
        double scaleWidth = pageFormat.getImageableWidth() / bounds.width;
        double scaleHeight = pageFormat.getImageableHeight() / bounds.height;
        double scale = Math.min(scaleWidth, scaleHeight);

        g2d.translate(pageFormat.getImageableX(), pageFormat.getImageableY());
        g2d.scale(scale, scale);
        disableDoubleBuffering(this);
        paint(g2d);
        enableDoubleBuffering(this);

        return PAGE_EXISTS;
    }

    /**
     * Diese Methode berechnet den Maßstab der Kartenansicht in Metern pro Pixel.
     */
    private double meterProPixel() {
        if (_showNothing) {
            return -1;
        }
        final Rectangle displayRectangle = _displayObjectManager.getDisplayRectangle(_gnd.getSystemObjects());
        if (null == displayRectangle) {
            return -1;
        }
        int minX = (int) displayRectangle.getMinX();
        int maxX = (int) displayRectangle.getMaxX();
        int minY = (int) displayRectangle.getMinY();
        int meterDistance = maxX - minX;

        AffineTransform affineTransform = (AffineTransform) _mapTransform.clone();
        affineTransform.scale(_zoomScale, _zoomScale);

        Point p1 = new Point(minX, minY);
        affineTransform.transform(p1, p1);
        Point p2 = new Point(maxX, minY);
        affineTransform.transform(p2, p2);
        final double pixelDistance = p2.getX() - p1.getX();

        return meterDistance / pixelDistance;
    }

    private void determineCurrentScale() {
        final double meterProPixel = meterProPixel();
        Double dpi = _gnd.getScreenResolution();
        if (dpi == null) {
            dpi = (double) Toolkit.getDefaultToolkit().getScreenResolution();
        }
        setMapScale((int) (meterProPixel * 100 * dpi / 2.54));
    }

    @Override
    protected void paintComponent(Graphics g) {
        // Dies ist die spezielle paintComponent-Methode für das MapPane. Sie sorgt im Wesentlichen dafür, dass
        // der richtige Ausschnitt gezeichnet wird. Dazu setzt sie zwei Arten von Informationen zusammen:
        // 1. die über die Lage der Komponenente auf dem Bildschirm und den Ausgangssituation bei Programmstart
        //      oder der letzten "Gehe-zu"-Aktion (beides aus Graphics2D.getTransform())
        // 2. die benutzer-verursachten Änderungen, die in den _zoom*-Parametern stecken (modifyAffineTransform)
        // Anschließend gibt läßt sei die wesentlichen Zeichenoperationen von LayerPanel.paintLayer ausführen.
        // Abschließend wird der Maßstabsfaktor unten links eingezeichnet.
        super.paintComponent(g);
        Graphics2D g2D = (Graphics2D) g;

        AffineTransform affineTransform = g2D.getTransform();
        if (_showNothing || (affineTransform == null)) {
            return;
        }
        modifyAffineTransform(affineTransform);
        AffineTransform oldTransform = g2D.getTransform();
        g2D.setTransform(affineTransform);

        // zur besseren Auflösung
        g2D.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        g2D.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BICUBIC);
        g2D.setRenderingHint(RenderingHints.KEY_ALPHA_INTERPOLATION, RenderingHints.VALUE_ALPHA_INTERPOLATION_QUALITY);
        g2D.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);

        // Rechteck vor dem Neuzeichnen leeren.
        g2D.setBackground(Color.WHITE);
        Rectangle bounds = getBounds();
//		System.out.println(bounds);
        g2D.clearRect(0, 0, (int) bounds.getWidth(), (int) bounds.getHeight()); // ALT. Aber warum?
        // Oder ist das folgende richtig? Siehe auch getUTMBounds!
//		g2D.clearRect((int) bounds.getX(), (int) bounds.getY(), (int) bounds.getWidth(), (int) bounds.getHeight());

        for (int i = 0; i <= highestLayer(); i++) {  // highestLayer gibt wohl den Wert für das SelectionPanel zurück ...
            Component[] components = getComponentsInLayer(i);
            if (0 == components.length) { // ... und wenn ein Layer leer ist, sind alle LayerPanels aktualisiert
                break;
            }
            for (Component component : components) {
                if (component.isVisible()) {
	                if (component instanceof LayerPanel la) {
                        la.paintLayer(g2D);
                    }
                }
            }
        }

        if (_gnd.areNamesToBeDisplayedForSelection()) {
            SelectionPainter selectionPainter =
                new SelectionPainter(g2D, getUTMBounds(), _selectedDisplayObjects, _tempSelectedDisplayObjects, _tempToggleDisplayObjects);
            selectionPainter.paintSelection();
        }

        g2D.setTransform(oldTransform);
        g2D.setColor(Color.black);
        g2D.setStroke(new BasicStroke(1));
        g2D.setFont(new Font("Default", Font.PLAIN, 10));

        drawScaling(g2D, 1 / meterProPixel());
    }

    private void drawScaling(final Graphics2D g, final double pixelPerMeter) {

        double lineLength = pixelPerMeter;
        if (lineLength <= 0) {
            return;
        }
        int factor = 10;
        while (lineLength < 20) {
            lineLength *= 10;
            factor *= 10;
        }
        if (lineLength > 100) {
            lineLength /= 5;
            factor /= 5;
        }
        if (lineLength > 40) {
            lineLength /= 2;
            factor /= 2;
        }

        if (factor >= 10000) {
            factor /= 10000;
            g.drawString(factor + " km", 10, getHeight() - 15);
        } else if (factor >= 10) {
            factor /= 10;
            g.drawString(factor + " m", 10, getHeight() - 15);
        } else {
            g.drawString(factor + "0 cm", 10, getHeight() - 15);
        }
        g.draw(new Line2D.Double(10, getHeight() - 10, 10 + lineLength, getHeight() - 10));
        g.draw(new Line2D.Double(10, getHeight() - 11, 10, getHeight() - 9));
        g.draw(new Line2D.Double(10 + lineLength, getHeight() - 11, 10 + lineLength, getHeight() - 9));
    }

    private void addToSelection(DisplayObject displayObject) {
        if (displayObject != null) {
            _selectedDisplayObjects.add(displayObject);
            redrawObject(displayObject);
        }
    }

    private void addToTempSelection(DisplayObject displayObject) {
        if (displayObject != null) {
            _tempSelectedDisplayObjects.add(displayObject);
            redrawObject(displayObject);
        }
    }

    private void addToTempToggle(DisplayObject displayObject) {
        if (displayObject != null) {
            _tempToggleDisplayObjects.add(displayObject);
            redrawObject(displayObject);
        }
    }

    private void clearSelection() {
        List<Rectangle> rectangles = new ArrayList<>();
        for (DisplayObject displayObject : _selectedDisplayObjects) {
            final Rectangle boundingRectangle = displayObject.getBoundingRectangle();
            if (boundingRectangle != null) {
                Rectangle transformed = getTransformedRectangle(boundingRectangle);
                if (transformed != null) {
                    rectangles.add(transformed);
                }
            }
        }
        _selectedDisplayObjects.clear();
        _selectionPanel.setRectangle(null, _zoomScale, _zoomTranslateX, _zoomTranslateY);
        for (Rectangle rectangle : rectangles) {
            repaint(rectangle);
        }
        notifySelectionChanged(Collections.unmodifiableCollection(_selectedDisplayObjects));
    }

    private void clearTempSelection() {
        clearSelection(_tempSelectedDisplayObjects);
    }

    private void clearTempToggle() {
        clearSelection(_tempToggleDisplayObjects);
    }

    private void clearSelection(final Set<DisplayObject> selectedDisplayObjects) {
        List<Rectangle> rectangles = new ArrayList<>();
        for (DisplayObject displayObject : selectedDisplayObjects) {
            final Rectangle boundingRectangle = displayObject.getBoundingRectangle();
            if (boundingRectangle != null) {
                Rectangle transformed = getTransformedRectangle(boundingRectangle);
                if (transformed != null) {
                    rectangles.add(transformed);
                }
            }
        }
        selectedDisplayObjects.clear();
        for (Rectangle rectangle : rectangles) {
            repaint(rectangle);
        }
    }

    private void approveTempSelection() {
        for (DisplayObject displayObject : _tempSelectedDisplayObjects) {
            addToSelection(displayObject);
        }
        notifySelectionChanged(Collections.unmodifiableCollection(_selectedDisplayObjects));
        _tempSelectedDisplayObjects.clear();
    }

    private void approveTempToggle() {
        for (DisplayObject displayObject : _tempToggleDisplayObjects) {
            toggleSelection(displayObject, false);
        }
        notifySelectionChanged(Collections.unmodifiableCollection(_selectedDisplayObjects));
        _tempToggleDisplayObjects.clear();
    }

    private void setSelection(Collection<DisplayObject> displayObjects) {
        if (displayObjects.equals(_selectedDisplayObjects)) {
            return;
        }
        clearSelection();
        for (DisplayObject displayObject : displayObjects) {
            addToSelection(displayObject);
        }
        notifySelectionChanged(Collections.unmodifiableCollection(_selectedDisplayObjects));
    }

    private void toggleSelection(DisplayObject displayObject, boolean notifySelectionChanged) {
        if (displayObject != null) {
            if (_selectedDisplayObjects.contains(displayObject)) {
                _selectedDisplayObjects.remove(displayObject);
            } else {
                _selectedDisplayObjects.add(displayObject);
            }
            if (notifySelectionChanged) {
                notifySelectionChanged(Collections.unmodifiableCollection(_selectedDisplayObjects));
            }
        }
    }

    @SuppressWarnings("OverlyLongMethod")
    private void doTempRectangleSelection(Rectangle2D rectangle, boolean toggle) {
        if (!toggle) {
            clearTempSelection();
        } else {
            clearTempToggle();
        }
        Rectangle2D utmRectangle = getUTMRectangle(rectangle);

        if (utmRectangle != null) {
            for (ViewEntry entry : _view.getAllViewEntries()) {
                if (entry.isVisible(getMapScale().intValue()) && entry.isSelectable()) {
                    Component component = entry.getComponent();
	                if (component instanceof LayerPanel layerPanel) {
                        for (DisplayObject displayObject : layerPanel.getDisplayObjects()) {
                            // Hier werden fünf Fälle unterschieden: PointWithArea und Point2D(.Double) für
                            // das Point- und NeedlePlugin, Path2D für das Line- und NeedlePlugin, Polygon für
                            // das AreaPlugin und als Default wird auf OnlineDisplayObject.getBoundingRectangle zurückgegriffen.
                            List<Object> coordinates = displayObject.getCoordinates();
                            if (coordinates == null) {
                                continue;
                            }
                            for (Object o : coordinates) {
	                            if (o instanceof PointWithAngle pwa) {
                                    Point2D p = pwa.getPoint();
                                    if (utmRectangle.contains(p)) {
                                        if (!toggle) {
                                            addToTempSelection(displayObject);
                                        } else {
                                            addToTempToggle(displayObject);
                                        }
                                    }
	                            } else if (o instanceof Point2D p) {
                                    if (utmRectangle.contains(p)) {
                                        if (!toggle) {
                                            addToTempSelection(displayObject);
                                        } else {
                                            addToTempToggle(displayObject);
                                        }
                                    }
	                            } else if (o instanceof Path2D path) {
                                    double[] coors = new double[6];
                                    double lastX = 0.d;
                                    double lastY = 0.d;
                                    for (PathIterator pit = path.getPathIterator(null); !pit.isDone(); pit.next()) {
                                        int type = pit.currentSegment(coors);
                                        if (type == PathIterator.SEG_LINETO) {
                                            if (utmRectangle.intersectsLine(lastX, lastY, coors[0], coors[1])) {
                                                if (!toggle) {
                                                    addToTempSelection(displayObject);
                                                } else {
                                                    addToTempToggle(displayObject);
                                                }
                                                break;
                                            }
                                        }
                                        lastX = coors[0];
                                        lastY = coors[1];
                                    }
	                            } else if (o instanceof Polygon pol) {
                                    if (pol.intersects(utmRectangle)) {
                                        if (!toggle) {
                                            addToTempSelection(displayObject);
                                        } else {
                                            addToTempToggle(displayObject);
                                        }
                                    }
                                } else {
                                    Rectangle boundingRectangle = displayObject.getBoundingRectangle();
                                    if (boundingRectangle != null && boundingRectangle.intersects(utmRectangle)) {
                                        if (!toggle) {
                                            addToTempSelection(displayObject);
                                        } else {
                                            addToTempToggle(displayObject);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    private void addListeners() {
        MyMouseListener myMouseListener = new MyMouseListener();
        addMouseListener(myMouseListener);
        addMouseMotionListener(myMouseListener);

        class ScaleHandler implements MouseWheelListener {
            @Override
            public void mouseWheelMoved(MouseWheelEvent e) {
                if (e.getScrollType() == MouseWheelEvent.WHEEL_UNIT_SCROLL) {
                    Double _mysticalFactor = 0.1 * _zoomScale;
//					if ( _mysticalFactor <= 0.03 * _zoomScale ) {
//						_mysticalFactor = 0.03 * _zoomScale;
//					}
                    int wheelRotation = e.getWheelRotation();
                    Double c = -_mysticalFactor * wheelRotation;
                    Double factor = _zoomScale / (_zoomScale + c);
                    if (factor > 2. || factor < 0. || factor.isInfinite() || factor.isNaN()) {    // Beschränkung des Herauszoomen auf das Doppelte
                        factor = 2.;
                        c = -_zoomScale / 2.;
                    }
                    if (factor < .5) {    // Beschränkung des Hineinzoomen auf die Hälfte
                        factor = .5;
                        c = _zoomScale;
                    }
                    final double nextMapScale = getMapScale() * factor;
                    // Weiteres rein- oder rauszoomen unterbinden!
                    // Die Grenzen sind etwas willkürlich gewählt. Tatsächlich wurden Probleme bei Werten
                    // um die 30 beobachtet (der Mechanismus geht dort kaputt!) ... wahrscheinlich, weil dann
                    // beim Integer-Runden immer wieder derselbe Wert kommt. 1 : 100 ist aber okay, weil dabei
                    // 1 cm auf dem Bildschirm genau 1 Meter in der Realität entspricht.
                    if ((nextMapScale < 200000000.) && (nextMapScale > 100.)) {
                        Point2D p = new Point2D.Double(e.getX(), e.getY());
                        AffineTransform at = new AffineTransform();
                        at.translate(_zoomTranslateX, _zoomTranslateY);
                        at.scale(_zoomScale, _zoomScale);
                        try {
                            at.inverseTransform(p, p);
                        } catch (NoninvertibleTransformException ignore) {
                            return;
                        }
                        _zoomScale += c;
                        _zoomTranslateX += -c * p.getX();
                        _zoomTranslateY += -c * p.getY();
                        setMapScale(nextMapScale);
                        _selectionPanel.transformRectangle(_zoomScale, _zoomTranslateX, _zoomTranslateY);
                    }
                }
            }
        }
        addMouseWheelListener(new ScaleHandler());
    }

    private void redrawObject(final DisplayObject displayObject) {
        final Rectangle boundingRectangle = displayObject.getBoundingRectangle();
        if (boundingRectangle != null) {
            final Rectangle transformedRectangle = getTransformedRectangle(boundingRectangle);
            if (transformedRectangle != null) {
                repaint(transformedRectangle);
            }
        }
    }

    /**
     * Gibt den aktuellen Maßstab zurück.
     *
     * @return der Maßstabsfaktor
     */
    public Double getMapScale() {
        return _mapScale;
    }

    /**
     * Aktualisiert den Maßstab der Kartenansicht, informiert alle MapScaleListeners und veranlaßt ein Neuzeichnen der Kartenansicht.
     *
     * @param mapScale der neue Maßstabsfaktor
     */
    private void setMapScale(double mapScale) {
        _mapScale = mapScale;
        for (MapScaleListener mapScaleListener : _mapScaleListeners) {
            mapScaleListener.mapScaleChanged(mapScale);
        }
//		final int h = highestLayer();
        final int h = _view.getNumberOfViewEntries();
        for (int i = 0; i < h; i++) {
            for (Component component : getComponentsInLayer(i)) {
                if (component instanceof LayerPanel) {
                    component.setVisible(_view.getAllViewEntries().get(h - 1 - i).isVisible(getMapScale().intValue()));
                }
            }
        }
        visibleObjectsChanged();
        repaint();
    }

    /**
     * Fügt die übergebenen Objekte der Menge der auf Änderungen des Maßstabs angemeldeten Objekte hinzu.
     *
     * @param listeners die neuen Listener
     */
    public void addMapScaleListeners(final Collection<MapScaleListener> listeners) {
        if (listeners != null) {
            _mapScaleListeners.addAll(listeners);
        }
    }

    /**
     * Entfernt die übergebenen Objekte aus der Menge der auf Änderungen des Maßstabs angemeldeten Objekte.
     *
     * @param listeners die zu löschenden Listener
     */
    private void removeMapScaleListeners(final Collection<MapScaleListener> listeners) {
        if (listeners == null) {
            return;
        }
        Runnable remover = () -> _mapScaleListeners.removeAll(listeners);
        Thread removerThread = new Thread(remover);
        removerThread.start();
    }

    private void removeAllMapScaleListeners() {
        _mapScaleListeners.clear();
    }

    private boolean getUTMPoint(Point p, Point utmP) {
        AffineTransform affineTransform = new AffineTransform();
        modifyAffineTransform(affineTransform);
        AffineTransform inverseT;
        try {
            inverseT = affineTransform.createInverse();
        } catch (NoninvertibleTransformException ignore) {
            return false;
        }
        inverseT.transform(p, utmP);
        return true;
    }

    private boolean getUTMPoint(Point2D p, Point2D utmP) {
        AffineTransform affineTransform = new AffineTransform();
        modifyAffineTransform(affineTransform);
        AffineTransform inverseT;
        try {
            inverseT = affineTransform.createInverse();
        } catch (NoninvertibleTransformException ignore) {
            return false;
        }
        inverseT.transform(p, utmP);
        return true;
    }

    @Nullable
    private Rectangle2D getUTMRectangle(final Rectangle2D rectangle) {
        Point2D p1 = new Point2D.Double(rectangle.getX(), rectangle.getY());
        Point2D utmP1 = new Point2D.Double();
        if (!getUTMPoint(p1, utmP1)) {
            return null;
        }
        Point2D p2 = new Point2D.Double(rectangle.getX() + rectangle.getWidth(), rectangle.getY() + rectangle.getHeight());
        Point2D utmP2 = new Point2D.Double();
        if (!getUTMPoint(p2, utmP2)) {
            return null;
        }
        return new Rectangle2D.Double(utmP1.getX(), utmP1.getY(), utmP2.getX() - utmP1.getX(), utmP2.getY() - utmP1.getY());
    }

    /**
     * Erzeugt den Tooltipp auf der Kartenansicht.
     *
     * @param e der Mouse-Event
     */
    @Override
    @Nullable
    public String getToolTipText(MouseEvent e) {
        if (!_isTooltipOn) {
            return null;
        }
        Point p = e.getPoint();
        Point utmPoint = new Point();
        if (!getUTMPoint(p, utmPoint)) {
            return "";
        }
        StringBuilder sb = new StringBuilder("<html>");
        boolean stringIsEmpty = true;
        final int maxNumberOfObjects = 24;
        int countNumberOfObjects = 0;
        final int maxNumberOfObjectsPerPanel = 5;
        for (ViewEntry entry : _view.getAllViewEntries()) {
            if (entry.isVisible(getMapScale().intValue()) && entry.isSelectable()) {
                Component component = entry.getComponent();
	            if (component instanceof LayerPanel layerPanel) {
                    int countTheObjectsOfThisPanel = 0;
                    final List<DisplayObject> displayObjectsCloseToPoint = layerPanel.getDisplayObjectsCloseToPoint(utmPoint, 2);
                    final List<NearestDisplayObject> nearestDisplayObjectsOfLayer =
                        layerPanel.getNearestDisplayObjects(utmPoint, displayObjectsCloseToPoint);
                    // Sortiere nach Distanz:
                    Collections.sort(nearestDisplayObjectsOfLayer);
                    // Einschränken auf nahegelegene Objekte:
                    List<DisplayObject> filteredDisplayObjects = new ArrayList<>();
                    double cut = Math.max(30. * meterProPixel(), 5.);
                    for (NearestDisplayObject nearestDisplayObject : nearestDisplayObjectsOfLayer) {
                        if (nearestDisplayObject.getDistance() < cut) {
                            filteredDisplayObjects.add(nearestDisplayObject.getDisplayObject());
                        } else {
                            break;
                        }
                    }
                    // Ausgabe erstellen:
                    for (DisplayObject displayObject : filteredDisplayObjects) {
	                    if (displayObject instanceof OnlineDisplayObject onlineDisplayObject) {
                            stringIsEmpty = false;
                            countTheObjectsOfThisPanel++;
                            if (countTheObjectsOfThisPanel > maxNumberOfObjectsPerPanel) {
                                sb.append("...").append("<br></br>");
                                break;
                            }
                            sb.append(entry.getLayer().getName()).append(": ").append(onlineDisplayObject.getSystemObject().getNameOrPidOrId())
                                .append("<br></br>");
                        }
                    }
                    countNumberOfObjects += countTheObjectsOfThisPanel;
                    if (countNumberOfObjects > maxNumberOfObjects) {
                        break;
                    }
                }
            }
        }
        sb.append("</html>");
        if (stringIsEmpty) {
            return null;
        }
        return sb.toString();
    }

    // Die Aufgabe dieser Methode besteht darin, die übergebene AffineTransform um die benutzer-verursachten
    // Veränderungen des Kartenausschnitts, die in den _zoom*-Variablen gespeichert sind, zu erweitern.
    // Die etwas überraschende Initialisierung von _mapTransform ist ein Artefakt, das mit den Notiz-Layern
    // gekommen ist: der DocumentListener ruft noch während der Initialisierung des MapPanes ein saveNotices
    // auf (w'lich weil der Text des Documents initialisiert wird). Das ist ziemlich unschön. Eine elementare
    // Beseitigung ist aber nicht trivial, da die entsprechende Initialisierung der Notices in einem eigenen
    // Thread durchgeführt wird. TODO
    private void modifyAffineTransform(AffineTransform affinTransform) {
        affinTransform.translate(_zoomTranslateX, _zoomTranslateY);
        affinTransform.scale(_zoomScale, _zoomScale);
        if (_mapTransform == null) {
            initAffineMapTransform();
            if (_showNothing) {
                return;
            }
            determineCurrentScale();
        }
        affinTransform.concatenate(_mapTransform);
    }

    /*
     * Diese Methode transformiert die 'bounding rectangles' von DisplayObjects, also
     * Rechtecke aus der UTM-Welt, in Rechtecke, die gezeicnet werden können, also
     * Rechtecke aus der Bildschirm-Welt.
     */
    @Nullable
    private Rectangle getTransformedRectangle(Rectangle rectangle) {
        if (rectangle == null) {
            return null;
        }
        Graphics2D g2D = (Graphics2D) getGraphics();
        if (g2D == null) {
            return null;
        }
        AffineTransform affineTransform = g2D.getTransform();
        modifyAffineTransform(affineTransform);
        Point p1 = new Point((int) rectangle.getMinX(), (int) rectangle.getMinY());
        affineTransform.transform(p1, p1);
        Rectangle transformedRect = new Rectangle(p1);
        Point p2 = new Point((int) rectangle.getMaxX(), (int) rectangle.getMaxY());
        affineTransform.transform(p2, p2);
        transformedRect.add(p2);
        return transformedRect;
    }

    /**
     * Gibt {@code true} zurück, wenn die Kartenansicht mit Anti-Aliasing gezeichnet wird.
     *
     * @return {@code true} genau dann, wenn die Kartenansicht mit Anti-Aliasing gezeichnet wird
     */
    private boolean isAntialising() {
        return _antialising;
    }

    /**
     * Setzt die interne Variable, die bestimmt, ob die Kartenansicht mit Anti-Aliasing gezeichnet wird.
     *
     * @param antialising die neue Einstellung von Anti-Aliasing
     */
    void setAntialising(boolean antialising) {
        _antialising = antialising;
    }

    /**
     * Gibt {@code true} zurück, falls der Tooltipp auf der Kartenansicht aktiviert ist.
     *
     * @return {@code true} genau dann, wenn der Tooltipp auf der Kartenansicht aktiviert ist
     */
    @SuppressWarnings("unused")
    public boolean isTooltipOn() {
        return _isTooltipOn;
    }

    /**
     * Schaltet den Tooltipp auf der Kartenansicht ab oder an.
     *
     * @param tooltip der neue Wert für die Tooltipp-Aktivität
     */
    public void setTooltip(boolean tooltip) {
        _isTooltipOn = tooltip;
    }

    /**
     * Veranlaßt eine Aktualisierung der Darstellung des übergebenen DisplayObjects.
     *
     * @param displayObject das OnlineDisplayObject
     */
    public void updateDisplayObject(OnlineDisplayObject displayObject) {
        if (displayObject != null) {
            final Rectangle boundingRectangle = displayObject.getBoundingRectangle();
            if (boundingRectangle != null) {
                final Rectangle rectangle = getTransformedRectangle(boundingRectangle);
                if (rectangle != null) {
                    repaint(rectangle);
                }
            }
        }
    }

    /*
     * Implementiert die Methode des Interfaces GenericNetDisplay.ResolutionListener.
     */
    @Override
    public void resolutionChanged(Double newValue, Double oldValue) {
        final Double mapScale = (_mapScale * newValue) / oldValue;
        setMapScale(mapScale.intValue());
    }

    /**
     * Gibt die Menge der aktuell in der Kartenansicht selektierten Objekte zurück.
     *
     * @return die Menge der aktuell in der Kartenansicht selektierten Objekte
     */
    public Collection<SystemObject> getSelectedSystemObjects() {
        final Set<SystemObject> systemObjects = new HashSet<>();
        for (DisplayObject displayObject : _selectedDisplayObjects) {
            if (displayObject instanceof OnlineDisplayObject) {
                systemObjects.add(((OnlineDisplayObject) displayObject).getSystemObject());
            }
        }
        return systemObjects;
    }

    /**
     * Gibt alles frei, so dass der Garbage-Collector zuschlagen kann.
     */
    void clearEverything() {
        // Abmelden beim Datenverteiler
        final Component[] components = getComponents();
        for (Component component : components) {
	        if (component instanceof LayerPanel layerPanel) {
                final Collection<DisplayObject> displayObjects = layerPanel.getDisplayObjects();
                _displayObjectManager.unsubscribeDisplayObjects(displayObjects);
            }
            remove(component);
        }
        // Freigeben selektierter Objekte
        clearSelection();
        clearTempSelection();
        // ViewEntries den Rückwärtsverweis auf die Component nehmen
        for (ViewEntry entry : _view.getAllViewEntries()) {
            entry.setComponent(null);
        }
        // Entferne die MapScaleListener
        removeAllMapScaleListeners();
        _mapScaleListeners.clear();
    }

    /*
     * Implementiert die Methode des Interfaces DOTManager.DOTChangeListener
     */
    @Override
    public void displayObjectTypeAdded(DisplayObjectType displayObjectType) {
        visibleObjectsChanged();
        repaint();
    }

    /*
     * Implementiert die Methode des Interfaces DOTManager.DOTChangeListener
     */
    @Override
    public void displayObjectTypeChanged(DisplayObjectType displayObjectType) {
        visibleObjectsChanged();
        repaint();
    }

    /*
     * Implementiert die Methode des Interfaces DOTManager.DOTChangeListener
     */
    @Override
    public void displayObjectTypeRemoved(String displayObjectTypeName) {
        visibleObjectsChanged();
        repaint();
    }

    /**
     * Fügt das übergebene Objekt der Liste der auf Änderungen der Selektion angemeldeten Objekte hinzu.
     *
     * @param listener der neue Listener
     */
    public void addSelectionListener(final SelectionListener listener) {
        _selectionListeners.add(listener);
    }

    /**
     * Entfernt das übergebene Objekt aus der Liste der auf Änderungen der Selektion angemeldeten Objekte.
     *
     * @param listener der zu entfernende Listener
     */
    @SuppressWarnings("unused")
    public boolean removeSelectionListener(final SelectionListener listener) {

        return _selectionListeners.remove(listener);
    }

    /**
     * Benachrichtigt alle Objekte, die auf Änderungen der Selektion angemeldet sind.
     *
     * @param objects die neue Selektion
     */
    private void notifySelectionChanged(final Collection<DisplayObject> objects) {
        for (SelectionListener listener : _selectionListeners) {
            listener.selectionChanged(objects);
        }
    }

    // Implementation des ReferenceHierarchyManager.RhmChangeListener
    @Override
    public void referenceHierarchyAdded(final ReferenceHierarchy referenceHierarchy) {
        repaint();
    }

    @Override
    public void referenceHierarchyChanged(final ReferenceHierarchy referenceHierarchy) {
        repaint();
    }

    @Override
    public void referenceHierarchyRemoved(final String name) {
        repaint();
    }

    @SuppressWarnings({"UseOfSystemOutOrSystemErr", "unused"})
    private void printLayers() {
        System.out.println();
        System.out.println("Print Layers");
        for (int i = 0; i < highestLayer(); ++i) {
            Component[] components = getComponentsInLayer(i);
            for (Component component : components) {
	            if (component instanceof LayerPanel layerPanel) {
                    System.out.println(i + ": " + layerPanel.getLayer().getName());
                }
            }
        }
    }

    @Override
    public String toString() {
        return "MapPane{}";
    }

    private enum SelectionMode { SingleSelection, SingleToggle, RectangleSelection, RectangleToggle, NoMode }

    /**
     * Ein Interface für Listener, die an Maßstabs-Änderungen der Kartenansicht interessiert sind.
     *
     * @author Kappich Systemberatung
     */
    public interface MapScaleListener {
        /**
         * Diese Methode wird für die Listener aufgerufen, wenn eine Maßstabsänderung mitgeteilt werden muss.
         *
         * @param scale der neue Maßstab
         */
        void mapScaleChanged(double scale);
    }

    static class NearestDisplayObject implements Comparable<NearestDisplayObject> {
        private final DisplayObject _displayObject;
        private final Double _distance;
        private final Point _point;
        private final Layer _layer;

        NearestDisplayObject(final DisplayObject displayObject, Double distance, Point p, Layer layer) {
            _displayObject = displayObject;
            _distance = distance;
            _point = p;
            _layer = layer;
        }

        public DisplayObject getDisplayObject() {
            return _displayObject;
        }

        public Double getDistance() {
            return _distance;
        }

        public Point getPoint() {
            return _point;
        }

        public Layer getLayer() {
            return _layer;
        }

        /**
         * Der Abstand ist das primäre Sortierkriterium, das DisplayObject das sekundäre. Point und Layer werden als abhängige Daten betrachtet, und
         * nicht verglichen.
         *
         * @param o das andere Objekt
         *
         * @return eine Ganzzahl
         */
        @Override
        public int compareTo(@NotNull final NearestDisplayObject o) {
	        return _distance.compareTo(o._distance);
        }

        @Override
        public boolean equals(Object o) {
	        if (!(o instanceof NearestDisplayObject other)) {
                return false;
            }
            return _distance.equals(other._distance) && _displayObject.equals(other._displayObject);
        }

        @Override
        public int hashCode() {
            int result = _displayObject.hashCode();
            result = 31 * result + _distance.hashCode();
            return result;
        }

        @Override
        public String toString() {
            return "   Object: " + _displayObject.getName() + System.lineSeparator() + "Distanz: " + _distance + System.lineSeparator() + "Layer: " +
                   _layer.getName();
        }
    }

    class LayerPanel extends JPanel {

        /* Es gibt Layer mit DisplayObjects, die jeweils ein SystemObject haben, und
            solche, wo dies nicht der Fall ist.	*/
        private final Layer _layer;
        private final Map<SystemObject, DisplayObject> _displayObjectMap;
        private final List<DisplayObject> _displayObjectList;

        LayerPanel(Layer layer, List<DisplayObject> displayObjects) {
            _layer = layer;
            _displayObjectMap = new HashMap<>();
            _displayObjectList = new ArrayList<>();
            for (DisplayObject displayObject : displayObjects) {
                if (displayObject instanceof OnlineDisplayObject) {
                    _displayObjectMap.put(((OnlineDisplayObject) displayObject).getSystemObject(), displayObject);
                } else {
                    _displayObjectList.add(displayObject);
                }
            }
            if (!_displayObjectMap.isEmpty() && !_displayObjectList.isEmpty()) {
                throw new RuntimeException("Conceptual error in LayerPanel");
            }
        }

        @Override
        public String toString() {
            return "LayerPanel{layer='" + _layer.getName() + "'}";
        }

        /**
         * Gibt des Layer des LayerPanels zurück.
         *
         * @return der Layer
         */
        public Layer getLayer() {
            return _layer;
        }

        /**
         * Gibt die DisplayObjects des LayerPanels zurück.
         *
         * @return die DisplayObjects
         */
        Collection<DisplayObject> getDisplayObjects() {
            if (!_displayObjectMap.isEmpty()) {
                return _displayObjectMap.values();
            } else if (!_displayObjectList.isEmpty()) {
                return _displayObjectList;
            } else {
                return new ArrayList<>();
            }
        }

        private DisplayObject getDisplayObject(SystemObject systemObject) {
            return _displayObjectMap.get(systemObject);
        }

        private boolean intersect(Rectangle rectangle, List<Object> coordinates) {
            for (Object object : coordinates) {
	            if (object instanceof Path2D.Double polyline) {
                    final PathIterator pathIterator = polyline.getPathIterator(null);
                    double[] coors = new double[6];
                    double oldX = Double.MAX_VALUE;
                    double oldY = Double.MAX_VALUE;
                    while (!pathIterator.isDone()) {
                        int type = pathIterator.currentSegment(coors);
                        switch (type) {
                            case PathIterator.SEG_MOVETO:
                                oldX = coors[0];
                                oldY = coors[1];
                                break;
                            case PathIterator.SEG_LINETO:
                                if (oldX != Double.MAX_VALUE) {
                                    Line2D.Double line = new Line2D.Double(oldX, oldY, coors[0], coors[1]);
                                    if (line.intersects(rectangle)) {
                                        return true;
                                    }
                                }
                                oldX = coors[0];
                                oldY = coors[1];
                                break;
                            default:
                                break;
                        }
                        pathIterator.next();
                    }
                }
            }
            return false;
        }

        /**
         * Diese Methode erzeugt eine Liste nächstgelegener DisplayObjects in diesem LayerPanel mit Abständen.
         *
         * @param p der Punkt
         *
         * @return nächstgelegenen DisplayObjects und ihre Abstände
         */
        List<NearestDisplayObject> getNearestDisplayObjects(Point p, List<DisplayObject> displayObjects) {
            List<NearestDisplayObject> nearDisplayObjects = new ArrayList<>();
            Double minDist = Double.MAX_VALUE;
            for (DisplayObject displayObject : displayObjects) {
                Double dist = distance(p, displayObject);
                if (dist < minDist) {
                    List<NearestDisplayObject> oldList = nearDisplayObjects;
                    nearDisplayObjects = new ArrayList<>();
                    nearDisplayObjects.add(new NearestDisplayObject(displayObject, dist, p, _layer));
                    for (NearestDisplayObject nearestDisplayObject : oldList) {
                        if (nearestDisplayObject.getDistance() < 1.5 * dist || nearestDisplayObject.getDistance() < 2.) {
                            nearDisplayObjects.add(nearestDisplayObject);
                        }
                    }
                    minDist = dist;
                } else if (dist < 1.5 * minDist || dist < 2.) {
                    nearDisplayObjects.add(new NearestDisplayObject(displayObject, dist, p, _layer));
                }
            }
            return nearDisplayObjects;
        }

        private Double distance(final Point point, final DisplayObject displayObject) {
            Double minDistance = Double.MAX_VALUE;
            final List<Object> coordinates = displayObject.getCoordinates();
            for (Object object : coordinates) {
                Double distance = distance(point, object);
                if (distance < minDistance) {
                    minDistance = distance;
                }
            }
            return minDistance;
        }

        private Double distance(final Point point, final Object object) {
	        if (object instanceof Path2D.Double path) {
                return distance(point, path);
	        } else if (object instanceof PointWithAngle pwa) {
                return distance(point, pwa);
	        } else if (object instanceof Polygon polygon) {
                return distance(point, polygon);
            }
            return Double.MAX_VALUE;
        }

        private Double distance(final Point point, final PathIterator pathIterator) {
            Double minDistance = Double.MAX_VALUE;
            double currentX = Double.MAX_VALUE;
            double currentY = Double.MAX_VALUE;
            while (!pathIterator.isDone()) {
                double[] coordinates = new double[6];
                int type = pathIterator.currentSegment(coordinates);
                switch (type) {
                    case PathIterator.SEG_MOVETO:
                        currentX = coordinates[0];
                        currentY = coordinates[1];
                        break;
                    case PathIterator.SEG_LINETO:
                        double newX = coordinates[0];
                        double newY = coordinates[1];
                        if (currentX != Double.MAX_VALUE) {
                            Double distance = distancePointToSegment(point, currentX, currentY, newX, newY);
                            if (distance < minDistance) {
                                minDistance = distance;
                            }
                        }
                        currentX = coordinates[0];
                        currentY = coordinates[1];
                        break;
                }
                pathIterator.next();
            }
            return minDistance;
        }

        private Double distance(final Point point, final Path2D.Double path) {
            final PathIterator pathIterator = path.getPathIterator(null);
            return distance(point, pathIterator);
        }

        private Double distance(final Point point, final Polygon polygon) {
            if (polygon.contains(point)) {
                return 0.;
            }
            final PathIterator pathIterator = polygon.getPathIterator(null);
            return distance(point, pathIterator);
        }

        private Double distancePointToSegment(Point point, Double x1, Double y1, Double x2, Double y2) {
            Double deltaX = x1 - x2;
            Double deltaY = y1 - y2;
            Double d1 = deltaX * deltaX + deltaY * deltaY;
            if (d1 > 0) {
                Double lambda = ((x1 - point.getX()) * deltaX + (y1 - point.getY()) * deltaY) / d1;
                if (lambda <= 0.) {
                    return distance(point, x1, y1);
                } else if (lambda >= 1.) {
                    return distance(point, x2, y2);
                } else {
                    return distance(point, x1 + lambda * (x2 - x1), y1 + lambda * (y2 - y1));
                }
            } else {
                return distance(point, x1, y1);
            }
        }

        private Double distance(final Point point, Double x, Double y) {
            Double deltaX = point.getX() - x;
            Double deltaY = point.getY() - y;
            return Math.sqrt(deltaX * deltaX + deltaY * deltaY);
        }

        private Double distance(final Point point, final PointWithAngle pwa) {
            return distance(point, pwa.getPoint().getX(), pwa.getPoint().getY());
        }

        /**
         * Erzeugt eine Liste von DisplayObjects, die in der Nähe des Punktes p liegen. Mit preferredUpperSize kann man eine angestrebte obere Grenze
         * für die Anzahl der Objekte angeben. Ist die Anzahl der Objekte zunächst zu groß, so wird versucht eine kleinere Liste zu erzeugen, bis die
         * angestrebte obere Grenze unterschritten wird. Gelingt aber keine Verkleinerung der Liste, so erhält man die größere Liste.
         *
         * @param p                  der Punkt
         * @param preferredUpperSize die angestrebte obere Grenze der Rückgabeliste
         *
         * @return eine Liste mit DisplayObjects
         */
        List<DisplayObject> getDisplayObjectsCloseToPoint(Point p, int preferredUpperSize) {
            double factor = 45.;
            List<DisplayObject> returnList = getDisplayObjectsCloseToPoint(p, factor);
            int size = returnList.size();
            int counter = 0;
            while (returnList.isEmpty()) {
                factor *= 1.5;
                returnList = getDisplayObjectsCloseToPoint(p, factor);
                ++counter;
                if (counter > 10) {    // Vermeidung einer Endlosschleife
                    break;
                }
            }
            while (size > preferredUpperSize) {
                factor *= 0.65;
                returnList = getDisplayObjectsCloseToPoint(p, factor);
                if (size == returnList.size()) {
                    break;
                }

                size = returnList.size();
            }
            returnList.sort(Comparator.comparing(DisplayObject::getName));
            return returnList;
        }

        private List<DisplayObject> getDisplayObjectsCloseToPoint(Point p, double factor) {
            List<DisplayObject> returnList = new ArrayList<>();
            double mpp = factor * meterProPixel();
            Rectangle rectangle = new Rectangle((int) (p.getX() - mpp / 2.), (int) (p.getY() - mpp / 2.), (int) mpp, (int) mpp);
            Collection<DisplayObject> collection;
            if (!_displayObjectList.isEmpty()) {
                collection = _displayObjectList;
            } else if (!_displayObjectMap.isEmpty()) {
                collection = _displayObjectMap.values();
            } else {
                collection = new ArrayList<>();
            }
            for (DisplayObject displayObject : collection) {
                final List<Object> coordinates = displayObject.getCoordinates();
                if (!coordinates.isEmpty()) {
                    Object firstCoordinate = coordinates.get(0);
                    if ((firstCoordinate instanceof Path2D.Double) && intersect(rectangle, coordinates)) {
                        returnList.add(displayObject);
                    } else if (firstCoordinate instanceof PointWithAngle pwa) {
                        if (rectangle.contains(pwa.getPoint())) {
                            returnList.add(displayObject);
                        }
                    } else if (firstCoordinate instanceof Polygon polygon) {
                        if (polygon.intersects(rectangle)) {
                            returnList.add(displayObject);
                        }
                    }
                }
            }
            return returnList;
        }

        /**
         * Malt einen LayerPanel.
         *
         * @param g das Graphics-Objekt
         */
        void paintLayer(Graphics g) {

            final Graphics2D g2D = (Graphics2D) g;
            if (isAntialising()) {
                g2D.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
            } else {
                g2D.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_OFF);
            }
//			g2D.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BICUBIC);
//			g2D.setRenderingHint(RenderingHints.KEY_ALPHA_INTERPOLATION, RenderingHints.VALUE_ALPHA_INTERPOLATION_SPEED);
//			g2D.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_SPEED);

            Rectangle filterRectangle = getUTMBounds();

            Collection<DisplayObject> collection;
            if (!_displayObjectList.isEmpty()) {
                collection = _displayObjectList;
            } else if (!_displayObjectMap.isEmpty()) {
                collection = _displayObjectMap.values();
            } else {
                collection = new ArrayList<>();
            }
            // Seelektierte Objekte werden nach den anderen gezeichnet:
            Collection<DisplayObject> selectedObjects = new ArrayList<>();
            for (DisplayObject displayObject : collection) {
                // Die folgende if-Bedingung sieht fehlerhaft aus; tatsächlich aber liefern einige DisplayObjects
                // kein 'BoundingRectangle' zurück! Hier kann also noch optimiert werden.
                if (displayObject.getBoundingRectangle() == null || displayObject.getBoundingRectangle().intersects(filterRectangle)) {
                    boolean selected = _selectedDisplayObjects.contains(displayObject) || _tempSelectedDisplayObjects.contains(displayObject);
                    if (_tempToggleDisplayObjects.contains(displayObject)) {
                        selected = !selected;
                    }
                    if (selected) {
                        selectedObjects.add(displayObject);
                    } else {
                        displayObject.getPainter().paintDisplayObject(MapPane.this, g2D, displayObject, false);
                    }
                    // Draw bounding box
//					g2D.setColor(Color.red);
//					if(displayObject.getBoundingRectangle() != null) {
//						g2D.draw(displayObject.getBoundingRectangle());
//					}
                }
            }
            for (DisplayObject displayObject : selectedObjects) {
                displayObject.getPainter().paintDisplayObject(MapPane.this, g2D, displayObject, true);
            }
        }
    }

    private class MyMouseListener extends MouseAdapter {
        private JPopupMenu _popup;
        private HashMap<String, DisplayObject> _displayObjectHash;
        private int _lastOffsetX = Integer.MIN_VALUE;
        private int _lastOffsetY = Integer.MIN_VALUE;
        private SelectionMode _selectionMode = SelectionMode.NoMode;

        @Override
        public void mouseDragged(MouseEvent e) {
            if (!SwingUtilities.isLeftMouseButton(e)) {
                return;
            }
            if (e.isShiftDown()) {   // Rechteck-Selektion akkumulativ oder Rechteck-Toggeln
                if (_selectionMode == SelectionMode.RectangleSelection) { // Rechteck-Selektion akkumulativ fortsetzen
                    if (_startPoint == null) {      // wenn mousePressed zuvor nicht ausgeführt worden ist
                        _startPoint = e.getPoint();
                    }
                    Rectangle2D.Double rectangle =
                        new Rectangle2D.Double(Math.min(_startPoint.getX(), e.getX()), Math.min(_startPoint.getY(), e.getY()),
                                               Math.abs(e.getPoint().getX() - _startPoint.getX()),
                                               Math.abs(e.getPoint().getY() - _startPoint.getY()));
                    rectangle.add(e.getPoint());
                    _selectionPanel.setRectangle(rectangle, _zoomScale, _zoomTranslateX, _zoomTranslateY);
                    doTempRectangleSelection(rectangle, false);
                    repaint();
                } else if (_selectionMode == SelectionMode.RectangleToggle) { // Rechteck-Toggeln fortsetzen
                    if (_startPoint == null) {      // wenn mousePressed zuvor nicht ausgeführt worden ist
                        _startPoint = e.getPoint();
                    }
                    Rectangle2D.Double rectangle =
                        new Rectangle2D.Double(Math.min(_startPoint.getX(), e.getX()), Math.min(_startPoint.getY(), e.getY()),
                                               Math.abs(e.getPoint().getX() - _startPoint.getX()),
                                               Math.abs(e.getPoint().getY() - _startPoint.getY()));
                    rectangle.add(e.getPoint());
                    _selectionPanel.setRectangle(rectangle, _zoomScale, _zoomTranslateX, _zoomTranslateY);
                    doTempRectangleSelection(rectangle, true);
                    repaint();
                }
            } else {  // Verschiebe den Kartenausschnitt
                int newX = e.getX() - _lastOffsetX;
                int newY = e.getY() - _lastOffsetY;
                _lastOffsetX += newX;
                _lastOffsetY += newY;
                _zoomTranslateX += newX;
                _zoomTranslateY += newY;
                visibleObjectsChanged();
                _selectionPanel.transformRectangle(_zoomScale, _zoomTranslateX, _zoomTranslateY);
                repaint();
            }
        }

        @Override
        public void mousePressed(MouseEvent e) {
            if (!SwingUtilities.isLeftMouseButton(e)) {
                return;
            }
            if (e.isAltDown()) {    // Single-Selektion unter Alt-Taste hat absoluten Vorrang
                _selectionMode = SelectionMode.SingleSelection;
                _selectionPanel.setRectangle(null, _zoomScale, _zoomTranslateX, _zoomTranslateY);
                singleSelection(e);
            } else if (e.isShiftDown()) { // Rechteck-Selektion oder Rechteck-Toggle beginnen
                if (e.getClickCount() == 1) {
                    _selectionPanel.setRectangle(null, _zoomScale, _zoomTranslateX, _zoomTranslateY);
                    repaint();
                } else {
                    clearSelection();
                    clearTempSelection();
                    clearTempToggle();
                }
                if (isCtrlOrCommandDown(e)) {
                    _selectionMode = SelectionMode.RectangleToggle;
                } else {
                    _selectionMode = SelectionMode.RectangleSelection;
                }
                _startPoint = e.getPoint();
                // Hack: Wenn im GenericNetDisplay ein Notiz-Objekt selektiert ist, und dieser Teil des JTabbedPane
                // den Fokus hat, käme es unmittelbar nach dem Abschluß der Rechteck-Selektion zu deren Löschung.
                // Deshalb wird der Fokus hierher geholt.
                requestFocus();
            } else if (isCtrlOrCommandDown(e)) {     // Mehrfachselektion/Toggle-Modus für einzelne Elemente
                _selectionMode = SelectionMode.SingleToggle;
                // Hack: Wenn im GenericNetDisplay ein Notiz-Objekt selektiert ist, und dieser Teil des JTabbedPane
                // den Fokus hat, käme es unmittelbar nach jeder Toggle-Selektion hier zu einer ausschließlichen
                // Selektion des Notiz-Objekts. Deshalb wird der Fokus hierher geholt.
                requestFocus();

                if (_popup != null) {             // das Popup-Menü ausblenden
                    _popup.setVisible(false);
                }
                toggleModus(e);
            }
            _lastOffsetX = e.getX();
            _lastOffsetY = e.getY();
        }

        @Override
        public void mouseReleased(MouseEvent e) {
            if (!SwingUtilities.isLeftMouseButton(e)) {
                return;
            }
            if (_selectionMode == SelectionMode.RectangleSelection || _selectionMode == SelectionMode.RectangleToggle) {
                if (_startPoint != null) {
                    Rectangle2D.Double rectangle =
                        new Rectangle2D.Double(Math.min(_startPoint.getX(), e.getX()), Math.min(_startPoint.getY(), e.getY()),
                                               Math.abs(e.getPoint().getX() - _startPoint.getX()),
                                               Math.abs(e.getPoint().getY() - _startPoint.getY()));
                    rectangle.add(e.getPoint());
                    if (!rectangle.isEmpty()) {
                        _selectionPanel.setRectangle(rectangle, _zoomScale, _zoomTranslateX, _zoomTranslateY);
                        if (_selectionMode == SelectionMode.RectangleSelection) {
                            doTempRectangleSelection(rectangle, false);
                            approveTempSelection();
                        } else {                            // Rechteck-Toggeln
                            doTempRectangleSelection(rectangle, true);
                            approveTempToggle();
                        }
                        _startPoint = null;
                        repaint();
                    }
                }
            }
        }

        private void toggleModus(MouseEvent e) {
            Point point = e.getPoint();
            Point utmPoint = new Point();
            if (!getUTMPoint(point, utmPoint)) {
                return;
            }
            Map<String, DisplayObject> displayObjectsHash = searchDisplayObjectsCloseToPoint(utmPoint, 1);
            // Nun das Toggeln:
            if (displayObjectsHash.isEmpty()) {
                //noinspection UnnecessaryReturnStatement
                return;    // vermeidet ein Artefakt
            } else if (displayObjectsHash.size() == 1) {
                toggleSelection(displayObjectsHash.values().iterator().next(), true);
            } else {
                _popup = new JPopupMenu();
                _displayObjectHash = new HashMap<>();
                for (final Map.Entry<String, DisplayObject> entry : displayObjectsHash.entrySet()) {
                    JCheckBoxMenuItem menuItem = new JCheckBoxMenuItem(entry.getKey(), _selectedDisplayObjects.contains(entry.getValue()));
                    _popup.add(menuItem);
                    menuItem.addActionListener(actionEvent -> toggleSelection(entry.getValue(), true));
                }
                _popup.show(e.getComponent(), point.x, point.y);
                _popup.setVisible(true);
            }
        }

        private void singleSelection(MouseEvent e) {
            clearSelection();
            clearTempSelection();
            if (_popup != null) {
                _popup.setVisible(false);
            }
            Point point = e.getPoint();
            Point utmPoint = new Point();
            if (!getUTMPoint(point, utmPoint)) {
                return;
            }
            Map<String, DisplayObject> displayObjectsHash = searchDisplayObjectsCloseToPoint(utmPoint, 5);
            if (displayObjectsHash.isEmpty()) {
                //noinspection UnnecessaryReturnStatement
                return;    // vermeidet ein Artefakt
            } else if (displayObjectsHash.size() == 1) {
                addToSelection(displayObjectsHash.values().iterator().next());
                notifySelectionChanged(Collections.unmodifiableCollection(_selectedDisplayObjects));
            } else {
                _popup = new JPopupMenu();
                _displayObjectHash = new HashMap<>();
                for (final Map.Entry<String, DisplayObject> entry : displayObjectsHash.entrySet()) {
                    JMenuItem menuItem = new JMenuItem(entry.getKey());
                    _popup.add(menuItem);
                    menuItem.addActionListener(new MenuItemSelector());
                    _displayObjectHash.put(entry.getKey(), entry.getValue());
                }
                _popup.show(e.getComponent(), point.x, point.y);
                _popup.setVisible(true);
            }
        }

        private Map<String, DisplayObject> searchDisplayObjectsCloseToPoint(Point utmPoint, int preferredNumber) {

            List<NearestDisplayObject> nearestDisplayObjectList = new ArrayList<>();
            for (ViewEntry entry : _view.getAllViewEntries()) {
                if (entry.isVisible(getMapScale().intValue()) && entry.isSelectable()) {
                    Component component = entry.getComponent();
	                if (component instanceof LayerPanel layerPanel) {
                        final List<DisplayObject> displayObjectsCloseToPoint = layerPanel.getDisplayObjectsCloseToPoint(utmPoint, preferredNumber);
                        List<NearestDisplayObject> nearestDisplayObjectsOfLayer =
                            layerPanel.getNearestDisplayObjects(utmPoint, displayObjectsCloseToPoint);
                        nearestDisplayObjectList.addAll(nearestDisplayObjectsOfLayer);
                    }
                }
            }
            // Sortiere nach Distanz:
            Collections.sort(nearestDisplayObjectList);

            // Einschränken auf nahegelegene Objekte unter Beachtung von preferredNumber:
            List<NearestDisplayObject> filteredNearestDisplayObjectList = new ArrayList<>();
            double cut = Math.max(30. * meterProPixel(), 5.);
            for (int i = 0; i < nearestDisplayObjectList.size(); ++i) {
                NearestDisplayObject nearestDisplayObject = nearestDisplayObjectList.get(i);
                if (nearestDisplayObject.getDistance() < cut) {
                    filteredNearestDisplayObjectList.add(nearestDisplayObject);
                }
                if (filteredNearestDisplayObjectList.size() >= preferredNumber) {
                    // füge noch die Objekte in gleicher Entfernung hinzu
                    Double lastDistance = nearestDisplayObject.getDistance();
                    for (int j = i + 1; j < nearestDisplayObjectList.size(); ++j) {
                        NearestDisplayObject furtherNearestDO = nearestDisplayObjectList.get(j);
                        if (furtherNearestDO.getDistance().equals(lastDistance)) {
                            filteredNearestDisplayObjectList.add(furtherNearestDO);
                        } else {
                            break;
                        }
                    }
                    break;
                }
            }

            // Sortiere nach Layer und Distanz:
            filteredNearestDisplayObjectList.sort((o1, o2) -> {
                if (!o1.getLayer().equals(o2.getLayer())) {
                    return o1.getLayer().getName().compareTo(o2.getLayer().getName());
                } else {
                    if (o1.getDistance() < o2.getDistance()) {
                        return -1;
                    } else if (o1.getDistance() > o2.getDistance()) {
                        return 1;
                    } else {
                        return 0;
                    }
                }
            });

//			for(NearestDisplayObject nearestDisplayObject: filteredNearestDisplayObjectList) {
//				System.out.println(nearestDisplayObject.toString());
//			}

            Map<String, DisplayObject> displayObjectsHash = new LinkedHashMap<>();

            for (NearestDisplayObject nearestDisplayObject : filteredNearestDisplayObjectList) {
                Layer layer = nearestDisplayObject.getLayer();
                DisplayObject displayObject = nearestDisplayObject.getDisplayObject();
	            if (displayObject instanceof OnlineDisplayObject onlineDisplayObject) {
                    displayObjectsHash.put(layer.getName() + ": " + onlineDisplayObject.getSystemObject().getNameOrPidOrId() + " (" +
                                           nearestDisplayObject.getDistance().intValue() + "m)", onlineDisplayObject);
	            } else if (displayObject instanceof CsvDisplayObject csvDisplayObject) {
                    displayObjectsHash
                        .put(layer.getName() + ": " + csvDisplayObject.getName() + " (" + nearestDisplayObject.getDistance().intValue() + "m)",
                             csvDisplayObject);
                }

            }
            return displayObjectsHash;
        }

        private boolean isCtrlOrCommandDown(MouseEvent e) {
            final int ctrlOrCommandDown;
            if (!_isMac) {
                ctrlOrCommandDown = InputEvent.CTRL_DOWN_MASK;
            } else {
                ctrlOrCommandDown = InputEvent.META_DOWN_MASK;
            }
            return (e.getModifiersEx() & ctrlOrCommandDown) != 0;
        }

        class MenuItemSelector implements ActionListener {
            @Override
            public void actionPerformed(ActionEvent ae) {
                final String actionCommand = ae.getActionCommand();
                addToSelection(_displayObjectHash.get(actionCommand));
                notifySelectionChanged(Collections.unmodifiableCollection(_selectedDisplayObjects));
            }
        }

    }
}
