/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.gnd;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.sys.funclib.application.AbstractGUIApplication;
import de.bsvrz.sys.funclib.application.StandardApplicationRunner;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList.Argument;
import de.bsvrz.sys.funclib.debug.Debug;
import de.kappich.pat.gnd.colorManagement.ColorManager;
import de.kappich.pat.gnd.extLocRef.ReferenceHierarchyManager;
import de.kappich.pat.gnd.viewManagement.View;
import de.kappich.pat.gnd.viewManagement.ViewManager;
import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.List;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JProgressBar;

/**
 * Die Start-Klasse für die GND im Stand-Alone-Betrieb.
 *
 * @author Kappich Systemberatung
 */
public class RunGND extends AbstractGUIApplication {

    private static Debug _debug;
    private final List<String> _plugins = new ArrayList<>();
    private boolean _deletePreferences;

    /** @param args die Aufrufparemeter */

    public static void main(String[] args) {
        System.getProperties().put("apple.laf.useScreenMenuBar", "true");
        Thread.setDefaultUncaughtExceptionHandler(new MyUncaughtExceptionHandler());
        StandardApplicationRunner.run(new RunGND(), args);
    }

    private static JDialog getProgressDialog() {
        final JDialog progressDialog = new JDialog();
        progressDialog.setTitle("Die GND wird initialisiert");
        progressDialog.setLayout(new BorderLayout());
        final JLabel textLabel = new JLabel("Initialisierung der Präferenzen.");
        textLabel.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
        progressDialog.add(textLabel, BorderLayout.NORTH);
        final JLabel counterLabel = new JLabel();
        counterLabel.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
        progressDialog.add(counterLabel, BorderLayout.WEST);
        final JProgressBar progressBar = new JProgressBar();
        progressBar.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
        progressBar.setIndeterminate(true);
        progressDialog.add(progressBar, BorderLayout.CENTER);
        final JButton cancelButton = new JButton("Abbrechen");
        ActionListener cancelButtonListener = e -> {
            progressDialog.dispose();
            System.exit(0);
        };
        cancelButton.addActionListener(cancelButtonListener);
        final JPanel cancelPanel = new JPanel();
        cancelPanel.add(cancelButton);
        progressDialog.add(cancelPanel, BorderLayout.SOUTH);
        progressDialog.setPreferredSize(new Dimension(400, 150));
        progressDialog.pack();

        return progressDialog;
    }

    @Override
    public void parseArguments(ArgumentList argumentList) throws Exception {
        _debug = Debug.getLogger();
        _debug.fine("argumentList = " + argumentList);
        final Argument pluginArgument = argumentList.fetchArgument("-plugins=");
        if (pluginArgument != null && pluginArgument.hasValue()) {
            final String value = pluginArgument.getValue();
            if ((value != null) && (!value.isEmpty())) {
                final String[] valueParts = value.split(",");
                for (String valuePart : valueParts) {
                    if ((valuePart != null) && (!valuePart.isEmpty())) {
                        _plugins.add(valuePart);
                    }
                }
            }
        }
        if (argumentList.hasArgument("-löschePräferenzen")) {
            argumentList.fetchArgument("-löschePräferenzen");
            _deletePreferences = true;
        }
    }

    @Override
    public void initialize(ClientDavInterface connection) throws Exception {
        PreferencesHandler.setKvPid(connection.getLocalConfigurationAuthority().getPid());
        if (_deletePreferences) {
            Preferences gndPrefs = PreferencesHandler.getInstance().getPreferenceStartPath();
            try {
                gndPrefs.removeNode();
                JOptionPane.showMessageDialog(null, "Die Präferenzen wurden gelöscht.", "Information", JOptionPane.INFORMATION_MESSAGE);

            } catch (BackingStoreException ignore) {
                JOptionPane.showMessageDialog(null, "Das Löschen der Präferenzen war nicht erfolgreich.", "Fehlermeldung", JOptionPane.ERROR_MESSAGE);
            }
        }

        JDialog progressDialog = getProgressDialog();
        progressDialog.setVisible(true);

        // Der ReferenceHierarchyManager wird schon im LayerManager benötigt, wo allerdings kein ClientDavInterface
        // zur Verfügung steht; deshalb Initialiesierung hier.
        ReferenceHierarchyManager.getInstance(connection);
        // Der ColorManager wird schon hier initialisiert, weil er ansonsten erst bei paintDisplayObject initialisiert
        // wird, und wenn es da zu Problemen käme, gibt es eine NPE.
        //noinspection ResultOfMethodCallIgnored
        ColorManager.getInstance();

        ViewManager viewManager = ViewManager.getInstance();
        String viewName = GenericNetDisplay.INTERNAL_START_VIEW_NAME;
        final View view = viewManager.getView(viewName);
        if (view == null) {
            final String x = "RunGND: ein View namens '" + viewName + "' kann nicht gefunden werden.";
            _debug.error(x);
            System.err.println(x);
            System.exit(1); // RunGND darf beendet werden!
        }
        if (!_plugins.isEmpty()) {
            GenericNetDisplay.addPlugins(_plugins);
        }
        progressDialog.dispose();

        GenericNetDisplay gnd = new GenericNetDisplay(view, connection, true, null);
        gnd.setVisible(true);
    }

    @Override
    protected String getApplicationName() {
        return "Kappich Systemberatung - Generische Netzdarstellung";
    }

    @Override
    public String toString() {
        return "RunGND{" + "_plugins=" + _plugins + '}';
    }

    private static class MyUncaughtExceptionHandler implements Thread.UncaughtExceptionHandler {

        @Override
        public void uncaughtException(Thread t, Throwable e) {
            System.err.println("Laufzeitfehler: Ein Thread hat sich wegen einer Exception beendet:");
            System.err.println(t);
            e.printStackTrace(System.err);
            _debug.error("Laufzeitfehler: " + t + " hat sich wegen einer Exception beendet", e);
            System.exit(1);    // RunGND darf beendet werden!
        }
    }
}
