/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.gnd;

import de.kappich.pat.gnd.displayObjectToolkit.DisplayObject;
import de.kappich.pat.gnd.displayObjectToolkit.OnlineDisplayObject;
import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.font.FontRenderContext;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.List;
import java.util.Set;

/**
 * Ein {@code SelectionPainter} übernimmt plugin-übergreifende Aufgaben beim Zeichnen der {@link DisplayObject DisplayObjects}. Im Moment ist dies die
 * Anzeige von Namen oder Pids für die selektierten Objekte.
 *
 * @author Kappich Systemberatung
 */
class SelectionPainter {

    private static final AffineTransform IDENTITY = new AffineTransform();

    private final Graphics2D _g2D;
    private final Rectangle _utmBounds;
    private final Set<DisplayObject> _selectedObjects;
    private final Set<DisplayObject> _tempSelectedObjects;
    private final Set<DisplayObject> _tempToggleSelectionObjects;

    /**
     * Der SelectionPainter bekommt drei Mengen übergeben, wovon nur eine der letzten beiden nicht leer sein sollte, der man sich entweder im Modus
     * einer temporären Selektion oder im Modus des temporären Toggeln befindet. Sind doch beide dieser Mengen nicht leer, so ist das Verhalten nicht
     * festgelegt (s. paintSelection).
     *
     * @param g2D                        ein Graphics2D-Objekt
     * @param utmBounds                  ein UTM-Rechteck
     * @param selectedObjects            die Menge der bereits selektierten Objekte
     * @param tempSelectedObjects        die Menge der temporär selektierten Objekte
     * @param tempToggleSelectionObjects die Menge der temporär zu toggelnden Objekte
     */
    SelectionPainter(final Graphics2D g2D, final Rectangle utmBounds, Set<DisplayObject> selectedObjects, Set<DisplayObject> tempSelectedObjects,
                     Set<DisplayObject> tempToggleSelectionObjects) {
        _g2D = g2D;
        _utmBounds = utmBounds;
        _selectedObjects = selectedObjects;
        _tempSelectedObjects = tempSelectedObjects;
        _tempToggleSelectionObjects = tempToggleSelectionObjects;
    }

    private static String getNameOrPid(final DisplayObject displayObject) {
	    if (displayObject instanceof OnlineDisplayObject onlineDisplayObject) {
            return onlineDisplayObject.getSystemObject().getNameOrPidOrId();
        } else {
            return displayObject.getName();
        }
    }

    void paintSelection() {
        for (DisplayObject displayObject : _selectedObjects) {
            if (_tempToggleSelectionObjects.contains(displayObject)) {
                continue;
            }
            paintNameOrPid(displayObject);
        }
        for (DisplayObject displayObject : _tempToggleSelectionObjects) {
            if (_selectedObjects.contains(displayObject)) {
                continue;
            }
            paintNameOrPid(displayObject);
        }
        for (DisplayObject displayObject : _tempSelectedObjects) {
            paintNameOrPid(displayObject);
        }
    }

    /*
    Der SelektionPainter zeichnet den Namen oder die Pid für jeden Referenzpunkt!
     */
    private void paintNameOrPid(final DisplayObject displayObject) {
        List<Point2D> referencePoints = displayObject.getReferencePoints(_utmBounds);
        for (Point2D point : referencePoints) {
            String text = getNameOrPid(displayObject);
            if (null == text || text.isEmpty()) {
                return;
            }
            Font font = new Font(Font.DIALOG, Font.PLAIN, 12);
            _g2D.setFont(font);
            _g2D.setColor(Color.BLACK);
            Double xAsDouble = point.getX();
            Double yAsDouble = point.getY();
            Point2D.Double translation = new Point2D.Double(xAsDouble, yAsDouble);

            AffineTransform at = _g2D.getTransform();
            _g2D.setTransform(IDENTITY);
            FontRenderContext fontRenderContext = _g2D.getFontRenderContext();
            Rectangle2D textBoundingRectangle = font.getStringBounds(text, fontRenderContext);
            final Point2D.Double deviceTextPoint = new Point2D.Double();
            at.transform(translation, deviceTextPoint);
            final Double textX = deviceTextPoint.getX() - textBoundingRectangle.getWidth() / 2;
            final Double textY = deviceTextPoint.getY() + textBoundingRectangle.getHeight() / 2;
            textBoundingRectangle.setRect(textBoundingRectangle.getX(), textBoundingRectangle.getY() + textBoundingRectangle.getHeight() / 3,
                                          textBoundingRectangle.getWidth(),
                                          textBoundingRectangle.getHeight() - textBoundingRectangle.getHeight() / 3);
            _g2D.drawString(text, textX.floatValue(), textY.floatValue());
            _g2D.setTransform(at);
        }
    }
}
