/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.kappich.pat.gnd.layerManagement;

import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.asbNodePlugin.DOTAsbNode;
import de.kappich.pat.gnd.asbNodePlugin.DOTAsbNodePlugin;
import de.kappich.pat.gnd.displayObjectToolkit.DOTManager;
import de.kappich.pat.gnd.gnd.PreferencesHandler;
import de.kappich.pat.gnd.kmPlugin.DOTKm;
import de.kappich.pat.gnd.kmPlugin.DOTKmPlugin;
import de.kappich.pat.gnd.linePlugin.DOTLinePlugin;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.pointPlugin.DOTPointPlugin;
import de.kappich.pat.gnd.rnPlugin.DOTRn;
import de.kappich.pat.gnd.rnPlugin.DOTRnPlugin;
import de.kappich.pat.gnd.statPlugin.DOTStat;
import de.kappich.pat.gnd.statPlugin.DOTStatPlugin;
import de.kappich.pat.gnd.utils.view.PreferencesDeleter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import javax.swing.table.AbstractTableModel;

/**
 * Eine Singleton-Klasse zur Verwaltung aller Layer, die das Interface TableModel implementiert, damit sie in einem JTable angezeigt werden kann.
 * <p>
 * Dass diese Klasse (wie z.B. auch der {@link DOTManager} als Singleton implementiert ist, ist hinsichtlich denkbarer Erweiterungen sicherlich keine
 * optimale Lösung, aber erspart gegenwärtig die Implementation der Kommunikation zwischen verschiedenen Instanzen dieser Klasse.
 *
 * @author Kappich Systemberatung
 */
@SuppressWarnings("serial")
public final class LayerManager extends AbstractTableModel {

    // Statische statt lazy Initialisierung da lazy nicht thread-safe. Statische Initialisierung
    // hat allerdings den Nachteil bei Exceptions. Deshalb immer auch Debug benutzen.
    private static final LayerManager _instance = new LayerManager();

    private static final Debug _debug = Debug.getLogger();
    private static final String[] COLUMN_NAMES = {"Name des Layers"};
	private final List<Layer> _layers = new ArrayList<>() {
        public boolean add(Layer layer) {
            int index = Collections.binarySearch(this, layer);
            if (index < 0) {
                index = ~index;
            }
            super.add(index, layer);
            return true;
        }
    };
    private final Set<String> _unchangeables = new HashSet<>();
    private final Map<String, Layer> _layerHashMap = new HashMap<>();
    private final List<LayerManagerChangeListener> _listeners = new CopyOnWriteArrayList<>();

    /*
     * Ein LayerManager verwaltet alle zur GND gehörenden Layer in einer Liste. Darin
     * stehen einige in seinem Kode definierte Layer, die unveränderbar sind, sowie
     * alle vom Benutzer definierten Layer, die in den Präferenzen abgespeichert werden.
     */
    private LayerManager() {
        initDefaultLayers();
        initUserDefinedLayers();
    }

    /**
     * Die für ein Singleton übliche Methode, um an die einzige Instanz der Klasse zu gelangen.
     *
     * @return den LayerManager
     */
    public static LayerManager getInstance() {
        return _instance;
    }

    /**
     * Mit Hilfe dieser Methode kann man den LayerManager dazu zwingen, sich erneut zu konstruieren, was etwa nach dem Importieren von Präferenzen
     * sinnvoll ist.
     */
    public static void refreshInstance() {
        getInstance()._layers.clear();
        getInstance()._unchangeables.clear();
        getInstance()._layerHashMap.clear();
        getInstance().initDefaultLayers();
        getInstance().initUserDefinedLayers();
    }

    /**
     * Gibt das Preferences-Objekt für den Ausgangspunkt zur Ablage der Präferenzen des Layermanagers zurück.
     *
     * @return den Ausgangsknoten
     */
    private static Preferences getPreferenceStartPath() {
//		return Preferences.userRoot().node("de/kappich/pat/gnd/Layer");
        return PreferencesHandler.getInstance().getPreferenceStartPath().node("Layer");
    }

    /**
     * Gibt eine sortierte Kopie der Liste aller Layer zurück.
     *
     * @return die Liste aller Layer
     */
    public List<Layer> getLayers() {
        return Collections.unmodifiableList(_layers);
    }

    /**
     * Gibt den Index des ersten Layers, dessen Name mit dem übergebenen Zeichen anfängt, zurück. Gibt es einen solchen Layer nicht, so wird -1
     * zurückgegeben.
     *
     * @param c ein Buchstabe
     *
     * @return ein gültiger Index oder -1
     */
    public int getIndexOfFirstLayer(final char c) {
        char s = Character.toLowerCase(c);
        for (int index = 0; index < _layers.size(); ++index) {
            Layer layer = _layers.get(index);
            char t = Character.toLowerCase(layer.getName().charAt(0));
            if (s == t) {
                return index;
            }
        }
        return -1;
    }

    /**
     * Gibt den Layer mit dem übergebenen Namen zurück.
     *
     * @param layerName der Name
     *
     * @return den geforderten Layer
     */
    public Layer getLayer(String layerName) {
        return _layerHashMap.get(layerName);
    }

    /**
     * Gibt den Layer an der i-ten Stelle der Layerliste zurück, wobei die Zählung mit 0 beginnt.
     *
     * @param i ein Index
     *
     * @return den geforderten Layer
     */
    public Layer getLayer(int i) {
        return _layers.get(i);
    }

    /**
     * Gibt {@code true} zurück, wenn es einen Layer gibt, dessen Name sich allenfalls bezüglich Klein-Groß-Schreibung unterseheidet. Sonst {@code
     * false}.
     *
     * @param layerName
     *
     * @return s.o.
     */
    public boolean hasLayerToLowerCase(String layerName) {
        for (Layer l : _layers) {
            if (l.getName().toLowerCase().equals(layerName.toLowerCase())) {
                return true;
            }
        }
        return false;
    }

    /*
     * Gehört zur Implementation des TableModel.
     */
    @Override
    public int getColumnCount() {
        return COLUMN_NAMES.length;
    }

    /*
     * Gehört zur Implementation des TableModel.
     */
    @Override
    public int getRowCount() {
        return _layers.size();
    }

    /*
     * Gehört zur Implementation des TableModel.
     */
    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
        return _layers.get(rowIndex).getName();
    }

    /**
     * Definiert den Tooltipp für die Felder der Tabelle. Gehört zur Implementation des TableModel.
     *
     * @param rowIndex ein Zeilenindex
     *
     * @return ein Tooltipp
     */
    @Nullable
    public String getTooltipAt(int rowIndex) {
        if (rowIndex >= 0 && rowIndex < getRowCount()) {
            return _layers.get(rowIndex).getInfo();
        } else {
            return null;
        }
    }

    /*
     * Gehört zur Implementation des TableModel.
     */
    @Override
    public String getColumnName(int columnIndex) {
        return COLUMN_NAMES[columnIndex];
    }

    /**
     * Fügt den Layer der Layerliste an, wenn nicht schon ein gleichnamiger Layer existiert.
     *
     * @param layer ein Layer
     *
     * @throws IllegalArgumentException wenn bereits ein gleichnamiger Layer existiert
     */
    public void addLayer(Layer layer) throws IllegalArgumentException {
        if (_layerHashMap.containsKey(layer.getName())) {
            throw new IllegalArgumentException("Ein Layer mit diesem Namen existiert bereits.");
        }
        // Klein-/Großschreibung nicht (mehr) signifikant:
        for (Layer l : _layers) {
            if (layer.equals(l)) {
                throw new IllegalArgumentException(
                    "Es existiert bereits ein Layer, dessen Name sich nur bezüglich Klein-/Großschreibung unterscheidet.");
            }
        }
        _layers.add(layer);
        _layerHashMap.put(layer.getName(), layer);
        layer.putPreferences(getPreferenceStartPath());
//		final int lastIndex = _layers.size()-1;
//		fireTableRowsInserted( lastIndex, lastIndex);   // das ist falsch, seit _layers sortiert
        fireTableDataChanged();
        notifyChangeListenersLayerAdded(layer);
    }

    /**
     * Ändert den gleichnamigen Layer.
     *
     * @param layer ein Layer
     *
     * @throws IllegalArgumentException wenn der Layer nicht bekannt ist
     */
    public void changeLayer(Layer layer) throws IllegalArgumentException {
        final String name = layer.getName();
        if (!_layerHashMap.containsKey(name)) {
            throw new IllegalArgumentException("Ein Layer mit diesem Namen existiert nicht.");
        }
        final Layer existingLayer = _layerHashMap.get(name);
        existingLayer.deletePreferences(getPreferenceStartPath());
        existingLayer.setInfo(layer.getInfo());
        existingLayer.setConfigurationObjectType(layer.getConfigurationObjectType());
        existingLayer.setDotCollection(layer.getDotCollection());
        existingLayer.setReferenceHierarchy(layer.getReferenceHierarchy());
        for (int i = 0; i < _layers.size(); i++) {
            if (_layers.get(i).getName().equals(name)) {
                fireTableRowsUpdated(i, i);
                break;
            }
        }
        layer.putPreferences(getPreferenceStartPath());
        notifyChangeListenersLayerDefinitionChanged(layer);
    }

    /**
     * Entfernt den übergebenen Layer auf Basis eines Namensvergleichs aus der Liste aller Layer und damit auch aus den Präferenzen. Entspricht einer
     * kompletten Löschung des Layers. Wirkt aber nicht für im Kode definierte Layer. Wird ein Layer gelöscht, so erhält man den Rückgabewert {@code
     * true}, sonst false.
     *
     * @param layer ein Layer
     *
     * @return {@code true} genau dann, wenn der Layer gelöscht wurde
     */
    public boolean removeLayer(Layer layer) {
        final String name = layer.getName();
        if (!_unchangeables.contains(name)) {
            final int index = remove(name);
            if (index > -1) {
                layer.deletePreferences(getPreferenceStartPath());
                fireTableRowsDeleted(index, index);
                notifyChangeListenersLayerRemoved(name);
                return true;
            }
        }
        return false;
    }

    /**
     * Löscht alle benutzerdefinierten Layer.
     */
    public void clearLayers() {
        for (Layer layer : _layers) {
            removeLayer(layer);
        }
    }

    private void initDefaultLayers() {
        // Reihenfolge wie in der Default-View! S. ViewManager.initDefaultViews.
        initRnLayer();
        initKmLayer();
        initStatLayer();
        initAsbNodeLayer();
        initDefaultLayersLines();
        initDefaultLayersPoints();
    }

    private void initDefaultLayersLines() {
        Layer streetNetLayer =
            new Layer("Straßennetz", "Dargestellt werden alle Straßenteilsegmente", new DOTLinePlugin(), "typ.straßenTeilSegment", null, null, null);
        final DisplayObjectType dotBlackLine = DOTManager.getInstance().getDisplayObjectType("Konfigurationslinie schwarz");
        if (dotBlackLine != null) {
            streetNetLayer.addDisplayObjectType(dotBlackLine, Integer.MAX_VALUE, 1000000);
        } else {
            _debug
                .warning("Fehler in LayerManager.initDefaultLayersLines: das Darstellungsobjekt 'Konfigurationslinie' konnte nicht gefunden werden.");
        }
        final DisplayObjectType dotGreyLine = DOTManager.getInstance().getDisplayObjectType("Konfigurationslinie hellgrau");
        if (dotGreyLine != null) {
            streetNetLayer.addDisplayObjectType(dotGreyLine, 1000000, 1);
        } else {
            _debug
                .warning("Fehler in LayerManager.initDefaultLayersLines: das Darstellungsobjekt 'Konfigurationslinie' konnte nicht gefunden werden.");
        }
        _layers.add(streetNetLayer);
        _layerHashMap.put(streetNetLayer.getName(), streetNetLayer);
        _unchangeables.add(streetNetLayer.getName());

        Layer olsim1Layer =
            new Layer("Störfällzustand OLSIM 1", "Berechnet an Straßensegmenten", new DOTLinePlugin(), "typ.straßenSegment", null, null, null);
        String dotName = "Störfallzustand OLSIM 1 (grob)";
        final DisplayObjectType dotOlsim1Rough = DOTManager.getInstance().getDisplayObjectType(dotName);
        if (dotOlsim1Rough != null) {
            olsim1Layer.addDisplayObjectType(dotOlsim1Rough, Integer.MAX_VALUE, 500000);
        } else {
            _debug.warning("Fehler in LayerManager.initDefaultLayersLines: das Darstellungsobjekt '" + dotName + "' konnte nicht gefunden werden.");
        }
        dotName = "Störfallzustand OLSIM 1 (fein)";
        final DisplayObjectType dotOlsim1Smooth = DOTManager.getInstance().getDisplayObjectType(dotName);
        if (dotOlsim1Smooth != null) {
            olsim1Layer.addDisplayObjectType(dotOlsim1Smooth, 500000, 1);
        } else {
            _debug.warning("Fehler in LayerManager.initDefaultLayersLines: das Darstellungsobjekt '" + dotName + "' konnte nicht gefunden werden.");
        }
        _layers.add(olsim1Layer);
        _layerHashMap.put(olsim1Layer.getName(), olsim1Layer);
        _unchangeables.add(olsim1Layer.getName());
    }

    private void initDefaultLayersPoints() {
        Layer simpleDetectionSiteLayer = new Layer("Messquerschnitte", null, new DOTPointPlugin(), "typ.messQuerschnitt", null, null, null);
        String dotName = "Verkehrsdatenanalyse kurz";
        final DisplayObjectType dotConstraint = DOTManager.getInstance().getDisplayObjectType(dotName);
        if (dotConstraint != null) {
            simpleDetectionSiteLayer.addDisplayObjectType(dotConstraint, 300000, 1);
        } else {
            _debug.warning("Fehler in LayerManager.initDefaultLayersPoints: das Darstellungsobjekt '" + dotName + "' konnte nicht gefunden werden.");
        }
        _layers.add(simpleDetectionSiteLayer);
        _layerHashMap.put(simpleDetectionSiteLayer.getName(), simpleDetectionSiteLayer);
        _unchangeables.add(simpleDetectionSiteLayer.getName());

        Layer advancedDetectionSiteLayer =
            new Layer("Messquerschnitte (erweitert)", null, new DOTPointPlugin(), "typ.messQuerschnitt", null, null, null);
        dotName = "MQ, einfach kombinierte Darstellung";
        final DisplayObjectType dotCombined = DOTManager.getInstance().getDisplayObjectType(dotName);
        if (dotCombined != null) {
            advancedDetectionSiteLayer.addDisplayObjectType(dotCombined, 300000, 1);
        } else {
            _debug.warning("Fehler in LayerManager.initDefaultLayersPoints: das Darstellungsobjekt '" + dotName + "' konnte nicht gefunden werden.");
        }
        _layers.add(advancedDetectionSiteLayer);
        _layerHashMap.put(advancedDetectionSiteLayer.getName(), advancedDetectionSiteLayer);
        _unchangeables.add(advancedDetectionSiteLayer.getName());

        Layer complexTestLayer = new Layer("Messquerschnitte (Testlayer)", "MQ", new DOTPointPlugin(), "typ.messQuerschnitt", null, null, null);
        dotName = "MQ, Testdarstellung 1";
        final DisplayObjectType dotDetectionSiteTest = DOTManager.getInstance().getDisplayObjectType(dotName);
        if (dotDetectionSiteTest != null) {
            complexTestLayer.addDisplayObjectType(dotDetectionSiteTest, 500000, 1);
        } else {
            _debug.warning("Fehler in LayerManager.initDefaultLayersPoints: das Darstellungsobjekt '" + dotName + "' konnte nicht gefunden werden.");
        }
        _layers.add(complexTestLayer);
        _layerHashMap.put(complexTestLayer.getName(), complexTestLayer);
        _unchangeables.add(complexTestLayer.getName());
    }

    private void initAsbNodeLayer() {
        DOTAsbNodePlugin plugin = new DOTAsbNodePlugin();
        Layer asbNodeLayer = new Layer("ASB-Knotennummern", "Die ASB-Knotennummern", plugin, plugin.getGeometryType(), null, null, null);
        {
            DOTAsbNode asbNode0 = (DOTAsbNode) DOTManager.getInstance().getDisplayObjectType("ASB-Knotennummern V0");
            if (asbNode0 != null) {
                asbNodeLayer.addDisplayObjectType(asbNode0, 20000, 1);
            } else {
                _debug.warning(
                    "Fehler in LayerManager.initAsbNodeLayer: das Darstellungsobjekt 'ASB-Knotennummern V0' " + "konnte nicht gefunden werden.");
            }
        }
        {
            DOTAsbNode asbNode1 = (DOTAsbNode) DOTManager.getInstance().getDisplayObjectType("ASB-Knotennummern V1");
            if (asbNode1 != null) {
                asbNodeLayer.addDisplayObjectType(asbNode1, 50000, 20001);
            } else {
                _debug.warning(
                    "Fehler in LayerManager.initAsbNodeLayer: das Darstellungsobjekt 'ASB-Knotennummern V1' " + "konnte nicht gefunden werden.");
            }
        }
        {
            DOTAsbNode asbNode2 = (DOTAsbNode) DOTManager.getInstance().getDisplayObjectType("ASB-Knotennummern V2");
            if (asbNode2 != null) {
                asbNodeLayer.addDisplayObjectType(asbNode2, 150000, 50001);
            } else {
                _debug.warning(
                    "Fehler in LayerManager.initAsbNodeLayer: das Darstellungsobjekt 'ASB-Knotennummern V2' " + "konnte nicht gefunden werden.");
            }
        }

        _layers.add(asbNodeLayer);
        _layerHashMap.put(asbNodeLayer.getName(), asbNodeLayer);
        _unchangeables.add(asbNodeLayer.getName());
    }

    private void initKmLayer() {
        Layer kmLayer =
            new Layer("Betriebskilometrierung", "Die Betriebskilometrierung der StraßenTeilSegmente", new DOTKmPlugin(), "typ.straßenTeilSegment",
                      null, null, null);
        DOTKm km10 = (DOTKm) DOTManager.getInstance().getDisplayObjectType("Betriebskilometrierung 10 m");
        DOTKm km50 = (DOTKm) DOTManager.getInstance().getDisplayObjectType("Betriebskilometrierung 50 m");
        DOTKm km100 = (DOTKm) DOTManager.getInstance().getDisplayObjectType("Betriebskilometrierung 100 m");
        DOTKm km500 = (DOTKm) DOTManager.getInstance().getDisplayObjectType("Betriebskilometrierung 500 m");
        DOTKm km1000 = (DOTKm) DOTManager.getInstance().getDisplayObjectType("Betriebskilometrierung 1 km");
        DOTKm km5000 = (DOTKm) DOTManager.getInstance().getDisplayObjectType("Betriebskilometrierung 5 km");
        DOTKm km10000 = (DOTKm) DOTManager.getInstance().getDisplayObjectType("Betriebskilometrierung 10 km");
        DOTKm km50000 = (DOTKm) DOTManager.getInstance().getDisplayObjectType("Betriebskilometrierung 50 km");
        if (km10 != null) {
            kmLayer.addDisplayObjectType(km10, 450, 1);
        } else {
            _debug.warning(
                "Fehler in LayerManager.initKmLayer: das Darstellungsobjekt 'Betriebskilometrierung 50 km' " + "konnte nicht gefunden werden.");
        }
        if (km50 != null) {
            kmLayer.addDisplayObjectType(km50, 1300, 451);
        } else {
            _debug.warning(
                "Fehler in LayerManager.initKmLayer: das Darstellungsobjekt 'Betriebskilometrierung 100 km' " + "konnte nicht gefunden werden.");
        }
        if (km100 != null) {
            kmLayer.addDisplayObjectType(km100, 3000, 1301);
        } else {
            _debug.warning(
                "Fehler in LayerManager.initKmLayer: das Darstellungsobjekt 'Betriebskilometrierung 100 m' " + "konnte nicht gefunden werden.");
        }
        if (km500 != null) {
            kmLayer.addDisplayObjectType(km500, 10000, 3001);
        } else {
            _debug.warning(
                "Fehler in LayerManager.initKmLayer: das Darstellungsobjekt 'Betriebskilometrierung 500 m' " + "konnte nicht gefunden werden.");
        }
        if (km1000 != null) {
            kmLayer.addDisplayObjectType(km1000, 40000, 10001);
        } else {
            _debug.warning(
                "Fehler in LayerManager.initKmLayer: das Darstellungsobjekt 'Betriebskilometrierung 1 km' " + "konnte nicht gefunden werden.");
        }
        if (km5000 != null) {
            kmLayer.addDisplayObjectType(km5000, 60000, 40001);
        } else {
            _debug.warning(
                "Fehler in LayerManager.initKmLayer: das Darstellungsobjekt 'Betriebskilometrierung 5 km' " + "konnte nicht gefunden werden.");
        }
        if (km10000 != null) {
            kmLayer.addDisplayObjectType(km10000, 500000, 60001);
        } else {
            _debug.warning(
                "Fehler in LayerManager.initKmLayer: das Darstellungsobjekt 'Betriebskilometrierung 10 km' " + "konnte nicht gefunden werden.");
        }
        if (km50000 != null) {
            kmLayer.addDisplayObjectType(km50000, 1000000, 500001);
        } else {
            _debug.warning(
                "Fehler in LayerManager.initKmLayer: das Darstellungsobjekt 'Betriebskilometrierung 10 km' " + "konnte nicht gefunden werden.");
        }

        _layers.add(kmLayer);
        _layerHashMap.put(kmLayer.getName(), kmLayer);
        _unchangeables.add(kmLayer.getName());
    }

    private void initStatLayer() {
        Layer kmLayer =
            new Layer("ASB-Stationierung", "Die ASB-Stationierung der StraßenTeilSegmente", new DOTStatPlugin(), "typ.straßenTeilSegment", null, null,
                      null);
        DOTStat stat10 = (DOTStat) DOTManager.getInstance().getDisplayObjectType("ASB-Stationierung 10 m");
        DOTStat stat50 = (DOTStat) DOTManager.getInstance().getDisplayObjectType("ASB-Stationierung 50 m");
        DOTStat stat100 = (DOTStat) DOTManager.getInstance().getDisplayObjectType("ASB-Stationierung 100 m");
        DOTStat stat500 = (DOTStat) DOTManager.getInstance().getDisplayObjectType("ASB-Stationierung 500 m");
        DOTStat stat1000 = (DOTStat) DOTManager.getInstance().getDisplayObjectType("ASB-Stationierung 1 km");
        DOTStat stat5000 = (DOTStat) DOTManager.getInstance().getDisplayObjectType("ASB-Stationierung 5 km");
        DOTStat stat10000 = (DOTStat) DOTManager.getInstance().getDisplayObjectType("ASB-Stationierung 10 km");
        if (stat10 != null) {
            kmLayer.addDisplayObjectType(stat10, 450, 1);
        } else {
            _debug
                .warning("Fehler in LayerManager.initStatLayer: das Darstellungsobjekt 'ASB-Stationierung 50 km' " + "konnte nicht gefunden werden.");
        }
        if (stat50 != null) {
            kmLayer.addDisplayObjectType(stat50, 1300, 451);
        } else {
            _debug.warning(
                "Fehler in LayerManager.initStatLayer: das Darstellungsobjekt 'ASB-Stationierung 100 km' " + "konnte nicht gefunden werden.");
        }
        if (stat100 != null) {
            kmLayer.addDisplayObjectType(stat100, 3000, 1301);
        } else {
            _debug
                .warning("Fehler in LayerManager.initStatLayer: das Darstellungsobjekt 'ASB-Stationierung 100 m' " + "konnte nicht gefunden werden.");
        }
        if (stat500 != null) {
            kmLayer.addDisplayObjectType(stat500, 10000, 3001);
        } else {
            _debug
                .warning("Fehler in LayerManager.initStatLayer: das Darstellungsobjekt 'ASB-Stationierung 500 m' " + "konnte nicht gefunden werden.");
        }
        if (stat1000 != null) {
            kmLayer.addDisplayObjectType(stat1000, 40000, 10001);
        } else {
            _debug.warning("Fehler in LayerManager.initKmLayer: das Darstellungsobjekt 'ASB-Stationierung 1 km' " + "konnte nicht gefunden werden.");
        }
        if (stat5000 != null) {
            kmLayer.addDisplayObjectType(stat5000, 60000, 40001);
        } else {
            _debug.warning("Fehler in LayerManager.initKmLayer: das Darstellungsobjekt 'ASB-Stationierung 5 km' " + "konnte nicht gefunden werden.");
        }
        if (stat10000 != null) {
            kmLayer.addDisplayObjectType(stat10000, 70000, 60001);
        } else {
            _debug.warning("Fehler in LayerManager.initKmLayer: das Darstellungsobjekt 'ASB-Stationierung 10 km' " + "konnte nicht gefunden werden.");
        }
        _layers.add(kmLayer);
        _layerHashMap.put(kmLayer.getName(), kmLayer);
        _unchangeables.add(kmLayer.getName());
    }

    private void initRnLayer() {
        Layer rnLayer = new Layer("Autobahnschilder", "Die Autobahnschilder", new DOTRnPlugin(), "typ.äußeresStraßenSegment", null, null, null);
        {
            String s = "Autobahnschilder Kategorie 0";
            DOTRn dotRn0 = (DOTRn) DOTManager.getInstance().getDisplayObjectType(s);
            if (dotRn0 != null) {
                rnLayer.addDisplayObjectType(dotRn0, 1000000, 250000);
            } else {
                _debug.warning("Fehler in LayerManager.initRnLayer: der Darstellungsobjekttyp '" + s + "' konnte nicht gefunden werden.");
            }
        }
        {
            String s = "Autobahnschilder Kategorie 0 bis 1";
            DOTRn dotRn1 = (DOTRn) DOTManager.getInstance().getDisplayObjectType(s);
            if (dotRn1 != null) {
                rnLayer.addDisplayObjectType(dotRn1, 249999, 100000);
            } else {
                _debug.warning("Fehler in LayerManager.initRnLayer: der Darstellungsobjekttyp '" + s + "' konnte nicht gefunden werden.");
            }
        }
        {
            String s = "Autobahnschilder Kategorie 0 bis 2";
            DOTRn dotRn2 = (DOTRn) DOTManager.getInstance().getDisplayObjectType(s);
            if (dotRn2 != null) {
                rnLayer.addDisplayObjectType(dotRn2, 99999, 50000);
            } else {
                _debug.warning("Fehler in LayerManager.initRnLayer: der Darstellungsobjekttyp '" + s + "' konnte nicht gefunden werden.");
            }
        }
        {
            String s = "Autobahnschilder Kategorie 0 bis 3";
            DOTRn dotRn3 = (DOTRn) DOTManager.getInstance().getDisplayObjectType(s);
            if (dotRn3 != null) {
                rnLayer.addDisplayObjectType(dotRn3, 49999, 20000);
            } else {
                _debug.warning("Fehler in LayerManager.initRnLayer: der Darstellungsobjekttyp '" + s + "' konnte nicht gefunden werden.");
            }
        }
        {
            String s = "Autobahnschilder Kategorie 0 bis 4";
            DOTRn dotRn4 = (DOTRn) DOTManager.getInstance().getDisplayObjectType(s);
            if (dotRn4 != null) {
                rnLayer.addDisplayObjectType(dotRn4, 19999, 1);
            } else {
                _debug.warning("Fehler in LayerManager.initRnLayer: der Darstellungsobjekttyp '" + s + "' konnte nicht gefunden werden.");
            }
        }
        _layers.add(rnLayer);
        _layerHashMap.put(rnLayer.getName(), rnLayer);
        _unchangeables.add(rnLayer.getName());
    }

    private void initUserDefinedLayers() {
        Preferences classPrefs = getPreferenceStartPath();
        String[] layers;
        try {
            layers = classPrefs.childrenNames();
        } catch (BackingStoreException | IllegalStateException ignored) {
            PreferencesDeleter pd = new PreferencesDeleter("Die benutzer-definierten Layer können nicht geladen werden.", classPrefs);
            pd.run();
            return;
        }
        for (String layerName : layers) {
            Preferences layerPrefs = classPrefs.node(layerName);
            Layer layer = new Layer();
            if (layer.initializeFromPreferences(layerPrefs)) {
                _layers.add(layer);
                _layerHashMap.put(layer.getName(), layer);
            }
        }
    }

    private int remove(final String layerName) {
        int index = 0;
        for (Layer layer : _layers) {
            if (layerName.equals(layer.getName())) {
                _layers.remove(index);
                _layerHashMap.remove(layerName);
                return index;
            }
            index++;
        }
        return -1;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("[LayerManager: ");
        for (Layer layer : _layers) {
            sb.append(layer.toString());
        }
        sb.append("[Unveränderebare Layer: ");
        for (String name : _unchangeables) {
            sb.append("[").append(name).append("]");
        }
        sb.append("]");
        return sb.toString();
    }

    /**
     * Gibt {@code true} zurück, wenn der Layer veränderbar ist. Im Moment ist ein Layer genau dann unveränderbar, wenn er im Kode definiert ist.
     *
     * @param layer ein Layer
     *
     * @return {@code true} genau dann, wenn der Layer veränderbar ist
     */
    public boolean isChangeable(Layer layer) {
        return !_unchangeables.contains(layer.getName());
    }

    /**
     * Fügt das übergebene Objekt der Liste der auf Layeränderungen angemeldeten Objekte hinzu.
     *
     * @param listener ein Listener
     */
    public void addChangeListener(LayerManagerChangeListener listener) {
        _listeners.add(listener);
    }

    /**
     * Entfernt das übergebene Objekt aus der Liste der auf Layeränderungen angemeldeten Objekte.
     *
     * @param listener ein Listener
     */
    public void removeChangeListener(LayerManagerChangeListener listener) {
        _listeners.remove(listener);
    }

    /**
     * Informiert die auf Layeränderungen angemeldeten Objekte über einen neu hinzugefügten Layer.
     *
     * @param layer ein Layer
     */
    private void notifyChangeListenersLayerAdded(final Layer layer) {
        for (LayerManagerChangeListener changeListener : _listeners) {
            changeListener.layerAdded(layer);
        }
    }

    /**
     * Informiert die auf Layeränderungen angemeldeten Objekte über einen substanziell geänderten Layer.
     *
     * @param layer ein Layer
     */
    private void notifyChangeListenersLayerDefinitionChanged(final Layer layer) {
        for (LayerManagerChangeListener changeListener : _listeners) {
            changeListener.layerDefinitionChanged(layer);
        }
    }

    /**
     * Informiert die auf Layeränderungen angemeldeten Objekte über einen marginal geänderten Layer.
     *
     * @param layer ein Layer
     */
    private void notifyChangeListenersLayerPropertyChanged(final Layer layer) {
        for (LayerManagerChangeListener changeListener : _listeners) {
            changeListener.layerPropertyChanged(layer);
        }
    }

    /**
     * Informiert die auf Layeränderungen angemeldeten Objekte über einen gelöschten Layer.
     *
     * @param layerName ein Layername
     */
    private void notifyChangeListenersLayerRemoved(final String layerName) {
        for (LayerManagerChangeListener changeListener : _listeners) {
            changeListener.layerRemoved(layerName);
        }
    }

    /**
     * Gibt {@code true} zurück, falls der durch den Namen angegebene Darstellungstyp von einem der Layer verwendet wrd.
     *
     * @param displayObjectTypeName der Name eines Darstellungstyps
     *
     * @return {@code true} genau dann, wenn der Darstellungstyp benutzt wird
     */
    public boolean displayObjectTypeIsUsed(final String displayObjectTypeName) {
        for (Layer layer : _layers) {
            if (layer.getDotCollection().displayObjectTypeIsUsed(displayObjectTypeName)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Gibt die Namen all der Layer zurück, die den durch den Namen angegebene Darstellungstyp verwenden.
     *
     * @param displayObjectTypeName der Name eines Darstellungstyps
     *
     * @return eine Liste von Darstellungstypen
     */
    public List<String> getLayersUsingTheDisplayObjectType(final String displayObjectTypeName) {
        List<String> returnList = new ArrayList<>();
        for (Layer layer : _layers) {
            if (layer.getDotCollection().displayObjectTypeIsUsed(displayObjectTypeName)) {
                returnList.add(layer.getName());
            }
        }
        return returnList;
    }

    /**
     * Ein Interface für Listener, die über das Hinzufügen, Löschen und Ändern von Layern informiert werden wollen.
     *
     * @author Kappich Systemberatung
     */
    public interface LayerManagerChangeListener {
        /**
         * Diese Methode wird aufgerufen, wenn der Layer hinzugefügt wurde.
         *
         * @param layer ein Layer
         */
        void layerAdded(final Layer layer);

        /**
         * Diese Methode wird aufgerufen, wenn der Layer substanziell geändert wurde.
         *
         * @param layer ein Layer
         */
        void layerDefinitionChanged(final Layer layer);

        /**
         * Diese Methode wird aufgerufen, wenn der Layer marginal geändert wurde.
         *
         * @param layer ein Layer
         */
        void layerPropertyChanged(final Layer layer);

        /**
         * Diese Methode wird aufgerufen, wenn der Layer gelöscht wurde.
         *
         * @param layerName ein Layername
         */
        void layerRemoved(final String layerName);
    }
}
