/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.pluginInterfaces;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.displayObjectToolkit.DOTDefinitionDialog;
import de.kappich.pat.gnd.displayObjectToolkit.DOTItemManager;
import de.kappich.pat.gnd.displayObjectToolkit.DynamicDOTItem;
import de.kappich.pat.gnd.displayObjectToolkit.DynamicDOTItemManager;
import de.kappich.pat.gnd.displayObjectToolkit.DynamicDefinitionComponent;
import de.kappich.pat.gnd.kmPlugin.DOTKm;
import de.kappich.pat.gnd.properties.Property;
import de.kappich.pat.gnd.properties.PropertyPanel;
import de.kappich.pat.gnd.properties.PropertyPanelListener;
import de.kappich.pat.gnd.utils.SpringUtilities;
import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.ActionListener;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSpinner;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SpinnerModel;
import javax.swing.SpinnerNumberModel;
import javax.swing.SpringLayout;
import javax.swing.event.ChangeListener;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.TableModel;
import javax.swing.table.TableRowSorter;

/**
 * Ein DefaultDOTPanel ist eine Klasse, die eine Implementation von DOTDefinitionPanel ist.
 *
 * @author Kappich Systemberatung
 */
public class DefaultDOTDefinitionPanel implements DOTDefinitionPanel {

    private static final Debug _debug = Debug.getLogger();

    private final DisplayObjectTypePlugin _plugin;
    private final DOTDefinitionDialog _dotDefinitionDialog;
    private final boolean _staticOnly;
    private final DataModel _configuration;
    private final Map<Property, PropertyPanel> _staticPropertyPanels = new HashMap<>();
    private final Map<Property, JPanel> _dynamicPropertyPanels = new HashMap<>();
    private final Map<Property, JTable> _dynamicPropertyTables = new HashMap<>();

    /**
     * Konstruiert den Dialog.
     *
     * @param plugin              das zugehörige Plugin
     * @param dotDefinitionDialog der umgebende Rahmen
     */
    public DefaultDOTDefinitionPanel(final DisplayObjectTypePlugin plugin, final DOTDefinitionDialog dotDefinitionDialog) {
        _plugin = plugin;
        _dotDefinitionDialog = dotDefinitionDialog;
        _staticOnly = !_plugin.isDynamicsPossible();
        final ClientDavInterface connection = _dotDefinitionDialog.getConnection();
        _configuration = connection.getDataModel();
        if (!(dotDefinitionDialog.getDisplayObjectType() instanceof DefaultDisplayObjectType)) {
            throw new IllegalArgumentException();
        }
        // Die Strategie ist bei den folgenden Membern wie folgt: es werden alle Center-Panels,
        // also alle statischen und nicht-statischen initialisiert, damit beim Umschalten möglichst
        // wenig Informationen verloren gehen. Das Scratch-DOT dient den Dialogen zunächst zur
        // Initialisierung, und anschließend werden alle Änderungen, die duch die Dialoge
        // vorgenommen werden, dort nachgezogen. Beim Speichern müssen die Informationen aus
        // beiden Quellen kombiniert werden.
        initAllPanelsAndTables();
    }

    private void initAllPanelsAndTables() {
        _staticPropertyPanels.clear();
        if (!_staticOnly) {
            _dynamicPropertyPanels.clear();
            _dynamicPropertyTables.clear();
        }
        Property[] properties = _plugin.getProperties(null);
        if (properties != null) {
            for (Property property : properties) {
                initStaticPanel(property);
                if (!_staticOnly) {
                    initDynamicPanel(property);
                }
            }
        }
    }

    @SuppressWarnings("OverlyLongMethod")
    private void addButtonListeners(final Property property, final JButton newDOTItemButton, final JButton deleteDOTItemButton,
                                    final JButton showConflictsButton) {

        @SuppressWarnings("serial")
        final class DOTItemDialog extends JDialog {

            private final Property _property;

            private final DynamicDefinitionComponent _dynamicDefinitionComponent;

            private final PropertyPanel _propertyPanel;

            private DOTItemDialog() {
                super();
                _property = property;
                _dynamicDefinitionComponent = new DynamicDefinitionComponent(_configuration);
                final Object propertyValue = initDynamicDefinitionComponent();
                _propertyPanel = _property.getPropertyPanel(propertyValue, _dotDefinitionDialog.isEditable());

                JButton saveButton = new JButton("Speichern");
                saveButton.setEnabled(_dotDefinitionDialog.isEditable());
                JButton cancelButton = new JButton("Dialog schließen");

                JPanel buttonsPanel = new JPanel();
                buttonsPanel.setLayout(new SpringLayout());

                buttonsPanel.add(saveButton);
                buttonsPanel.add(cancelButton);

                buttonsPanel.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));
                SpringUtilities.makeCompactGrid(buttonsPanel, 2, 20, 5);
                addButtonListeners(saveButton, cancelButton);

                setTitle("GND: Darstellungsfestlegung für Linien");
                setLayout(new BorderLayout());
                add(new JScrollPane((JPanel) _propertyPanel), BorderLayout.NORTH);
                add(new JScrollPane(_dynamicDefinitionComponent), BorderLayout.CENTER);
                add(buttonsPanel, BorderLayout.SOUTH);
                pack();
                setSize(700, 550);
                final Rectangle bounds = getBounds();
                setLocation(new Point((int) (bounds.getHeight() / 1.5 + 300), (int) (bounds.getWidth() / 1.5)));
            }

            private void runDialog() {
                setVisible(true);
            }

            @SuppressWarnings("unchecked")
            @Nullable
            private Object initDynamicDefinitionComponent() {
                JTable workWithThisTable = _dynamicPropertyTables.get(_property);
                if (workWithThisTable == null) {
                    return null;
                }

                int selectedRow = workWithThisTable.getSelectedRow();
                if (selectedRow == -1) {
                    if (workWithThisTable.getModel().getRowCount() > 0) {
                        selectedRow = 0;
                    } else {
                        return null;
                    }
                }
                selectedRow = workWithThisTable.convertRowIndexToModel(selectedRow);
                final TableModel model = workWithThisTable.getModel();
                DynamicDOTItemManager dynamicDOTItemManager = (DynamicDOTItemManager) model;
                final DOTItemManager<DynamicDOTItem>.DisplayObjectTypeItemWithInterval dotItemWithInterval = dynamicDOTItemManager.get(selectedRow);
                _dynamicDefinitionComponent.fillComponents(dotItemWithInterval);
                return dotItemWithInterval.getItem().getPropertyValue();
            }

            private void addButtonListeners(JButton saveButton, JButton cancelButton) {
                ActionListener actionListenerSave = e -> {

                    final String description = _dynamicDefinitionComponent.getInfoText();
                    if (description == null) {
                        JOptionPane.showMessageDialog(new JFrame(), "Bitte geben Sie einen Info-Text ein!", "Fehler", JOptionPane.ERROR_MESSAGE);
                        return;
                    }

                    final Object propertyValue = _propertyPanel.getPropertyValue();
                    if (null == propertyValue) {
                        JOptionPane.showMessageDialog(new JFrame(), "Bitte wählen Sie einen Wert zur Eigenschaft '" + _property.getName() + "'aus!",
                                                      "Fehler", JOptionPane.ERROR_MESSAGE);
                        return;
                    }

                    final String attributeGroupName = _dynamicDefinitionComponent.getAttributeGroupName();
                    if (attributeGroupName == null) {
                        JOptionPane.showMessageDialog(new JFrame(), "Bitte wählen Sie eine Attributgruppe aus!", "Fehler", JOptionPane.ERROR_MESSAGE);
                        return;
                    }

                    final String aspectName = _dynamicDefinitionComponent.getAspectName();
                    if (aspectName == null) {
                        JOptionPane.showMessageDialog(new JFrame(), "Bitte wählen Sie einen Aspekt aus!", "Fehler", JOptionPane.ERROR_MESSAGE);
                        return;
                    }

                    final String attributeName = _dynamicDefinitionComponent.getAttributeName();
                    if (attributeName == null) {
                        int error = _dynamicDefinitionComponent.checkAttributeName();
                        if (error == 1) {
                            JOptionPane.showMessageDialog(new JFrame(), "Bitte wählen Sie ein Attribut aus!", "Fehler", JOptionPane.ERROR_MESSAGE);
                            return;
                        } else if (error == 2) {
                            JOptionPane.showMessageDialog(new JFrame(), "Der Attributname ist ungültig!", "Fehler", JOptionPane.ERROR_MESSAGE);
                            return;
                        }
                        return; // kann gar nicht passieren; ergänzt, damit IDE attributeName != null erkennt
                    }

                    final Double fromValue = _dynamicDefinitionComponent.getFromValue();
                    if (fromValue == null || fromValue.isNaN()) {
                        int error = _dynamicDefinitionComponent.checkFromValue();
                        if (error == 1) {
                            JOptionPane.showMessageDialog(new JFrame(), "Bitte tragen Sie einen Von-Wert ein!", "Fehler", JOptionPane.ERROR_MESSAGE);
                            return;
                        } else if ((error == 2) || (error == 3)) {
                            JOptionPane.showMessageDialog(new JFrame(), "Bitte korrigieren Sie den Von-Wert!", "Fehler", JOptionPane.ERROR_MESSAGE);
                            return;
                        }
                    }

                    final Double toValue = _dynamicDefinitionComponent.getToValue();
                    if (toValue == null || toValue.isNaN()) {
                        int error = _dynamicDefinitionComponent.checkToValue();
                        if (error == 1) {
                            JOptionPane.showMessageDialog(new JFrame(), "Bitte tragen Sie einen Bis-Wert ein!", "Fehler", JOptionPane.ERROR_MESSAGE);
                            return;
                        } else if ((error == 2) || (error == 3)) {
                            JOptionPane.showMessageDialog(new JFrame(), "Bitte korrigieren Sie den Bis-Wert!", "Fehler", JOptionPane.ERROR_MESSAGE);
                            return;
                        }
                    }

                    if ((fromValue != null) && (toValue != null) && (fromValue > toValue)) {
                        JOptionPane.showMessageDialog(new JFrame(), "Der Von-Wert ist größer als der Bis-Wert!", "Fehler", JOptionPane.ERROR_MESSAGE);
                        return;
                    }

                    DynamicDOTItem dItem = new DynamicDOTItem(attributeGroupName, aspectName, attributeName, description, propertyValue);
                    _dotDefinitionDialog.getScratchDisplayObjectType().setValueOfDynamicProperty(null, _property, dItem, fromValue, toValue);
                    _dotDefinitionDialog.setSomethingChanged(true);
                };
                saveButton.addActionListener(actionListenerSave);

                ActionListener actionListenerCancel = e -> dispose();
                cancelButton.addActionListener(actionListenerCancel);
            }

            @Override
            public String toString() {
                return "DOTItemDialog{" + "_property=" + _property + '}';
            }
        }

        ActionListener actionListenerNewButton = e -> {
            DOTItemDialog dotItemDialog = new DOTItemDialog();
            dotItemDialog.runDialog();
        };
        newDOTItemButton.addActionListener(actionListenerNewButton);

        ActionListener actionListenerDeleteButton = e -> {
            final Property currentProperty = _dotDefinitionDialog.getSelectedProperty();
            JTable workForThisTable = _dynamicPropertyTables.get(currentProperty);
            if (workForThisTable == null) {
                return;
            }
            final int[] selectedRows = workForThisTable.getSelectedRows();
            if (selectedRows.length == 0) {
                JOptionPane.showMessageDialog(new JFrame(), "Bitte wählen Sie eine Zeile aus!", "Fehler", JOptionPane.ERROR_MESSAGE);
                return;
            }
            final TableModel model = workForThisTable.getModel();
            DynamicDOTItemManager dynamicDOTItemManager = (DynamicDOTItemManager) model;
            Set<Integer> selectedModelIndexes = new HashSet<>();
            for (final int selectedRow : selectedRows) {
                selectedModelIndexes.add(workForThisTable.convertRowIndexToModel(selectedRow));
            }
            final Object[] array = selectedModelIndexes.toArray();
            for (int i = array.length - 1; i >= 0; i--) {
                dynamicDOTItemManager.remove((Integer) array[i]);
                _dotDefinitionDialog.setSomethingChanged(true);
            }
        };
        deleteDOTItemButton.addActionListener(actionListenerDeleteButton);

        ActionListener actionListenerShowConflictsButton = e -> {
            final Property currentProperty = _dotDefinitionDialog.getSelectedProperty();
            JTable workForThisTable = _dynamicPropertyTables.get(currentProperty);
            if (workForThisTable == null) {
                return;
            }
            workForThisTable.clearSelection();

            DefaultDisplayObjectType dot = (DefaultDisplayObjectType) _dotDefinitionDialog.getScratchDisplayObjectType();
            final Set<Integer> conflictingRows = dot.getConflictingRows(property);
            ListSelectionModel listSelectionModel = workForThisTable.getSelectionModel();
            if (conflictingRows != null) {
                for (Integer row : conflictingRows) {
                    Integer rowInView = workForThisTable.convertRowIndexToView(row);
                    listSelectionModel.addSelectionInterval(rowInView, rowInView);
                }
                workForThisTable.setSelectionModel(listSelectionModel);
            }
        };
        showConflictsButton.addActionListener(actionListenerShowConflictsButton);
    }

    private void addListSelectionListener(final JTable table, final JButton deleteDOTItemButton) {
        final ListSelectionModel selectionModel = table.getSelectionModel();
        ListSelectionListener listSelectionListener = e -> {
            final int[] selectedRows = table.getSelectedRows();
            if (selectedRows.length == 0) {
                deleteDOTItemButton.setEnabled(false);
            } else {
                deleteDOTItemButton.setEnabled(_dotDefinitionDialog.isEditable());
                if (selectedRows.length == 1) {
                    deleteDOTItemButton.setText("Zeile löschen");
                } else {
                    deleteDOTItemButton.setText("Zeilen löschen");
                }
            }
        };
        selectionModel.addListSelectionListener(listSelectionListener);
    }

    @Override
    @Nullable
    public JPanel getDOTItemDefinitionPanel() {
        final Property property = _dotDefinitionDialog.getSelectedProperty();
        if (property == null) {
            JOptionPane.showMessageDialog(new JFrame(), "Bitte wählen Sie eine Eigenschaft aus!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return null;
        }
        if (_dotDefinitionDialog.isPropertyStatic(null, property)) {
            return (JPanel) getStaticCenterPanel();
        } else {
            return getNonStaticCenterPanel();
        }
    }

    @Override
    public boolean saveDisplayObjectType() {
        DefaultDisplayObjectType newDot = (DefaultDisplayObjectType) _dotDefinitionDialog.getScratchDisplayObjectType().getCopy(null);
        final String name = _dotDefinitionDialog.getNameText();
        if ((name == null) || (name.isEmpty())) {
            JOptionPane.showMessageDialog(new JFrame(), "Bitte geben Sie einen Namen an!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return false;
        }
        if (!_dotDefinitionDialog.isReviseOnly()) {
            if (_dotDefinitionDialog.getDotManager().containsDisplayObjectType(name)) {
                JOptionPane
                    .showMessageDialog(new JFrame(), "Ein Darstellungstyp mit diesem Namen existiert bereits!", "Fehler", JOptionPane.ERROR_MESSAGE);
                return false;
            }
        }
        newDot.setName(name);
        newDot.setInfo(_dotDefinitionDialog.getInfoText());
        _dotDefinitionDialog.getDotManager().saveDisplayObjectType(newDot);
        _dotDefinitionDialog.setDisplayObjectType(newDot, true);
        return true;
    }

    @Nullable
    private PropertyPanel getStaticCenterPanel() {
        final Property property = _dotDefinitionDialog.getSelectedProperty();
        return _staticPropertyPanels.get(property);
    }

    private void initStaticPanel(Property property) {
        final Object value;
        if (_dotDefinitionDialog.isPropertyStatic(null, property)) {
            value = _dotDefinitionDialog.getValueOfStaticProperty(null, property);
        } else {
            value = property.getDefaultValue();
        }
        PropertyPanel propertyPanel = property.getPropertyPanel(value, _dotDefinitionDialog.isEditable());
        if (propertyPanel instanceof JPanel) {
            ((JPanel) propertyPanel).setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
        }
        _staticPropertyPanels.put(property, propertyPanel);

        PropertyPanelListener listener = () -> {
            final Object selectedItem = propertyPanel.getPropertyValue();
            _dotDefinitionDialog.getScratchDisplayObjectType().setValueOfStaticProperty(null, property, selectedItem);
        };
        propertyPanel.addListener(listener);
    }

    @Nullable
    private JPanel getNonStaticCenterPanel() {
        final Property property = _dotDefinitionDialog.getSelectedProperty();
        JPanel panel = _dynamicPropertyPanels.get(property);
        if (panel != null) {
            return panel;
        } else {
            _debug.warning("DefaultDOTDefinitionPanel: Die Eigenschaft " + property.getName() + " wird nicht unterstützt.");
            return null;
        }
    }

    private void initDynamicPanel(final Property property) {
        JPanel panel = new JPanel();
        JTable table = new JTable();

        DefaultDisplayObjectType dot = (DefaultDisplayObjectType) _dotDefinitionDialog.getScratchDisplayObjectType();
        final DynamicDOTItemManager tableModel = (DynamicDOTItemManager) dot.getTableModel(property);
        table.setModel(tableModel);

        class NumberComparator implements Comparator<Number> {
            @Override
            public int compare(Number o1, Number o2) {
                final double d1 = o1.doubleValue();
                final double d2 = o2.doubleValue();
                return Double.compare(d1, d2);
            }
        }

        TableRowSorter<DynamicDOTItemManager> tableRowSorter = new TableRowSorter<>();
        tableRowSorter.setModel(tableModel);
        tableRowSorter.setComparator(4, new NumberComparator());
        tableRowSorter.setComparator(5, new NumberComparator());
        table.setRowSorter(tableRowSorter);

        JButton newDOTItemButton = new JButton("Neue Zeile");
        newDOTItemButton.setEnabled(_dotDefinitionDialog.isEditable());
        JButton deleteDOTItemButton = new JButton("Löschen");
        deleteDOTItemButton.setEnabled(false);
        JButton showConflictsButton = new JButton("Zeige Konflikte");

        addButtonListeners(property, newDOTItemButton, deleteDOTItemButton, showConflictsButton);
        addListSelectionListener(table, deleteDOTItemButton);

        JPanel dotButtonsPanel = new JPanel();
        dotButtonsPanel.setLayout(new SpringLayout());

        dotButtonsPanel.add(newDOTItemButton);
        dotButtonsPanel.add(deleteDOTItemButton);
        dotButtonsPanel.add(showConflictsButton);

        dotButtonsPanel.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));
        SpringUtilities.makeCompactGrid(dotButtonsPanel, 1, 5, 20);

        panel.setLayout(new SpringLayout());
        panel.setBorder(BorderFactory.createTitledBorder("Werte"));
        panel.add(new JScrollPane(table));
        panel.add(dotButtonsPanel);
        SpringUtilities.makeCompactGrid(panel, 2, 20, 5);

        /* Hier wird die bevorzugte Größe des Panels vorgegeben; als Breite wird die bevorzugte Breite
         *  des Tables genommen, als Höhe mindestens 160 Pixel wegen der drei Buttons auf der rechten
         *  Seite, und mindestens die bevorzugte Höhe des Tables plus der Höhe für zwei weitere Zeilen. */
        final Dimension tableDim = table.getPreferredSize();
        panel.setPreferredSize(new Dimension(tableDim.width, Math.max(tableDim.height + 50, 160)));

        JPanel outerPanel = new JPanel();
        outerPanel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
        outerPanel.add(panel);

        _dynamicPropertyPanels.put(property, outerPanel);
        _dynamicPropertyTables.put(property, table);
    }

    @Nullable
    @Override
    public JPanel getAdditionalCharacteristicsPanel(final DisplayObjectType displayObjectType) {
	    if (displayObjectType instanceof DOTKm dotKm) {
            final JSpinner _translationFactorSpinner = new JSpinner();
            final JCheckBox _joinByLineCheckBox = new JCheckBox();
            JLabel translationFactorLabel = new JLabel("Verschiebungsfaktor: ");
            SpinnerModel spinnerModel = new SpinnerNumberModel(100, 0, 5000, 1);
            _translationFactorSpinner.setModel(spinnerModel);
            _translationFactorSpinner.setMaximumSize(new Dimension(60, 30));
            _translationFactorSpinner.setEnabled(_dotDefinitionDialog.isEditable());
            JPanel translationPanel = new JPanel();
            translationPanel.setLayout(new BoxLayout(translationPanel, BoxLayout.X_AXIS));
            translationPanel.setBorder(BorderFactory.createEmptyBorder(0, 0, 10, 10));
            translationPanel.add(translationFactorLabel);
            translationPanel.add(_translationFactorSpinner);

            JLabel joinByLineLabel = new JLabel("Verbindungslinie: ");
            _joinByLineCheckBox.setSelected(false);
            _joinByLineCheckBox.setEnabled(_dotDefinitionDialog.isEditable());
            JPanel joinByLinePanel = new JPanel();
            joinByLinePanel.setLayout(new BoxLayout(joinByLinePanel, BoxLayout.X_AXIS));
            joinByLinePanel.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
            joinByLinePanel.add(joinByLineLabel);
            joinByLinePanel.add(_joinByLineCheckBox);

            JPanel invisiblePanel = new JPanel();
            invisiblePanel.add(new JTextField());
            invisiblePanel.setVisible(false);

            JPanel thePanel = new JPanel();
            thePanel.setBorder(BorderFactory.createEmptyBorder(10, 10, 0, 10));
            thePanel.setLayout(new SpringLayout());
            thePanel.add(translationPanel);
            thePanel.add(joinByLinePanel);
            thePanel.add(invisiblePanel);
            SpringUtilities.makeCompactGrid(thePanel, 3, 5, 5);

            _translationFactorSpinner.setValue(dotKm.getTranslationFactor());
            _joinByLineCheckBox.setSelected(dotKm.isJoinedByLine());
            // Erst jetzt die Listener, sonst werden die Setzungen von _translationFactorSpinner und _joinByLineCheckBox zuvor verarbeitet!
            ChangeListener translationChangeListener = e -> _dotDefinitionDialog.setSomethingChanged(true);
            _translationFactorSpinner.addChangeListener(translationChangeListener);
            ChangeListener joinByLineChangeListener = e -> _dotDefinitionDialog.setSomethingChanged(true);
            _joinByLineCheckBox.addChangeListener(joinByLineChangeListener);

            return thePanel;
        }
        return null;
    }

    @Override
    public String toString() {
        return "DefaultDOTDefinitionPanel{}";
    }
}
