/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.kappich.pat.gnd.pointPlugin;

import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.displayObjectToolkit.*;
import de.kappich.pat.gnd.gnd.LegendTreeNodes;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectTypePlugin;
import de.kappich.pat.gnd.properties.ColorProperty;
import de.kappich.pat.gnd.properties.FillingProperty;
import de.kappich.pat.gnd.properties.PropertiesManager;
import de.kappich.pat.gnd.properties.Property;
import de.kappich.pat.gnd.properties.TextSizeProperty;
import de.kappich.pat.gnd.properties.TextStyleProperty;
import de.kappich.pat.gnd.utils.Interval;
import de.kappich.pat.gnd.utils.view.PreferencesDeleter;
import java.awt.Color;
import java.awt.Font;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import javax.swing.JOptionPane;
import javax.swing.table.TableModel;

/**
 * Der Darstellungstyp für Punktobjekte.
 * <p>
 * Ein DOTPoint implementiert das Interface DisplayObjectType für das Plugin für Punktobjekte. Dieser GND-interne Darstellungstyp ist bei weitem der
 * umfangreichste und seine Implementation beruht NICHT auf DefaultDisplayObjectType.
 * <p>
 * Jeder DOTPoint hat einen Namen, einen Infotext, einen Verschiebungsfaktor (-länge) und eine interne Variable, die anzeigt, ob eine Verbingslinie
 * zwischen der Lage in der Karte und dem verschobenen Objekt gezeichnet werden soll. Weiterhin kann er beliebig viele {@link PrimitiveForm
 * Grundfiguren} enthalten, die je nach ihrem Typ statische oder dynamische Eigenschaften besitzen. Der DOTPoint besitzt selbst keine
 * Visualisierungs-Eigenschaften.
 *
 * @author Kappich Systemberatung
 */
public class DOTPoint implements DisplayObjectType, DOTManager.DOTChangeListener {

    static final Debug _debug = Debug.getLogger();
    private static final String LOWER_BOUND = "LOWER_BOUND";
    private static final String UPPER_BOUND = "UPPER_BOUND";
    private static final String TYPE = "TYPE";
    private static final String VALUE = "VALUE";
    private static final String INFO = "INFO";
    private static final String TRANSLATION_X = "TRANSLATION_X";
    private static final String TRANSLATION_Y = "TRANSLATION_Y";
    private static final String JOIN_BY_LINE = "JOIN_BY_LINE";
    private static final String TRANSLATION_FACTOR = "TRANSLATION_FACTOR";
    private final Map<String, PrimitiveForm> _primitiveForms = new HashMap<>();
    // Members von DOTPoint
    private String _name;
    private String _info;
    private Double _translationFactor;
    private Boolean _joinByLine;

    /**
     * Ein Konstruktor ohne Vorgaben.
     */
    public DOTPoint() {
        super();
        _name = "";
        _info = "";
        _translationFactor = 0.;
        _joinByLine = false;
        DOTManager.getInstance().addDOTChangeListener(this);
    }

    /**
     * Ein Konstruktor mit punkt-spezifischen Vorgaben.
     *
     * @param name              der Name des Darstellungstyp-Objekts
     * @param info              die Kurzinfo des Darstellungstyp-Objekts
     * @param translationFactor der globale Verschiebungsfaktor
     * @param joinByLine        {@code true} genau dann, wenn eine Verbindungslinie gezeichnet werden soll
     */
    public DOTPoint(String name, String info, double translationFactor, boolean joinByLine) {
        super();
        _name = name;
        _info = info;
        _translationFactor = translationFactor;
        _joinByLine = joinByLine;
        DOTManager.getInstance().addDOTChangeListener(this);
    }

    @Override
    public String getName() {
        return _name;
    }

    @Override
    public void setName(String name) {
        _name = name;
    }

    @Override
    public String getInfo() {
        return _info;
    }

    @Override
    public void setInfo(String info) {
        _info = info;
    }

    /**
     * Der Getter für den Verschiebungsfaktor bzw. -länge.
     *
     * @return der Verschiebungsfaktor
     */
    public Double getTranslationFactor() {
        return _translationFactor;
    }

    /**
     * Der Setter für den Verschiebungsfaktor bzw. -länge.
     *
     * @param translationFactor der Verschiebungsfaktor
     */
    public void setTranslationFactor(Double translationFactor) {
        _translationFactor = translationFactor;
    }

    /**
     * Gibt {@code true} zurück, wenn die Lage in der Karte mit dem verschobenen Objekt durch eine Verbindungslinie verbunden werden soll.
     *
     * @return soll eine Verbindungslinie gezeichnet werden
     */
    public boolean getJoinByLine() {
        return _joinByLine;
    }

    /**
     * Setzt die interne Variable, die bestimmt, ob die Lage in der Karte mit dem verschobenen Objekt durch eine Verbindungslinie verbunden werden
     * soll.
     *
     * @param joinByLine legt fest, ob eine Verbindungslinie gezeichnet werden soll
     */
    public void setJoinByLine(boolean joinByLine) {
        _joinByLine = joinByLine;
    }

    @Override
    public boolean isPropertyStatic(String primitiveFormName, Property property) {
        PrimitiveForm primitiveForm = getPrimitiveForm(primitiveFormName, "isPropertyStatic");
        final Boolean isStatic = primitiveForm.isPropertyStatic(property);
        if (isStatic == null) {
            throw new IllegalArgumentException("DOTPoint.isPropertyStatic wurde für eine unbekannte Eigenschaft aufgerufen.");
        }
        return isStatic;
    }

    @Override
    public void setPropertyStatic(String primitiveFormName, Property property, boolean b) {
        PrimitiveForm primitiveForm = getPrimitiveForm(primitiveFormName, "setPropertyStatic");
        primitiveForm.setPropertyStatic(property, b);
    }

    @Override
    @Nullable
    public Object getValueOfStaticProperty(String primitiveFormName, Property property) {
        PrimitiveForm primitiveForm = getPrimitiveForm(primitiveFormName, "getValueOfStaticProperty");
        if (primitiveForm == null) {
            return null;
        }
        return primitiveForm.getValueOfStaticProperty(property);
    }

    @Override
    public void setValueOfStaticProperty(String primitiveFormName, Property property, Object value) {
        PrimitiveForm primitiveForm = getPrimitiveForm(primitiveFormName, "setValueOfStaticProperty");
        primitiveForm.setValueOfStaticProperty(property, value);
    }

    @Override
    public void setValueOfDynamicProperty(String primitiveFormName, Property property, DisplayObjectTypeItem dItem, Double lowerBound,
                                          Double upperBound) {
        PrimitiveForm primitiveForm = getPrimitiveForm(primitiveFormName, "setValueOfDynamicProperty");
        primitiveForm.setValueOfDynamicProperty(property, dItem, lowerBound, upperBound);
    }

    @Override
    public void initializeFromPreferences(Preferences prefs) {
        _name = prefs.name();
        Preferences infoPrefs = prefs.node("info");
        _info = infoPrefs.get(INFO, "");
        Preferences translationFactorPrefs = prefs.node("translationFactor");
        _translationFactor = translationFactorPrefs.getDouble(TRANSLATION_FACTOR, 0.);
        Preferences joinByLinePrefs = prefs.node("joinByLine");
        _joinByLine = joinByLinePrefs.getBoolean(JOIN_BY_LINE, false);

        Preferences primitiveFormsPrefs = prefs.node("primitiveForms");
        String[] primitiveFormChilds;
        try {
            primitiveFormChilds = primitiveFormsPrefs.childrenNames();
        } catch (BackingStoreException ignore) {
            PreferencesDeleter pd =
                new PreferencesDeleter("Ein benutzer-definierter Darstellungstyp konnte nicht geladen werden.", primitiveFormsPrefs);
            pd.run();
            return;
        }
        for (String primitiveFormName : primitiveFormChilds) {
            PrimitiveForm primitiveForm = new PrimitiveForm();
            Preferences primitiveFormPrefs = primitiveFormsPrefs.node(primitiveFormName);
            primitiveForm.initializeFromPreferences(primitiveFormPrefs);
            addPrimitiveForm(primitiveForm);
        }
    }

    @Override
    public void deletePreferences(Preferences prefs) {
        Preferences classPrefs = prefs.node("DOTPoint");
        Preferences objectPrefs = classPrefs.node(getName());
        try {
            objectPrefs.removeNode();
        } catch (BackingStoreException ignore) {
            JOptionPane.showMessageDialog(null, "Das Löschen eines Darstellungstypen aus den Präferenzen war nicht erfolgreich.", "Fehlermeldung",
                                          JOptionPane.ERROR_MESSAGE);
        }
    }

    @Override
    public void putPreferences(Preferences prefs) {
        deletePreferences(prefs);
        Preferences classPrefs = prefs.node("DOTPoint");
        Preferences objectPrefs = classPrefs.node(getName());
        Preferences infoPrefs = objectPrefs.node("info");
        infoPrefs.put(INFO, getInfo());
        Preferences translationFactorPrefs = objectPrefs.node("translationFactor");
        translationFactorPrefs.putDouble(TRANSLATION_FACTOR, _translationFactor);
        Preferences joinByLinePrefs = objectPrefs.node("joinByLine");
        joinByLinePrefs.putBoolean(JOIN_BY_LINE, _joinByLine);

        Preferences primitiveFormPrefs = objectPrefs.node("primitiveForms");
        for (PrimitiveForm primitiveForm : _primitiveForms.values()) {
            primitiveForm.putPreferences(primitiveFormPrefs);
        }
    }

    @Override
    public DisplayObjectTypePlugin getDisplayObjectTypePlugin() {
        return new DOTPointPlugin();
    }

    @Override
    public LegendTreeNodes getLegendTreeNodes() {
        LegendTreeNodes legendTreeNodes = new LegendTreeNodes();
        LegendTreeNodes.LegendTreeNode node = null;
        int depth = 0;
        int newDepth;
        for (PrimitiveForm primitiveForm : _primitiveForms.values()) {
            if (!primitiveForm.hasDynamicProperties()) {
                continue;
            }
            newDepth = 0;
            if (node != null) {
                legendTreeNodes.add(node, depth - newDepth);
            }
            node = new LegendTreeNodes.LegendTreeNode(primitiveForm.getName(), primitiveForm.getInfo(), this);
            depth = newDepth;
            for (Property property : primitiveForm.getDynamicProperties()) {
                newDepth = 1;
                legendTreeNodes.add(node, depth - newDepth);
                node = new LegendTreeNodes.LegendTreeNode(property.toString(), null, this);
                depth = newDepth;
                final DynamicDOTItemManager dynamicDOTItemManager = primitiveForm.getDynamicDOTItemManager(property);
                final int size = dynamicDOTItemManager.size();
                for (int rowIndex = 0; rowIndex < size; rowIndex++) {
                    newDepth = 2;
                    legendTreeNodes.add(node, depth - newDepth);
                    String description = dynamicDOTItemManager.get(rowIndex).getItem().getDescription();
                    node = new LegendTreeNodes.LegendTreeNode(description, null, this);
                    depth = newDepth;
                }
            }
        }
        newDepth = 0;
        if (node != null) {
            legendTreeNodes.add(node, depth - newDepth);
        }
        // Vorläufig hinzugefügt (2017-05-18)
        if (legendTreeNodes.isEmpty()) {
            for (PrimitiveForm primitiveForm : _primitiveForms.values()) {
                final String colorName = (String) getValueOfStaticProperty(primitiveForm.getName(), ColorProperty.getInstance());
                if (null != colorName) {
                    node = new LegendTreeNodes.LegendTreeNode(primitiveForm.getName() + ": " + colorName, null, this);
                    legendTreeNodes.add(node, 0);
                }
            }
        }
        return legendTreeNodes;
    }

    @Override
    public Set<DOTSubscriptionData> getSubscriptionData() {
        Set<DOTSubscriptionData> sdSet = new HashSet<>();
        for (PrimitiveForm primitiveForm : _primitiveForms.values()) {
            for (Property property : primitiveForm.getDynamicProperties()) {
                final DynamicDOTItemManager dynamicDOTItemManager = primitiveForm.getDynamicDOTItemManager(property);
                sdSet.addAll(dynamicDOTItemManager.getSubscriptionData());
            }
        }
        return sdSet;
    }

    @Override
    public List<String> getAttributeNames(String primitiveFormName, Property property, DOTSubscriptionData subscriptionData) {
        final PrimitiveForm primitiveForm = getPrimitiveForm(primitiveFormName, "getAttributeNames");
        return primitiveForm.getAttributeNames(property, subscriptionData);
    }

    @Override
    @Nullable
    public DisplayObjectTypeItem getDOTItemForValue(String primitiveFormName, Property property, DOTSubscriptionData subscriptionData,
                                                    String attributeName, double value) {
        final PrimitiveForm primitiveForm = getPrimitiveForm(primitiveFormName, "getDisplayObjectTypeItem");
        return primitiveForm.getDisplayObjectTypeItem(property, subscriptionData, attributeName, value);
    }

    @Override
    @Nullable
    public DisplayObjectTypeItem getDisplayObjectTypeItemForState(final String primitiveFormName, final Property property,
                                                                  final DOTSubscriptionData subscriptionData, final DataState dataState) {
        final PrimitiveForm primitiveForm = getPrimitiveForm(primitiveFormName, "getDisplayObjectTypeItemForState");
        if (primitiveForm.isPropertyStatic(property)) {
            return null;
        }
        final DynamicDOTItemManager dynamicDOTItemManager = primitiveForm.getDynamicDOTItemManager(property);
        final String keyString1;
        if (dataState.equals(DataState.NO_DATA)) {
            keyString1 = dynamicDOTItemManager.getKeyString(subscriptionData, DynamicDefinitionComponent.KEINE_DATEN_STATUS);
        } else if (dataState.equals(DataState.NO_SOURCE)) {
            keyString1 = dynamicDOTItemManager.getKeyString(subscriptionData, DynamicDefinitionComponent.KEINE_QUELLE_STATUS);
        } else if (dataState.equals(DataState.NO_RIGHTS)) {
            keyString1 = dynamicDOTItemManager.getKeyString(subscriptionData, DynamicDefinitionComponent.KEINE_RECHTE_STATUS);
        } else {
            keyString1 = null;
        }
        if (keyString1 != null) {    // einer der Substati hat gezogen ...
            final TreeMap<Interval<Double>, DynamicDOTItem> treeMap1 = dynamicDOTItemManager.get(keyString1);
            if ((treeMap1 != null) && (treeMap1.size() == 1)) { // ... und ist definiert
                return treeMap1.values().toArray(new DisplayObjectTypeItem[1])[0];
            }
        }
        // den übergreifenden Status überprüfen
        final String keyString2 = dynamicDOTItemManager.getKeyString(subscriptionData, DynamicDefinitionComponent.LEERE_DATEN_STATUS);
        final TreeMap<Interval<Double>, DynamicDOTItem> treeMap2 = dynamicDOTItemManager.get(keyString2);
        if ((treeMap2 != null) && (treeMap2.size() == 1)) { // ... er ist definiert
            return treeMap2.values().toArray(new DisplayObjectTypeItem[1])[0];
        } else {    // ... dann bleibt nur der Default
            return DynamicDOTItem.NO_DATA_ITEM;
        }
    }

    @Override
    public DisplayObjectType getCopy(String name) {
        DOTPoint copy;
        if (name == null) {
            copy = new DOTPoint(_name, _info, _translationFactor, _joinByLine);
        } else {
            copy = new DOTPoint(name, _info, _translationFactor, _joinByLine);
        }
        for (PrimitiveForm primitiveForm : _primitiveForms.values()) {
            PrimitiveForm copyOfPrimitiveForm = primitiveForm.getCopy();
            copy._primitiveForms.put(copyOfPrimitiveForm.getName(), copyOfPrimitiveForm);
        }
        return copy;
    }

    /**
     * Fügt eine Grundfigur hinzu.
     *
     * @param primitiveForm die Grundfigur
     */
    public void addPrimitiveForm(PrimitiveForm primitiveForm) {
        if (_primitiveForms.containsKey(primitiveForm.getName())) {
            throw new IllegalArgumentException(
                "DOTPoint.addPrimitiveForm(): eine Grundfigur mit dem Namen '" + primitiveForm.getName() + "' existiert bereits.");
        }
        _primitiveForms.put(primitiveForm.getName(), primitiveForm);
    }

    /**
     * Macht ein Update auf die bereits vorhandene Grundfigur oder fügt sie andernfalls hinzu.
     *
     * @param primitiveForm die Grundfigur
     */
    @SuppressWarnings("unused")
    public void putPrimitiveForm(PrimitiveForm primitiveForm) {
        _primitiveForms.put(primitiveForm.getName(), primitiveForm);
    }

    /**
     * Gibt die Grundfigur zurück.
     *
     * @param primitiveFormName der Name der Grundfigur
     * @param methodName        der Name der aufrufenden Methode (ursprünglich für eine Fehlermeldung vorgesehen)
     *
     * @return die Grundfigur oder {@code null}
     */
    private PrimitiveForm getPrimitiveForm(String primitiveFormName, @SuppressWarnings("unused") final String methodName) {
        return _primitiveForms.get(primitiveFormName);
    }

    /**
     * Gibt die genannte Grundfigur zurück.
     *
     * @param primitiveFormName der Name der Grundfigur
     *
     * @return die Grundfigur oder {@code null}
     */
    public PrimitiveForm getPrimitiveForm(String primitiveFormName) {
        return _primitiveForms.get(primitiveFormName);
    }

    /**
     * Gibt alle Grundfiguren zurück.
     *
     * @return alle Grundfiguren
     */
    public Collection<PrimitiveForm> getPrimitiveForms() {
        return Collections.unmodifiableCollection(_primitiveForms.values());
    }

    /**
     * Gibt die Namen aller Grundfiguren zurück.
     *
     * @return alle Grundfigurnamen
     */
    @Override
    public Set<String> getPrimitiveFormNames() {
        return Collections.unmodifiableSet(_primitiveForms.keySet());
    }

    /**
     * Gibt den Typ der genannten Grundfigur zurück, oder aber einen leeren String.
     *
     * @param primitiveFormName der Grundfigurname
     *
     * @return der Grundfigurtyp
     */
    @Override
    public String getPrimitiveFormType(final String primitiveFormName) {
        final PrimitiveForm primitiveForm = _primitiveForms.get(primitiveFormName);
        if (primitiveForm == null) {
            return "";
        }
        return primitiveForm.getType().getName();
    }

    /**
     * Gibt den Infotext der benannten Grundfigur zurück, oder aber einen leeren String.
     *
     * @param primitiveFormName der Grundfigurname
     *
     * @return die Kurzinfo
     */
    @Override
    public String getPrimitiveFormInfo(final String primitiveFormName) {
        final PrimitiveForm primitiveForm = _primitiveForms.get(primitiveFormName);
        if (primitiveForm == null) {
            return "";
        }
        return primitiveForm.getInfo();
    }

    /**
     * Entfernt die benannte Grundfigur.
     *
     * @param primitiveFormName der Grundfigurname
     */
    @Override
    public void removePrimitiveForm(final String primitiveFormName) {
        _primitiveForms.remove(primitiveFormName);
    }

    /**
     * Gibt alle dynamischen Eigenschaften der benannten Grundfigur zurück.
     *
     * @param primitiveFormName der Grundfigurname
     *
     * @return alle dynamischen Eigenschaften der Grundfigur
     */
    @Override
    public List<Property> getDynamicProperties(final String primitiveFormName) {
        final PrimitiveForm primitiveForm = _primitiveForms.get(primitiveFormName);
        if (primitiveForm == null) {
            return new ArrayList<>();
        }
        return primitiveForm.getDynamicProperties();
    }

    /**
     * Gibt das Tabellenmodel der durch die übergebenen Werte beschriebenen Eigenschaft zurück.
     *
     * @param primitiveForm die Grundfigur
     * @param property      die Eigenschaft
     *
     * @return das TableModel
     */
    public TableModel getTableModel(PrimitiveForm primitiveForm, Property property) {
        return primitiveForm.getDynamicDOTItemManager(property);
    }

    /**
     * Gibt die Indizes aller in Konflikt stehenden Zeilen des Tabellenmodells an. Ein Konflikt besteht, wenn zwei Zeilen sich hinsichtlich der
     * Wertebereiche überlappen.
     *
     * @param primitiveForm die Grundfigur
     * @param property      die Eigenschaft
     *
     * @return die Indizes von in Konflikten stehenden Zeilen
     */
    @Nullable
    public Set<Integer> getConflictingRows(PrimitiveForm primitiveForm, Property property) {
        final DynamicDOTItemManager dynamicDOTItemManager = primitiveForm.getDynamicDOTItemManager(property);
        if (dynamicDOTItemManager == null) {
            return null;
        }
        return dynamicDOTItemManager.getConflictingRows();
    }

    @SuppressWarnings("NonFinalFieldReferencedInHashCode")
    @Override
    public int hashCode() {
        int result = _name.hashCode();
        result = 31 * result + _info.hashCode();
        result = 31 * result + _translationFactor.hashCode();
        result = 31 * result + _joinByLine.hashCode();
        result = 31 * result + _primitiveForms.hashCode();
        return result;
    }

    @Override
    public final boolean equals(Object o) {    // Nicht ändern!
        return super.equals(o);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("[DOTPoint: [Name:").append(_name).append("][Info:").append(_info).append("][Verschiebungsfaktor:").append(_translationFactor)
            .append("][Verbindungslinie:").append(_joinByLine).append("]");
        final Set<String> keySet = _primitiveForms.keySet();
        // Wir müssen die Reihenfolge normieren, damit auch der String-Vergleich geht!
        final SortedSet<String> sortedKeySet = new TreeSet<>();
        sortedKeySet.addAll(keySet);
        for (String key : sortedKeySet) {
            sb.append(_primitiveForms.get(key).toString());
        }
        sb.append("]");
        return sb.toString();
    }

    @Override
    public Set<String> getUsedColors() {
        Set<String> usedColors = new HashSet<>();
        for (PrimitiveForm primitiveForm : _primitiveForms.values()) {
            usedColors.addAll(primitiveForm.getUsedColors());
        }
        return usedColors;
    }

    /*
     * Implementiert diese Methode von DOTManager.DOTChangeListener leer.
     */
    @Override
    public void displayObjectTypeAdded(DisplayObjectType displayObjectType) {
    }

    /*
     * Implementiert diese Methode von DOTManager.DOTChangeListener.
     */
    @Override
    public void displayObjectTypeChanged(DisplayObjectType displayObjectType) {
        if (displayObjectType.equals(this)) {
            return;
        }
        if (displayObjectType.getName().equals(_name)) {
            DOTPoint dotPoint = (DOTPoint) displayObjectType;
            _info = dotPoint.getInfo();
            _translationFactor = dotPoint.getTranslationFactor();
            _joinByLine = dotPoint.getJoinByLine();
            _primitiveForms.clear();
            for (String s : dotPoint._primitiveForms.keySet()) {
                _primitiveForms.put(s, dotPoint._primitiveForms.get(s));
            }
        }
    }

    /*
     * Implementiert diese Methode von DOTManager.DOTChangeListener leer.
     */
    @Override
    public void displayObjectTypeRemoved(String displayObjectTypeName) {
    }

    /*
     * Implementiert Comparable<DisplayObjectType>.
     */
    @Override
    public int compareTo(DisplayObjectType o) {
        return getName().toLowerCase().compareTo(o.getName().toLowerCase());
    }

    /**
     * PrimitiveForm-Objekte sind die Grundfiguren in der Darstellung der DOTPoints.
     * <p>
     * Jede Grundfigur hat einen Namen, einen von fünf vorgegebenen Typen (Rechteck, Kreis, Halbkreis, Textdarstellung oder Punkt), einen Infotext,
     * einen Punkt in der Ebene, der einen Verschiebungvektor beschreibt, und abhängig vom Typ spezifische definierende Eigenschaften (Höhe, Breite,
     * Radius, Orientierung, Durchmesser usw.
     * <p>
     * Die Klasse ist statisch, damit sie statische Methoden haben kann (s. {@link #getDefaultSpecificInformation(String)}).
     *
     * @author Kappich Systemberatung
     */
    public static class PrimitiveForm {

        /**
         * A string.
         */
        public static final String HEIGHT = "height";
        /**
         * A string.
         */
        public static final String WIDTH = "width";
        /**
         * A string.
         */
        public static final String RADIUS = "radius";
        /**
         * A string.
         */
        public static final String ORIENTATION = "orientation";
        protected Map<Property, Boolean> _isStaticMap = new HashMap<>();
        protected Map<Property, Object> _staticPropertyValues = new HashMap<>();
        protected Map<Property, DynamicDOTItemManager> _dynamicDOTItemManagers = new HashMap<>();
        private String _nameOfPrimitiveForm;
        private PrimitiveFormType _typeOfPrimitiveForm;
        private String _infoOfPrimitiveForm;
        private Point2D.Double _translation;
        private Map<String, Object> _specificInformation = new HashMap<>();

        /**
         * Legt eine leere Grundfigur an.
         */
        public PrimitiveForm() {
            super();
        }

        /**
         * Legt eine Grundfigur mit den vorgegebenen Informationen an.
         *
         * @param name                der Name
         * @param type                der Typ
         * @param info                die Kurzinfo
         * @param translation         der lokale Verschiebungsvektor
         * @param specificInformation spezifische Informationen
         */
        public PrimitiveForm(String name, @Nullable PrimitiveFormType type, String info, Point2D.Double translation,
                             Map<String, Object> specificInformation) {
            super();
            _nameOfPrimitiveForm = name;
            _typeOfPrimitiveForm = type;
            _infoOfPrimitiveForm = info;
            _translation = translation;
            if (specificInformation != null) {
                _specificInformation = specificInformation;
            }
            checkSpecificInformation();
            initCollections();
        }

        private static Preferences getPropertyPreferences(Preferences prefs, Property property) {
            return prefs.node(property.getKey());
        }

        /**
         * Gibt Default-Werte für die spezifisch definiernden Informationen zurück.
         *
         * @param primitiveFormName der Name einer Grundfigur
         *
         * @return die Default-Werte
         */
        @SuppressWarnings("unused")
        public static Map<String, Object> getDefaultSpecificInformation(String primitiveFormName) {
            Map<String, Object> map = new HashMap<>();
            if (primitiveFormName.equals(PrimitiveFormType.RECHTECK.getName())) {
                map.put(HEIGHT, 0.);
                map.put(WIDTH, 0.);
            } else if (primitiveFormName.equals(PrimitiveFormType.KREIS.getName())) {
                map.put(RADIUS, 0.);
            } else if (primitiveFormName.equals(PrimitiveFormType.HALBKREIS.getName())) {
                map.put(RADIUS, 0.);
                map.put(ORIENTATION, DOTPointPainter.OBERER_HALBKREIS);
            }
            return map;
        }

        /**
         * Gibt den Namen der Grundfigur zurück.
         *
         * @return den Namen
         */
        public String getName() {
            return _nameOfPrimitiveForm;
        }

        /**
         * Setzt den Namen der Grundfigur.
         *
         * @param name der neue Name
         */
        public void setName(String name) {
            _nameOfPrimitiveForm = name;
        }

        /**
         * Gibt den Typ der Grundfigur zurück.
         *
         * @return der Grundfigurtyp
         */
        public PrimitiveFormType getType() {
            return _typeOfPrimitiveForm;
        }

        /**
         * Setzt den typ der Grundfigur.
         *
         * @param type der neue Grundfigurtyp
         */
        public void setType(PrimitiveFormType type) {
            _typeOfPrimitiveForm = type;
        }

        /**
         * Gibt den Infotext zurück.
         *
         * @return die Kurzinfo
         */
        public String getInfo() {
            return _infoOfPrimitiveForm;
        }

        /**
         * Setzt den Infotext.
         *
         * @param info die neue Kurzinfo
         */
        public void setInfo(String info) {
            _infoOfPrimitiveForm = info;
        }

        /**
         * Gibt den die lokale Verschiebung beschreibenden Vektor zurück.
         *
         * @return den Verschiebungsvektor
         */
        public Point2D.Double getTranslation() {
            return _translation;
        }

        /**
         * Setzt den die lokale Verschiebung beschreibenden Vektor.
         *
         * @param translation der neue Verschiebungsvektor
         */
        public void setTranslation(Point2D.Double translation) {
            _translation = translation;
        }

        /**
         * Gibt die spezifische definierende Eigenschaft mit dem übergebenen Namen zurück.
         *
         * @param name der Name der spezifischen Eigenschaft
         *
         * @return die spezifischen Eigenschaft
         */
        public Object getSpecificInformation(String name) {
            return _specificInformation.get(name);
        }

        /**
         * Setzt die spezifische definiernde Eigenschaft mit dem übergebenen Namen.
         *
         * @param name der Name
         * @param o    die Eigenschaft
         */
        public void setSpecificInformation(String name, Object o) {
            if (_specificInformation.containsKey(name)) {
                _specificInformation.put(name, o);
            }
        }

        /**
         * Gibt den Wert {@code true} zurück, wenn die übergebene Eigenschaft statisch ist, {@code false}, wenn sie dynamisch ist, und {@code null}
         * wenn sie nicht bei dieser Grundfigur auftritt.
         *
         * @return {@code true} genau dann, wenn die Eigenschaft statisch ist
         */
        public boolean isPropertyStatic(Property property) {
            return _isStaticMap.get(property);
        }

        /**
         * Legt fest, ob die übergebene Eigenschaft statisch oder dynamisch ist.
         *
         * @param property die Eigenschaft
         * @param b        der neue Wert
         */
        public void setPropertyStatic(Property property, boolean b) {
            if (b) {
                final boolean wasDynamic = _dynamicDOTItemManagers.containsKey(property);
                if (wasDynamic) {
                    Object value = property.getDefaultValue();
                    if (value == null) {
                        throw new IllegalArgumentException();
                    }
                    _staticPropertyValues.put(property, value);
                }
            }
            _isStaticMap.put(property, b);
        }

        /**
         * Gibt {@code true} zurück, wenn die Grundfigur mindestens ein dynamische Eigenschaft besitzt.
         *
         * @return gibt es dynamische Eigenschaften?
         */
        public boolean hasDynamicProperties() {
            for (boolean isStatic : _isStaticMap.values()) {
                if (!isStatic) {
                    return true;
                }
            }
            return false;
        }

        /**
         * Gibt eine Liste aller dynamischen Eigenschaften der Grundfigur zurück.
         *
         * @return alle dynamischen Eigenschaften
         */
        public List<Property> getDynamicProperties() {
            List<Property> dynamicPropertyList = new ArrayList<>();
            for (final Entry<Property, Boolean> dotPropertyBooleanEntry : _isStaticMap.entrySet()) {
                if (!dotPropertyBooleanEntry.getValue()) {
                    dynamicPropertyList.add(dotPropertyBooleanEntry.getKey());
                }
            }
            return dynamicPropertyList;
        }

        /**
         * Gibt den Wert (Farbe, Tranzparens, Textstil etc.) der statischen Eigenschaft der Grundfigur zurück.
         *
         * @param property die Eigenschaft
         *
         * @return den Wert der statischen Eigenschaft
         */
        @Nullable
        public Object getValueOfStaticProperty(Property property) {
            if (_isStaticMap.containsKey(property)) {
                return _staticPropertyValues.get(property);
            } else {
                return property.getDefaultValue();
            }
        }

        /**
         * Setzt den Wert (Farbe, Tranzparens, Textstil etc.) der statischen Eigenschaft der Grundfigur.
         *
         * @param property die Eigenschaft
         * @param value    der Wert der Eigenschaft
         */
        public void setValueOfStaticProperty(Property property, Object value) {
            if (property.equals(TextStyleProperty.getInstance())) {
	            if (value instanceof String s) {
                    switch (s) {
                        case DOTPointPanel.BOLD_FONT_STYLE:
                            _staticPropertyValues.put(property, Font.BOLD);
                            return;
                        case DOTPointPanel.ITALIC_FONT_STYLE:
                            _staticPropertyValues.put(property, Font.ITALIC);
                            return;
                        case DOTPointPanel.PLAIN_FONT_STYLE:
                            _staticPropertyValues.put(property, Font.PLAIN);
                            return;
                        default:
                            throw new IllegalArgumentException("DOTPoint.setValueOfStaticProperty(): unbekannter Fonttyp " + s);
                    }
                }
            } else if (property.equals(TextSizeProperty.getInstance())) {
	            if (value instanceof Double d) {
                    _staticPropertyValues.put(property, d.intValue());
                }
            }
            _staticPropertyValues.put(property, value);
        }

        /**
         * Setzt den Wert (Farbe, Tranzparens, Textstil etc.) der dynamsichen Eigenschaft der Grundfigur.
         *
         * @param property   die Eigenschaft
         * @param dItem      eine Item
         * @param lowerBound die untere Schranke
         * @param upperBound die obere Schranke
         */
        @SuppressWarnings("Java8ReplaceMapGet")
        public void setValueOfDynamicProperty(Property property, DisplayObjectTypeItem dItem, Double lowerBound, Double upperBound) {
            DynamicDOTItemManager dynamicDOTItemManager = _dynamicDOTItemManagers.get(property);
            if (dynamicDOTItemManager == null) {
                dynamicDOTItemManager = new DynamicDOTItemManager();
                _dynamicDOTItemManagers.put(property, dynamicDOTItemManager);

            }
	        if (!(dItem instanceof DynamicDOTItem ldItem)) {
                return;
            }
            final Interval<Double> interval;
            if (lowerBound == null || upperBound == null) {
                interval = new Interval<>(Double.NEGATIVE_INFINITY, Double.NEGATIVE_INFINITY);
            } else {
                interval = new Interval<>(lowerBound, upperBound);
            }
            dynamicDOTItemManager.insert(interval, ldItem);
        }

        /**
         * Initialisiert die Grundfigur aus den Präferenzen.
         *
         * @param prefs der Knoten, unter dem die Initialisierung beginnt
         */
        public void initializeFromPreferences(Preferences prefs) {
            _nameOfPrimitiveForm = prefs.name();
            Preferences typePrefs = prefs.node("type");
            _typeOfPrimitiveForm = PrimitiveFormType.getPrimitiveFormType(typePrefs.get(TYPE, ""));
            Preferences infoPrefs = prefs.node("info");
            _infoOfPrimitiveForm = infoPrefs.get(INFO, "");
            Preferences translationPrefs = prefs.node("translation");
            double x = translationPrefs.getDouble(TRANSLATION_X, 0.);
            double y = translationPrefs.getDouble(TRANSLATION_Y, 0.);
            _translation = new Point2D.Double(x, y);

            initPreferencesOfSpecificInformation(prefs);
            initPreferencesOfStaticProperties(prefs);
            initPreferencesOfDynamicProperties(prefs);
        }

        private void initPreferencesOfSpecificInformation(Preferences prefs) {
            Preferences specificPrefs = prefs.node("specific");
            String[] specificChilds;
            try {
                specificChilds = specificPrefs.childrenNames();
            } catch (BackingStoreException ignore) {
                PreferencesDeleter pd =
                    new PreferencesDeleter("Ein benutzer-definierter darstellungstyp konnte nur zum Teil geladen werden.", specificPrefs);
                pd.run();
                return;
            }
            for (String specificChild : specificChilds) {
                Preferences valuePrefs = specificPrefs.node(specificChild);
                String type = valuePrefs.get(TYPE, "");
	            Object value = switch (type) {
		            case "Integer" -> valuePrefs.getInt(VALUE, 0);
		            case "Double" -> valuePrefs.getDouble(VALUE, 0.);
		            case "String" -> valuePrefs.get(VALUE, "");
		            default -> null;
	            };
                if (value != null) {
                    _specificInformation.put(specificChild, value);
                }
            }
            checkSpecificInformation();
        }

        private void initPreferencesOfStaticProperties(Preferences prefs) {
            Preferences staticPrefs = prefs.node("static");
            String[] staticChilds;
            try {
                staticChilds = staticPrefs.childrenNames();
            } catch (BackingStoreException e) {
                _debug.error("Ein benutzer-definierter Darstellungstyp kann nicht initialisiert werden, " + "BackingStoreException:" + e.toString());
                PreferencesDeleter pd =
                    new PreferencesDeleter("Ein benutzer-definierter Darstellungstyp kann nicht initialisiert werden.", staticPrefs);
                pd.run();
                return;
            }
            for (String staticChild : staticChilds) {
                final Property property = PropertiesManager.INSTANCE.getProperty(staticChild);
                if (property == null) {
                    continue;
                }
                Preferences propertyPrefs = staticPrefs.node(staticChild);
                _isStaticMap.put(property, true);
                final DynamicDOTItem dynamicDOTItem;
                try {
                    dynamicDOTItem = new DynamicDOTItem(propertyPrefs, staticChild);
                } catch (BackingStoreException e) {
                    _debug.error(
                        "Ein benutzer-definierter Darstellungstyp kann nicht initialisiert werden, " + "BackingStoreException:" + e.toString());
                    PreferencesDeleter pd =
                        new PreferencesDeleter("Ein benutzer-definierter Darstellungstyp kann nicht initialisiert werden.", propertyPrefs);
                    pd.run();
                    return;
                }
                _staticPropertyValues.put(property, dynamicDOTItem.getPropertyValue());
            }
        }

        private void initPreferencesOfDynamicProperties(Preferences prefs) {
            Preferences dynamicPrefs = prefs.node("dynamic");
            String[] dynamicChilds;
            try {
                dynamicChilds = dynamicPrefs.childrenNames();
            } catch (BackingStoreException e) {
                _debug.error("Ein benutzer-definierter Darstellungstyp kann nicht initialisiert werden, " + "BackingStoreException:" + e.toString());
                PreferencesDeleter pd =
                    new PreferencesDeleter("Ein benutzer-definierter Darstellungstyp kann nicht initialisiert werden.", dynamicPrefs);
                pd.run();
                return;
            }
            for (String dynamicChild : dynamicChilds) {
                Property property = PropertiesManager.INSTANCE.getProperty(dynamicChild);
                if (property == null) {
                    _debug.error("Ein benutzer-definierter Darstellungstyp kann nicht initialisiert werden.");
                    continue;
                }
                Preferences propertyPrefs = dynamicPrefs.node(dynamicChild);
                _isStaticMap.put(property, false);
                DynamicDOTItemManager dynamicDOTItemManager = new DynamicDOTItemManager();
                _dynamicDOTItemManagers.put(property, dynamicDOTItemManager);

                String[] intervalNames;
                try {
                    intervalNames = propertyPrefs.childrenNames();
                } catch (BackingStoreException e) {
                    _debug.error(
                        "Ein benutzer-definierter Darstellungstyp kann nicht initialisiert werden, " + "BackingStoreException:" + e.toString());
                    PreferencesDeleter pd =
                        new PreferencesDeleter("Ein benutzer-definierter Darstellungstyp kann nicht initialisiert werden.", propertyPrefs);
                    pd.run();
                    continue;
                }
                for (String child : intervalNames) {
                    if (child.startsWith("interval")) {
                        Preferences objectItemPrefs = propertyPrefs.node(child);
                        final DynamicDOTItem dynamicItem;
                        try {
                            dynamicItem = new DynamicDOTItem(objectItemPrefs, dynamicChild);
                        } catch (BackingStoreException e) {
                            _debug.error("Ein benutzer-definierter Darstellungstyp kann nicht initialisiert werden, " + "BackingStoreException:" +
                                         e.toString());
                            PreferencesDeleter pd =
                                new PreferencesDeleter("Ein benutzer-definierter Darstellungstyp kann nicht initialisiert werden.", objectItemPrefs);
                            pd.run();
                            continue;
                        }
                        if (dynamicItem.isValid()) {
                            setValueOfDynamicProperty(property, dynamicItem, objectItemPrefs.getDouble(LOWER_BOUND, Double.MAX_VALUE),
                                                      objectItemPrefs.getDouble(UPPER_BOUND, Double.MIN_VALUE));
                        }
                    }
                }
            }
        }

        /**
         * Löscht die Präferenzen der Grundfigur.
         *
         * @param prefs der Knoten, unter dem gelöscht wird
         */
        public void deletePreferences(Preferences prefs) {
            Preferences objectPrefs = prefs.node(getName());
            try {
                objectPrefs.removeNode();
            } catch (BackingStoreException ignored) {
                PreferencesDeleter pd = new PreferencesDeleter("Es ist ein Fehler beim Löschen aus den Präferenzen aufgetreten.", objectPrefs);
                pd.run();
            }
        }

        /**
         * Speichert die Präferenzen der Grundfigur.
         *
         * @param prefs der Knoten, unter dem die Speicherung beginnt
         */
        public void putPreferences(Preferences prefs) {
            deletePreferences(prefs);
            Preferences objectPrefs = prefs.node(getName());
            Preferences typePrefs = objectPrefs.node("type");
            typePrefs.put(TYPE, getType().getName());
            Preferences infoPrefs = objectPrefs.node("info");
            infoPrefs.put(INFO, getInfo());
            Preferences translationPrefs = objectPrefs.node("translation");
            translationPrefs.putDouble(TRANSLATION_X, _translation.getX());
            translationPrefs.putDouble(TRANSLATION_Y, _translation.getY());

            Preferences specificPrefs = objectPrefs.node("specific");
            putPreferencesForSpecificInformation(specificPrefs);

            Preferences staticPrefs = objectPrefs.node("static");
            putPreferencesOfStaticProperties(staticPrefs);

            Preferences dynamicPrefs = objectPrefs.node("dynamic");
            putPreferencesOfDynamicProperties(dynamicPrefs);
        }

        private void putPreferencesForSpecificInformation(Preferences prefs) {
            for (final Entry<String, Object> entry : _specificInformation.entrySet()) {
                Preferences specificPrefs = prefs.node(entry.getKey());
                Object object = entry.getValue();
                if (object instanceof Integer) {
                    specificPrefs.put(TYPE, "Integer");
                    specificPrefs.putInt(VALUE, (Integer) object);
                } else if (object instanceof Double) {
                    specificPrefs.put(TYPE, "Double");
                    specificPrefs.putDouble(VALUE, (Double) object);
                } else if (object instanceof String) {
                    specificPrefs.put(TYPE, "String");
                    specificPrefs.put(VALUE, (String) object);
                } else {
                    _debug.warning("PrimitiveForm: eine spezifische Information konnte nicht gespeichert werden.");
                }
            }
        }

        private void putPreferencesOfStaticProperties(Preferences prefs) {
            for (final Entry<Property, Boolean> entry : _isStaticMap.entrySet()) {
                if (entry.getValue()) {
                    // Eine Statische Property wird als dynamische ohne Anmeldungsdaten weggeschrieben.
                    Preferences propertyPrefs = getPropertyPreferences(prefs, entry.getKey());
                    final DynamicDOTItem dynamicDOTItem = new DynamicDOTItem("", "", "", "", _staticPropertyValues.get(entry.getKey()));
                    dynamicDOTItem.putPreferences(propertyPrefs);
                }
            }
        }

        private void putPreferencesOfDynamicProperties(Preferences prefs) {
            for (final Entry<Property, Boolean> entry : _isStaticMap.entrySet()) {
                if (!entry.getValue()) {
                    Preferences propertyPrefs = getPropertyPreferences(prefs, entry.getKey());
                    DynamicDOTItemManager dynamicDOTItemManager = _dynamicDOTItemManagers.get(entry.getKey());
                    int i = 0;
                    for (TreeMap<Interval<Double>, DynamicDOTItem> treeMap : dynamicDOTItemManager.getTreeMaps()) {
                        for (final Entry<Interval<Double>, DynamicDOTItem> intervalDynamicDOTItemEntry : treeMap.entrySet()) {
                            Preferences objectForItemPrefs = propertyPrefs.node("interval" + i);
                            objectForItemPrefs.putDouble(LOWER_BOUND, intervalDynamicDOTItemEntry.getKey().getLowerBound());
                            objectForItemPrefs.putDouble(UPPER_BOUND, intervalDynamicDOTItemEntry.getKey().getUpperBound());
                            DynamicDOTItem dynamicItem = intervalDynamicDOTItemEntry.getValue();
                            if ((dynamicItem == null) && (treeMap.size() >= 1)) {
                                // weil es mit Double.NEGATIVE_INFINITY nicht geht, das get(); >= ist wichtig!
                                dynamicItem = treeMap.values().toArray(new DynamicDOTItem[1])[0];
                            }
                            if (dynamicItem != null) {
                                dynamicItem.putPreferences(objectForItemPrefs);
                            }
                            i++;
                        }
                    }
                }
            }
        }

        /**
         * Gibt eine Liste mit allen Attributnamen zurück, die für die Eigenschaft und die durch DOTSubscriptionData gekapselte Attributgruppe und den
         * Aspekt für diese Grundfigur relevant sind.
         *
         * @param property         die Eigenschaft
         * @param subscriptionData eine Anmeldung
         *
         * @return alle relevanten Attributnamen
         */
        public List<String> getAttributeNames(Property property, DOTSubscriptionData subscriptionData) {
            final DynamicDOTItemManager dynamicDOTItemManager = _dynamicDOTItemManagers.get(property);
            if (dynamicDOTItemManager == null) {
                return new ArrayList<>();
            }
            final List<String> list = dynamicDOTItemManager.getAttributeNames(subscriptionData);
            if (list != null) {
                return list;
            }
            return new ArrayList<>();
        }

        /**
         * Gibt das Item zurück, das für die übergebenen Werte verwendet werden kann, oder {@code null}, wenn ein solches nicht existiert.
         *
         * @param property         die Eigenschaft
         * @param subscriptionData eine Anmeldung
         * @param attributeName    ein Attributname
         * @param value            der Wert
         *
         * @return das Item oder {@code null}
         */
        @Nullable
        public DisplayObjectTypeItem getDisplayObjectTypeItem(Property property, DOTSubscriptionData subscriptionData, String attributeName,
                                                              double value) {
            if (_isStaticMap.get(property)) {
                return null;
            }
            final DynamicDOTItemManager dynamicDOTItemManager = _dynamicDOTItemManagers.get(property);
            if (dynamicDOTItemManager == null) {
                return null;
            }
            final List<String> list = dynamicDOTItemManager.getAttributeNames(subscriptionData);
            if (list == null) {
                return null;
            }
            if (!list.contains(attributeName)) {
                return null;
            }
            final TreeMap<Interval<Double>, DynamicDOTItem> treeMap =
                dynamicDOTItemManager.get(dynamicDOTItemManager.getKeyString(subscriptionData, attributeName));
            if (treeMap == null) {
                return null;
            }
            Interval<Double> valInterval = new Interval<>(value, value);
            final Entry<Interval<Double>, DynamicDOTItem> floorEntry = treeMap.floorEntry(valInterval);
            if (floorEntry != null) {
                final Interval<Double> floorKey = floorEntry.getKey();
                if ((floorKey.getLowerBound() <= value) && (value <= floorKey.getUpperBound())) {
                    return floorEntry.getValue();
                }
            }
            final Entry<Interval<Double>, DynamicDOTItem> ceilingEntry = treeMap.ceilingEntry(valInterval);
            if (ceilingEntry != null) {
                final Interval<Double> ceilingKey = ceilingEntry.getKey();
                if ((ceilingKey.getLowerBound() <= value) && (value <= ceilingKey.getUpperBound())) {
                    return ceilingEntry.getValue();
                }
            }
            return null;
        }

        /**
         * Erzeugt eine tiefe Kopie des Objekts.
         *
         * @return die Kopie
         */
        public PrimitiveForm getCopy() {
            Map<String, Object> specificInformation = new HashMap<>();
            for (final Entry<String, Object> stringObjectEntry : _specificInformation.entrySet()) {
                final Object object = stringObjectEntry.getValue();
                /* 2017: die folgenden Zeilen sehen stark vereinfachungswürdig aus. */
                Object newObject = null;
                if (object instanceof Integer) {
                    newObject = object;
                } else if (object instanceof Double) {
                    newObject = object;
                } else if (object instanceof String) {
                    newObject = object;
                }
                specificInformation.put(stringObjectEntry.getKey(), newObject);
            }
            PrimitiveForm copy = new PrimitiveForm(_nameOfPrimitiveForm, _typeOfPrimitiveForm, _infoOfPrimitiveForm,
                                                   new Point2D.Double(_translation.getX(), _translation.getY()), specificInformation);
            for (final Entry<Property, Boolean> dotPropertyBooleanEntry : _isStaticMap.entrySet()) {
                Boolean newBoolean = dotPropertyBooleanEntry.getValue();
                copy._isStaticMap.put(dotPropertyBooleanEntry.getKey(), newBoolean);
            }
            for (final Entry<Property, Object> entry : _staticPropertyValues.entrySet()) {
                Object object = entry.getValue();
                Object newObject;
                /* 2017: ein paar Zeilen siehen komisch aus. S.a. andere Klassen. */
                if (object instanceof Integer) {
                    newObject = object;
                } else if (object instanceof Double) {
                    newObject = object;
                } else if (object instanceof String) {
                    newObject = object;
                } else if (object instanceof Color color) {
                    newObject = new Color(color.getRed(), color.getGreen(), color.getBlue(), color.getAlpha());
                } else if (object instanceof TextStyleProperty.Styles) {
                    newObject = object;
                } else {
                    //noinspection ProhibitedExceptionThrown
                    throw new RuntimeException("Unbekannter Objekttyp in DOTPoint#PrimitiveForm.getCopy()!");
                }
                copy._staticPropertyValues.put(entry.getKey(), newObject);
            }
            for (final Entry<Property, DynamicDOTItemManager> dotPropertyDynamicDOTItemManagerEntry : _dynamicDOTItemManagers.entrySet()) {
                DynamicDOTItemManager dynamicDOTItemManager = dotPropertyDynamicDOTItemManagerEntry.getValue();
                DynamicDOTItemManager newDynamicDOTItemManager = dynamicDOTItemManager.getCopy();
                copy._dynamicDOTItemManagers.put(dotPropertyDynamicDOTItemManagerEntry.getKey(), newDynamicDOTItemManager);
            }
            return copy;
        }

        private void checkSpecificInformation() {
            if (_typeOfPrimitiveForm == PrimitiveFormType.RECHTECK) {
                if (_specificInformation.size() != 2) {
                    _debug.warning("PrimitiveForm.checkSpecificInformation() für " + getName() + ": ein ungültiges Rechteck.");
                }
                if (!_specificInformation.containsKey(HEIGHT)) {
                    _debug.warning("PrimitiveForm.checkSpecificInformation() für " + getName() + ": ein Rechteck ohne Höhe.");
                }
                if (!_specificInformation.containsKey(WIDTH)) {
                    _debug.warning("PrimitiveForm.checkSpecificInformation() für " + getName() + ": ein Rechteck ohne Breite.");
                }
            } else if (_typeOfPrimitiveForm == PrimitiveFormType.KREIS) {
                if (_specificInformation.size() != 1) {
                    _debug.warning("PrimitiveForm.checkSpecificInformation() für " + getName() + ": ein ungültiger Kreis.");
                }
                if (!_specificInformation.containsKey(RADIUS)) {
                    _debug.warning("PrimitiveForm.checkSpecificInformation() für " + getName() + ": ein Kreis ohne Radius.");
                }
            } else if (_typeOfPrimitiveForm == PrimitiveFormType.HALBKREIS) {
                if (_specificInformation.size() != 2) {
                    _debug.warning("PrimitiveForm.checkSpecificInformation() für " + getName() + ": ein ungültiger Halbkreis.");
                }
                if (!_specificInformation.containsKey(RADIUS)) {
                    _debug.warning("PrimitiveForm.checkSpecificInformation() für " + getName() + ": ein Halbkreis ohne Radius.");
                }
                if (!_specificInformation.containsKey(ORIENTATION)) {
                    _debug.warning("PrimitiveForm.checkSpecificInformation() für " + getName() + ": ein Halbkreis ohne Orientation.");
                }
            } else if (_typeOfPrimitiveForm == PrimitiveFormType.TEXTDARSTELLUNG) {
                if (!_specificInformation.isEmpty()) {
                    _debug.warning("PrimitiveForm.checkSpecificInformation() für " + getName() + ": eine ungültige Textdarstellung.");
                }
            } else if (_typeOfPrimitiveForm == PrimitiveFormType.PUNKT) {
                if (!_specificInformation.isEmpty()) {
                    _debug.warning("PrimitiveForm.checkSpecificInformation() für " + getName() + ": ein ungültiger Punkt.");
                }
            }
        }

        private void initCollections() {
            final DisplayObjectTypePlugin displayObjectTypePlugin = new DOTPointPlugin();
            final Property[] properties = displayObjectTypePlugin.getProperties(_typeOfPrimitiveForm);
            for (Property property : properties) {
                _isStaticMap.put(property, true);
                _staticPropertyValues.put(property, property.getDefaultValue());
                _dynamicDOTItemManagers.put(property, new DynamicDOTItemManager());
            }
        }

        /**
         * Gibt den Item-Manager der Eigenschaft zurück.
         *
         * @param property die Eigenschaft
         *
         * @return den ItemManager
         */
        public DynamicDOTItemManager getDynamicDOTItemManager(final Property property) {
            return _dynamicDOTItemManagers.get(property);
        }

        @SuppressWarnings("unchecked")
        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("[PrimitiveForm:[Name:").append(_nameOfPrimitiveForm).append("][Type:").append(_typeOfPrimitiveForm).append("][Info:")
                .append(_infoOfPrimitiveForm).append("][Verschiebung:(").append(_translation.getX()).append(",").append(_translation.getY())
                .append(")]").append("[Definition:");
            for (final Entry<String, Object> stringObjectEntry : _specificInformation.entrySet()) {
                sb.append("[").append(stringObjectEntry.getKey()).append(":").append(stringObjectEntry.getValue()).append("]");
            }
            sb.append("]").append("[Statische Eigenschaften:");
            for (final Entry<Property, Boolean> dotPropertyBooleanEntry : _isStaticMap.entrySet()) {
                if (dotPropertyBooleanEntry.getValue()) {
                    sb.append("[").append(dotPropertyBooleanEntry.getKey().toString()).append(":")
                        .append(_staticPropertyValues.get(dotPropertyBooleanEntry.getKey())).append("]");
                }
            }
            sb.append("]").append("[Dynamische Eigenschaften:");
            for (final Entry<Property, Boolean> dotPropertyBooleanEntry : _isStaticMap.entrySet()) {
                if (!dotPropertyBooleanEntry.getValue()) {
                    sb.append("[").append(dotPropertyBooleanEntry.getKey().toString()).append(":");
                    DynamicDOTItemManager dynamicDOTItemManager = _dynamicDOTItemManagers.get(dotPropertyBooleanEntry.getKey());
                    final int size = dynamicDOTItemManager.size();
                    for (int i = 0; i < size; i++) {
	                    final DOTItemManager<DynamicDOTItem>.DisplayObjectTypeItemWithInterval displayObjectTypeItemWithInterval = dynamicDOTItemManager.get(i);
                        sb.append(displayObjectTypeItemWithInterval.toString());
                    }
                    sb.append("]");
                }
            }
            sb.append("]]");
            return sb.toString();
        }

        /**
         * Gibt eine Menge mit allen von dieser Grundfigur für die übergebene Eigenschaft benutzten Farben zurück.
         *
         * @param property die Eigenschaft
         *
         * @return die Menge der benutzten Farben
         */
        @SuppressWarnings("unchecked")
        public Set<String> getUsedColors(final Property property) {
            Set<String> usedColors = new HashSet<>();
            if (isPropertyStatic(property)) {
                final String colorName = (String) getValueOfStaticProperty(property);
                if (colorName != null) {
                    usedColors.add(colorName.toLowerCase());
                }
            } else {
                DynamicDOTItemManager dynamicDOTItemManager = _dynamicDOTItemManagers.get(property);
                final int size = dynamicDOTItemManager.size();
                for (int index = 0; index < size; index++) {
	                final DOTItemManager<DynamicDOTItem>.DisplayObjectTypeItemWithInterval dotItemWithInterval = dynamicDOTItemManager.get(index);
                    final String colorName = (String) dotItemWithInterval.getItem().getPropertyValue();
                    usedColors.add(colorName.toLowerCase());
                }
            }
            return usedColors;
        }

        /**
         * Gibt eine Menge mit allen von dieser Grundfigur benutzten Farben zurück.
         *
         * @return die Menge der benutzten Farben
         */
        public Set<String> getUsedColors() {
            Set<String> usedColors = new HashSet<>();
            if (_isStaticMap.containsKey(ColorProperty.getInstance())) {
                usedColors.addAll(getUsedColors(ColorProperty.getInstance()));
            }
            if (_isStaticMap.containsKey(FillingProperty.getInstance())) {
                usedColors.addAll(getUsedColors(FillingProperty.getInstance()));
            }
            return usedColors;
        }
    }

    /**
     * Eine Enumeration aller Grundfigurtypen. Jeder Grundfigurtyp hat nur einen eindeutigen Namen.
     *
     * @author Kappich Systemberatung
     */
    public static final class PrimitiveFormType {
        public static final PrimitiveFormType PUNKT = new PrimitiveFormType("Punkt");
        public static final PrimitiveFormType RECHTECK = new PrimitiveFormType("Rechteck");
        public static final PrimitiveFormType KREIS = new PrimitiveFormType("Kreis");
        public static final PrimitiveFormType HALBKREIS = new PrimitiveFormType("Halbkreis");
        public static final PrimitiveFormType TEXTDARSTELLUNG = new PrimitiveFormType("Textdarstellung");
        private final String _name;

        private PrimitiveFormType(String name) {
            super();
            _name = name;
        }

        /**
         * Wandelt den String in ein PrimitiveFormType-Objekt.
         *
         * @param name der Name des Types
         *
         * @return der Typ
         */
        @Nullable
        public static PrimitiveFormType getPrimitiveFormType(final String name) {
	        return switch (name) {
		        case "Punkt" -> PUNKT;
		        case "Rechteck" -> RECHTECK;
		        case "Kreis" -> KREIS;
		        case "Halbkreis" -> HALBKREIS;
		        case "Textdarstellung" -> TEXTDARSTELLUNG;
		        default -> null;
	        };
        }

        /**
         * Gibt den Namen zurück.
         *
         * @return den Namen
         */
        public String getName() {
            return _name;
        }

        @Override
        public String toString() {
            return _name;
        }

        @Override
        public final boolean equals(Object o) {
            return super.equals(o);
        }

        @Override
        public final int hashCode() {
            return super.hashCode();
        }
    }

}
