/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.pointPlugin;

import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.colorManagement.ColorManager;
import de.kappich.pat.gnd.csvPlugin.CsvDisplayObject;
import de.kappich.pat.gnd.displayObjectToolkit.DisplayObject;
import de.kappich.pat.gnd.displayObjectToolkit.DynamicDOTItem;
import de.kappich.pat.gnd.displayObjectToolkit.OnlineDisplayObject;
import de.kappich.pat.gnd.displayObjectToolkit.PrimitiveFormPropertyPair;
import de.kappich.pat.gnd.gnd.MapPane;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectPainter;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType.DisplayObjectTypeItem;
import de.kappich.pat.gnd.pointPlugin.DOTPoint.PrimitiveForm;
import de.kappich.pat.gnd.pointPlugin.DOTPoint.PrimitiveFormType;
import de.kappich.pat.gnd.properties.ColorProperty;
import de.kappich.pat.gnd.properties.DiameterProperty;
import de.kappich.pat.gnd.properties.FillingProperty;
import de.kappich.pat.gnd.properties.StrokeWidthProperty;
import de.kappich.pat.gnd.properties.TextProperty;
import de.kappich.pat.gnd.properties.TextSizeProperty;
import de.kappich.pat.gnd.properties.TextStyleProperty;
import de.kappich.pat.gnd.properties.TransparencyProperty;
import de.kappich.pat.gnd.utils.PointWithAngle;
import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.Ellipse2D;
import java.awt.geom.GeneralPath;
import java.awt.geom.Line2D;
import java.awt.geom.PathIterator;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Die Implementation von DisplayObjectPainter zum Zeichnen von Punktobjekten.
 *
 * @author Kappich Systemberatung
 */
@SuppressWarnings("OverlyLongMethod")
public class DOTPointPainter implements DisplayObjectPainter {

    /**
     *
     */
    public static final String OBERER_HALBKREIS = "Oberer Halbkreis";
    public static final String RECHTER_HALBKREIS = "Rechter Halbkreis";
    public static final String UNTERER_HALBKREIS = "Unterer Halbkreis";
    public static final String LINKER_HALBKREIS = "Linker Halbkreis";
    public static final String DYNAMIC_ATTRIBUTE_SCALED = "Vordefinierte Funktion: Attribut skaliert anzeigen";
    @SuppressWarnings("unused")
    private static final Debug _debug = Debug.getLogger();
    private static final String DYNAMIC_ATTRIBUTE_UNSCALED = "Vordefinierte Funktion: Attribut unskaliert anzeigen";
    private static final String GET_NAME = "Vordefinierte Funktion: Name";
    private static final String GET_NAME_OR_PID_OR_ID = "Vordefinierte Funktion: Name, PID oder Id";
    private static final String GET_PID_OR_ID = "Vordefinierte Funktion: PID oder Id";
    private static final String GET_INFO_GET_DESCRIPTION = "Vordefinierte Funktion: Beschreibung";
    /**
     * Die Einträge der möglichen statischen Textausgaben.
     */
    @SuppressWarnings("PublicStaticArrayField")
    public static final String[] STATIC_TEXT_ITEMS =
        {GET_NAME_OR_PID_OR_ID, GET_NAME, GET_PID_OR_ID, GET_INFO_GET_DESCRIPTION, "Ein veränderbarer Text"};
    /**
     * Die Einträge der möglichen dynamischen Textausgaben.
     */
    @SuppressWarnings("PublicStaticArrayField")
    public static final String[] DYNAMIC_TEXT_ITEMS = {DYNAMIC_ATTRIBUTE_UNSCALED,
                                                       DYNAMIC_ATTRIBUTE_SCALED,
                                                       GET_NAME_OR_PID_OR_ID,
                                                       GET_NAME,
                                                       GET_PID_OR_ID,
                                                       GET_INFO_GET_DESCRIPTION,
                                                       "Ein veränderbarer Text"};

    /*
     * Gibt true zurück, wenn die übergebene Linie mindestens eines der Rechtecke, die von
     * Shape.getBounds() für dieübergebenen Shapes zurückgegeben werden, schneidet.
     */
    private static boolean lineIntersectsWithShapes(final Line2D line, final List<Shape> shapes) {
        for (Shape shape : shapes) {
            final Rectangle boundingRectangle = shape.getBounds();
            if (line.intersects(boundingRectangle)) {
                return true;
            }
        }
        return false;
    }

    /*
     * Verkürzt oder verlängert die übergebene Linie an ihrem Anfang um den übegebenen
     * Stretchfaktor. Beispiel: ein Strechtfaktor von 0,95 führt zu einer Linie, an deren
     * Anfang 5 % abgeschnitten wurden.
     */
    @SuppressWarnings("SameParameterValue")
    private static void stretchLineAtTheBeginning(final Line2D line, double stretchFactor) {
        final Point2D p1 = line.getP1();
        final Point2D p2 = line.getP2();
        line.setLine(p2.getX() + stretchFactor * (p1.getX() - p2.getX()), p2.getY() + stretchFactor * (p1.getY() - p2.getY()), p2.getX(), p2.getY());
    }

    /*
     * Verkürzt oder verlängert die übergebene Linie an ihrem Ende um den übegebenen
     * Stretchfaktor. Beispiel: ein Strechtfaktor von 0,95 führt zu einer Linie, an deren
     * Ende 5 % abgeschnitten wurden.
     */
    @SuppressWarnings("SameParameterValue")
    private static void stretchLineAtTheEnd(final Line2D line, double stretchFactor) {
        final Point2D p1 = line.getP1();
        final Point2D p2 = line.getP2();
        line.setLine(p1.getX(), p1.getY(), p1.getX() + stretchFactor * (p2.getX() - p1.getX()), p1.getY() + stretchFactor * (p2.getY() - p1.getY()));
    }

    /*
     * Liefert eine affine Transformation zurück, die eine Translation der Länge translationFactor
     * in Richtung des Winkels angle, der zur x-Achse berechnet wird, durchführt.
     */
    private static AffineTransform getAngleTransformation(final Double translationFactor, @Nullable final Double angle) {
        AffineTransform angleTransformation = new AffineTransform();
        if (null == angle || angle.isNaN()) {
            return angleTransformation;
        }
        angleTransformation.translate(-translationFactor * Math.sin(angle), translationFactor * Math.cos(angle));
        return angleTransformation;
    }

    /*
     * Liefert eine ShapeWithReferencePoint-Objekt für die übergebenen Daten zurück, das
     * in paintDisplayObject zum Zeichnen verwendet wird.
     */
    @Nullable
    private static ShapeWithReferencePoint getShape(Point2D point, PrimitiveForm primitiveForm) {
        final PrimitiveFormType type = primitiveForm.getType();
        final Point2D.Double translation = primitiveForm.getTranslation();
        if (type == PrimitiveFormType.RECHTECK) {
            double transX = 0;
            double transY = 0;
            if (translation != null) {
                transX = translation.getX();
                transY = -translation.getY();
            }
            final Double width = (Double) primitiveForm.getSpecificInformation("width");
            final Double height = (Double) primitiveForm.getSpecificInformation("height");
            Rectangle2D.Double rectangle =
                new Rectangle2D.Double(point.getX() + transX - width / 2., point.getY() + transY - height / 2., width, height);
            return new ShapeWithReferencePoint(rectangle, new Point2D.Double(point.getX() + transX, point.getY() + transY));    // Provisorium! ???
        } else if (type == PrimitiveFormType.KREIS) {
            double centerX = point.getX();
            double centerY = point.getY();
            if (translation != null) {
                centerX += translation.getX();
                centerY -= translation.getY();
            }
            double radius = (Double) primitiveForm.getSpecificInformation("radius");
            Ellipse2D.Double circle = new Ellipse2D.Double(centerX - radius, centerY - radius, 2 * radius, 2 * radius);
            return new ShapeWithReferencePoint(circle, new Point2D.Double(centerX, centerY));
        } else if (type == PrimitiveFormType.HALBKREIS) {
            double centerX = point.getX();
            double centerY = point.getY();
            if (translation != null) {
                centerX += translation.getX();
                centerY -= translation.getY();
            }
            double radius = (Double) primitiveForm.getSpecificInformation("radius");
            String orientation = (String) primitiveForm.getSpecificInformation("orientation");
            return new ShapeWithReferencePoint(getSemiCircle(centerX, centerY, radius, orientation), new Point2D.Double(centerX, centerY));
        }
        return null;
    }

    /*
     * Konstruiert einen Halbkreis zu den Aufrufparametern.
     */
    private static Shape getSemiCircle(double centerX, double centerY, Double radius, String orientation) {
        int[] take = {1, 1, 1, 0, 0, 1};
        Ellipse2D.Double circle = new Ellipse2D.Double(-radius, -radius, 2 * radius, 2 * radius);
        GeneralPath gPath = new GeneralPath();
        AffineTransform affineTransform = new AffineTransform();
        affineTransform.translate(centerX, centerY);
        switch (orientation) {
            case OBERER_HALBKREIS:
                affineTransform.rotate(Math.PI);
                break;
            case RECHTER_HALBKREIS:
                affineTransform.rotate(3 * Math.PI / 2);
                break;
            case LINKER_HALBKREIS:
                affineTransform.rotate(Math.PI / 2);
                break;
            case UNTERER_HALBKREIS:
                break;
        }
        final PathIterator pathIterator = circle.getPathIterator(affineTransform);
        for (int i = 0; i < 6; i++) {
            if (take[i] == 1) {
                double[] coords = new double[6];
                final int currentSegment = pathIterator.currentSegment(coords);
                if (currentSegment == PathIterator.SEG_MOVETO) {
                    gPath.moveTo(coords[0], coords[1]);
                } else if (currentSegment == PathIterator.SEG_CUBICTO) {
                    gPath.curveTo(coords[0], coords[1], coords[2], coords[3], coords[4], coords[5]);
                } else if (currentSegment == PathIterator.SEG_CLOSE) {
                    gPath.closePath();
                }
            }
            pathIterator.next();
        }
        return gPath;
    }

    /*
     * Gibt den Durchmesser zurück.
     */
    @Nullable
    private static Double determineDiameter(DisplayObject displayObject, DOTPoint dotPoint, String primitiveFormName) {
        final Double diameter;
        if (dotPoint.isPropertyStatic(primitiveFormName, DiameterProperty.getInstance())) {
            final Object valueOfStaticProperty = dotPoint.getValueOfStaticProperty(primitiveFormName, DiameterProperty.getInstance());
            if (valueOfStaticProperty != null) {
                if (valueOfStaticProperty instanceof Integer) {
                    diameter = ((Integer) valueOfStaticProperty).doubleValue();
                } else if (valueOfStaticProperty instanceof Double) {
                    diameter = (Double) valueOfStaticProperty;
                } else {
                    diameter = 0.;
                }
            } else {
                diameter = 0.;
            }
        } else {
            final PrimitiveFormPropertyPair primitiveFormPropertyPair =
                new PrimitiveFormPropertyPair(primitiveFormName, DiameterProperty.getInstance());
            final DisplayObjectTypeItem displayObjectTypeItem = displayObject.getDisplayObjectTypeItem(primitiveFormPropertyPair);
            if (displayObjectTypeItem == null) {
                return null;
            }
            final Object propertyValue = displayObjectTypeItem.getPropertyValue();
            if (propertyValue != null) {
                if (propertyValue instanceof Integer) {
                    diameter = ((Integer) propertyValue).doubleValue();
                } else if (propertyValue instanceof Double) {
                    diameter = (Double) displayObjectTypeItem.getPropertyValue();
                } else {
                    diameter = 0.;
                }
            } else if (Objects.equals(displayObjectTypeItem, DynamicDOTItem.NO_DATA_ITEM)) {
                diameter = 0.;
            } else if (Objects.equals(displayObjectTypeItem, DynamicDOTItem.NO_SOURCE_ITEM)) {
                diameter = 0.;
            } else {
                diameter = 0.;
            }
            // Die letzten drei Fälle werden noch gleichbehandelt, aber bei einer kommenden
            // Erweiterung kann hier unterschieden werden.

        }
        return diameter;
    }

    /*
     * Gibt die Strichbreite zurück.
     */
    @Nullable
    private static Double determineStrokeWidth(DisplayObject displayObject, DOTPoint dotPoint, String primitiveFormName) {
        final Double strokeWidth;
        if (dotPoint.isPropertyStatic(primitiveFormName, StrokeWidthProperty.getInstance())) {
            Object propertyValue = dotPoint.getValueOfStaticProperty(primitiveFormName, StrokeWidthProperty.getInstance());
            if (propertyValue != null) {
                if (propertyValue instanceof Double) {
                    strokeWidth = (Double) propertyValue;
                } else if (propertyValue instanceof Integer) {
                    strokeWidth = ((Integer) propertyValue).doubleValue();
                } else {
                    strokeWidth = 0.;
                }
            } else {
                strokeWidth = 0.;
            }
        } else {
            final PrimitiveFormPropertyPair primitiveFormPropertyPair =
                new PrimitiveFormPropertyPair(primitiveFormName, StrokeWidthProperty.getInstance());
            final DisplayObjectTypeItem displayObjectTypeItem = displayObject.getDisplayObjectTypeItem(primitiveFormPropertyPair);
            if (displayObjectTypeItem == null) {
                return null;
            }
            final Object propertyValue = displayObjectTypeItem.getPropertyValue();
            if (propertyValue != null) {
                if (propertyValue instanceof Double) {
                    strokeWidth = (Double) propertyValue;
                } else if (propertyValue instanceof Integer) {
                    strokeWidth = ((Integer) propertyValue).doubleValue();
                } else {
                    strokeWidth = 0.;
                }
            } else if (Objects.equals(displayObjectTypeItem, DynamicDOTItem.NO_DATA_ITEM)) {
                strokeWidth = 0.;
            } else if (Objects.equals(displayObjectTypeItem, DynamicDOTItem.NO_SOURCE_ITEM)) {
                strokeWidth = 0.;
            } else {
                strokeWidth = 0.;
            }
            // Die letzten drei Fälle werden noch gleichbehandelt, aber bei einer kommenden
            // Erweiterung muss hier unterschieden werden.
        }
        return strokeWidth;
    }

    /*
     * Gibt die Farbe zurück.
     */
    @Nullable
    private static Color determineFillColor(DisplayObject displayObject, DOTPoint dotPoint, String primitiveFormName) {
        final Color color;
        if (dotPoint.isPropertyStatic(primitiveFormName, FillingProperty.getInstance())) {
            final Object valueOfStaticProperty = dotPoint.getValueOfStaticProperty(primitiveFormName, FillingProperty.getInstance());
            if (valueOfStaticProperty != null) {
                if (valueOfStaticProperty instanceof Color) {
                    color = (Color) valueOfStaticProperty;
                } else if (valueOfStaticProperty instanceof String) {
                    color = ColorManager.getInstance().getColor((String) valueOfStaticProperty);
                } else {
                    color = null;
                }
            } else {
                color = null;
            }
        } else {
            final PrimitiveFormPropertyPair primitiveFormPropertyPair =
                new PrimitiveFormPropertyPair(primitiveFormName, FillingProperty.getInstance());
            final DisplayObjectTypeItem displayObjectTypeItem = displayObject.getDisplayObjectTypeItem(primitiveFormPropertyPair);
            if (displayObjectTypeItem == null) {
                return null;
            }
            final Object propertyValue = displayObjectTypeItem.getPropertyValue();
            if (propertyValue != null) {
                if (propertyValue instanceof Color) {
                    color = (Color) propertyValue;
                } else if (propertyValue instanceof String) {
                    color = ColorManager.getInstance().getColor((String) propertyValue);
                } else {
                    color = null;
                }
            } else if (Objects.equals(displayObjectTypeItem, DynamicDOTItem.NO_DATA_ITEM)) {
                color = ColorManager.getInstance().getColor("keine");
            } else if (Objects.equals(displayObjectTypeItem, DynamicDOTItem.NO_SOURCE_ITEM)) {
                color = ColorManager.getInstance().getColor("keine");
            } else {
                color = ColorManager.getInstance().getColor("keine");
            }
            // Die letzten drei Fälle werden noch gleichbehandelt, aber bei einer kommenden
            // Erweiterung kann hier unterschieden werden.
        }
        return color;
    }

    /*
     * Gibt die Tranzparenz zurück, und zwar als Wert zwischen 0 und 255, so wie er in einem
     * Color-Konstruktor gebraucht wird.
     */
    @Nullable
    private static Integer determineTransparency(DisplayObject displayObject, DOTPoint dotPoint, String primitiveFormName) {
        Integer transparency = null;
        if (dotPoint.isPropertyStatic(primitiveFormName, TransparencyProperty.getInstance())) {
            final Object propertyValue = dotPoint.getValueOfStaticProperty(primitiveFormName, TransparencyProperty.getInstance());
            if (propertyValue != null) {
                if (propertyValue instanceof Double) {
                    transparency = ((Double) propertyValue).intValue();
                } else if (propertyValue instanceof Integer) {
                    transparency = (Integer) propertyValue;
                }
                if (transparency != null && transparency <= 100) { // skaliere von % auf 0...255
                    transparency = 255 - (transparency * 255 / 100);
                }
            }
        } else {
            final PrimitiveFormPropertyPair primitiveFormPropertyPair =
                new PrimitiveFormPropertyPair(primitiveFormName, TransparencyProperty.getInstance());
            final DisplayObjectTypeItem displayObjectTypeItem = displayObject.getDisplayObjectTypeItem(primitiveFormPropertyPair);
            if (displayObjectTypeItem == null) {
                return null;
            }
            final Object propertyValue = displayObjectTypeItem.getPropertyValue();
            if (propertyValue != null) {
                if (propertyValue instanceof Double) {
                    transparency = ((Double) propertyValue).intValue();
                } else if (propertyValue instanceof Integer) {
                    transparency = (Integer) propertyValue;
                }
                if (transparency != null && transparency <= 100) { // skaliere von % auf 0...255
                    transparency = 255 - (transparency * 255 / 100);
                }
            } else if (Objects.equals(displayObjectTypeItem, DynamicDOTItem.NO_DATA_ITEM)) {
                transparency = 0;
            } else if (Objects.equals(displayObjectTypeItem, DynamicDOTItem.NO_SOURCE_ITEM)) {
                transparency = 0;
            } else {
                transparency = 0;
            }
            // Die letzten drei Fälle werden noch gleichbehandelt, aber bei einer kommenden
            // Erweiterung kann hier unterschieden werden.
        }
        return transparency;
    }

    /*
     * Gibt den Textstil zurück.
     */
    private static int determineTextStyle(DisplayObject displayObject, DOTPoint dotPoint, String primitiveFormName) {
        final int style;
        if (dotPoint.isPropertyStatic(primitiveFormName, TextStyleProperty.getInstance())) {
            Object valueOfStaticProperty = dotPoint.getValueOfStaticProperty(primitiveFormName, TextStyleProperty.getInstance());
            style = getValidTextStyle(valueOfStaticProperty);
        } else {
            final PrimitiveFormPropertyPair primitiveFormPropertyPair =
                new PrimitiveFormPropertyPair(primitiveFormName, TextStyleProperty.getInstance());
            final DisplayObjectTypeItem displayObjectTypeItem = displayObject.getDisplayObjectTypeItem(primitiveFormPropertyPair);
            if (displayObjectTypeItem == null) {
                return Font.PLAIN;
            }
            final Object propertyValue = displayObjectTypeItem.getPropertyValue();
            if (propertyValue != null && propertyValue instanceof Integer) {
                style = getValidTextStyle(propertyValue);
            } else if (Objects.equals(displayObjectTypeItem, DynamicDOTItem.NO_DATA_ITEM)) {
                style = Font.PLAIN;
            } else if (Objects.equals(displayObjectTypeItem, DynamicDOTItem.NO_SOURCE_ITEM)) {
                style = Font.PLAIN;
            } else {
                style = Font.PLAIN;
            }
            // Die letzten drei Fälle werden noch gleichbehandelt, aber bei einer kommenden
            // Erweiterung muss hier unterschieden werden.
        }
        return style;
    }

    private static int getValidTextStyle(@Nullable final Object object) {
        if (object != null) {
	        if (object instanceof Integer value) {
                if (value >= 0 && value <= 2) {
                    return value;
                } else {
                    return Font.PLAIN;
                }
            } else if (object instanceof TextStyleProperty.Styles) {
                return ((TextStyleProperty.Styles) object).getIntValue();
            } else {
                return Font.PLAIN;
            }
        } else {
            return Font.PLAIN;
        }
    }

    /*
     * Gibt die Textgröße zurück.
     */
    private static int determineTextSize(DisplayObject displayObject, DOTPoint dotPoint, String primitiveFormName) {
        final int size;
        if (dotPoint.isPropertyStatic(primitiveFormName, TextSizeProperty.getInstance())) {
            final Object valueOfStaticProperty = dotPoint.getValueOfStaticProperty(primitiveFormName, TextSizeProperty.getInstance());
            if (valueOfStaticProperty != null) {
                if (valueOfStaticProperty instanceof Double) {
                    size = ((Double) valueOfStaticProperty).intValue();
                } else if (valueOfStaticProperty instanceof Integer) {
                    size = (Integer) valueOfStaticProperty;
                } else {
                    size = 12;
                }
            } else {
                size = 12;
            }
        } else {
            final PrimitiveFormPropertyPair primitiveFormPropertyPair =
                new PrimitiveFormPropertyPair(primitiveFormName, TextSizeProperty.getInstance());
            final DisplayObjectTypeItem displayObjectTypeItem = displayObject.getDisplayObjectTypeItem(primitiveFormPropertyPair);
            if (displayObjectTypeItem == null) {
                return 0;
            }
            final Object propertyValue = displayObjectTypeItem.getPropertyValue();
            if (propertyValue != null) {
                if (propertyValue instanceof Double) {
                    size = ((Double) propertyValue).intValue();
                } else if (propertyValue instanceof Integer) {
                    size = (Integer) propertyValue;
                } else {
                    size = 12;
                }
            } else if (Objects.equals(displayObjectTypeItem, DynamicDOTItem.NO_DATA_ITEM)) {
                size = 0;
            } else if (Objects.equals(displayObjectTypeItem, DynamicDOTItem.NO_SOURCE_ITEM)) {
                size = 0;
            } else {
                size = 0;
            }
            // Die letzten drei Fälle werden noch gleichbehandelt, aber bei einer kommenden
            // Erweiterung kann hier unterschieden werden.
        }
        return size;
    }

    /*
     * Gibt die Farbe zurück.
     */
    @Nullable
    private static Color determineColor(DisplayObject displayObject, DOTPoint dotPoint, String primitiveFormName) {
        final Color color;
        if (dotPoint.isPropertyStatic(primitiveFormName, ColorProperty.getInstance())) {
            final Object valueOfStaticProperty = dotPoint.getValueOfStaticProperty(primitiveFormName, ColorProperty.getInstance());
            if (valueOfStaticProperty != null) {
                if (valueOfStaticProperty instanceof Color) {
                    color = (Color) valueOfStaticProperty;
                } else if (valueOfStaticProperty instanceof String) {
                    color = ColorManager.getInstance().getColor((String) valueOfStaticProperty);
                } else {
                    color = null;
                }
            } else {
                color = null;
            }
        } else {
            final PrimitiveFormPropertyPair primitiveFormPropertyPair = new PrimitiveFormPropertyPair(primitiveFormName, ColorProperty.getInstance());
            final DisplayObjectTypeItem displayObjectTypeItem = displayObject.getDisplayObjectTypeItem(primitiveFormPropertyPair);
            if (displayObjectTypeItem == null) {
                return null;
            }
            final Object propertyValue = displayObjectTypeItem.getPropertyValue();
            if (propertyValue != null) {
                if (propertyValue instanceof Color) {
                    color = (Color) propertyValue;
                } else if (propertyValue instanceof String) {
                    color = ColorManager.getInstance().getColor((String) propertyValue);
                } else {
                    color = null;
                }
            } else if (Objects.equals(displayObjectTypeItem, DynamicDOTItem.NO_DATA_ITEM)) {
                color = ColorManager.getInstance().getColor("keine");
            } else if (Objects.equals(displayObjectTypeItem, DynamicDOTItem.NO_SOURCE_ITEM)) {
                color = ColorManager.getInstance().getColor("keine");
            } else {
                color = ColorManager.getInstance().getColor("keine");
            }
            // Die letzten drei Fälle werden noch gleichbehandelt, aber bei einer kommenden
            // Erweiterung kann hier unterschieden werden.
        }
        return color;
    }

    /*
     * Gibt den Text zurück.
     */
    @SuppressWarnings("OverlyNestedMethod")
    @Nullable
    private static String determineText(DisplayObject displayObject, DOTPoint dotPoint, String primitiveFormName) {
        // Text hat eine Besonderheit: im Fall einer statischen Eigenschaft kommen die 'vordefinierten
        // Funktionen' und ein konstanter, vom Benutzer definierter Text in Frage, während im Fall
        // einer dynamischen Eigenschaft auch das angemeldete Attribut dargestellt werden kann.
        final String text;
        final PrimitiveFormPropertyPair primitiveFormPropertyPair;
        if (dotPoint.isPropertyStatic(primitiveFormName, TextProperty.getInstance())) {
            Object propertyValue = dotPoint.getValueOfStaticProperty(primitiveFormName, TextProperty.getInstance());
            if (propertyValue != null && propertyValue instanceof String) {
                text = (String) propertyValue;
            } else {
                text = "";
            }
            primitiveFormPropertyPair = null;
            if (text.equals(DYNAMIC_ATTRIBUTE_UNSCALED) || text.equals(DYNAMIC_ATTRIBUTE_SCALED)) {    // das wäre ein Fehler
                return "";
            }
        } else {
            primitiveFormPropertyPair = new PrimitiveFormPropertyPair(primitiveFormName, TextProperty.getInstance());
            final DisplayObjectTypeItem displayObjectTypeItem = displayObject.getDisplayObjectTypeItem(primitiveFormPropertyPair);
            if (displayObjectTypeItem == null) {
                return null;
            }
            final Object propertyValue = displayObjectTypeItem.getPropertyValue();
            if (propertyValue != null && propertyValue instanceof String) {
                text = (String) propertyValue;
            } else if (Objects.equals(displayObjectTypeItem, DynamicDOTItem.NO_DATA_ITEM)) {
                return "Keine Daten";
            } else if (Objects.equals(displayObjectTypeItem, DynamicDOTItem.NO_SOURCE_ITEM)) {
                return "Keine Quelle";
            } else {
                return null;
            }
        }
	    if (displayObject instanceof OnlineDisplayObject onlineDisplayObject) {
            switch (text) {
                case DOTPointPainter.DYNAMIC_ATTRIBUTE_UNSCALED: {
                    final Data value = onlineDisplayObject.getValue(primitiveFormPropertyPair);
                    if (value != null) {
                        return value.asUnscaledValue().getValueText();
                    } else {
                        return "";
                    }
                }
                case DOTPointPainter.DYNAMIC_ATTRIBUTE_SCALED: {
                    final Data value = onlineDisplayObject.getValue(primitiveFormPropertyPair);
                    if (value != null) {
                        final String name = value.asScaledValue().getText();
                        if (!name.isEmpty()) {
                            return name;
                        } else {
                            final Double doubleValue = value.asUnscaledValue().doubleValue();
                            Integer intValue = doubleValue.intValue();
                            if (doubleValue.equals(intValue.doubleValue())) {
                                return intValue.toString();
                            } else {
                                return doubleValue.toString();
                            }
                        }
                    } else {
                        return "";
                    }
                }
                case DOTPointPainter.GET_NAME:
                    return onlineDisplayObject.getSystemObject().getName();
                case DOTPointPainter.GET_NAME_OR_PID_OR_ID:
                    return onlineDisplayObject.getSystemObject().getNameOrPidOrId();
                case DOTPointPainter.GET_PID_OR_ID:
                    return onlineDisplayObject.getSystemObject().getPidOrId();
                case DOTPointPainter.GET_INFO_GET_DESCRIPTION:
                    return onlineDisplayObject.getSystemObject().getInfo().getDescription();
            }
        } else if (displayObject instanceof CsvDisplayObject) {
            if (text.equals(DOTPointPainter.GET_NAME) || text.equals(DOTPointPainter.GET_NAME_OR_PID_OR_ID)) {
                return displayObject.getName();
            }
        }
        return text;    // Greift im Fall eines konstanten benutzer-definierten Textes und bei unsinnigen Kombinationen
        // (z.B. CsvDisplayObject und DOTPointPainter.GET_PID_OR_ID).
    }

    @Override
    public void paintDisplayObject(MapPane mapPane, Graphics2D g2D, DisplayObject displayObject, boolean selected) {
        DisplayObjectType dot = displayObject.getDOTCollection().getDisplayObjectType(mapPane.getMapScale().intValue());
	    if (!(dot instanceof DOTPoint dotPoint)) {
            return;
        }
        List<Object> coor = displayObject.getCoordinates(0);
	    if (coor.isEmpty() || !(coor.get(0) instanceof PointWithAngle pointWithAngle)) {
            return;
        }
        AffineTransform angleTransformation = getAngleTransformation(dotPoint.getTranslationFactor(), pointWithAngle.getAngle());
        final Point2D point = pointWithAngle.getPoint();
        final Double xAsDouble = point.getX();
        final Double yAsDouble = point.getY();
        List<Shape> drawnShapes = new ArrayList<>();
        Ellipse2D.Double pointCircle = null;
        for (PrimitiveForm primitiveForm : dotPoint.getPrimitiveForms()) {
            final PrimitiveFormType type = primitiveForm.getType();
            if (type.equals(PrimitiveFormType.PUNKT)) {
                final Color color = determineColor(displayObject, dotPoint, primitiveForm.getName());
                final Double diameter = determineDiameter(displayObject, dotPoint, primitiveForm.getName());
                if ((diameter == null) || (color == null)) {
                    continue;
                }
                if (!selected) {
                    g2D.setColor(color);
                } else {
                    g2D.setColor(color.darker());
                }
                final Double radius = diameter / 2.;
                pointCircle = new Ellipse2D.Double(xAsDouble - radius, yAsDouble - radius, 2 * radius, 2 * radius);
                g2D.fill(pointCircle);
            } else if (type.equals(PrimitiveFormType.RECHTECK) || type.equals(PrimitiveFormType.KREIS) || type.equals(PrimitiveFormType.HALBKREIS)) {
                final Color fillColor = determineFillColor(displayObject, dotPoint, primitiveForm.getName());
                final Color fillColorWithTransparency;
                if (fillColor == null) {    // keine Farbe, deshalb transparent setzen
                    fillColorWithTransparency = new Color(0, 0, 0, 0);
                } else if (fillColor.getAlpha() == 0) { // Farbe ist schon transparent
                    fillColorWithTransparency = fillColor;
                } else {
                    final Integer transparency = determineTransparency(displayObject, dotPoint, primitiveForm.getName());
                    if (transparency == null) {
                        fillColorWithTransparency = new Color(fillColor.getRed(), fillColor.getGreen(), fillColor.getBlue());
                    } else {
                        fillColorWithTransparency = new Color(fillColor.getRed(), fillColor.getGreen(), fillColor.getBlue(), transparency);
                    }
                }
                if (!selected) {
                    g2D.setColor(fillColorWithTransparency);
                } else {
                    g2D.setColor(fillColorWithTransparency.brighter());
                }
                final ShapeWithReferencePoint shapeWithRefPoint = getShape(point, primitiveForm);
                if (shapeWithRefPoint == null) {
                    continue;
                }
                final Shape shape = angleTransformation.createTransformedShape(shapeWithRefPoint.getShape());
                g2D.fill(shape);
                if (!selected) {
                    g2D.setColor(Color.gray);
                } else {
                    g2D.setColor(Color.black);
                }
                final Double strokeWidth = determineStrokeWidth(displayObject, dotPoint, primitiveForm.getName());
                if (strokeWidth != null && strokeWidth.floatValue() != 0.0f) {
                    if (!selected) {
                        g2D.setStroke(new BasicStroke(strokeWidth.floatValue()));
                    } else {
                        g2D.setStroke(new BasicStroke(strokeWidth.floatValue() + 1));
                    }
                    g2D.draw(shape);
                }
                drawnShapes.add(shape);
            } else if (type.equals(PrimitiveFormType.TEXTDARSTELLUNG)) {
                final Color color = determineColor(displayObject, dotPoint, primitiveForm.getName());
                if (!selected) {
                    g2D.setColor(color);
                } else {
                    if (color != null) {
                        g2D.setColor(color.brighter());
                    }
                }
                final int style = determineTextStyle(displayObject, dotPoint, primitiveForm.getName());
                final int size = determineTextSize(displayObject, dotPoint, primitiveForm.getName());
                Font font = new Font(null, style, size);
                g2D.setFont(font);
                String text = determineText(displayObject, dotPoint, primitiveForm.getName());
                if (text == null) {
                    continue;
                }
                Point2D.Double translation1 = primitiveForm.getTranslation();
                Point2D.Double translation2 = new Point2D.Double(xAsDouble + translation1.getX(), yAsDouble - translation1.getY());
                final Point2D.Double translation3 = new Point2D.Double();
                angleTransformation.transform(translation2, translation3);
//				final Double x = translation3.getX(); // ALT
//				final Double y = translation3.getY(); // ALT
//					g2D.drawString( text, x.intValue(), y.intValue());  // ALT
                /* Beginn NEU */
                AffineTransform at = g2D.getTransform();
                final Point2D.Double devicePoint = new Point2D.Double();
                at.transform(translation3, devicePoint);
                g2D.setTransform(new AffineTransform());
                final Double deviceX = devicePoint.getX();
                final Double deviceY = devicePoint.getY();
                g2D.drawString(text, deviceX.floatValue(), deviceY.floatValue());
                g2D.setTransform(at);
                /* Ende NEU */
            }
        }
        if (dotPoint.getJoinByLine()) {
            Point2D.Double translation1 = new Point2D.Double(xAsDouble, yAsDouble);
            final Point2D.Double translation2 = new Point2D.Double();
            angleTransformation.transform(translation1, translation2);
            final Double x = translation2.getX();
            final Double y = translation2.getY();
            Line2D.Double line = new Line2D.Double(xAsDouble.intValue(), yAsDouble.intValue(), x.intValue(), y.intValue());
            if (pointCircle != null) {
                List<Shape> shapes = new ArrayList<>();
                shapes.add(pointCircle);
                int counter = 0;
                while (lineIntersectsWithShapes(line, shapes) && counter < 10) {
                    stretchLineAtTheBeginning(line, 0.9);
                    counter++;
                }
            }
            int counter = 0;
            while (lineIntersectsWithShapes(line, drawnShapes) && counter < 10) {
                stretchLineAtTheEnd(line, 0.9);
                counter++;
            }
            g2D.setStroke(new BasicStroke(1.f));
            g2D.draw(line);
        }
    }

    /*
     * Gibt zu dem Darstellungsobjekt und dem Typ das umschließende achsen-parallele Rechteck
     * zurück.
     */
    @Nullable
    @Override
    public Rectangle getBoundingRectangle(DisplayObject displayObject, int type) {
        // Provisorische Implementation
        Rectangle rectangle = null;
        for (Object o : displayObject.getCoordinates(type)) {
	        if (o instanceof PointWithAngle pwa) {
                rectangle = new Rectangle((int) (pwa.getPoint().getX() - 250), (int) (pwa.getPoint().getY() - 150), 500, 300);
	        } else if (o instanceof Point2D.Double p) {
                rectangle = new Rectangle((int) (p.getX() - 250), (int) (p.getY() - 150), 500, 300);
            }
        }
        return rectangle;

        // Anfang einer optimierten Implementation; optional besteht auch die Möglichkeit, den
        // Kode von paintDisplayObject() dahingehend zu gebrauchen, dass man aus all den dortigen
        // Shapes das umfassende Rechteck bereits berechnet, dem OnlineDisplayObject es verrät, und es
        // so wieder zurückbekommt.
		/*
		Rectangle primitiveFormsRect = null;
		for ( DisplayObjectType displayObjectType : displayObject.getDOTCollection().values()) {
			DOTPoint dotPoint = (DOTPoint) displayObjectType;
			final Double translationFactor = dotPoint.getTranslationFactor();
			for ( PrimitiveForm primitiveForm : dotPoint.getPrimitiveForms()) {
				// Hier muss die lokale Translation berechnet werden, die sich aus Translation,
				// TranslationsFaktor und gegebenfalls dem Winkel ergibt!
				final Point2D.Double translation = primitiveForm.getTranslation();
				final PrimitiveFormType pfType = primitiveForm.getType();
				if ( pfType == PrimitiveFormType.PUNKT) {
					if ( primitiveFormsRect == null) {
						primitiveFormsRect = new Rectangle();
					} else {
						Point p = new Point(0,0);
						primitiveFormsRect.add( p);
					}
				} else if ( pfType == PrimitiveFormType.RECHTECK) {
					// To-Do
				}
				// Weitere Fälle To-Do
			}
		}*/
    }

    /**
     * Gibt zu dem Darstellungsobjekt und dem Typ die Koordinaten zurück.
     */
    @Override
    public List<Object> getCoordinates(List<Object> coordinates, int type) {
        // Bisher gibt es beim Punkt-Plugin keine Typ-Unterscheidung.
        return coordinates;
    }

    /*
     * Ein ShapeWithReferencePoint kapselt ein Paar bestehend aus einem Shape und einem
     * Referenzpunkt.
     *
     * @author Kappich Systemberatung
     *
     */
    private static class ShapeWithReferencePoint {

        private final Shape _shape;
        private final Point2D.Double _referencePoint;

        /**
         * Konstruiert ein ShapeWithReferencePoint aus den übergebenen Informationen
         */
        public ShapeWithReferencePoint(Shape shape, Point2D.Double referencePoint) {
            super();
            _shape = shape;
            _referencePoint = referencePoint;
        }

        /**
         * Gibt das Shape zurück.
         */
        public Shape getShape() {
            return _shape;
        }

        /**
         * Gibt den ReferenzPunkt zurück.
         */
        @SuppressWarnings("unused")
        public Point2D.Double getReferencePoint() {
            return _referencePoint;
        }
    }
}
