/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.pointPlugin;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.colorManagement.ColorManager;
import de.kappich.pat.gnd.displayObjectToolkit.*;
import de.kappich.pat.gnd.pluginInterfaces.DOTDefinitionPanel;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.pointPlugin.DOTPoint.PrimitiveForm;
import de.kappich.pat.gnd.pointPlugin.DOTPoint.PrimitiveFormType;
import de.kappich.pat.gnd.properties.ColorProperty;
import de.kappich.pat.gnd.properties.DiameterProperty;
import de.kappich.pat.gnd.properties.FillingProperty;
import de.kappich.pat.gnd.properties.Property;
import de.kappich.pat.gnd.properties.StrokeWidthProperty;
import de.kappich.pat.gnd.properties.TextProperty;
import de.kappich.pat.gnd.properties.TextSizeProperty;
import de.kappich.pat.gnd.properties.TextStyleProperty;
import de.kappich.pat.gnd.properties.TransparencyProperty;
import de.kappich.pat.gnd.utils.SpringUtilities;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.ComboBoxEditor;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSpinner;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SpinnerModel;
import javax.swing.SpinnerNumberModel;
import javax.swing.SpringLayout;
import javax.swing.event.ChangeListener;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.TableModel;
import javax.swing.table.TableRowSorter;

/**
 * Der Definitionsdialog für Darstellungstypen von Punktobjekten.
 * <p>
 * DOTPointPanel implementiert das Interface DOTDefinitionPanel für das Punkt-Plugin. Hierzu interagiert es intensiv und software-technisch unsauber
 * mit dem umschließenden DOTDefinitionPanel.
 *
 * @author Kappich Systemberatung
 */
public class DOTPointPanel implements DOTDefinitionPanel {

    /**
     * Der Standard-Fontstil {@code PLAIN}.
     */
    public static final String PLAIN_FONT_STYLE = "Standard";
    /**
     * Der kursive Fontstil {@code ITALIC}.
     */
    public static final String ITALIC_FONT_STYLE = "Kursiv";
    /**
     * Der kursive Fontstil {@code BOLD}.
     */
    public static final String BOLD_FONT_STYLE = "Fett";
    private static final Object[] FONT_ITEMS = {PLAIN_FONT_STYLE, ITALIC_FONT_STYLE, BOLD_FONT_STYLE};
    private final DOTDefinitionDialog _dotDefinitionDialog;
    private final DataModel _configuration;
    private final JSpinner _translationFactorSpinner = new JSpinner();
    private final JCheckBox _joinByLineCheckBox = new JCheckBox();
    private final Map<PrimitiveFormPropertyPair, JPanel> _staticPanels = new HashMap<>();
    private final Map<PrimitiveFormPropertyPair, JPanel> _dynamicPanels = new HashMap<>();
    private final Map<PrimitiveFormPropertyPair, JTable> _dynamicTables = new HashMap<>();

    /**
     * Konstruiert einen DOTPointPanel.
     *
     * @param dotDefinitionDialog das umgebende Fenster
     */
    DOTPointPanel(final DOTDefinitionDialog dotDefinitionDialog) {
        _dotDefinitionDialog = dotDefinitionDialog;
        final ClientDavInterface connection = _dotDefinitionDialog.getConnection();
        _configuration = connection.getDataModel();
        if (!(dotDefinitionDialog.getDisplayObjectType() instanceof DOTPoint)) {
            throw new IllegalArgumentException();
        }
        addToolTips();
    }

    private static SpinnerNumberModel getNewDiameterSpinnerModel() {
        return new SpinnerNumberModel(0, 0, 10000, 1);
    }

    private static SpinnerNumberModel getNewStrokeWidthSpinnerModel() {
        return new SpinnerNumberModel(0.5, 0.0, 10000.0, 0.1);
    }

    private static SpinnerNumberModel getNewTransparencySpinnerModel() {
        return new SpinnerNumberModel(0, 0, 100, 1);
    }

    private static SpinnerNumberModel getNewTextSizeSpinnerModel() {
        return new SpinnerNumberModel(20, 0, 10000, 1);
    }

    @Override
    public JPanel getAdditionalCharacteristicsPanel(final DisplayObjectType displayObjectType) {
        JLabel translationFactorLabel = new JLabel("Verschiebungsfaktor: ");
        SpinnerModel spinnerModel = new SpinnerNumberModel(100, 0, 5000, 1);
        _translationFactorSpinner.setModel(spinnerModel);
        _translationFactorSpinner.setMaximumSize(new Dimension(60, 30));
        _translationFactorSpinner.setEnabled(_dotDefinitionDialog.isEditable());
        JPanel translationPanel = new JPanel();
        translationPanel.setLayout(new BoxLayout(translationPanel, BoxLayout.X_AXIS));
        translationPanel.setBorder(BorderFactory.createEmptyBorder(0, 0, 10, 10));
        translationPanel.add(translationFactorLabel);
        translationPanel.add(_translationFactorSpinner);

        JLabel joinByLineLabel = new JLabel("Verbindungslinie: ");
        _joinByLineCheckBox.setSelected(false);
        _joinByLineCheckBox.setEnabled(_dotDefinitionDialog.isEditable());
        JPanel joinByLinePanel = new JPanel();
        joinByLinePanel.setLayout(new BoxLayout(joinByLinePanel, BoxLayout.X_AXIS));
        joinByLinePanel.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
        joinByLinePanel.add(joinByLineLabel);
        joinByLinePanel.add(_joinByLineCheckBox);

        JPanel invisiblePanel = new JPanel();
        invisiblePanel.add(new JTextField());
        invisiblePanel.setVisible(false);

        JPanel thePanel = new JPanel();
        thePanel.setBorder(BorderFactory.createEmptyBorder(10, 10, 0, 10));
        thePanel.setLayout(new SpringLayout());
        thePanel.add(translationPanel);
        thePanel.add(joinByLinePanel);
        thePanel.add(invisiblePanel);
        SpringUtilities.makeCompactGrid(thePanel, 3, 5, 5);

        if (displayObjectType != null) {
            DOTPoint dotPoint = (DOTPoint) displayObjectType;
            _translationFactorSpinner.setValue(dotPoint.getTranslationFactor());
            _joinByLineCheckBox.setSelected(dotPoint.getJoinByLine());
        }
        addChangeListeners(); // Erst jetzt, denn sonst werden die Setzungen von _translationFactorSpinner und _joinByLineCheckBox schon verarbeitet!

        return thePanel;
    }

    private void addChangeListeners() {
        ChangeListener translationChangeListener = e -> _dotDefinitionDialog.setSomethingChanged(true);
        _translationFactorSpinner.addChangeListener(translationChangeListener);

        ChangeListener joinByLineChangeListener = e -> _dotDefinitionDialog.setSomethingChanged(true);
        _joinByLineCheckBox.addChangeListener(joinByLineChangeListener);
    }

    @Override
    @Nullable
    public JPanel getDOTItemDefinitionPanel() {
        DOTPoint dotPoint = (DOTPoint) _dotDefinitionDialog.getScratchDisplayObjectType();
        final PrimitiveForm primitiveForm = dotPoint.getPrimitiveForm(_dotDefinitionDialog.getSelectedPrimitiveForm());
        final Property property = _dotDefinitionDialog.getSelectedProperty();
        if ((primitiveForm == null) || (property == null)) {
            return null;
        }
        PrimitiveFormPropertyPair pfpPair = new PrimitiveFormPropertyPair(primitiveForm.getName(), property);
        final boolean isStatic = _dotDefinitionDialog.getStaticCheckBoxState();
        if (isStatic) {
            if (!_staticPanels.containsKey(pfpPair)) {
                _staticPanels.put(pfpPair, createStaticCenterPanel(primitiveForm, property));
            }
            return _staticPanels.get(pfpPair);
        } else {
            if (!_dynamicPanels.containsKey(pfpPair)) {
                _dynamicPanels.put(pfpPair, createDynamicCenterPanel(primitiveForm, property));
            }
            return _dynamicPanels.get(pfpPair);
        }
    }

    @Override
    public boolean saveDisplayObjectType() {
        DOTPoint newDOTPoint = (DOTPoint) _dotDefinitionDialog.getScratchDisplayObjectType().getCopy(null);
        final String name = _dotDefinitionDialog.getNameText();
        if ((name == null) || (name.isEmpty())) {
            JOptionPane.showMessageDialog(new JFrame(), "Bitte geben Sie einen Namen an!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return false;
        }
        if (!_dotDefinitionDialog.isReviseOnly()) {
            if (_dotDefinitionDialog.getDotManager().containsDisplayObjectType(name)) {
                JOptionPane
                    .showMessageDialog(new JFrame(), "Ein Darstellungstyp mit diesem Namen existiert bereits!", "Fehler", JOptionPane.ERROR_MESSAGE);
                return false;
            }
        }
        newDOTPoint.setName(name);
        newDOTPoint.setInfo(_dotDefinitionDialog.getInfoText());
        final Object value = _translationFactorSpinner.getValue();
	    if (value instanceof Number number) {
            newDOTPoint.setTranslationFactor(number.doubleValue());
        }
        newDOTPoint.setJoinByLine(_joinByLineCheckBox.isSelected());
        _dotDefinitionDialog.getDotManager().saveDisplayObjectType(newDOTPoint);
        _dotDefinitionDialog.setDisplayObjectType(newDOTPoint, true);
        return true;
    }

    @Nullable
    private JPanel createStaticCenterPanel(PrimitiveForm primitiveForm, Property property) {
        final PrimitiveFormType primitiveFormType = primitiveForm.getType();
        if (primitiveFormType.equals(PrimitiveFormType.PUNKT)) {
            if (property.equals(DiameterProperty.getInstance())) {
                return createStaticDiameterPanel(primitiveForm, property);
            } else if (property.equals(ColorProperty.getInstance())) {
                return createStaticColorPanel(primitiveForm, property);
            }
        } else if (primitiveFormType.equals(PrimitiveFormType.RECHTECK) || primitiveFormType.equals(PrimitiveFormType.KREIS) ||
                   primitiveFormType.equals(PrimitiveFormType.HALBKREIS)) {
            if (property.equals(FillingProperty.getInstance())) {
                return createStaticColorPanel(primitiveForm, property);
            } else if (property.equals(StrokeWidthProperty.getInstance())) {
                return createStaticStrokeWidthPanel(primitiveForm, property);
            } else if (property.equals(TransparencyProperty.getInstance())) {
                return createStaticTransparencyPanel(primitiveForm, property);
            }
        } else if (primitiveFormType.equals(PrimitiveFormType.TEXTDARSTELLUNG)) {
            if (property.equals(ColorProperty.getInstance())) {
                return createStaticColorPanel(primitiveForm, property);
            } else if (property.equals(TextSizeProperty.getInstance())) {
                return createStaticTextsizePanel(primitiveForm, property);
            } else if (property.equals(TextProperty.getInstance())) {
                return createStaticTextPanel(primitiveForm, property);
            } else if (property.equals(TextStyleProperty.getInstance())) {
                return createStaticTextstylePanel(primitiveForm, property);
            }
        }
        return null;
    }

    private StaticPanel createStaticDiameterPanel(final PrimitiveForm primitiveForm, final Property property) {
        return createStaticSpinnerPanel(primitiveForm, property, "Durchmesser: ", getNewDiameterSpinnerModel());
    }

    private StaticPanel createStaticColorPanel(PrimitiveForm primitiveForm, Property property) {
        return createStaticComboBoxPanel(primitiveForm, property, "Farbe: ", ColorManager.getInstance().getColorNames());
    }

    private StaticPanel createStaticStrokeWidthPanel(PrimitiveForm primitiveForm, Property property) {
        return createStaticSpinnerPanel(primitiveForm, property, "Strichbreite: ", getNewStrokeWidthSpinnerModel());
    }

    private StaticPanel createStaticTransparencyPanel(PrimitiveForm primitiveForm, Property property) {
        return createStaticSpinnerPanel(primitiveForm, property, "Tranzparenz in %: ", getNewTransparencySpinnerModel());
    }

    private StaticPanel createStaticTextsizePanel(PrimitiveForm primitiveForm, Property property) {
        return createStaticSpinnerPanel(primitiveForm, property, "Schriftgröße: ", getNewTextSizeSpinnerModel());
    }

    private StaticPanel createStaticTextPanel(PrimitiveForm primitiveForm, Property property) {
        StaticPanel theTextPanel = createStaticComboBoxPanel(primitiveForm, property, "Text: ", DOTPointPainter.STATIC_TEXT_ITEMS);
        final JComboBox<Object> theComboBox = theTextPanel.getComboBox();
        if (theComboBox == null) {
            return theTextPanel;
        }

        ItemListener comboBoxItemListener = e -> {
            if (e.getStateChange() == ItemEvent.SELECTED) {
                if (theComboBox.getSelectedIndex() == DOTPointPainter.STATIC_TEXT_ITEMS.length - 1) {
                    theComboBox.setEditable(true);
                } else {
                    theComboBox.setEditable(false);
                }
            }
        };
        theComboBox.addItemListener(comboBoxItemListener);

        ActionListener editorActionListener = e -> {
            final int length = DOTPointPainter.STATIC_TEXT_ITEMS.length;
            final int index = length - 1;
            final ComboBoxEditor editor = theComboBox.getEditor();
            String editedItem = (String) editor.getItem();
            theComboBox.insertItemAt(editedItem, index);
            final DefaultComboBoxModel<Object> model = (DefaultComboBoxModel<Object>) theComboBox.getModel();
            if (model.getSize() > length) { // Sieht komisch aus, aber der direkte Weg ging nicht.
                model.removeElementAt(length);
            }
        };
        theComboBox.getEditor().addActionListener(editorActionListener);

        return theTextPanel;
    }

    private StaticPanel createStaticTextstylePanel(PrimitiveForm primitiveForm, Property property) {
        return createStaticComboBoxPanel(primitiveForm, property, "Textstil: ", FONT_ITEMS);
    }

    private StaticPanel createStaticComboBoxPanel(PrimitiveForm primitiveForm, Property property, String labelText, Object[] items) {
        JLabel label = new JLabel(labelText);
        JComboBox<Object> comboBox = new JComboBox<>(items);
        comboBox.setMaximumSize(new Dimension(300, 25));
        comboBox.setEnabled(_dotDefinitionDialog.isEditable());

        StaticPanel thePanel = new StaticPanel(primitiveForm, property);
        thePanel.setMaximumSize(new Dimension(300, 25));
        thePanel.setLayout(new SpringLayout());
//		thePanel.setBorder(BorderFactory.createMatteBorder(2, 2, 2, 2, Color.BLACK));
        thePanel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
        thePanel.add(label);
        thePanel.add(comboBox);
        SpringUtilities.makeCompactGrid(thePanel, 2, 5, 5);

        thePanel.setValue(comboBox);
        thePanel.addListener(comboBox);

        return thePanel;
    }

    private StaticPanel createStaticSpinnerPanel(PrimitiveForm primitiveForm, Property property, String labelText, SpinnerModel spinnerModel) {
        JLabel label = new JLabel(labelText);
        JSpinner spinner = new JSpinner(spinnerModel);
        spinner.setMaximumSize(new Dimension(200, 20));
        spinner.setEnabled(_dotDefinitionDialog.isEditable());

        StaticPanel thePanel = new StaticPanel(primitiveForm, property);
        thePanel.setLayout(new SpringLayout());
//		thePanel.setBorder(BorderFactory.createMatteBorder(2, 2, 2, 2, Color.BLACK));
        thePanel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
        thePanel.add(label);
        thePanel.add(spinner);
        SpringUtilities.makeCompactGrid(thePanel, 2, 5, 5);

        thePanel.setValue(spinner);
        thePanel.addListener(spinner);

        return thePanel;
    }

    private JPanel createDynamicCenterPanel(PrimitiveForm primitiveForm, Property property) {
        final JTable theTable = new JTable();
        PrimitiveFormPropertyPair pfpPair = new PrimitiveFormPropertyPair(primitiveForm.getName(), property);
        _dynamicTables.put(pfpPair, theTable);
        DOTPoint dotPoint = (DOTPoint) _dotDefinitionDialog.getScratchDisplayObjectType();
        final DynamicDOTItemManager tableModel = (DynamicDOTItemManager) dotPoint.getTableModel(primitiveForm, property);
        theTable.setModel(tableModel);

        class NumberComparator implements Comparator<Number> {
            @Override
            public int compare(Number o1, Number o2) {
                final double d1 = o1.doubleValue();
                final double d2 = o2.doubleValue();
                return Double.compare(d1, d2);
            }
        }
        TableRowSorter<DynamicDOTItemManager> tableRowSorter = new TableRowSorter<>();
        tableRowSorter.setModel(tableModel);
        tableRowSorter.setComparator(4, new NumberComparator());
        tableRowSorter.setComparator(5, new NumberComparator());
        theTable.setRowSorter(tableRowSorter);

        JButton newDOTItemButton = new JButton("Neue Zeile");
        newDOTItemButton.setEnabled(_dotDefinitionDialog.isEditable());
        JButton deleteDOTItemButton = new JButton("Zeile löschen");
        deleteDOTItemButton.setEnabled(false);
        JButton showConflictsButton = new JButton("Zeige Konflikte");

        addButtonListeners(primitiveForm, property, newDOTItemButton, deleteDOTItemButton, showConflictsButton);
        addListSelectionListener(theTable, deleteDOTItemButton);

        JPanel dotButtonsPanel = new JPanel();
        dotButtonsPanel.setLayout(new SpringLayout());

        dotButtonsPanel.add(newDOTItemButton);
        dotButtonsPanel.add(deleteDOTItemButton);
        dotButtonsPanel.add(showConflictsButton);

        dotButtonsPanel.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));
        SpringUtilities.makeCompactGrid(dotButtonsPanel, 1, 5, 20);

        JPanel thePanel = new JPanel();
        thePanel.setLayout(new SpringLayout());
//		thePanel.setBorder(BorderFactory.createMatteBorder(2, 2, 2, 2, Color.BLACK));
        thePanel.add(new JScrollPane(theTable));
        thePanel.add(dotButtonsPanel);
        SpringUtilities.makeCompactGrid(thePanel, 2, 20, 5);

        /* Hier wird die bevorzugte Größe des Panels vorgegeben; als Breite wird die bevorzugte Breite
         *  des Tables genommen, als Höhe mindestens 150 Pixel wegen der drei Buttons auf der rechten
         *  Seite, und mindestens die bevorzugte Höhe des Tables plus der Höhe für zwei weitere Zeilen. */
        final Dimension tableDim = theTable.getPreferredSize();
        thePanel.setPreferredSize(new Dimension(tableDim.width, Math.max(tableDim.height + 50, 150)));

        return thePanel;
    }

    @SuppressWarnings("OverlyLongMethod")
    private void addButtonListeners(final PrimitiveForm primitiveForm, final Property property, final JButton newDOTItemButton,
                                    final JButton deleteDOTItemButton, final JButton showConflictsButton) {

        @SuppressWarnings("serial")
        final class DOTItemDialog extends JDialog {

            private final PrimitiveForm _primitiveForm;

            private final Property _property;
            private final DynamicDefinitionComponent _dynamicDefinitionComponent;
            private final JComboBox<Object> _colorComboBox = new JComboBox<>(ColorManager.getInstance().getColorNames());
            private final JComboBox<Object> _textStyleComboBox = new JComboBox<>(FONT_ITEMS);
            private final JComboBox<String> _textComboBox = new JComboBox<>(DOTPointPainter.DYNAMIC_TEXT_ITEMS);
            private final JSpinner _diameterSpinner = new JSpinner(getNewDiameterSpinnerModel());
            private final JSpinner _strokeWidthSpinner = new JSpinner(getNewStrokeWidthSpinnerModel());
            private final JSpinner _transparencySpinner = new JSpinner(getNewTransparencySpinnerModel());
            private final JSpinner _textSizeSpinner = new JSpinner(getNewTextSizeSpinnerModel());
            private JComponent _component;

            private DOTItemDialog() {
                super();
                _primitiveForm = primitiveForm;
                _property = property;

                _dynamicDefinitionComponent = new DynamicDefinitionComponent(_configuration);
                final Object propertyValue = initDynamicDefinitionComponent();

                final JPanel panel = new JPanel();
                panel.setLayout(new SpringLayout());

                JLabel aLabel = null;
                if (_property.equals(DiameterProperty.getInstance())) {
                    aLabel = new JLabel("Durchmesser: ");
                    if (propertyValue != null) {
                        _diameterSpinner.setValue(propertyValue);
                    }
                    _component = _diameterSpinner;
                } else if (_property.equals(ColorProperty.getInstance()) || _property.equals(FillingProperty.getInstance())) {
                    aLabel = new JLabel("Farbe: ");
                    if (propertyValue != null) {
                        _colorComboBox.setSelectedItem(propertyValue);
                    }
                    _component = _colorComboBox;
                } else if (_property.equals(TextSizeProperty.getInstance())) {
                    aLabel = new JLabel("Schriftgröße: ");
                    if (propertyValue != null) {
                        _textSizeSpinner.setValue(propertyValue);
                    }
                    _component = _textSizeSpinner;
                } else if (_property.equals(StrokeWidthProperty.getInstance())) {
                    aLabel = new JLabel("Strichbreite: ");
                    if (propertyValue != null) {
                        _strokeWidthSpinner.setValue(propertyValue);
                    }
                    _component = _strokeWidthSpinner;
                } else if (_property.equals(TextProperty.getInstance())) {
                    aLabel = new JLabel("Text: ");
                    prepareTextComboBox(propertyValue);
                    _component = _textComboBox;
                } else if (_property.equals(TextStyleProperty.getInstance())) {
                    aLabel = new JLabel("Textstil: ");
                    if (propertyValue != null) {
                        _textStyleComboBox.setSelectedItem(propertyValue);
                    }
                    _component = _textStyleComboBox;
                } else if (_property.equals(TransparencyProperty.getInstance())) {
                    aLabel = new JLabel("Tranzparenz: ");
                    if (propertyValue != null) {
                        _transparencySpinner.setValue(propertyValue);
                    }
                    _component = _transparencySpinner;
                }
                panel.add(aLabel);
                panel.add(_component);

                panel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 0));
                SpringUtilities.makeCompactGrid(panel, 2, 20, 5);

                final JButton saveButton = new JButton("Speichern");
                saveButton.setEnabled(_dotDefinitionDialog.isEditable());
                final JButton cancelButton = new JButton("Dialog schließen");

                final JPanel buttonsPanel = new JPanel();
                buttonsPanel.setLayout(new SpringLayout());

                buttonsPanel.add(saveButton);
                buttonsPanel.add(cancelButton);

                buttonsPanel.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));
                SpringUtilities.makeCompactGrid(buttonsPanel, 2, 20, 5);
                addButtonListeners(saveButton, cancelButton);

                setTitle("GND: Darstellungsfestlegung für Punkte");
                setLayout(new BorderLayout());
                add(new JScrollPane(panel), BorderLayout.NORTH);
                add(new JScrollPane(_dynamicDefinitionComponent), BorderLayout.CENTER);
                add(buttonsPanel, BorderLayout.SOUTH);
                pack();
                setSize(700, 550);
                final Rectangle bounds = getBounds();
                setLocation(new Point((int) (bounds.getHeight() / 1.5 + 300), (int) (bounds.getWidth() / 1.5)));
            }

            @Nullable
            private Object initDynamicDefinitionComponent() {
                final String selectedPrimitiveFormName = _dotDefinitionDialog.getSelectedPrimitiveForm();
                final Property currentProperty = _dotDefinitionDialog.getSelectedProperty();
                PrimitiveFormPropertyPair pfpPair = new PrimitiveFormPropertyPair(selectedPrimitiveFormName, currentProperty);
                final JTable workWithThisTable = _dynamicTables.get(pfpPair);

                int selectedRow = workWithThisTable.getSelectedRow();
                if (selectedRow == -1) {
                    if (workWithThisTable.getModel().getRowCount() > 0) {
                        selectedRow = 0;
                    } else {
                        return null;
                    }
                }
                selectedRow = workWithThisTable.convertRowIndexToModel(selectedRow);
                final TableModel model = workWithThisTable.getModel();
                DynamicDOTItemManager dynamicDOTItemManager = (DynamicDOTItemManager) model;
	            final DOTItemManager<DynamicDOTItem>.DisplayObjectTypeItemWithInterval dotItemWithInterval = dynamicDOTItemManager.get(selectedRow);
                _dynamicDefinitionComponent.fillComponents(dotItemWithInterval);
                return dotItemWithInterval.getItem().getPropertyValue();
            }

            private void addButtonListeners(JButton saveButton, JButton cancelButton) {
                ActionListener actionListenerSave = e -> {

                    Object propertyValue = null;
                    String errorString;
                    if (_property.equals(DiameterProperty.getInstance())) {
                        propertyValue = _diameterSpinner.getValue();
                        errorString = "Bitte wählen Sie einen Durchmesser aus!";
                    } else if (_property.equals(ColorProperty.getInstance()) || _property.equals(FillingProperty.getInstance())) {
                        propertyValue = _colorComboBox.getSelectedItem();
                        errorString = "Bitte wählen Sie eine Farbe aus!";
                    } else if (_property.equals(TextSizeProperty.getInstance())) {
                        propertyValue = _textSizeSpinner.getValue();
                        errorString = "Bitte wählen Sie eine Schriftgröße aus!";
                    } else if (_property.equals(StrokeWidthProperty.getInstance())) {
                        propertyValue = _strokeWidthSpinner.getValue();
                        errorString = "Bitte wählen Sie eine Strichbreite  aus!";
                    } else if (_property.equals(TextProperty.getInstance())) {
                        propertyValue = _textComboBox.getSelectedItem();
                        errorString = "Bitte wählen Sie einen Text aus!";
                    } else if (_property.equals(TextStyleProperty.getInstance())) {
                        propertyValue = _textStyleComboBox.getSelectedItem();
                        errorString = "Bitte wählen Sie einen Textstil aus!";
                    } else if (_property.equals(TransparencyProperty.getInstance())) {
                        propertyValue = _transparencySpinner.getValue();
                        errorString = "Bitte wählen Sie eine Tranparenz aus!";
                    } else {
                        errorString = "DOTPointPanel.addButtonListeners(): unbehandelte Eigenschaft!";
                    }
                    if (propertyValue == null) {
                        JOptionPane.showMessageDialog(new JFrame(), errorString, "Fehler", JOptionPane.ERROR_MESSAGE);
                        return;
                    }

                    final String description = _dynamicDefinitionComponent.getInfoText();
                    if (description == null) {
                        JOptionPane.showMessageDialog(new JFrame(), "Bitte geben Sie einen Info-Text ein!", "Fehler", JOptionPane.ERROR_MESSAGE);
                        return;
                    }

                    final String attributeGroupName = _dynamicDefinitionComponent.getAttributeGroupName();
                    if (attributeGroupName == null) {
                        JOptionPane.showMessageDialog(new JFrame(), "Bitte wählen Sie eine Attributgruppe aus!", "Fehler", JOptionPane.ERROR_MESSAGE);
                        return;
                    }

                    final String aspectName = _dynamicDefinitionComponent.getAspectName();
                    if (aspectName == null) {
                        JOptionPane.showMessageDialog(new JFrame(), "Bitte wählen Sie einen Aspekt aus!", "Fehler", JOptionPane.ERROR_MESSAGE);
                        return;
                    }

                    final String attributeName = _dynamicDefinitionComponent.getAttributeName();
                    if (attributeName == null) {
                        int error = _dynamicDefinitionComponent.checkAttributeName();
                        if (error == 1) {
                            JOptionPane.showMessageDialog(new JFrame(), "Bitte wählen Sie ein Attribut aus!", "Fehler", JOptionPane.ERROR_MESSAGE);
                            return;
                        } else if (error == 2) {
                            JOptionPane.showMessageDialog(new JFrame(), "Der Attributname ist ungültig!", "Fehler", JOptionPane.ERROR_MESSAGE);
                            return;
                        }
                    }

                    final Double fromValue = _dynamicDefinitionComponent.getFromValue();
                    if (fromValue == null) {
                        int error = _dynamicDefinitionComponent.checkFromValue();
                        if (error == 1) {
                            JOptionPane.showMessageDialog(new JFrame(), "Bitte tragen Sie einen Von-Wert ein!", "Fehler", JOptionPane.ERROR_MESSAGE);
                            return;
                        } else if ((error == 2) || (error == 3)) {
                            JOptionPane.showMessageDialog(new JFrame(), "Bitte korrigieren Sie den Von-Wert!", "Fehler", JOptionPane.ERROR_MESSAGE);
                            return;
                        }
                    }

                    final Double toValue = _dynamicDefinitionComponent.getToValue();
                    if (toValue == null) {
                        int error = _dynamicDefinitionComponent.checkToValue();
                        if (error == 1) {
                            JOptionPane.showMessageDialog(new JFrame(), "Bitte tragen Sie einen Bis-Wert ein!", "Fehler", JOptionPane.ERROR_MESSAGE);
                            return;
                        } else if ((error == 2) || (error == 3)) {
                            JOptionPane.showMessageDialog(new JFrame(), "Bitte korrigieren Sie den Bis-Wert!", "Fehler", JOptionPane.ERROR_MESSAGE);
                            return;
                        }
                    }

                    if ((fromValue != null) && (toValue != null) && fromValue > toValue) {
                        JOptionPane.showMessageDialog(new JFrame(), "Der Von-Wert ist größer als der Bis-Wert!", "Fehler", JOptionPane.ERROR_MESSAGE);
                        return;
                    }

                    DynamicDOTItem dItem = new DynamicDOTItem(attributeGroupName, aspectName, attributeName, description, propertyValue);
                    _dotDefinitionDialog.getScratchDisplayObjectType()
                        .setValueOfDynamicProperty(_primitiveForm.getName(), _property, dItem, fromValue, toValue);
                    _dotDefinitionDialog.setSomethingChanged(true);
                };
                saveButton.addActionListener(actionListenerSave);

                ActionListener actionListenerCancel = e -> dispose();
                cancelButton.addActionListener(actionListenerCancel);
            }

            private void prepareTextComboBox(@Nullable final Object propertyValue) {
                // _textComboBox wird in der letzten Zeile editierbar gemacht, und es wird
                // der übergebene Wert in diese Zeile eingetragen und selektiert, wenn er nicht
                // einem der anderen Einträge entspricht.

                if (propertyValue != null) {
                    String propertyValueAsString = (String) propertyValue;
                    final int size = _textComboBox.getModel().getSize();
                    boolean propertyValueFound = false;
                    for (int i = 0; i < size - 1; i++) {
                        if (propertyValueAsString.equals(_textComboBox.getItemAt(i))) {
                            propertyValueFound = true;
                            _textComboBox.setSelectedIndex(i);
                            break;
                        }
                    }
                    if (!propertyValueFound) {
                        _textComboBox.removeItemAt(size - 1);
                        _textComboBox.addItem(propertyValueAsString);
                        _textComboBox.setSelectedIndex(size - 1);
                    }
                }

                ItemListener textComboBoxItemListener = e -> {
                    if (e.getStateChange() == ItemEvent.SELECTED) {
                        if (_textComboBox.getSelectedIndex() == DOTPointPainter.DYNAMIC_TEXT_ITEMS.length - 1) {
                            _textComboBox.setEditable(true);
                        } else {
                            _textComboBox.setEditable(false);
                        }
                    }
                };
                _textComboBox.addItemListener(textComboBoxItemListener);

                ActionListener editorActionListener = e -> {
                    final int length = DOTPointPainter.DYNAMIC_TEXT_ITEMS.length;
                    final int index = length - 1;
                    final ComboBoxEditor editor = _textComboBox.getEditor();
                    String editedItem = (String) editor.getItem();
                    _textComboBox.insertItemAt(editedItem, index);
                    final DefaultComboBoxModel<String> model = (DefaultComboBoxModel<String>) _textComboBox.getModel();
                    if (model.getSize() > length) { // Sieht komisch aus, aber der direkte Weg ging nicht.
                        model.removeElementAt(length);
                    }
                };
                _textComboBox.getEditor().addActionListener(editorActionListener);
            }

            public void runDialog() {
                setVisible(true);
            }

            @Override
            public String toString() {
                return "DOTItemDialog{}";
            }
        }

        ActionListener actionListenerNewButton = e -> {
            DOTItemDialog dotItemDialog = new DOTItemDialog();
            dotItemDialog.runDialog();
        };
        newDOTItemButton.addActionListener(actionListenerNewButton);

        ActionListener actionListenerDeleteButton = e -> {
            final String selectedPrimitiveFormName = _dotDefinitionDialog.getSelectedPrimitiveForm();
            final Property currentProperty = _dotDefinitionDialog.getSelectedProperty();
            PrimitiveFormPropertyPair pfpPair = new PrimitiveFormPropertyPair(selectedPrimitiveFormName, currentProperty);
            final JTable workForThisTable = _dynamicTables.get(pfpPair);
            final int[] selectedRows = workForThisTable.getSelectedRows();
            if (selectedRows.length == 0) {
                JOptionPane.showMessageDialog(new JFrame(), "Bitte wählen Sie eine Zeile aus!", "Fehler", JOptionPane.ERROR_MESSAGE);
                return;
            }
            final TableModel model = workForThisTable.getModel();
            DynamicDOTItemManager dynamicDOTItemManager = (DynamicDOTItemManager) model;
            Set<Integer> selectedModelIndexes = new TreeSet<>();
            for (final int selectedRow : selectedRows) {
                selectedModelIndexes.add(workForThisTable.convertRowIndexToModel(selectedRow));
            }
            final Object[] array = selectedModelIndexes.toArray();
            for (int i = array.length - 1; i >= 0; i--) {
                dynamicDOTItemManager.remove((Integer) array[i]);
                _dotDefinitionDialog.setSomethingChanged(true);
            }
        };
        deleteDOTItemButton.addActionListener(actionListenerDeleteButton);

        ActionListener actionListenerShowConflictsButton = e -> {
            final String selectedPrimitiveFormName = _dotDefinitionDialog.getSelectedPrimitiveForm();
            final Property currentProperty = _dotDefinitionDialog.getSelectedProperty();
            PrimitiveFormPropertyPair pfpPair = new PrimitiveFormPropertyPair(selectedPrimitiveFormName, currentProperty);
            final JTable workForThisTable = _dynamicTables.get(pfpPair);
            workForThisTable.clearSelection();

            DOTPoint dotPoint = (DOTPoint) _dotDefinitionDialog.getScratchDisplayObjectType();
            final PrimitiveForm selectedPrimitiveForm = dotPoint.getPrimitiveForm(selectedPrimitiveFormName);
            final Set<Integer> conflictingRows = dotPoint.getConflictingRows(selectedPrimitiveForm, currentProperty);
            ListSelectionModel listSelectionModel = workForThisTable.getSelectionModel();
            if (null != conflictingRows) {
                for (Integer row : conflictingRows) {
                    Integer rowInView = workForThisTable.convertRowIndexToView(row);
                    listSelectionModel.addSelectionInterval(rowInView, rowInView);
                }
            }
            workForThisTable.setSelectionModel(listSelectionModel);
        };
        showConflictsButton.addActionListener(actionListenerShowConflictsButton);
    }

    private void addListSelectionListener(final JTable table, final JButton deleteDOTItemButton) {
        final ListSelectionModel selectionModel = table.getSelectionModel();
        ListSelectionListener listSelectionListener = e -> {
            final int[] selectedRows = table.getSelectedRows();
            if (selectedRows.length == 0) {
                deleteDOTItemButton.setEnabled(false);
            } else {
                deleteDOTItemButton.setEnabled(_dotDefinitionDialog.isEditable());
                if (selectedRows.length == 1) {
                    deleteDOTItemButton.setText("Zeile löschen");
                } else {
                    deleteDOTItemButton.setText("Zeilen löschen");
                }
            }
        };
        selectionModel.addListSelectionListener(listSelectionListener);
    }

    private void addToolTips() {
        _translationFactorSpinner
            .setToolTipText("Für Punkte, die auf einer Straße liegen, wird die Darstellung orthogonal um diese Länge verschoben.");
        _joinByLineCheckBox.setToolTipText("Hier wird festgelegt, ob die verschobene Darstellung mit dem Lagepunkt durch eine Linie verbunden wird.");
    }

    @Override
    public String toString() {
        return "DOTPointPanel{}";
    }

    @SuppressWarnings("serial")
    class StaticPanel extends JPanel {

        private final PrimitiveForm _primitiveForm;
        private final Property _property;

        /**
         * Konstruiert aus Grundfigur und Eigenschaft ein Objekt.
         *
         * @param primitiveForm die Grundfigur
         * @param property      die Eigenschaft
         */
        public StaticPanel(final PrimitiveForm primitiveForm, final Property property) {
            super();
            _primitiveForm = primitiveForm;
            _property = property;
        }

        /**
         * Gibt die einzige Combobox oder {@code null} zurück.
         *
         * @return die einzige Combobox oder {@code null}
         */
        @Nullable
        public JComboBox<Object> getComboBox() {
            final Component[] components = getComponents();
            for (Component component : components) {
                if (component instanceof JComboBox) {
                    return (JComboBox<Object>) component;
                }
            }
            return null;
        }

        private void setValue(final JComponent component) {
            final String primitiveFormName = _primitiveForm.getName();
            if (_dotDefinitionDialog.isPropertyStatic(primitiveFormName, _property)) {
                Object object = _dotDefinitionDialog.getValueOfStaticProperty(primitiveFormName, _property);
                if (component instanceof JComboBox) {
                    final String currentValue;
                    if (object instanceof Color) {
                        currentValue = ColorManager.getInstance().getColorName((Color) object);
                    } else if (object instanceof Integer i) {
                        if (i.equals(Font.PLAIN)) {
                            currentValue = PLAIN_FONT_STYLE;
                        } else if (i.equals(Font.BOLD)) {
                            currentValue = BOLD_FONT_STYLE;
                        } else if (i.equals(Font.ITALIC)) {
                            currentValue = ITALIC_FONT_STYLE;
                        } else {
                            currentValue = "Unbekannter Font";
                        }
                    } else if (object instanceof TextStyleProperty.Styles) {
                        currentValue = object.toString();
                    } else {
                        currentValue = (String) object;
                    }
                    JComboBox<String> comboBox = (JComboBox<String>) component;
                    comboBox.setSelectedItem(currentValue);
                } else if (component instanceof JSpinner spinner) {
                    if (object instanceof Integer) {
                        spinner.setValue(object);
                    } else {
                        spinner.setValue(object);
                    }
                }
            } else {
                if (component instanceof JComboBox) {
                    JComboBox<Object> comboBox = (JComboBox<Object>) component;
                    comboBox.setSelectedItem(_property.getDefaultValue());
                } else if (component instanceof JSpinner spinner) {
                    spinner.setValue(_property.getDefaultValue());
                }
            }
        }

        private void addListener(final JComponent component) {
            if (component instanceof JComboBox) {
                JComboBox<Object> comboBox = (JComboBox<Object>) component;
                comboBox.addItemListener(e -> {
                    final Object selectedItem = comboBox.getSelectedItem();
                    _dotDefinitionDialog.getScratchDisplayObjectType().setValueOfStaticProperty(_primitiveForm.getName(), _property, selectedItem);
                });
            } else if (component instanceof JSpinner) {
                javax.swing.event.ChangeListener changeListener = e -> {
                    JSpinner spinner = (JSpinner) component;
                    final Object value = spinner.getValue();
                    _dotDefinitionDialog.getScratchDisplayObjectType().setValueOfStaticProperty(_primitiveForm.getName(), _property, value);
                };
                JSpinner spinner = (JSpinner) component;
                spinner.addChangeListener(changeListener);
            }
        }
    }
}
