/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.pointPlugin;

import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.debug.Debug;
import de.kappich.pat.gnd.displayObjectToolkit.DOTCollection;
import de.kappich.pat.gnd.displayObjectToolkit.DisplayObject;
import de.kappich.pat.gnd.displayObjectToolkit.GeoInitializer;
import de.kappich.pat.gnd.displayObjectToolkit.OnlineDisplayObject;
import de.kappich.pat.gnd.displayObjectToolkit.PrimitiveFormPropertyPair;
import de.kappich.pat.gnd.gnd.MapPane;
import de.kappich.pat.gnd.layerManagement.Layer;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectPainter;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectTypePlugin;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectsInitializer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import javax.swing.JProgressBar;
import javax.swing.SwingUtilities;

/**
 * Dies ist die Implementierung des {@code DisplayObjectsInitializers} im Point-Plugin.
 *
 * @author Kappich Systemberatung
 */
public class PointInitializer implements DisplayObjectsInitializer {

    private static final Debug _debug = Debug.getLogger();

    @SuppressWarnings("OverlyLongMethod")
    @Override
    public void initializeDisplayObjects(final DataModel configuration, final Layer layer, final MapPane mapPane, final JProgressBar progressBar,
                                         List<DisplayObject> returnList) {
        final String geoReferenceType = layer.getConfigurationObjectType();
        final SystemObjectType systemObjectType = configuration.getType(geoReferenceType);
        if (systemObjectType == null) {
            _debug.warning("Der Typ " + geoReferenceType + " ist in der Konfiguration unbekannt.");
            return;
        }
        SystemObjectType pointType = configuration.getType("typ.punkt");
        if (!systemObjectType.inheritsFrom(pointType)) {
            _debug.warning("Der Typ " + pointType.getName() + " ist kein Punkttyp.");
            return;
        }

        final List<SystemObject> systemObjects = systemObjectType.getElements();
        SwingUtilities.invokeLater(() -> {
            progressBar.setIndeterminate(false);
            progressBar.setMaximum(systemObjects.size());
        });

        final AttributeGroup pointCoordinateAttributeGroup = configuration.getAttributeGroup("atg.punktKoordinaten");
        final AttributeGroup pointOnLineAttributeGroup = configuration.getAttributeGroup("atg.punktLiegtAufLinienObjekt");
        final AttributeGroup _composedOfLinesAttributeGroup = configuration.getAttributeGroup("atg.bestehtAusLinienObjekten");

        final int chunkSize = 100;
        for (int i = 0; i < systemObjects.size(); i += chunkSize) {
            final int val = i;
            SwingUtilities.invokeLater(() -> progressBar.setValue(val));
            final List<SystemObject> subList = systemObjects.subList(i, Math.min(i + chunkSize, systemObjects.size()));
            configuration.getConfigurationData(subList, pointCoordinateAttributeGroup);
            final Data[] configurationData = configuration.getConfigurationData(subList, pointOnLineAttributeGroup);
            final Collection<Long> lines = new ArrayList<>();
            for (Data data : configurationData) {
                if (data != null) {
                    final long line = data.getReferenceValue("LinienReferenz").getId();
                    if (line != 0) {
                        lines.add(line);
                    }
                }
            }
            if (!lines.isEmpty()) {
                final Collection<SystemObject> objects = GeoInitializer.getInstance().getObjects(lines);
                configuration.getConfigurationData(objects, _composedOfLinesAttributeGroup);
                GeoInitializer.getInstance().preloadLines(objects);
            }
        }
        SwingUtilities.invokeLater(() -> progressBar.setValue(systemObjects.size()));

        final DOTCollection dotCollection = layer.getDotCollection();
        final Iterator<DisplayObjectType> iterator = dotCollection.values().iterator();
        if (!iterator.hasNext()) {
            return;
        }
        final DisplayObjectType dot = iterator.next();
        final DisplayObjectTypePlugin displayObjectTypePlugin = dot.getDisplayObjectTypePlugin();
        DisplayObjectPainter painter = displayObjectTypePlugin.getPainter();
        Map<DisplayObjectType, List<PrimitiveFormPropertyPair>> pfPropertyPairs = dotCollection.getPrimitiveFormPropertyPairs();
        for (SystemObject systemObject : systemObjects) {
            final List<Object> pointCoordinates = GeoInitializer.getInstance().getPointCoordinates(systemObject);
            OnlineDisplayObject displayObject =
                new OnlineDisplayObject(systemObject, pointCoordinates, painter, dotCollection, pfPropertyPairs, mapPane);
            returnList.add(displayObject);
        }
        SwingUtilities.invokeLater(() -> {
            progressBar.setValue(0);
            progressBar.setIndeterminate(true);
        });
    }
}
