/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.properties;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.kmPlugin.KmFormat;
import de.kappich.pat.gnd.rnPlugin.RnFormat;
import de.kappich.pat.gnd.statPlugin.StatFormat;
import java.util.prefs.Preferences;

/**
 * <p>{@code PropertyValuesManager} hilft bei der Initialisierung von Werten jener Eigenschaften, die
 * nicht den Standardtypen {@code Integer}, {@code Double}, {@code String} oder {@code Color} zugehören.</p>
 * <p>Jede Klasse, die das Interface {@code PropertyValue} implementiert, muss auch hier behandelt
 * werden, damit ihre Werte aus den Preferences gelesen werden können.</p>
 *
 * @author Kappich Systemberatung
 */
public final class PropertyValuesManager {

	/* Die Entscheidung, dass alle Klassen, die <code>PropertyValue</code> implementieren, auch hier
	behandelt werden müssen, war alles andere als leicht. Im Moment dieser Entscheidung waren alle
	diese Klassen <code>Enums</code> und für solche ist eine vollkommen generische Lösung mittels
	Reflection möglich. Doch für Klassen mit unendlich vielen Objekten nicht. Deshalb wäre eine
	generische Lösung nicht nur komplexer, sondern auch konzeptionell unvollständig.
	 */

    /**
     * Gibt einen Wert der {@code PropertyValue} implementierenden Klasse zurück.
     *
     * @param className der Klassenname
     * @param prefs     die Preferenzen
     *
     * @return der Wert
     */
    @Nullable
    public static Object getPropertyValue(final String className, final Preferences prefs) {
	    return switch (className) {
		    case "de.kappich.pat.gnd.kmPlugin.KmFormat" -> {
                int value = prefs.getInt("PROPERTY_VALUE", -1);
			    yield KmFormat.getFormat(value);
            }
		    case "de.kappich.pat.gnd.properties.DistanceRasterType" -> {
                int value = prefs.getInt("PROPERTY_VALUE", -1);
			    yield DistanceRasterType.getType(value);
            }
		    case "de.kappich.pat.gnd.properties.TextStyleProperty$Styles" -> {
                int style = prefs.getInt("PROPERTY_VALUE", -1);
			    yield TextStyleProperty.Styles.getStyle(style);
            }
		    case "de.kappich.pat.gnd.statPlugin.StatFormat" -> {
                int value = prefs.getInt("PROPERTY_VALUE", -1);
			    yield StatFormat.getFormat(value);
            }
		    case "de.kappich.pat.gnd.rnPlugin.RnFormat" -> {
                int value = prefs.getInt("PROPERTY_VALUE", -1);
			    yield RnFormat.getFormat(value);
            }
		    default -> null;
	    };
    }
}
