/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.properties;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.utils.SpringUtilities;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.event.ItemListener;
import java.util.prefs.Preferences;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SpringLayout;

/**
 * Diese Klasse verkapselt die Eigenschaft Textstil.
 *
 * @author Kappich Systemberatung
 */
public final class TextStyleProperty extends AbstractProperty {
    private static final TextStyleProperty _instance = new TextStyleProperty();
    private static final String KEY = "textstyle";
    private static final Styles DEFAULT_TEXT_STYLE = Styles.STANDARD;

    private TextStyleProperty() {
        super("Textstil");
    }

    public static Property getInstance() {
        return _instance;
    }

    @Override
    public Object getDefaultValue() {
        return DEFAULT_TEXT_STYLE;
    }

    @Override
    public String getKey() {
        return KEY;
    }

    @Override
    public PropertyPanel getPropertyPanel(@Nullable Object value, boolean changeable) {
        if (value != null) {
            if (value instanceof Integer) {
                return new TextStylePropertyPanel((Integer) value, changeable);
            } else if (value instanceof Styles) {
                return new TextStylePropertyPanel(((Styles) value).getIntValue(), changeable);
            }
        }
        return new TextStylePropertyPanel(0, changeable);
    }

    /* Die TextStyleProperty ist mit diesem Enum eine Besonderheit und muss deshalb gelegentlich besonders behandelt
    werden, siehe z.B. {@link de.kappich.pat.gnd.pointPlugin#DOTPoint.PrimitiveForm.getCopy}.
     */
    public enum Styles implements PropertyValue {
        STANDARD("Standard", Font.PLAIN),   // 0
        BOLD("Fett", Font.BOLD),            // 1
        ITALIC("Kursiv", Font.ITALIC);      // 2

        private final String _description;
        private final int _intValue;

        Styles(String description, int intValue) {
            _description = description;
            _intValue = intValue;
        }

        @Nullable
        public static Styles getStyle(int intValue) {
            if (0 == intValue) {
                return STANDARD;
            } else if (1 == intValue) {
                return BOLD;
            } else if (2 == intValue) {
                return ITALIC;
            } else {
                return null;
            }
        }

        @Override
        public void putPreferences(final Preferences prefs) {
            prefs.putInt("PROPERTY_VALUE", getIntValue());
        }

        @Override
        public String toString() {
            return _description;
        }

        public int getIntValue() {
            return _intValue;
        }
    }

    private static class TextStylePropertyPanel extends JPanel implements PropertyPanel {
        private final JComboBox<Styles> _textStyleComboBox = new JComboBox<>(Styles.values());

        public TextStylePropertyPanel(Integer textstyle, boolean changeable) {
            JLabel label = new JLabel("Textstil: ");
            _textStyleComboBox.setMaximumSize(new Dimension(200, 20));
            _textStyleComboBox.setSelectedIndex(textstyle);
            _textStyleComboBox.setEnabled(changeable);
            setLayout(new SpringLayout());
            add(label);
            add(_textStyleComboBox);
            SpringUtilities.makeCompactGrid(this, 2, 5, 5);
        }

        @Override
        public void addListener(final PropertyPanelListener listener) {
            ItemListener itemListener = e -> listener.stateChanged();
            _textStyleComboBox.addItemListener(itemListener);
        }

        @Override
        public Object getPropertyValue() {
            return _textStyleComboBox.getSelectedItem();
        }

        @Override
        public void setSelectionEnabled(final boolean enabled) {
            _textStyleComboBox.setEnabled(enabled);
        }
    }
}
