/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.statPlugin;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.displayObjectToolkit.DOTCollection;
import de.kappich.pat.gnd.displayObjectToolkit.DisplayObject;
import de.kappich.pat.gnd.displayObjectToolkit.PrimitiveFormPropertyPair;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectPainter;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.properties.DistanceRasterType;
import de.kappich.pat.gnd.utils.PointWithAngle;
import java.awt.Rectangle;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * {@code StatDisplayObject} ist die Implementation des Interfaces {@link DisplayObject} im Stationierungs-Plugin.
 *
 * @author Kappich Systemberatung
 */
public class StatDisplayObject implements DisplayObject {
    private final Map<Integer, Rectangle> _boundingRectangles = new HashMap<>();
    private final DOTCollection _dotCollection;
    private final String _beginNode;
    private final String _endNode;
    private final long _statValue;
    private final List<Object> _coordinates;
    private DisplayObjectPainter _painter;
    private int _defaultType;

    /**
     * Ein Konstruktor, der nur Kilometrierungs-Informationen enthält. Objekte, dei mit diesem Konstruktor erstellt wurden, sind nur eingeschränkt
     * verwendbar.
     *
     * @param statValue der ASB-Stationierungswert
     */
    public StatDisplayObject(final long statValue) {
        super();
        _painter = null;
        _dotCollection = null;
        _defaultType = 0;
        _beginNode = null;
        _endNode = null;
        _statValue = statValue;
        _coordinates = new ArrayList<>();
    }

    /**
     * Ein Konstruktor mit Painter, DOTCollection und Koordinaten. Objekte, die so konstruiert wurden, sind vollständig verwendungsfähig.
     *
     * @param painter       ein Painter, der dieses Object zeichnen kann, in der Regel ein {@link DOTStatPainter}
     * @param dotCollection die Kollektion, die beim Zeichnen verwendet werden soll
     * @param statValue     der ASB-Stationierungswert
     * @param point         die Koordinateninformation
     */
    public StatDisplayObject(DisplayObjectPainter painter, final DOTCollection dotCollection, final String beginNode, final String endNode,
                             final long statValue, final PointWithAngle point) {
        super();
        _painter = painter;
        _dotCollection = dotCollection;
        _defaultType = 0;
        _beginNode = beginNode;
        _endNode = endNode;
        _statValue = statValue;
        _coordinates = new ArrayList<>(1);
        _coordinates.add(point);
    }

    @Override
    public String getName() {
        return _beginNode + " -> " + _endNode + ", " + _statValue + " m";
    }

    @Override
    public List<Object> getCoordinates(final int type) {
        return Collections.unmodifiableList(_coordinates);
    }

    @Override
    public List<Object> getCoordinates() {
        return Collections.unmodifiableList(_coordinates);
    }

    @Override
    public List<Point2D> getReferencePoints(final Rectangle utmBounds) {
        List<Point2D> pointList = new ArrayList<>(1);
        if (!_coordinates.isEmpty()) {
            Point2D point = ((PointWithAngle) _coordinates.get(0)).getPoint();
            if (utmBounds.contains(point)) {
                pointList.add(point);
            }
        }
        return pointList;
    }

    @Override
    public DisplayObjectPainter getPainter() {
        return new DOTStatPainter();
    }

    @Override
    public void setPainter(DisplayObjectPainter painter) {
        _painter = painter;
    }

    @Override
    public DOTCollection getDOTCollection() {
        return _dotCollection;
    }

    @Override
    public Rectangle getBoundingRectangle(final int type) {
        if (!_boundingRectangles.containsKey(type)) {
            _boundingRectangles.put(type, _painter.getBoundingRectangle(this, type));
        }
        return _boundingRectangles.get(type);
    }

    @Override
    public Rectangle getBoundingRectangle() {
        return getBoundingRectangle(_defaultType);
    }

    @Override
    public void setDefaultType(final int defaultType) {
        _defaultType = defaultType;
    }

    @Override
    @Nullable
    public DisplayObjectType.DisplayObjectTypeItem getDisplayObjectTypeItem(final PrimitiveFormPropertyPair pair) {
        return null;
    }

    /*
     * Das ist die Methode, die für den MapPane.MapScaleListener implementiert werden muss. Sie kann hier
     * leer bleiben, da StatDisplayObjects davon keinen Gebrauch machen.
     */
    @Override
    public void mapScaleChanged(double scale) {
    }

    /**
     * Diese Methode prüft, ob {@code this} bei dem übergebenen {@code DistanceRasterType} gezeigt werden kann, d.h. ob der Kilometrierungswert zum
     * Raster gehört.
     *
     * @param distanceRasterType der {@code StatType}
     *
     * @return {@code true} genau dann, wenn der Kilometrierungswert zum Raster gehört
     */
    public boolean checkType(@Nullable DistanceRasterType distanceRasterType) {
        return distanceRasterType != null && (_statValue % distanceRasterType.getValue() == 0);
    }

    /**
     * Gibt den zu den Übergabeparametern gehörigen Text zurück.
     *
     * @param distanceRasterType der {@code DistanceRasterType}
     * @param statFormat         das {@code StatFormat}
     *
     * @return der Text
     */
    @SuppressWarnings("SuspiciousIndentAfterControlStatement")
    @Nullable
    public String getText(@Nullable final DistanceRasterType distanceRasterType, @Nullable final StatFormat statFormat) {
        if (distanceRasterType == null || statFormat == null) {
            return null;
        }
	    return switch (statFormat) {
		    default -> distanceRasterType.getFormat().format((double) _statValue / 1000.);
	    };
    }
}
