/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.utils.view;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.gnd.PreferencesHandler;
import java.awt.Component;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.LayoutManager;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import javax.swing.JDialog;
import javax.swing.JMenuBar;

/**
 * <p>Diese Klasse erlaubt es einen {@link JDialog} mit Zusatzfunktionalität zur Speicherung und Wiederbenutzung von Größe und Location
 * auszustatten. In allen Konstruktoren wird ein Identifier angegeben. GndDialog-Objekte mit gleichem Identifier greifen auf die gleichen Größen- und
 * Location-Angeben zurück.</p>
 *
 * @author Kappich Systemberatung
 */
public class GndDialog {

    private static final boolean LOG = false;

    private final JDialog _dialog;
    private final String _identifier;
    private boolean _boundsFromPrefs;
    private int _preferencesX = Integer.MIN_VALUE;
    private int _preferencesY = Integer.MIN_VALUE;
    private int _preferencesWidth = Integer.MIN_VALUE;
    private int _preferencesHeight = Integer.MIN_VALUE;

    /**
     * Der Konstruktor mit Identifier und den zwei Argumenten eines korrespondierenden {@link JDialog}-Konstruktors. Hier wird ein solches {@code
     * JDialog}-Objekt intern gebildet.
     *
     * @param identifier der Identifier
     * @param owner      der Owner-Frame
     * @param modal      der Modalitätsparameter
     */
    public GndDialog(final String identifier, Frame owner, boolean modal) {
        _dialog = new JDialog(owner, modal);
        _identifier = identifier;
        readPreferenceBounds();
        _dialog.addWindowListener(new MyWindowListener());
    }

    /**
     * Der Konstruktor mit Identifier und einem {@link JDialog}-Objekt, das intern verwendet wird.
     *
     * @param dialog     ein JDialog
     * @param identifier der Identifier
     */
    public GndDialog(final JDialog dialog, final String identifier) {
        _dialog = dialog;
        _identifier = identifier;
        readPreferenceBounds();
        _dialog.addWindowListener(new MyWindowListener());
    }

    /**
     * Löscht in den Präferenzen die Informationen zu Position und Größe zu diesem Dialog (und damit zu allen Dialogen mit dem gleichen Identifier).
     */
    public static void removePreferenceBounds() {
        Preferences prefs = getPreferenceNode();
        try {
            prefs.removeNode();
        } catch (BackingStoreException ignored) {
            PreferencesDeleter pd =
                new PreferencesDeleter("Für ein Fenster konnten die Einstellungen der letzten Sitzungen nicht gelöscht werden.", prefs);
            pd.run();
        }
    }

    private static Preferences getPreferenceNode() {
        return PreferencesHandler.getInstance().getPreferenceStartPath().node("GNDDialogOld");
    }

    /**
     * @return der JDialog
     */
    public JDialog getDialog() {
        return _dialog;
    }

    /**
     * Liegen Bounds (x,y, width und height) in den Präferenzen vor?
     *
     * @return die Antwort
     */
    public boolean hasBoundsFromPrefs() {
        return _boundsFromPrefs;
    }

    /**
     * Der x-Wert aus den Präferenzen. Die Methode darf nur benutzt werden, wenn {@code hasPreferences} den Wert {@code true} zurückgeliefert hat.
     *
     * @return x
     */
    public int getPreferencesX() {
        return _preferencesX;
    }

    /**
     * Der y-Wert aus den Präferenzen. Die Methode darf nur benutzt werden, wenn {@code hasPreferences} den Wert {@code true} zurückgeliefert hat.
     *
     * @return y
     */
    public int getPreferencesY() {
        return _preferencesY;
    }

    /**
     * Der width-Wert aus den Präferenzen. Die Methode darf nur benutzt werden, wenn {@code hasPreferences} den Wert {@code true} zurückgeliefert
     * hat.
     *
     * @return width
     */
    public int getPreferencesWidth() {
        return _preferencesWidth;
    }

    /**
     * Der height-Wert aus den Präferenzen. Die Methode darf nur benutzt werden, wenn {@code hasPreferences} den Wert {@code true} zurückgeliefert
     * hat.
     *
     * @return height
     */
    public int getPreferencesHeight() {
        return _preferencesHeight;
    }

    /**
     * Liefert ein die Bounds beschreibendes Rechteck zurück, falls diese Bounds in den Präferenzen existieren, und {@code null} sonst.
     *
     * @return ein Rechteck oder {@code null}
     */
    @Nullable
    public Rectangle getPreferenceBounds() {
        if (_boundsFromPrefs) {
            return new Rectangle(_preferencesX, _preferencesY, _preferencesWidth, _preferencesHeight);
        } else {
            return null;
        }
    }

    /**
     * Dies ist eine Methode, mit deren Hilfe man sicherstellt, dass folgende Dinge erfüllt sind: 1. Das Fenster erscheint innerhalb des Bildschirms.
     * 2. Gibt es Präferenzen, so wird zumindest eine Minimalgröße garantiert. 3. Gibt es keine Präferenzen 3.1 so werden für die Location defaultX
     * und defaultY verwendet, 3.2 und ist pack true, so wird die Größe mittels pack() bestimmt, 3.3 und ist pack false, werden die defaultWidth und
     * defaultHeight benutzt.
     *
     * @param minWidth
     * @param minHeight
     * @param defaultX
     * @param defaultY
     * @param pack
     * @param defaultWidth
     * @param defaultHeight
     */
    public final void setPositionAndSize(final int minWidth, final int minHeight, final int defaultX, final int defaultY, final boolean pack,
                                         final int defaultWidth, final int defaultHeight) {
        readPreferenceBounds();
        if (hasBoundsFromPrefs()) {
            /* Hier wird sichergestellt, dass diese Instanz im Screen sichtbar wird. */
            final Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
            int x = Math.max(getPreferencesX(), 0);
            int y = Math.max(getPreferencesY(), 0);
            int width = Math.max(getPreferencesWidth(), minWidth);
            int height = Math.max(getPreferencesHeight(), minHeight);

            if (x > screenSize.width - width) {  // zu weit rechts
                x = Math.max(0, screenSize.width - width);
            }
            if (y > screenSize.height - height) { // zu weit unten
                y = Math.max(0, screenSize.height - height);
            }

            _dialog.setLocation(x, y);
            _dialog.setSize(width, height);
        } else {
            _dialog.setLocation(defaultX, defaultY);
            if (pack) {
                _dialog.pack();
            } else {
                _dialog.setSize(defaultWidth, defaultHeight);
            }
        }
    }

    /**
     * Liest die Bounds des Frames aus den Präferenzen.
     */
    public final void readPreferenceBounds() {
        if (null == _identifier || _identifier.isEmpty()) {
            return;
        }
        Preferences prefs = getPreferenceNode();
        try {
            if (prefs.nodeExists(_identifier)) {
                Preferences node = prefs.node(_identifier);
                _preferencesX = node.getInt("x", Integer.MIN_VALUE);
                _preferencesY = node.getInt("y", Integer.MIN_VALUE);
                _preferencesWidth = node.getInt("width", Integer.MIN_VALUE);
                _preferencesHeight = node.getInt("height", Integer.MIN_VALUE);
                if (LOG) {
                    System.out.println("Read " + _identifier + ": " + getPreferenceBounds());
                }
                _boundsFromPrefs =
                    _preferencesX != Integer.MIN_VALUE && _preferencesY != Integer.MIN_VALUE && _preferencesWidth != Integer.MIN_VALUE &&
                    _preferencesHeight != Integer.MIN_VALUE;
            }
        } catch (BackingStoreException ignored) {
            _preferencesX = Integer.MIN_VALUE;
            _preferencesY = Integer.MIN_VALUE;
            _preferencesWidth = Integer.MIN_VALUE;
            _preferencesHeight = Integer.MIN_VALUE;
            _boundsFromPrefs = false;

            PreferencesDeleter pd =
                new PreferencesDeleter("Für ein Fenster konnten die Einstellungen der letzten Sitzungen nicht wiederhergestellt werden.", prefs);
            pd.run();
        }
    }

    /**
     * Speichert die Bounds des Dialogs in den Präferenzen.
     */
    public void storePreferenceBounds() {
        if (null == _identifier || _identifier.isEmpty()) {
            return;
        }
        Preferences prefs = getPreferenceNode();
        Preferences node = prefs.node(_identifier);
        node.putInt("x", _dialog.getX());
        _preferencesX = _dialog.getX();
        node.putInt("y", _dialog.getY());
        _preferencesY = _dialog.getY();
        node.putInt("width", _dialog.getWidth());
        _preferencesWidth = _dialog.getWidth();
        node.putInt("height", _dialog.getHeight());
        _preferencesHeight = _dialog.getHeight();

        if (LOG) {
            System.out.println("Stored " + _identifier + ": " + _dialog.getBounds());
        }
    }

    @Override
    public String toString() {
        return "GNDDialogOld{" + "_identifier='" + _identifier + '\'' + ", _boundsFromPrefs=" + _boundsFromPrefs + ", _preferencesX=" +
               _preferencesX + ", _preferencesY=" + _preferencesY + ", _preferencesWidth=" + _preferencesWidth + ", _preferencesHeight=" +
               _preferencesHeight + '}';
    }

    public void add(Component component) {
        _dialog.add(component);
    }

    /* *******	Und hier die Methoden, die an den JFrame delegiert werden. ******* */

    public void add(Component component, Object constraints) {
        _dialog.add(component, constraints);
    }

    public void addWindowListener(WindowListener listener) {
        _dialog.addWindowListener(listener);
    }

    public Rectangle getBounds() {
        return _dialog.getBounds();
    }

    public void dispose() {
        _dialog.dispose();
    }

    public Container getContentPane() {
        return _dialog.getContentPane();
    }

    public Point getLocation() {
        return _dialog.getLocation();
    }

    public WindowListener[] getWindowListeners() {
        return _dialog.getWindowListeners();
    }

    public void pack() {
        _dialog.pack();
    }

    public void remove(Component component) {
        _dialog.remove(component);
    }

    public void removeWindowListener(WindowListener listener) {
        _dialog.removeWindowListener(listener);
    }

    public void repaint() {
        _dialog.repaint();
    }

    public void requestFocus() {
        _dialog.requestFocus();
    }

    public void revalidate() {
        _dialog.revalidate();
    }

    public void setDefaultCloseOperation(int operation) {
        _dialog.setDefaultCloseOperation(operation);
    }

    public void setJMenuBar(JMenuBar menuBar) {
        _dialog.setJMenuBar(menuBar);
    }

    public void setLayout(LayoutManager manager) {
        _dialog.setLayout(manager);
    }

    public void setCursor(Cursor cursor) {
        _dialog.setCursor(cursor);
    }

    public void setLocation(int x, int y) {
        _dialog.setLocation(x, y);
    }

    public void setLocationRelativeTo(Component component) {
        _dialog.setLocationRelativeTo(component);
    }

    public void setModal(boolean modal) {
        _dialog.setModal(modal);
    }

    public void setPreferredSize(Dimension d) {
        _dialog.setPreferredSize(d);
    }

    public void setSize(int x, int y) {
        _dialog.setSize(x, y);
    }

    public void setTitle(String title) {
        _dialog.setTitle(title);
    }

    public void setVisible(boolean b) {
        _dialog.setVisible(b);
    }

    public void toFront() {
        _dialog.toFront();
    }

    private class MyWindowListener extends WindowAdapter {
        @Override
        public void windowClosing(WindowEvent e) {
            storePreferenceBounds();
        }
    }

}
