/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.kappich.pat.gnd.viewManagement;

import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.layerManagement.Layer;
import de.kappich.pat.gnd.utils.view.PreferencesDeleter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Die Klasse für die Ansichten der Generischen Netzdarstellung.
 * <p>
 * Eine Ansicht hat einen eindeutigen Namen unter dem sie von der {@link ViewManager Ansichtsverwaltung} geführt wird. Sie besteht aus einer Liste von
 * {@link ViewEntry ViewEntries}, also Layern mit ansichts-spezifischen Einstellungen.
 *
 * @author Kappich Systemberatung
 */
public class View implements Comparable<View> {


    /* Eine kurze Beschreibung, wie das Update eines Views funktioniert. Klassen, die an
     * den Änderungen interesiert sind, implementieren das Interface View.ViewChangeListener.
     * Dies sind z.B. das MapPane, das LegendPane, aber auch der ViewTableModelAdapter
     * in ViewDialog. Dieser ist gleichzeitig aber auch TableModel und ändert seinerseits
     * den View in seiner setValueAt-Methode.
     */

    private static final Debug _debug = Debug.getLogger();
    private static final Pattern ENTRY = Pattern.compile("entry", Pattern.LITERAL);
    private final List<ViewEntry> _viewEntries = new ArrayList<>();
    private final List<ViewChangeListener> _listeners = new CopyOnWriteArrayList<>();
    private String _name;
    private boolean _somethingChanged;    // true, wenn seit dem letzten Speichern etwas geändert wurde

    /**
     * Konstruiert eine leere Ansicht.
     */
    public View() {
    }

    /**
     * Konstruiert eine Ansicht mit Namen und ViewEntries.
     *
     * @param name        der Name der Ansicht
     * @param viewEntries die ViewEntries
     */
    public View(String name, List<ViewEntry> viewEntries) {
        _name = name;
        for (ViewEntry viewEntry : viewEntries) {
            ViewEntry newViewEntry =
                new ViewEntry(viewEntry.getLayer(), viewEntry.getZoomIn(), viewEntry.getZoomOut(), viewEntry.isSelectable(), viewEntry.isVisible());
            newViewEntry.setView(this);
            _viewEntries.add(newViewEntry);
        }
    }

    /**
     * Ein Getter für den Namen.
     *
     * @return den Namen
     */
    public String getName() {
        return _name;
    }

    /**
     * Ein Setter für den Namen.
     *
     * @param name der neue Name
     */
    public void setName(String name) {
        _name = name;
    }

    /**
     * Gibt {@code true} zurück, wenn seit dem Konstruieren der Ansicht oder seit dem letzten Aufruf von setSomethingChanged() mit dem Wert false,
     * eine Veränderung an der Ansicht vorgenommen wurde (ob durch eine verändernde Methode oder durch setSomethingChanged() mit dem Wert {@code
     * true}).
     *
     * @return hat sich etwas geändert?
     */
    public boolean hasSomethingChanged() {
        return _somethingChanged;
    }

    /**
     * Setzt den Wert der internen Variablen, die zum Verwalten von Änderungen seit der letzten Speicherung dient.
     *
     * @param b setzt den Wert des Änderungsstatus
     */
    public void setSomethingChanged(final boolean b) {
        _somethingChanged = b;
    }

    /**
     * Gibt die Menge der ViewEntries inklusive Notiz-Layern zurück.
     *
     * @return die Menge der ViewEntries
     */
    public List<ViewEntry> getAllViewEntries() {
        return getViewEntries(true);
    }

    /**
     * Gibt die Anzahl aller ViewEntries inklusive Notiz-Layern zurück.
     *
     * @return die Anzahl
     */
    public int getNumberOfViewEntries() {
        return 2 * _viewEntries.size();
    }

    /**
     * Gibt die Menge der ViewEntries ohne Notiz-Layer zurück.
     *
     * @return die Menge der ViewEntries
     */
    public List<ViewEntry> getViewEntriesWithoutNoticeEntries() {
        return getViewEntries(false);
    }

    private List<ViewEntry> getViewEntries(final boolean withNoticeLayers) {
        if (withNoticeLayers) {
            final List<ViewEntry> viewEntries = new ArrayList<>(2 * _viewEntries.size());
            int viewEntriesSize = _viewEntries.size();
            for (int i = viewEntriesSize - 1; i >= 0; i--) {
                final ViewEntry viewEntry = _viewEntries.get(i);
                final ViewEntry noticeViewEntry = NoticeViewEntry.create(viewEntry);
                noticeViewEntry.setView(this);
                viewEntries.add(0, noticeViewEntry);
            }
            viewEntries.addAll(_viewEntries);
            return viewEntries;
        } else {
            return Collections.unmodifiableList(_viewEntries);
        }
    }

    /**
     * Gibt den Index des ViewEntries innerhalb der Liste der ViewEntries zurück. Gehört der ViewEntry gemäß Object.equals() nicht zu dieser Ansicht,
     * so ist der Rückgabewert -1.
     *
     * @param viewEntry ein ViewEntry
     *
     * @return der Index des Entrys
     */
    public int getIndex(ViewEntry viewEntry) {
        int i = 0;
        for (ViewEntry entry : _viewEntries) {
            if (entry.equals(viewEntry)) {
                return i;
            }
            i++;
        }
        return -1;
    }

    /**
     * Fügt einen ViewEntry mit dem übergebenen Layer und Default-Einstellungen am Ende der Liste der ViewEntries hinzu, falls der übergebene {@code
     * Layer} noch nicht zu der Ansicht gehört.
     *
     * @param layer ein Layer
     */
    public void addLayer(final Layer layer) {
        addLayer(layer, Integer.MAX_VALUE, 1, true, true);
    }

    /**
     * Fügt einen ViewEntry mit dem übergebenen Layer und Default-Einstellungen am Ende der Liste der ViewEntries hinzu, falls der übergebene {@code
     * Layer} noch nicht zu der Ansicht gehört.
     *
     * @param layer      ein Layer
     * @param zoomIn     Zoomstufe des Einblendens
     * @param zoomOut    Zoomstufe des Ausblendens
     * @param selectable Sollen die Objekte des Layers selektierbar sein?
     * @param visible    Soll der Layer eingeblendet werden?
     */
    @SuppressWarnings("SameParameterValue")
    public void addLayer(final Layer layer, int zoomIn, int zoomOut, boolean selectable, boolean visible) {
        for (ViewEntry viewEntry : _viewEntries) {
            if (viewEntry.getLayer().getName().equals(layer.getName())) {
                return;
            }
        }
        ViewEntry viewEntry = new ViewEntry(layer, zoomIn, zoomOut, selectable, visible);
        viewEntry.setView(this);
        _viewEntries.add(viewEntry);
        try {
            notifyChangeListenersViewEntryAppended();
        } catch (RuntimeException e) {
            _debug.error("Fehler in View.addLayer(): ", e.toString());
        }
        _somethingChanged = true;
    }

    /**
     * Informiert die {@link ViewChangeListener View.ViewChangeListeners} über einen geänderte Layer. Kann von außen z.B. von der {@link ViewManager
     * Ansichtsverwaltung} aufgerufen werden.
     *
     * @param layer ein Layer
     */
    public void layerDefinitionChanged(final Layer layer) {
        final String layerName = layer.getName();
        for (int i = 0; i < _viewEntries.size(); i++) {
            final ViewEntry entry = _viewEntries.get(i);
            if (entry.getLayer().getName().equals(layerName)) {
                notifyChangeListenersViewEntryDefinitionChanged(i);
            }
        }
    }

    /**
     * Informiert die {@link ViewChangeListener View.ViewChangeListeners} über einen geänderte Layer. Kann von außen z.B. von der {@link ViewManager
     * Ansichtsverwaltung} aufgerufen werden.
     *
     * @param layer ein Layer
     */
    public void layerPropertyChanged(final Layer layer) {
        final String layerName = layer.getName();
        for (int i = 0; i < _viewEntries.size(); i++) {
            final ViewEntry entry = _viewEntries.get(i);
            if (entry.getLayer().getName().equals(layerName)) {
                notifyChangeListenersViewEntryPropertyChanged(i);
            }
        }
    }

    /**
     * Entfernt alle ViewEntries aus der Ansicht, die den Layer mit dem übergebenen Namen verwenden.
     *
     * @param layerName ein Layername
     */
    public void removeLayer(final String layerName) {
        final int size = _viewEntries.size();
        for (int i = size - 1; i >= 0; i--) {
            if (_viewEntries.get(i).getLayer().getName().equals(layerName)) {
                remove(i);
                _somethingChanged = true;
            }
        }
    }

    /**
     * Da jeder {@code Layer} nur einmal in einer Ansicht auftreten kann, kann der zugehörige {@code ViewEntry} berechnet werden.
     *
     * @param layer
     *
     * @return
     */
    @Nullable
    public ViewEntry getViewEntry(final Layer layer) {
        for (ViewEntry entry : _viewEntries) {
            if (entry.getLayer().getName().equals(layer.getName())) {
                return entry;
            }
        }
        return null;
    }

    /**
     * Fügt den Listener den auf Änderungen angemeldeten Listenern hinzu.
     *
     * @param listener ein Listener
     */
    public void addChangeListener(ViewChangeListener listener) {
        _listeners.add(listener);
    }

    /**
     * Entfernt den Listener aus der Menge der auf Änderungen angemeldeten Listenern.
     *
     * @param listener ein Listener
     */
    public void removeChangeListener(ViewChangeListener listener) {
        _listeners.remove(listener);
    }

    /**
     * Entfernt alle Listener.
     */
    public void removeAllChangeListeners() {
        _listeners.clear();
    }

    /**
     * Entfernt den ViewEntry, der an der Stelle {@code index} in der Liste der ViewEntries steht.
     *
     * @param index ein Index
     */
    public void remove(int index) {
        if ((index >= 0) && (index < _viewEntries.size())) {
            _viewEntries.get(index).setView(null);
            _viewEntries.remove(index);
            notifyChangeListenersViewEntryRemoved(index);
            _somethingChanged = true;
        }
    }

    /**
     * Vertauscht die durch die Indizes i und j angegebenen ViewEntries in der Liste aller ViewEntries.
     *
     * @param i ein Index
     * @param j ein Index
     */
    public void switchTableLines(int i, int j) {
        if ((i >= 0) && (i < _viewEntries.size()) && (j >= 0) && (j < _viewEntries.size())) {
            final ViewEntry iRow = _viewEntries.set(i, _viewEntries.get(j));
            _viewEntries.set(j, iRow);
            notifyChangeListenersViewEntriesSwitched(i, j);
            _somethingChanged = true;
        }
    }

    /*
     * Informiert die ChangeListener darüber, dass ein neuer ViewEntry angehängt wurde.
     */
    private void notifyChangeListenersViewEntryAppended() {
        for (ViewChangeListener changeListener : _listeners) {
            changeListener.viewEntryInserted(this, _viewEntries.size() - 1);
            changeListener.viewEntryInserted(this, _viewEntries.size() * 2 - 1);
        }
    }

    /**
     * Informiert die ChangeListener darüber, dass die Definition von ViewEntry i geändert wurde.
     *
     * @param i ein Index
     */
    public void notifyChangeListenersViewEntryDefinitionChanged(int i) {
        _somethingChanged = true;
        for (ViewChangeListener changeListener : _listeners) {
            changeListener.viewEntryDefinitionChanged(this, i);
            changeListener.viewEntryDefinitionChanged(this, i + _viewEntries.size());
        }
    }

    /**
     * Informiert die ChangeListener darüber, dass eine Eigenschaft von ViewEntry i geändert wurde.
     *
     * @param i ein Index
     */
    public void notifyChangeListenersViewEntryPropertyChanged(int i) {
        _somethingChanged = true;
        for (ViewChangeListener changeListener : _listeners) {
            changeListener.viewEntryPropertyChanged(this, i);
            changeListener.viewEntryPropertyChanged(this, i + _viewEntries.size());
        }
    }

    /**
     * Informiert die ChangeListener darüber, dass ViewEntry i entfernt wurde.
     *
     * @param i ein Index
     */
    private void notifyChangeListenersViewEntryRemoved(int i) {
        for (ViewChangeListener changeListener : _listeners) {
            // ACHTUNG: Reihenfolge wichtig. Wenn die Listener zuerst den Entry mit dem kleineren Index
            // entfernen würden, so gäbe es Probleme.
            changeListener.viewEntryRemoved(this, i + _viewEntries.size());
            changeListener.viewEntryRemoved(this, i);
        }
    }

    /**
     * Informiert die ChangeListener darüber, dass ViewEntries i und j vertauscht wurden.
     *
     * @param i ein Index
     * @param j ein Index
     */
    private void notifyChangeListenersViewEntriesSwitched(int i, int j) {
        for (ViewChangeListener changeListener : _listeners) {
            changeListener.viewEntriesSwitched(this, i, j);
            changeListener.viewEntriesSwitched(this, i + _viewEntries.size(), j + _viewEntries.size());
        }
    }

    /**
     * Speichert die Ansicht in den übergebenen Knoten.
     *
     * @param prefs der Knoten, unter dem die Speicherung beginnt
     */
    public void putPreferences(Preferences prefs) {
        deletePreferences(prefs);
        Preferences objectPrefs = prefs.node(getName());
        int i = 0;
        for (ViewEntry viewEntry : _viewEntries) {
            Preferences entryPrefs = objectPrefs.node("entry" + i);
            viewEntry.putPreferences(entryPrefs);
            i++;
        }
        _somethingChanged = false;
    }

    /**
     * Initialisiert die Ansicht aus dem übergebenen Knoten.
     *
     * @param prefs der Knoten, unter dem die Initialisierung beginnt
     */
    public void initializeFromPreferences(Preferences prefs) {
        _name = prefs.name();
        String[] entries;
        try {
            entries = prefs.childrenNames();
        } catch (BackingStoreException | IllegalStateException ignored) {
            PreferencesDeleter pd = new PreferencesDeleter("Eine benutzer-definierte Ansicht kann nicht geladen werden.", prefs);
            pd.run();
            return;
        }
        final Map<Integer, ViewEntry> orderedViewEntries = new TreeMap<>();
        for (String entryName : entries) {
            if (!entryName.startsWith("entry")) {    // jetzt ein Fehler, aber später vielleicht okay
                continue;
            }
            Preferences entryPrefs = prefs.node(entryName);
            ViewEntry entry = new ViewEntry();
            if (entry.initializeFromPreferences(entryPrefs)) {
                final Integer index = Integer.valueOf(ENTRY.matcher(entryName).replaceAll(Matcher.quoteReplacement("")));
                orderedViewEntries.put(index, entry);
                entry.setView(this);
            }
        }
        _viewEntries.addAll(orderedViewEntries.values());
    }

    /**
     * Entfernt die Ansicht unterhalb des übergebenen Knotens.
     *
     * @param prefs der Knoten, unter dem gelöscht wird
     */
    public void deletePreferences(Preferences prefs) {
        Preferences objectPrefs = prefs.node(getName());
        try {
            objectPrefs.removeNode();
        } catch (BackingStoreException ignored) {
            PreferencesDeleter pd = new PreferencesDeleter("Es ist ein Fehler beim Löschen aus den Präferenzen aufgetreten.", objectPrefs);
            pd.run();
        }
    }

    /**
     * Gibt die Menge aller Farben, die von allen ViewEntries der Ansicht benutzt werden, zurück.
     *
     * @return die Menge aller benutzten Farben
     */
    public Set<String> getUsedColors() {
        Set<String> usedColors = new HashSet<>();
        for (ViewEntry entry : _viewEntries) {
            usedColors.addAll(entry.getUsedColors());
        }
        return usedColors;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("[View: " + _name);
        for (ViewEntry viewEntry : _viewEntries) {
            sb.append(viewEntry.toString());
        }
        sb.append("]");
        return sb.toString();
    }

    /**
     * Erzeugt eine tiefe Kopie des aufrufenden Objekts und setzt den Namen auf den übergebenen Wert, falls dieser nicht {@code null} ist.
     *
     * @param name der neue Name
     *
     * @return die Kopie
     */
    public View getCopy(final String name) {
        View copy = new View();
        if (name != null) {
            copy._name = name;
        } else {
            copy._name = _name;
        }
        for (ViewEntry viewEntry : _viewEntries) {
            final ViewEntry viewEntryCopy = viewEntry.getCopy();
            viewEntryCopy.setView(copy);
            copy._viewEntries.add(viewEntryCopy);
        }
        return copy;
    }

    @Override
    public int compareTo(final View o) {
        return getName().toLowerCase().compareTo(o.getName().toLowerCase());
    }

    @Override
    public boolean equals(Object o) {
	    if (!(o instanceof View other)) {
            return false;
        }
        return getName().toLowerCase().equals(other.getName().toLowerCase());
    }

    @Override
    public int hashCode() {
        return getName().toLowerCase().hashCode();
    }

    /**
     * Ein Interface für Listener, die über Änderungen der Ansicht informiert werden wollen. Bei der Implementation dieses Interfaces ist zu beachten,
     * dass die View die ViewChangeListener über alle Änderungen informiert, d.h. auch über solche der Notiz-Layer. Damit werden den
     * ViewChangeListener möglicherweise Indizes mitgeteilt, die sie nicht kennen, nämlich wenn sie {@link #getViewEntries(boolean)} mit {@code false}
     * aufgerufen haben.
     *
     * @author Kappich Systemberatung
     */
    public interface ViewChangeListener {
        /**
         * Der Ansicht wurde ein Layer am Ende an der angegebenen Stelle hinzugefügt.
         *
         * @param view     die Ansicht
         * @param newIndex der Index
         */
        void viewEntryInserted(View view, final int newIndex);

        /**
         * Die Definition des Layers an der i-ten Stelle der Ansicht wurde geändert. Diese Methode soll benutzt werden, wenn eine erneute
         * Initialisierung des Layers notwendig ist.
         *
         * @param view die Ansicht
         * @param i    ein Index
         */
        void viewEntryDefinitionChanged(View view, int i);

        /**
         * Eine Eigenschaft des Layers an der i-ten Stelle der Ansicht wurde geändert. Diese Methode soll benutzt werden, wenn keine erneute
         * Initialisierung des Layers notwendig ist.
         *
         * @param view die Ansicht
         * @param i    ein Index
         */
        void viewEntryPropertyChanged(View view, int i);

        /**
         * Der Layer an der i-ten Stelle der Ansicht wurde gelöscht.
         *
         * @param view die Ansicht
         * @param i    ein Index
         */
        void viewEntryRemoved(View view, int i);

        /**
         * Die Layer an der i-ten und j-ten Stelle der Ansicht wurden miteinander vertauscht.
         *
         * @param view die Ansicht
         * @param i    ein Index
         * @param j    ein Index
         */
        void viewEntriesSwitched(View view, int i, int j);
    }

}
