/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.viewManagement;

import de.kappich.pat.gnd.documentation.GndHelp;
import de.kappich.pat.gnd.gnd.GenericNetDisplay;
import de.kappich.pat.gnd.utils.SpringUtilities;
import de.kappich.pat.gnd.utils.view.GndDialog;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Rectangle;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.SpringLayout;
import javax.swing.UIDefaults;
import javax.swing.UIManager;
import javax.swing.WindowConstants;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.DefaultTableCellRenderer;

/**
 * Eine Dialog, der alle Ansichten auflistet und über Schaltflächen Möglichkeiten bietet, diese zu betrachten, zu bearbeiten, zu löschen oder neue
 * einzuführen. Der ViewManagerDialog zeigt die Inhalte der {@link ViewManager Ansichtsverwaltung}.
 *
 * @author Kappich Systemberatung
 */
@SuppressWarnings("OverlyLongMethod")
public final class ViewManagerDialog {

    private final GenericNetDisplay _gnd;

    private final GndDialog _dialog;

    private final JTable _table;
    private final JButton _aktivierenButton = new JButton("Ansicht aktivieren");
    private final JButton _startAnsichtButton = new JButton("Als Startansicht festlegen");
    private final JButton _schliessenButton = new JButton("Dialog schließen");
    private final JButton _neueAnsichtButton = new JButton("Neue Ansicht");
    private final JButton _bearbeitenButton = new JButton("Bearbeiten");
    private final JButton _kopierenButton = new JButton("Kopieren");
    private final JButton _loeschenButton = new JButton("Löschen");
    private final JButton _helpButton = new JButton("Hilfe");
    private final Object _lock = new Object();
    private boolean _modal;

    /**
     * Konstruiert den Dialog zur Anzeige bzw. Auswahl aller Ansichten. Der Dialog kennt das die {@link GenericNetDisplay Generische-Netzdarstellung},
     * zu der er gehört, und kann bei Bedarf modal gestartet werden.
     *
     * @param gnd   die Netzdarstellung
     * @param modal ist der Dialog modal?
     */
    public ViewManagerDialog(GenericNetDisplay gnd, boolean modal) {
        _gnd = gnd;
        _modal = modal;

        JPanel buttonPanelEast = new JPanel();
        buttonPanelEast.setLayout(new BoxLayout(buttonPanelEast, BoxLayout.Y_AXIS));
        buttonPanelEast.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));

        Dimension d = new Dimension(15, 15);
        buttonPanelEast.add(Box.createRigidArea(d));

        buttonPanelEast.add(_neueAnsichtButton);
        buttonPanelEast.add(Box.createRigidArea(d));

        buttonPanelEast.add(_bearbeitenButton);
        buttonPanelEast.add(Box.createRigidArea(d));

        buttonPanelEast.add(_kopierenButton);
        buttonPanelEast.add(Box.createRigidArea(d));

        buttonPanelEast.add(_loeschenButton);
        buttonPanelEast.add(Box.createRigidArea(d));

        buttonPanelEast.add(_helpButton);
        buttonPanelEast.add(Box.createRigidArea(d));

        JPanel buttonPanelSouth = new JPanel();
        buttonPanelSouth.setLayout(new SpringLayout());
        buttonPanelSouth.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));

        buttonPanelSouth.add(_aktivierenButton);
        buttonPanelSouth.add(_startAnsichtButton);
        buttonPanelSouth.add(_schliessenButton);
        SpringUtilities.makeCompactGrid(buttonPanelSouth, 3, 10, 5);

        _bearbeitenButton.setEnabled(false);
        _kopierenButton.setEnabled(false);
        _loeschenButton.setEnabled(false);
        _aktivierenButton.setEnabled(false);
        _startAnsichtButton.setEnabled(false);

        _dialog = new GndDialog("ViewManagerDialog", _gnd.getFrame(), _modal);
        setDialogModalSettings(_modal);
        _dialog.setTitle("GND: Ansichtsverwaltung");

        _dialog.add(buttonPanelEast, BorderLayout.EAST);
        _dialog.add(buttonPanelSouth, BorderLayout.SOUTH);

        _table = new JTable(ViewManager.getInstance());
        _table.setDefaultRenderer(Object.class, new MyDefaultTableCellRenderer());

        // Der folgende Versuch ist zwecklos: der Font steht schon auf Bold.
//		DefaultTableCellRenderer defaultTableCellRenderer = (DefaultTableCellRenderer)_table.getTableHeader().getDefaultRenderer();
//		Font font = defaultTableCellRenderer.getFont();
//		((DefaultTableCellRenderer)_table.getTableHeader().getDefaultRenderer()).setFont(new Font(font.getName(),Font.BOLD,font.getSize()));

        _table.setPreferredScrollableViewportSize(new Dimension(500, 70));
        _table.setFillsViewportHeight(true);
        final JScrollPane comp = new JScrollPane(_table);
        _dialog.add(comp, BorderLayout.CENTER);

        _dialog.setPositionAndSize(0, 0, 950, 50, true, 0, 0);

        final ListSelectionModel selectionModel = _table.getSelectionModel();
        selectionModel.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        // vordefinierte Darstellungsobjekttypen dürfen nicht bearbeitet oder gelöscht werden
        ListSelectionListener listSelectionListener = e -> {
            final int selectedRow = _table.getSelectedRow();
            if (selectedRow == -1) {
                _bearbeitenButton.setToolTipText("Es ist keine Ansicht ausgewählt worden");
                _bearbeitenButton.setEnabled(false);
                _kopierenButton.setToolTipText("Es ist keine Ansicht ausgewählt worden");
                _kopierenButton.setEnabled(false);
                _loeschenButton.setToolTipText("Es ist keine Ansicht ausgewählt worden");
                _loeschenButton.setEnabled(false);
                _aktivierenButton.setToolTipText("Es ist keine Ansicht ausgewählt worden");
                _aktivierenButton.setEnabled(false);
                _startAnsichtButton.setToolTipText("Es ist keine Ansicht ausgewählt worden");
                _startAnsichtButton.setEnabled(false);
            } else {
                final ViewManager instance = ViewManager.getInstance();
                final boolean changeable = instance.isChangeable(instance.getView(selectedRow));
                if (!changeable) {
                    _bearbeitenButton.setText("Betrachten");
                    _bearbeitenButton.setToolTipText("Die Details der Ansicht betrachten");
                    _loeschenButton.setToolTipText("Die ausgewählte Ansicht ist nicht löschbar");
                    _loeschenButton.setEnabled(false);
                } else {
                    _bearbeitenButton.setText("Bearbeiten");
                    _bearbeitenButton.setToolTipText("Die Details der ausgewählten Ansicht bearbeiten");
                    _loeschenButton.setToolTipText("Die ausgewählte Ansicht löschen");
                    _loeschenButton.setEnabled(!_modal);
                }
                _bearbeitenButton.setEnabled(!_modal);
                _kopierenButton.setToolTipText("Kopie der ausgewählten Ansicht erstellen und bearbeiten");
                _kopierenButton.setEnabled(!_modal);
                _aktivierenButton.setToolTipText("Die ausgewählte Ansicht zur aktuellen Ansicht machen");
                _aktivierenButton.setEnabled(true);
                _startAnsichtButton.setToolTipText("Die ausgewählte Ansicht zur Startansicht machen");
                _startAnsichtButton.setEnabled(!_modal);
            }
        };
        selectionModel.addListSelectionListener(listSelectionListener);

        final KeyAdapter keyAdapter = new KeyAdapter() {
            @Override
            public void keyPressed(KeyEvent e) {
                if (e.getKeyCode() == KeyEvent.VK_DELETE) {
                    final int selectedRow = _table.getSelectedRow();
                    if (selectedRow >= 0) {
                        final View view = ViewManager.getInstance().getView(selectedRow);
                        if (ViewManager.getInstance().isChangeable(view)) {
                            Object[] options = {"Ja", "Nein"};
                            int n = JOptionPane
                                .showOptionDialog(_dialog.getDialog(), "Wollen Sie die Ansicht " + view.getName() + " wirklich löschen?",
                                                  "Ansicht " + view.getName() + "löschen", JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE,
                                                  null, options, options[0]);
                            if (n == 0) {
                                ViewManager.getInstance().removeView(view);
                            }
                        }
                    }
                } else {
                    char c = e.getKeyChar();
                    if (Character.isLetter(c)) {
                        int index = ViewManager.getInstance().getIndexOfFirstView(c);
                        if (index >= 0) {
                            _table.getSelectionModel().setSelectionInterval(index, index);
                            _table.scrollRectToVisible(new Rectangle(_table.getCellRect(index, 0, true)));
                        }
                    }
                }
            }
        };
        _table.addKeyListener(keyAdapter);
        _aktivierenButton.addKeyListener(keyAdapter);
        _startAnsichtButton.addKeyListener(keyAdapter);
        _schliessenButton.addKeyListener(keyAdapter);
        _neueAnsichtButton.addKeyListener(keyAdapter);
        _bearbeitenButton.addKeyListener(keyAdapter);
        _kopierenButton.addKeyListener(keyAdapter);
        _loeschenButton.addKeyListener(keyAdapter);
        _helpButton.addKeyListener(keyAdapter);

        _table.addMouseListener(new MouseAdapter() {
            @Override
            public void mousePressed(final MouseEvent e) {
                if (e.getClickCount() > 1) {
                    editOrInspectAction();
                }
            }
        });

        ActionListener actionListenerAnsichtAuswaehlen = e -> {
            final int selectedRow = _table.getSelectedRow();
            if (selectedRow == -1) {
                JOptionPane.showMessageDialog(new JFrame(), "Bitte wählen Sie eine Zeile aus der Liste aus!", "Fehler", JOptionPane.ERROR_MESSAGE);
                return;
            }
            synchronized (_lock) {
                _dialog.setVisible(false);    // damit niemand auf die Idee kommt, nochmal zu drücken

                // Altes aufräumen:
                _gnd.storePreferenceBounds();
                _gnd.getView().removeAllChangeListeners();

                Runnable doSetSplitPaneFromView = () -> {
                    final View view = ViewManager.getInstance().getView(selectedRow);
                    _gnd.setSplitPaneFromView(view);
                    _gnd.updateCsvMenu();
                };
                Thread setSplitPaneFromViewThread = new Thread(doSetSplitPaneFromView);
                setSplitPaneFromViewThread.start();

                setDialogModalSettings(false);
            }
        };
        _aktivierenButton.addActionListener(actionListenerAnsichtAuswaehlen);

        ActionListener actionListenerStartViewFestlegen = e -> {
            final int selectedRow = _table.getSelectedRow();
            if (selectedRow != -1) {
                final View view = ViewManager.getInstance().getView(selectedRow);
                if (view != null) {
                    _gnd.writeStartViewNamePreference(view.getName());
                }
            }
            _dialog.repaint();
        };
        _startAnsichtButton.addActionListener(actionListenerStartViewFestlegen);

        ActionListener actionListenerNeueAnsicht = e -> newAction();
        _neueAnsichtButton.addActionListener(actionListenerNeueAnsicht);

        ActionListener actionListenerAnsichtBearbeiten = e -> editOrInspectAction();
        _bearbeitenButton.addActionListener(actionListenerAnsichtBearbeiten);

        ActionListener actionListenerAnsichtKopieren = e -> copyAction();
        _kopierenButton.addActionListener(actionListenerAnsichtKopieren);

        ActionListener actionListenerAnsichtLoeschen = e -> {
            final int selectedRow = _table.getSelectedRow();
            if (!ViewManager.getInstance().removeView(ViewManager.getInstance().getView(selectedRow))) {
                JOptionPane.showMessageDialog(new JFrame(), "Der Ansicht " + ViewManager.getInstance().getView(selectedRow).getName() +
                                                            " kann nicht gelöscht werden!", "Fehler", JOptionPane.ERROR_MESSAGE);
            }
        };
        _loeschenButton.addActionListener(actionListenerAnsichtLoeschen);

        ActionListener actionListenerHilfe = e -> GndHelp.openHelp("#theViewManagerDialog");
        _helpButton.addActionListener(actionListenerHilfe);
    }

    private static void newAction() {
        View view = new View();
        final String title = "GND: eine neue Ansicht bearbeiten";
        ViewDialog.runDialog(ViewManager.getInstance(), view, true, true, true, title);
    }

    /**
     * Zeigt den Dialog an.
     */
    public void showDialog() {
        _dialog.toFront();
        _dialog.setVisible(true);
    }

    /**
     * Schließt den Dialog.
     */
    public void closeDialog() {
        _dialog.setVisible(false);
        _dialog.storePreferenceBounds();
        _dialog.dispose();
    }

    private void setDialogModalSettings(boolean modal) {
        _modal = modal;
        if (_modal) {
            _dialog.setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
            _neueAnsichtButton.setEnabled(false);
            _loeschenButton.setEnabled(false);
            _helpButton.setEnabled(false);
            final class MyWindowListenerModal extends WindowAdapter {
                private MyWindowListenerModal() {
                    super();
                }

                @Override
                public void windowClosing(WindowEvent e) {
                    Object[] options = {"GND beenden", "Abbrechen"};
                    int n = JOptionPane.showOptionDialog(new JFrame(), "Entweder Sie aktivieren eine Ansicht oder die GND wird beendet.",
                                                         "Ansicht aktivieren oder beenden", JOptionPane.YES_NO_CANCEL_OPTION,
                                                         JOptionPane.QUESTION_MESSAGE, null, options, options[1]);
                    if (n == 0) {
                        _dialog.storePreferenceBounds();
                        if (_gnd.isStandAlone()) {
                            System.exit(0);
                        } else {
                            _gnd.dispose();
                        }
                    }
                }
            }
            _dialog.addWindowListener(new MyWindowListenerModal());

            for (ActionListener listener : _schliessenButton.getActionListeners()) {
                _schliessenButton.removeActionListener(listener);
            }
            final ActionListener cancelActionListener = e -> {
                Object[] options = {"GND beenden", "Abbrechen"};
                int n = JOptionPane.showOptionDialog(new JFrame(), "Entweder Sie aktivieren eine Ansicht oder die GND wird beendet.",
                                                     "Ansicht aktivieren oder beenden", JOptionPane.YES_NO_CANCEL_OPTION,
                                                     JOptionPane.QUESTION_MESSAGE, null, options, options[1]);
                if (n == 0) {
                    _dialog.storePreferenceBounds();
                    if (_gnd.isStandAlone()) {
                        System.exit(0);
                    } else {
                        _gnd.dispose();
                    }
                }
            };
            _schliessenButton.addActionListener(cancelActionListener);

        } else {
            _dialog.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
            _neueAnsichtButton.setEnabled(true);
            _helpButton.setEnabled(true);
            for (WindowListener listener : _dialog.getWindowListeners()) {
                _dialog.removeWindowListener(listener);
            }
            for (ActionListener listener : _schliessenButton.getActionListeners()) {
                _schliessenButton.removeActionListener(listener);
            }
            final class MyWindowListenerNotModal extends WindowAdapter {
                private MyWindowListenerNotModal() {
                    super();
                }

                @Override
                public void windowClosing(WindowEvent e) {
                    _dialog.storePreferenceBounds();
                }
            }
            _dialog.addWindowListener(new MyWindowListenerNotModal());

            ActionListener actionListenerDialogSchliessen = e -> closeDialog();
            _schliessenButton.addActionListener(actionListenerDialogSchliessen);
        }
        _dialog.setModal(_modal);
    }

    private void editOrInspectAction() {
        final int selectedRow = _table.getSelectedRow();
        if (selectedRow != -1) {
            final View view = ViewManager.getInstance().getView(selectedRow);
            if (view != null) {
                final boolean changeable = ViewManager.getInstance().isChangeable(view);
                final String title;
                if (changeable) {
                    title = "GND: eine Ansicht bearbeiten";
                } else {
                    title = "GND: eine Ansicht betrachten";
                }
                ViewDialog.runDialog(ViewManager.getInstance(), view, changeable, changeable, false, title);
            }
        }
    }

    private void copyAction() {
        final int selectedRow = _table.getSelectedRow();
        if (selectedRow != -1) {
            final View view = ViewManager.getInstance().getView(selectedRow);
            if (view != null) {
                final View copiedView = view.getCopy(view.getName() + " (Kopie)");
                final String title = "GND: eine kopierte Ansicht bearbeiten";
                ViewDialog.runDialog(ViewManager.getInstance(), copiedView, true, true, true, title);
            }
        }
    }

    @Override
    public String toString() {
        return "ViewManagerDialog";
    }

    class MyDefaultTableCellRenderer extends DefaultTableCellRenderer {

        private Color _background;

        MyDefaultTableCellRenderer() {
            UIDefaults defaults = UIManager.getDefaults();
            _background = defaults.getColor("Table.selectionBackground");
            if (_background == null) {
                _background = Color.lightGray;
            }
        }

        @Override
        public Component getTableCellRendererComponent(final JTable table, final Object value, final boolean isSelected, final boolean hasFocus,
                                                       final int row, final int column) {
            Component cell = super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
            if (null != value) { // null passiert auf Macs
	            if (value instanceof String viewName) {
                    if (ViewManager.getInstance().isChangeable(ViewManager.getInstance().getView(viewName))) {
                        setForeground(Color.BLACK);
                        Font font = getFont();
                        setFont(new Font(font.getName(), Font.PLAIN, font.getSize()));
                    } else {
                        setForeground(Color.GRAY);
                        Font font = getFont();
                        setFont(new Font(font.getName(), Font.ITALIC, font.getSize()));
                    }
                    if (viewName.equals(_gnd.getStartViewName())) {
                        if (isSelected) {
                            setBackground(new Color(220, 220, 70));
                        } else {
                            setBackground(new Color(255, 255, 100));
                        }
                    } else {
                        if (isSelected) {
                            setBackground(_background);
                        } else {
                            setBackground(Color.white);
                        }
                    }
                }
            }
            return cell;
        }
    }
}
