/*
 * Copyright 2019-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.accessControl;

import de.bsvrz.dav.daf.accessControl.internal.DummyAccessControlManager;
import de.bsvrz.dav.daf.communication.protocol.UserLogin;
import de.bsvrz.dav.daf.main.config.SystemObject;

/**
 * Interface für Klassen, die Abfragen nach Benutzerrechten ermöglichen. Hierzu muss zuerst für einen Benutzer, dessen Rechte grpüft werden sollen,
 * das entsprechende {@linkplain UserInfo Benutzerobjekt} geholt werden. Dieses Objekt kann man dann nach Berechtigungen für den angegebenen Benutzer
 * fragen.
 *
 * @author Kappich Systemberatung
 */
public interface AccessControlManager {

    /**
     * Gibt die Benutzerrechte zu dem angegebenen Benutzer zurück. Ist der Benutzer ein Administrator ist jede Aktion erlaubt. Ist der Benutzer nicht
     * eingeloggt ist keine Aktion erlaubt.
     *
     * @param userLogin Benutzer
     *
     * @return Das geforderte UserInfo-Objekt
     */
    default UserInfo getUserPermissions(UserLogin userLogin) {
        if (userLogin.isRegularUser()) {
            return getUserPermissions(userLogin.toLong());
        } else if (userLogin.isSystemUser()) {
            return new DummyAccessControlManager.AllUserPermissions(0);
        } else {
            return new DummyAccessControlManager.NoUserPermissions(-1);
        }
    }

    /**
     * Gibt die Benutzerrechte zu dem Benutzer mit der angegebenen ID zurück. Handelt es sich bei der ID nicht um einen gültigen Benutzer, werden
     * keine Rechte vergeben.
     *
     * @param userId Angegebene BenutzerId
     *
     * @return Das geforderte UserInfo-Objekt
     */
    UserInfo getUserPermissions(long userId);

    /**
     * Gibt die Benutzerrechte zu dem angegebenen Benutzer zurück. Handelt es sich dabei nicht um einen gültigen Benutzer, werden keine Rechte
     * vergeben.
     *
     * @param userObject Benutzer-Objekt
     *
     * @return Das geforderte UserInfo-Objekt
     */
    default UserInfo getUserPermissions(SystemObject userObject) {
        return getUserPermissions(userObject.getId());
    }

    /**
     * Gibt die Benutzerrechte zum aktuell angemeldeten Benutzer zurück.
     *
     * @return Ein UserInfo-Objekt
     */
    UserInfo getUserPermissions();

    /**
     * Meldet einen Listener auf geänderte Nutzerrechte an
     *
     * @param listener Listener
     */
    void addChangeListener(AccessControlChangeListener listener);

    /**
     * Entfernt einen Listener auf geänderte Benutzerrechte
     *
     * @param listener Listener
     */
    void removeChangeListener(AccessControlChangeListener listener);
}
