/*
 * Copyright 2018-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.accessControl;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.ConfigurationObject;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;

/**
 * Empfängt den Status der Rechteprüfung über die Attributgruppe atg.datenverteilerRechteprüfung am Datenverteilerobjekt. Falls das Datenmodell zu alt
 * ist, tut diese Klasse nichts.
 *
 * @author Kappich Systemberatung
 */
public class DafAccessControlReceiver {
    private static final Debug _debug = Debug.getLogger();
    private static final int TIMEOUT_SECONDS = 10;
    private final ClientDavInterface _connection;
    private final ConfigurationObject _davObject;
    private final DataModel _dataModel;
    private DataDescription _dataDescription;

    /**
     * Erstellt einen neuen DafAccessControlReceiver, der den angeschlossenen Datenverteiler nach der Art der Rechteprüfung fragt
     *
     * @param connection Verbindung
     */
    public DafAccessControlReceiver(final ClientDavInterface connection) {
        this(connection, connection.getLocalDav());
    }

    /**
     * Erstellt einen neuen DafAccessControlReceiver, der einen beliebigen Datenverteiler nach der Art der Rechteprüfung fragt
     *
     * @param connection Verbindung
     * @param davObject  Datenverteiler-Objekt
     */
    public DafAccessControlReceiver(final ClientDavInterface connection, final ConfigurationObject davObject) {
        _connection = connection;
        _davObject = davObject;
        _dataModel = _connection.getDataModel();
        AttributeGroup atg = _dataModel.getAttributeGroup("atg.datenverteilerRechteprüfung");
        Aspect asp = _dataModel.getAspect("asp.standard");
        if (atg == null || asp == null) {
            return;
        }
        _dataDescription = new DataDescription(atg, asp);
    }

    @Nullable
    private static AccessControlMode extractMode(final Data data) {
	    return switch (data.getTextValue("Rechteprüfung").getText()) {
		    case "Alte Rechteprüfung" -> AccessControlMode.OldDataModel;
		    case "Neue Rechteprüfung" -> AccessControlMode.NewDataModel;
		    case "Deaktiviert" -> AccessControlMode.Disabled;
		    default -> {
                _debug.warning("Art der Rechteprüfung konnte nicht bestimmt werden, unbekannter Wert: " + data);
			    yield null;
            }
	    };
    }

    /**
     * Gibt die Art der Rechteprüfung zurück
     *
     * @return Art der Rechteprüfung oder null, falls diese nicht bestimmt werden konnte (z. B. zu alte Kernsoftware)
     */
    @Nullable
    public AccessControlMode getAccessControlMode() {
        if (_dataDescription == null) {
            _debug.warning("Art der Rechteprüfung konnte nicht bestimmt werden, Datenmodell veraltet.");
            return null;
        }
        int i = 0;
        ResultData data = null;
        while (i < TIMEOUT_SECONDS) {
            data = _connection.getData(_davObject, _dataDescription, 10000);
            if (data.hasData()) {
                return extractMode(data.getData());
            }
            i++;
        }
        _debug.warning("Art der Rechteprüfung konnte nicht bestimmt werden, timeout beim Warten auf Daten: " + data);
        return null;
    }

}
