/*
 * Copyright 2019-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.accessControl.internal;

import java.util.concurrent.CopyOnWriteArrayList;

/**
 * Abstrakte Klasse, die die Listener-Funktionen der ObjectSelectionBlock-Klassen ("AuswahlBereich", "AuswahlRegion" usw.) kapselt
 *
 * @author Kappich Systemberatung
 * @version $Revision: 0000 $
 */
abstract class AbstractObjectSelectionBlock implements ObjectSelectionBlock {

    private final CopyOnWriteArrayList<ObjectCollectionChangeListener> _objectCollectionChangeListeners = new CopyOnWriteArrayList<>();

    /**
     * Fügt einen neuen Listener hinzu, der über Änderungen an dieser Objektauswahl informiert wird
     *
     * @param object Listener Listener
     */
    @Override
    public void addChangeListener(ObjectCollectionChangeListener object) {
        if (_objectCollectionChangeListeners.isEmpty()) {
            startChangeListener();
        }
        _objectCollectionChangeListeners.add(object);
    }

    /**
     * Entfernt einen Listener wieder
     *
     * @param object Listener Listener
     */
    @Override
    public void removeChangeListener(ObjectCollectionChangeListener object) {
        _objectCollectionChangeListeners.remove(object);
        if (_objectCollectionChangeListeners.isEmpty()) {
            stopChangeListener();
        }
    }

    @Override
    public void dispose() {
        _objectCollectionChangeListeners.clear();
        stopChangeListener();
    }

    /**
     * Wird benutzt um Listener über Änderungen dieses Blocks zu informieren.
     */
    protected void notifyBlockChanged() {
        for (final ObjectCollectionChangeListener objectCollectionChangeListener : _objectCollectionChangeListeners) {
            objectCollectionChangeListener.blockChanged();
        }
    }

    /** Startet, falls nötig, eventuelle interne Listener, die den Änderungsstatus der verwalteten Objekte überwachen */
    void startChangeListener() {
    }

    /** Stoppt die mit {@link #startChangeListener()} gestarteten Überwachungen */
    void stopChangeListener() {
    }
}
