/*
 * Copyright 2019-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.accessControl.internal;

import de.bsvrz.sys.funclib.debug.Debug;
import java.util.ArrayList;
import java.util.List;

/**
 * Klasse, die Basisberechtigungen enthält, die jeder Benutzer implizit zugewiesen bekommt. Es gibt keine Möglichkeit, einem Benutzer über die
 * Parametrierung diese Rechte zu nehmen. Bei Bedarf können weitere Berechtigungen hier hinzugefügt werden. Damit eine Aktion erlaubt ist, müssen
 * Attributgruppe und Aspekt übereinstimmen und die jeweilige Aktion muss über den Boolean-Parameter erlaubt sein. Die Angabe einer leeren Liste für
 * Attributgruppen oder Aspekte bedeutet wie üblich, dass alle Attributgruppen bzw. Aspekte ausgewählt sind.
 *
 * @author Kappich Systemberatung
 * @version $Revision: 0000 $
 */
final class ImplicitAccessUnitManager {

    private static final Debug _debug = Debug.getLogger();

    private static final List<AccessUnit> _accessUnits = new ArrayList<>();

    static {
        // Neue Berechtigungen können ab hier eingefügt werden.

        // Basis Anfragen (aus kb.objekteZugriffsrechteBasis)
        _accessUnits.add(new AccessUnit(
            // Attributgruppen
		        List.of("atg.konfigurationsBenutzerverwaltungsAnfrageSchnittstelle", "atg.konfigurationsAnfrageSchnittstelleSchreibend",
                             "atg.betriebsMeldung", "atg.konfigurationsSchreibAnfrage", "atg.konfigurationsAnfrage",
                             "atg.konfigurationsAnfrageSchnittstelleLesend", "atg.konfigurationsBereichsverwaltungsAnfrageSchnittstelle"),
            // Aspekte
		        List.of("asp.anfrage", "asp.information"), true,    // Sender
            false,   // Empfänger
            false,   // Quelle
            false    // Senke
        ));
        // Basis Antworten (aus kb.objekteZugriffsrechteBasis)
        _accessUnits.add(new AccessUnit(
            // Attributgruppen
		        List.of("atg.konfigurationsBenutzerverwaltungsAnfrageSchnittstelle", "atg.konfigurationsAnfrageSchnittstelleSchreibend",
                             "atg.konfigurationsAntwort", "atg.konfigurationsAnfrageSchnittstelleLesend",
                             "atg.konfigurationsBereichsverwaltungsAnfrageSchnittstelle", "atg.konfigurationsSchreibAntwort"),
            // Aspekte
		        List.of("asp.antwort"), false,   // Sender
            false,   // Empfänger
            false,   // Quelle
            true     // Senke
        ));
        // Basis Fertigmeldung (aus kb.objekteZugriffsrechteBasis)
        _accessUnits.add(new AccessUnit(
            // Attributgruppen
		        List.of("atg.applikationsFertigmeldung"),
            // Aspekte
		        List.of("asp.standard"), false,   // Sender
            true,    // Empfänger
            true,    // Quelle
            false    // Senke
        ));
        // Implizite Rechte zum Lesen der Berechtigungen von Benutzern
        _accessUnits.add(new AccessUnit(
            // Attributgruppen
		        List.of("atg.berechtigungsklassen", "atg.rollenRegionenPaare", "atg.rollenAktivität", "atg.region", "atg.benutzerParameter",
                             "atg.zugriffsRechte", "atg.rollenRegionenPaareParameter"),
            // Aspekte
		        List.of("asp.parameterSoll"), false,   // Sender
            true,    // Empfänger
            false,   // Quelle
            false    // Senke
        ));
        // Implizite Rechte zum Lesen der Rechteprüfungsart
        _accessUnits.add(new AccessUnit(
            // Attributgruppen
		        List.of("atg.datenverteilerRechteprüfung"),
            // Aspekte
		        List.of("asp.standard"), false,   // Sender
            true,    // Empfänger
            false,   // Quelle
            false    // Senke
        ));
    }

    public static boolean isAllowed(final String atg, final String asp, final UserAction action) {
        for (AccessUnit accessUnit : _accessUnits) {
            if (accessUnit.isAllowed(atg, asp, action)) {
                return true;
            }
        }
        return false;
    }

    private static final class AccessUnit {

        private final List<String> _attributeGroups;

        private final List<String> _aspects;

        private final boolean _allowSender;

        private final boolean _allowReceiver;

        private final boolean _allowSource;

        private final boolean _allowDrain;

        private AccessUnit(final List<String> attributeGroups, final List<String> aspects, final boolean allowSender, final boolean allowReceiver,
                           final boolean allowSource, final boolean allowDrain) {
            _attributeGroups = attributeGroups;
            _aspects = aspects;
            _allowSender = allowSender;
            _allowReceiver = allowReceiver;
            _allowSource = allowSource;
            _allowDrain = allowDrain;
        }

        public boolean isAllowed(final String atg, final String asp, final UserAction action) {
            if (!_attributeGroups.isEmpty() && !_attributeGroups.contains(atg)) {
                return false; // Keine Aussage, weil ATG nicht passt.
            }
            if (!_aspects.isEmpty() && !_aspects.contains(asp)) {
                return false; // Keine Aussage, weil ASP nicht passt.
            }
            return switch (action) {
                case RECEIVER -> _allowReceiver;
                case SENDER -> _allowSender;
                case SOURCE -> _allowSource;
                case DRAIN -> _allowDrain;
                default -> {
                    _debug.error("Unbekannte Aktion: " + action);
                    yield false;
                }
            };
        }
    }
}
