/*
 * Copyright 2016-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.communication.dataRepresentation.data.info.version1;

import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import java.util.HashMap;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.function.Function;

/**
 * Map-Ähnliche Klasse, die Systemobjekte als Keys pro Datenmodell verwaltet und so die Objekte eines einzelnen Datenmodells schnell löschen kann.
 * <p>
 * Die Implementierung ist aktuell nicht threadsafe, es muss also extern synchronisiert werden.
 *
 * @author Kappich Systemberatung
 */
public class SystemObjectCache<K extends SystemObject, V> {

    private final Map<DataModel, Map<K, V>> _dataModels = new WeakHashMap<>();

    public int size() {
        return _dataModels.values().stream().mapToInt(Map::size).sum();
    }

    public boolean isEmpty() {
        return _dataModels.values().stream().allMatch(Map::isEmpty);
    }

    private Map<K, V> getDataModelMap(final Object key) {
        SystemObject systemObject = (SystemObject) key;
        DataModel dataModel = systemObject.getDataModel();

	    Map<K, V> dataModelMap = _dataModels.computeIfAbsent(dataModel, k -> new HashMap<>());
        return dataModelMap;
    }

    public boolean containsKey(final Object key) {
        return getDataModelMap(key).containsKey(key);
    }

    public boolean containsValue(final Object value) {
        return _dataModels.values().stream().allMatch(map -> map.containsValue(value));
    }

    /**
     * Synchronisierter Zugriff auf das dem angegebenen Schlüssel zugeordnete Objekt. Wenn das gesuchte Objekt nicht enthalten ist, dann wird es mit
     * der angegebenen `mappingFunction` erzeugt und in die Datenstruktur eingetragen. Dabei findet das Erzeugen unsynchronisiert statt. Dadurch kann
     * es passieren, dass das Erzeugen durch zwei Threads gleichzeitig stattfindet. Von den beiden erzeugten Objekten wird nur genau eines in die
     * Datenstruktur eingetragen. Beide Threads liefern dann das selbe Objekt zurück.
     *
     * @param key             Schlüssel des gesuchten Objekts.
     * @param mappingFunction Funktion, die verwendet wird, um ein neues Objekt zu erzeugen, falls noch kein Objekt unter dem angegeben Schlüssel
     *                        gefunden wurde.
     *
     * @return Das dem Key zugeordnete Objekt.
     */
    public V computeIfAbsent(K key, Function<? super K, ? extends V> mappingFunction) {
        V v;
        // Wenn schon ein Objekt zugeordnet, dann dieses zurückliefern.
        synchronized (this) {
            v = get(key);
            if (v != null) {
                return v;
            }
        }
        // Unsynchronisiertes Erzeugen des Objekts. (Kann unter Umständen mehrfach ausgeführt werden).
        v = mappingFunction.apply(key);
        // Eintragen des Erzeugten Objekts in die Datenstruktur, falls immer noch kein Eintrag enthalten ist.
        synchronized (this) {
            V v2 = get(key);
            if (v2 != null) {
                return v2;
            }
            put(key, v);
            return v;
        }
    }

    public V get(final Object key) {
        return getDataModelMap(key).get(key);
    }

    public V put(final K key, final V value) {
        return getDataModelMap(key).put(key, value);
    }

    public V remove(final Object key) {
        return getDataModelMap(key).remove(key);
    }

    public void putAll(final Map<? extends K, ? extends V> m) {
        m.forEach(this::put);
    }

    public void clear() {
        _dataModels.clear();
    }

    /**
     * Löscht alle Keys, die das angegeben Datenmodell verwenden
     *
     * @param dataModel Datenmodell
     */
    public void forgetDataModel(final DataModel dataModel) {
        _dataModels.remove(dataModel);
    }
}
