/*
 * Copyright 2004 by Kappich+Kniß Systemberatung, Aachen
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.communication.dataRepresentation.datavalue;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.Arrays;

/**
 * Diese Klasse stellt die Attribute und Funktionalitäten des Datentyps ListArray zur Verfügung.
 *
 * @author Kappich Systemberatung
 */
public class AttributeListArrayAttribute extends DataValue {

    private AttributeListAttribute[] _values;

    private DataValue[] _attributeListValues;

    /** Erzeugt ein neues Objekt ohne Parameter. Die Parameter werden zu einem Späteren Zeitpunkt über die read-Methode eingelesen. */
    public AttributeListArrayAttribute() {
        _type = ATTRIBUTE_LIST_ARRAY_TYPE;
    }

    /**
     * Erzeugt ein neues Objekt mit den gegebenen Parametern.
     *
     * @param values feld mit Werten
     */
    public AttributeListArrayAttribute(AttributeListAttribute[] values) {
        _type = ATTRIBUTE_LIST_ARRAY_TYPE;
        _values = values;
    }

    /**
     * Gibt den Wert zurrück.
     *
     * @return der Wert
     */
    @Override
    public final Object getValue() {
        return _values;
    }

    public final void setValue(DataValue[] values) {
        _attributeListValues = values;
    }

    /**
     * Gibt die Anzahl der Attributlisten in diesem Array zurück.
     *
     * @return Anzahl der Attributlisten
     */
    public final int getAttributeListsLength() {
        return _values == null ? 0 : _values.length;
    }

    @Override
    public final DataValue cloneObject() {
        AttributeListAttribute[] _values = null;
        if (this._values != null) {
            _values = new AttributeListAttribute[this._values.length];
            for (int i = 0; i < this._values.length; ++i) {
                if (this._values[i] != null) {
                    _values[i] = (AttributeListAttribute) this._values[i].cloneObject();
                }
            }
        }
        AttributeListArrayAttribute clone = new AttributeListArrayAttribute(_values);
        clone.setValue(cloneAttributeListValues());
        return clone;
    }

    @Override
    public final String parseToString() {
        StringBuilder str = new StringBuilder("Attributliste Array: [\n");
        if (_values != null) {
            for (final AttributeListAttribute _value : _values) {
                if (_value != null) {
                    str.append(_value.parseToString());
                }
            }
        }
        str.append("]\n");
        return str.toString();
    }

    @Override
    public final void write(DataOutputStream out) throws IOException {
        if (_values == null) {
            out.writeInt(0);
        } else {
            out.writeInt(_values.length);
            for (final AttributeListAttribute _value : _values) {
                if (_value != null) {
                    _value.write(out);
                }
            }
        }
    }

    @Override
    public final void read(DataInputStream in) throws IOException {
        int length = in.readInt();
        if (length >= 0) {
            _values = new AttributeListAttribute[length];
            for (int i = 0; i < length; ++i) {
                AttributeListAttribute v = (AttributeListAttribute) DataValue.getObject(ATTRIBUTE_LIST_TYPE);
                if (v != null) {
                    v.setValue(cloneAttributeListValues());
                    v.read(in);
                    _values[i] = v;
                }
            }
        }
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        final AttributeListArrayAttribute that = (AttributeListArrayAttribute) o;

        return Arrays.equals(_values, that._values);
    }

    @Override
    public int hashCode() {
        return Arrays.hashCode(_values);
    }

    /**
     * Kopiert die Werte in der Attributsliste und gibt die Kopie zurück.
     *
     * @return Kopie der Werte in der Attributsliste
     */
    private DataValue[] cloneAttributeListValues() {
        if (_attributeListValues == null) {
            return null;
        }
        DataValue[] clone = new DataValue[_attributeListValues.length];
        for (int i = 0; i < _attributeListValues.length; ++i) {
            if (_attributeListValues[i] != null) {
                clone[i] = _attributeListValues[i].cloneObject();
            }
        }
        return clone;
    }
}
