/*
 * Copyright 2004 by Kappich+Kniß Systemberatung, Aachen
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.dav.daf.communication.lowLevel;

import de.bsvrz.dav.daf.communication.lowLevel.telegrams.ApplicationDataTelegram;
import de.bsvrz.dav.daf.communication.lowLevel.telegrams.BaseSubscriptionInfo;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Diese Klasse stellt einen Mechanismus zur Verfügung, der zerstückelte Datensätze zu vollständigen Datensätzen zusammenbaut und dann zur Verfügung
 * stellt.
 *
 * @author Kappich Systemberatung
 */
public class SplittedApplicationTelegramsTable {

    /** Die Tabelle wo je nach Datum eine Liste der zerstückelten Telegramme gehalten wird. */
    private final ConcurrentHashMap<BaseSubscriptionInfo, ConcurrentHashMap<Long, ApplicationDataTelegram[]>> dataTable;

    /** Erzeugt ein Objekt dieser Klasse. */
    public SplittedApplicationTelegramsTable() {
        dataTable = new ConcurrentHashMap<>();
    }

    /**
     * Diese Methode sammelt alle Teiltelegramme. Wurden alle Teiltelegramme empfangen, werden alle Teiltelegramme zurückgegeben.
     *
     * @param telegram Teiltelegramm, das ein Telegramm vervollständigen soll oder ein komplettes Telegramm, das als ganzes übergeben wurde und somit
     *                 nicht zusammengebaut werden muss.
     *
     * @return Alle Teiltelegramme, aus denen ein vollständiges Telegramm rekonstruiert werden kann (und damit ein Datenatz) oder aber {@code null}.
     *     {@code null} bedeutet, dass noch nicht alle Teiltelegramme empfangen wurden, die nötig sind um das gesamte Telegramm zusammen zu bauen.
     *
     * @throws IllegalArgumentException Das übergebene Telegramm konnte keinem bisher empfangenen Teil zugeordnet werden oder war {@code null}.
     */
    public final ApplicationDataTelegram[] put(ApplicationDataTelegram telegram) {
        if (telegram == null) {
            throw new IllegalArgumentException("Der Parameter ist null");
        }
        int totalTelegramCount = telegram.getTotalTelegramsCount();
        int index = telegram.getTelegramNumber();
        if (index >= totalTelegramCount) {
            throw new IllegalArgumentException(
                "Der Telegramm-Index ist grösser als die maximale Anzahl der zerstückelten Telegramme dieses Datensatzes");
        }
        if ((index == 0) && (totalTelegramCount == 1)) {
            return new ApplicationDataTelegram[] {telegram};
        }
        BaseSubscriptionInfo key = telegram.getBaseSubscriptionInfo();
        if (key == null) {
            throw new IllegalArgumentException("Das Telegramm ist inkonsistent");
        }
        ConcurrentHashMap<Long, ApplicationDataTelegram[]> table = dataTable.get(key);
        Long subKey = telegram.getDataNumber();
        if (table == null) {
            table = new ConcurrentHashMap<>();
            ApplicationDataTelegram[] list = new ApplicationDataTelegram[totalTelegramCount];
            list[index] = telegram;
            table.put(subKey, list);
            dataTable.put(key, table);
            return null;
        } else {
            ApplicationDataTelegram[] list = table.get(subKey);
            if (list == null) {
                list = new ApplicationDataTelegram[totalTelegramCount];
                list[index] = telegram;
                table.put(subKey, list);
                return null;
            } else {
                synchronized (list) {
                    list[index] = telegram;
                    for (int i = 0; i < list.length; ++i) {
                        ApplicationDataTelegram tmpTelegram = list[i];
                        if (tmpTelegram == null) {
                            return null;
                        }
                        if (i == tmpTelegram.getTelegramNumber()) {
                            continue;
                        } else {
                            throw new IllegalArgumentException("Falsche Daten in der Cache-Tabelle der zerstückelten Telegramme");
                        }
                    }
                }
                table.remove(subKey);
                if (table.isEmpty()) {
                    dataTable.remove(key);
                }
                return list;
            }
        }
    }
}
