/*
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.communication.lowLevel;

import java.util.Collection;
import java.util.LinkedList;

/**
 * Klasse, die zum gepufferten Austausch von Telegrammen zwischen verschiedenen Threads verwendet werden kann. Die Gesamtgröße der gepufferten
 * Telegramme ist beschränkt. Es werden verschiedene Telegrammprioritäten unterstützt.
 * <p>
 * Telegramme können mit der Methode {@link #put} gespeichert werden und mit der Methode {@link #take} wieder ausgelesen werden. Die Methoden
 * blockieren, wenn beim Speichern nicht genügend Platz vorhanden ist, bzw., wenn beim Auslesen kein Telegramm mehr zur Verfügung steht. Der Methode
 * {@link #close} dient zum Schließen der Queue. blockiert keine der beiden Methoden mehr.
 *
 * @author Kappich Systemberatung
 */
public class TelegramQueue<Telegram extends QueueableTelegram> {

    /** Maximale Gesamtgröße für zwischengespeicherte Telegramme. */
    private final int _capacity;
    /**
     * Array, das je mögliche Priorität eine verkettete Liste mit den zwischengespeicherten Telegrammen enthält. Es dient außerdem der Synchronisation
     * von Threads beim lesenden und schreibenden Zugriff.
     */
    private final LinkedList<Telegram>[] _priorityLists;
    /** Gesamtgröße der aktuell zwischengespeicherten Telegramme. */
    private int _size;
    private boolean _closed;

    /**
     * Erzeugt eine neue Queue mit den angegebenen Eigenschaften.
     *
     * @param capacity        Maximale Gesamtgröße der gepufferten Telegramme.
     * @param maximumPriority Maximale von Telegrammen verwendete Priorität.
     */
    public TelegramQueue(int capacity, int maximumPriority) {
        if (capacity <= 0) {
            throw new IllegalArgumentException("capacity muss positiv sein: " + capacity);
        }
        if (maximumPriority < 0) {
            throw new IllegalArgumentException("maximumPriority darf nicht negativ sein: " + maximumPriority);
        }
        if (maximumPriority > 127) {
            throw new IllegalArgumentException("maximumPriority darf nicht größer als 127 sein: " + maximumPriority);
        }
        _capacity = capacity;
        _size = 0;
        _priorityLists = (LinkedList<Telegram>[]) new LinkedList[maximumPriority + 1]; // Compiler-Warnung nicht vermeidbar
        for (int i = 0; i < _priorityLists.length; i++) {
            _priorityLists[i] = new LinkedList<>();
        }
    }

    /**
     * Gibt das älteste in der Queue gespeicherte Telegramm mit der höchsten Priorität zurück. Wenn die Queue noch nicht geschlossen wurde, wartet
     * diese Methode, bis ein Telegramm in der Queue zur Verfügung steht.
     *
     * @return Nächstes gespeicherte Telegramm mit der höchsten Priorität. Wenn die Queue geschlossen wurde und kein gespeichertes Telegramm mehr
     *     verfügbar ist wird {@code null} zurückgegeben.
     *
     * @throws InterruptedException Wenn der Thread während des Wartens unterbrochen wurde.
     */
    public Telegram take() throws InterruptedException {
        synchronized (this) {
            while (_size == 0) {
                // Wenn die Queue leer ist und geschlossen wurde, wird null zurückgegeben
                if (_closed) {
                    return null;
                }
                // Wenn die Queue leer ist und nicht geschlossen wurde, wird gewartet
                wait();
            }
            for (int i = _priorityLists.length - 1; i >= 0; i--) {
                LinkedList<Telegram> priorityList = _priorityLists[i];
                if (!priorityList.isEmpty()) {
                    final Telegram telegram = priorityList.removeFirst();
                    _size -= telegram.getSize();
                    notifyAll();
                    return telegram;
                }
            }
        }
        throw new IllegalStateException("Interner Fehler: Es wurde kein Telegramm gefunden, obwohl die Gesamtgröße " + _size + " ist");
    }

    /**
     * Gibt die ältesten in der Queue gespeicherten Telegramme zurück, so lange bis die summierte Telegrammlänge das angegebene Limit überschreitet
     * oder die Queue leer ist. Das erste Telegramm, was dafür sorgt, dass das Limit überschritten wird, wird mit zurückgegeben. Das bedeutet, dass
     * auch bei einem Limit von 0 immer (genau) ein Telegramm zurückgegeben wird.
     *
     * @param sizeLimit Anzahl Bytes, die diese Methode versucht mindestens zurückzugeben (solange in der Queue genug Daten da sind)
     * @param result    Liste in der die Telegramme zurückgegeben werden. Wird übergeben, damit nicht ständig eine neue Liste erstellt werden muss.
     *                  Die Liste wird beim Aufruf der Methode geleert.
     *
     * @return Anzahl Bytes der zurückgegebenen Telegramme oder -1 wenn die Queue geschlossen wurde
     *
     * @throws InterruptedException
     */
    public int takeMultiple(int sizeLimit, final Collection<Telegram> result) throws InterruptedException {
        result.clear();
        int aggregatedSize = 0;
        synchronized (this) {
            while (_size == 0) {
                // Wenn die Queue leer ist und geschlossen wurde, wird -1 zurückgegeben
                if (_closed) {
                    return -1;
                }
                // Wenn die Queue leer ist und nicht geschlossen wurde, wird gewartet
                wait();
            }
            for (int i = _priorityLists.length - 1; i >= 0; i--) {
                LinkedList<Telegram> priorityList = _priorityLists[i];
                while (!priorityList.isEmpty()) {
                    final Telegram telegram = priorityList.removeFirst();
                    _size -= telegram.getSize();
                    aggregatedSize += telegram.getSize();
                    notifyAll();
                    result.add(telegram);
                    if (aggregatedSize > sizeLimit) {
                        return aggregatedSize;
                    }
                }
            }
        }
        if (result.isEmpty()) {
            throw new IllegalStateException("Interner Fehler: Es wurde kein Telegramm gefunden, obwohl die Gesamtgröße " + _size + " ist");
        }
        return aggregatedSize;
    }

    /**
     * Speichert das angegebene Telegramm in der Queue. Bei Bedarf wartet diese Methode bis genügend Platz in der Queue für das zu speichernde
     * Telegramm zur Verfügung steht.
     *
     * @param telegram Das zu speichernde Telegramm
     *
     * @throws InterruptedException Wenn der Thread während des Wartens unterbrochen wurde.
     */
    public void put(Telegram telegram) throws InterruptedException {
        if (_closed) {
            return;
        }
        final int length = telegram.getSize();
        if (length <= 0) {
            throw new IllegalArgumentException("Telegrammlänge muss größer 0 sein, ist aber " + length + ": " + telegram);
        }
        final byte priority = telegram.getPriority();
        synchronized (this) {
            if (length > _capacity) {
                // Telegramm passt nicht in Queue, solange warten bis _size == 0 und dann senden
                while (!_closed && _size > 0) {
                    wait();
                }
            } else {
                while (!_closed && _size + length > _capacity) {
                    wait();
                }
            }
            if (_closed) {
                return;
            }
            _priorityLists[priority].add(telegram);
            _size += length;
            notifyAll();
        }
    }

    /**
     * Bestimmt die maximale Gesamtgröße für zwischengespeicherte Telegramme.
     *
     * @return Maximale Gesamtgröße für zwischengespeicherte Telegramme.
     */
    public int getCapacity() {
        return _capacity;
    }

    /**
     * Bestimmt die Gesamtgröße der aktuell zwischengespeicherten Telegramme.
     *
     * @return Gesamtgröße der aktuell zwischengespeicherten Telegramme.
     */
    public int getSize() {
        synchronized (this) {
            return _size;
        }
    }

    /**
     * Diese Methode schließt die Verbindung. Danach ignoriert die Methode {@link #put} sämtliche weitere zu speichernde Telegramme und die Methode
     * {@link #take} liefert noch alle bisher gespeicherten Telegramme und danach {@code null} zurück. Eventuell blockierte Threads werden geweckt.
     */
    public void close() {
        synchronized (this) {
            _closed = true;
            notifyAll();
        }
    }

    /**
     * Diese Methode schließt die Verbindung und löscht alle noch gespeicherten Telegramme. Danach ignoriert die Methode {@link #put} sämtliche
     * weitere zu speichernde Telegramme und die Methode {@link #take} liefert anschließend immer {@code null} zurück. Eventuell blockierte Threads
     * werden geweckt.
     */
    public void abort() {
        synchronized (this) {
            _closed = true;
            notifyAll();
            try {
                while (take() != null) {
                    ;
                }
            } catch (InterruptedException ignored) {
            }
        }
    }
}
