/*
 * Copyright 2018-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.communication.lowLevel.telegrams;

import de.bsvrz.dav.daf.communication.lowLevel.TelegramUtility;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

/**
 * Erweiterung des {@link TransmitterDataTelegram}, bei dem zusätzlich die ID der Sender-Applikation übertragen wird.
 * <p>
 * Diese Information ist in einigen Fällen erforderlich, um zerstückelte Telegramme von mehreren Sendern wieder zusammensetzen zu können.
 * <p>
 * Außerdem kan diese Information bei der Problemdiagnose helfen.
 *
 * @author Kappich Systemberatung
 */
public class TransmitterDataTelegramV2 extends TransmitterDataTelegram {

    /**
     * ID der sendenden Applikation
     */
    private long _applicationId;

    public TransmitterDataTelegramV2() {
        type = TRANSMITTER_DATA_TELEGRAM_V2_TYPE;
    }

    public TransmitterDataTelegramV2(final ApplicationDataTelegram applicationDataTelegram, final byte direction, final long applicationId) {
        super(applicationDataTelegram, direction);
        _applicationId = applicationId;
        type = TRANSMITTER_DATA_TELEGRAM_V2_TYPE;
        length += 8;
    }

    public TransmitterDataTelegramV2(final BaseSubscriptionInfo baseSubscriptionInfo, final long dataNumber, final boolean delayedDataFlag,
                                     final byte errorFlag, final byte[] attributesIndicator, final byte[] data, final int totalTelegramCount,
                                     final int telegramNumber, final long dataTime, final byte direction, final long applicationId) {
        super(baseSubscriptionInfo, dataNumber, delayedDataFlag, errorFlag, attributesIndicator, data, totalTelegramCount, telegramNumber, dataTime,
              direction);
        _applicationId = applicationId;
        type = TRANSMITTER_DATA_TELEGRAM_V2_TYPE;
        length += 8;
    }

    public TransmitterDataTelegramV2(final BaseSubscriptionInfo baseSubscriptionInfo, final long dataNumber, final boolean delayedDataFlag,
                                     final byte errorFlag, final byte[] attributesIndicator, final byte prior, final byte[] data,
                                     final int totalTelegramCount, final int telegramNumber, final long dataTime, final byte direction,
                                     final long applicationId) {
        super(baseSubscriptionInfo, dataNumber, delayedDataFlag, errorFlag, attributesIndicator, prior, data, totalTelegramCount, telegramNumber,
              dataTime, direction);
        _applicationId = applicationId;
        type = TRANSMITTER_DATA_TELEGRAM_V2_TYPE;
        length += 8;
    }

    public void write(DataOutputStream out) throws IOException {
        out.writeShort(length);
        out.writeShort(_telegramNumber);
        out.writeShort(_totalTelegramCount);
        _baseSubscriptionInfo.write(out);
        out.writeByte(_direction);
        out.writeLong(_dataNumber);
        out.writeBoolean(_delayedDataFlag);
        if (_telegramNumber == 0) {
            out.writeLong(_dataTime);
            out.writeByte(_errorFlag);
            if (_attributesIndicator == null) {
                out.writeByte(0);
            } else {
                out.writeByte(_attributesIndicator.length);
                out.write(_attributesIndicator);
            }
        }
        if (_data == null) {
            out.writeInt(0);
        } else {
            out.writeInt(_data.length);
            out.write(_data);
        }
        out.writeLong(_applicationId);
    }

    public void read(DataInputStream in) throws IOException {
        int _length = in.readShort();
        length = 40;
        if (_length < length) {
            throw new IOException("Falsche Telegrammlänge (zu kurz)");
        }
        _telegramNumber = in.readShort();
        _totalTelegramCount = in.readShort();
        _baseSubscriptionInfo = new BaseSubscriptionInfo();
        _baseSubscriptionInfo.read(in);
        _direction = in.readByte();
        _dataNumber = in.readLong();
        _delayedDataFlag = in.readBoolean();
        int size;
        if (_telegramNumber == 0) {
            _dataTime = in.readLong();
            _errorFlag = in.readByte();
            size = in.readUnsignedByte();
            length += 10;
            length += size;
            if (_length < length) {
                throw new IOException("Falsche Telegrammlänge (Indikatorbitfeld passt nicht ins Anwendungsdatentelegramm)");
            }
            if (size > 0) {
                _attributesIndicator = new byte[size];
                in.readFully(_attributesIndicator);
            }
        }
        size = in.readInt();
        length += size;
        if (_length != length) {
            throw new IOException("Falsche Telegrammlänge (Länge " + _length + " passt nicht zu size " + size + ")");
        }
        if (size > 0) {
            _data = new byte[size];
            in.readFully(_data);
        }
        _applicationId = in.readLong();
        priority = TelegramUtility.getPriority(this);
        checkConsistency();
    }

    private void checkConsistency() {
        if (_data == null && _errorFlag == 0) {
            String message = "Inkonsistentes Datenverteilertelegramm: " + _telegramNumber + "/" + _totalTelegramCount + ", " + " dataNumber: " +
                             (_dataNumber >>> 32) + "#" + ((_dataNumber & 0xffffffffL) >> 2) + "#" + (_dataNumber & 3) + ", " +
                             _baseSubscriptionInfo.toString() + ", errorFlag: " + _errorFlag + ", data: " +
                             (_data == null ? "null" : String.valueOf(_data.length) + " Bytes") + ", richtung: " + _direction;
            _debug.warning(message, new Exception("Inkonsistentes Datenverteilertelegramm"));
        }
    }

    /**
     * Gibt die ID der ursprünglich sendenden Applikation zurück
     *
     * @return Applikations-ID
     */
    public long getApplicationId() {
        return _applicationId;
    }
}
