/*
 * Copyright 2005 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.main.archive;

import java.util.AbstractSet;
import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * Diese Klasse ermöglicht es, eine Kombination von mehreren {@link ArchiveDataKind} zu erzeugen.
 * <p>
 * Seit Version 3.9.7 implementiert diese Klasse {@code Set<ArchiveDataKind>}.
 * <p>
 * Objekte dieser Klasse sind unveränderlich (immutable)
 *
 * @author Kappich Systemberatung
 */
public class ArchiveDataKindCombination extends AbstractSet<ArchiveDataKind> {

    // vordefinierte Konstanten zum schnellen Zugriff
    private static final ArchiveDataKindCombination ALL =
        new ArchiveDataKindCombination(ArchiveDataKind.ONLINE, ArchiveDataKind.ONLINE_DELAYED, ArchiveDataKind.REQUESTED,
                                       ArchiveDataKind.REQUESTED_DELAYED);
    private static final ArchiveDataKindCombination ONLINE = new ArchiveDataKindCombination(ArchiveDataKind.ONLINE);
    private static final ArchiveDataKindCombination ONLINE_DELAYED = new ArchiveDataKindCombination(ArchiveDataKind.ONLINE_DELAYED);
    private static final ArchiveDataKindCombination REQUESTED = new ArchiveDataKindCombination(ArchiveDataKind.REQUESTED);
    private static final ArchiveDataKindCombination REQUESTED_DELAYED = new ArchiveDataKindCombination(ArchiveDataKind.REQUESTED_DELAYED);

    /**
     * Gesetzte Bits (wie in EnumSet)
     */
    private final int _bits;

    /**
     * Copy-Konstruktor
     *
     * @param values Archivdatenarten, die gespeichert werden sllen.
     */
    public ArchiveDataKindCombination(Iterable<ArchiveDataKind> values) {
        int bits = 0;
        for (ArchiveDataKind adk : values) {
            if (adk == ArchiveDataKind.ONLINE) {
                bits |= 1;
            } else if (adk == ArchiveDataKind.ONLINE_DELAYED) {
                bits |= 2;
            } else if (adk == ArchiveDataKind.REQUESTED) {
                bits |= 4;
            } else if (adk == ArchiveDataKind.REQUESTED_DELAYED) {
                bits |= 8;
            }
        }
        _bits = bits;
    }

    /**
     * Dieser Konstruktor kann eine Kombination von vier ArchiveDataKind Objekten verarbeiten und ein entsprechendes Objekt zur Verfügung stellen.
     *
     * @param dataKind1 ArchiveDataKind
     * @param dataKind2 ArchiveDataKind
     * @param dataKind3 ArchiveDataKind
     * @param dataKind4 ArchiveDataKind
     */
    public ArchiveDataKindCombination(ArchiveDataKind dataKind1, ArchiveDataKind dataKind2, ArchiveDataKind dataKind3, ArchiveDataKind dataKind4) {
        boolean online = dataKind1 == ArchiveDataKind.ONLINE || dataKind2 == ArchiveDataKind.ONLINE || dataKind3 == ArchiveDataKind.ONLINE ||
                         dataKind4 == ArchiveDataKind.ONLINE;
        boolean onlineDelayed = dataKind1 == ArchiveDataKind.ONLINE_DELAYED || dataKind2 == ArchiveDataKind.ONLINE_DELAYED ||
                                dataKind3 == ArchiveDataKind.ONLINE_DELAYED || dataKind4 == ArchiveDataKind.ONLINE_DELAYED;
        boolean requested =
            dataKind1 == ArchiveDataKind.REQUESTED || dataKind2 == ArchiveDataKind.REQUESTED || dataKind3 == ArchiveDataKind.REQUESTED ||
            dataKind4 == ArchiveDataKind.REQUESTED;
        boolean requestedDelayed = dataKind1 == ArchiveDataKind.REQUESTED_DELAYED || dataKind2 == ArchiveDataKind.REQUESTED_DELAYED ||
                                   dataKind3 == ArchiveDataKind.REQUESTED_DELAYED || dataKind4 == ArchiveDataKind.REQUESTED_DELAYED;
        int bits = 0;
        if (online) {
            bits |= 1;
        }
        if (onlineDelayed) {
            bits |= 2;
        }
        if (requested) {
            bits |= 4;
        }
        if (requestedDelayed) {
            bits |= 8;
        }
        _bits = bits;
    }

    /**
     * Siehe Konstruktor mit 4 Eingabeparametern, dieser ist identisch nur mit 3 Objekten.
     *
     * @param dataKind1 ArchiveDataKind
     * @param dataKind2 ArchiveDataKind
     * @param dataKind3 ArchiveDataKind
     */
    public ArchiveDataKindCombination(ArchiveDataKind dataKind1, ArchiveDataKind dataKind2, ArchiveDataKind dataKind3) {
        this(dataKind1, dataKind2, dataKind3, dataKind3);
    }

    /**
     * Siehe Konstruktor mit 4 Eingabeparametern, dieser ist identisch nur mit 2 Objekten.
     *
     * @param dataKind1 ArchiveDataKind
     * @param dataKind2 ArchiveDataKind
     */
    public ArchiveDataKindCombination(ArchiveDataKind dataKind1, ArchiveDataKind dataKind2) {
        this(dataKind1, dataKind2, dataKind2, dataKind2);
    }

    /**
     * Siehe Konstruktor mit 4 Eingabeparametern, dieser ist identisch nur mit einem Objekt.
     *
     * @param dataKind1 ArchiveDataKind
     */
    public ArchiveDataKindCombination(ArchiveDataKind dataKind1) {
        this(dataKind1, dataKind1, dataKind1, dataKind1);
    }

    /**
     * Gibt eine ArchiveDataKindCombination zurück, die alle Datenarten enthält.
     *
     * @return ALL
     *
     * @since 3.9.7
     */
    public static ArchiveDataKindCombination all() {
        return ALL;
    }

    /**
     * Gibt eine ArchiveDataKindCombination zurück, die nur {@link ArchiveDataKind#ONLINE} enthält.
     *
     * @return ONLINE
     *
     * @since 3.9.7
     */
    public static ArchiveDataKindCombination online() {
        return ONLINE;
    }

    /**
     * Gibt eine ArchiveDataKindCombination zurück, die nur {@link ArchiveDataKind#ONLINE_DELAYED} enthält.
     *
     * @return ONLINE_DELAYED
     *
     * @since 3.9.7
     */
    public static ArchiveDataKindCombination onlineDelayed() {
        return ONLINE_DELAYED;
    }

    /**
     * Gibt eine ArchiveDataKindCombination zurück, die nur {@link ArchiveDataKind#REQUESTED} enthält.
     *
     * @return REQUESTED
     *
     * @since 3.9.7
     */
    public static ArchiveDataKindCombination requested() {
        return REQUESTED;
    }

    /**
     * Gibt eine ArchiveDataKindCombination zurück, die nur {@link ArchiveDataKind#REQUESTED_DELAYED} enthält.
     *
     * @return REQUESTED_DELAYED
     *
     * @since 3.9.7
     */
    public static ArchiveDataKindCombination requestedDelayed() {
        return REQUESTED_DELAYED;
    }

    /**
     * Diese Methode gibt an, ob {@code ArchiveDataKind.ONLINE} im Konstruktor übergeben wurde
     *
     * @return true = {@code ArchiveDataKind.ONLINE} wurde gewählt;
     */
    public boolean isOnline() {
        return (_bits & 1) != 0;
    }

    /**
     * Diese Methode gibt an, ob {@code ArchiveDataKind.ONLINE_DELAYED} im Konstruktor übergeben wurde
     *
     * @return true = {@code ArchiveDataKind.ONLINE_DELAYED} wurde gewählt;
     */
    public boolean isOnlineDelayed() {
        return (_bits & 2) != 0;
    }

    /**
     * Diese Methode gibt an, ob {@code ArchiveDataKind.REQUESTED} im Konstruktor übergeben wurde
     *
     * @return true = {@code ArchiveDataKind.REQUESTED} wurde gewählt;
     */
    public boolean isRequested() {
        return (_bits & 4) != 0;
    }

    /**
     * Diese Methode gibt an, ob {@code ArchiveDataKind.REQUESTED_DELAYED} im Konstruktor übergeben wurde
     *
     * @return true = {@code ArchiveDataKind.REQUESTED_DELAYED} wurde gewählt;
     */
    public boolean isRequestedDelayed() {
        return (_bits & 8) != 0;
    }

    @Override
    public Iterator<ArchiveDataKind> iterator() {
        return new ADKIterator();
    }

    @Override
    public int size() {
        return Integer.bitCount(_bits);
    }

    @Override
    public boolean contains(final Object o) {
	    if (o instanceof ArchiveDataKind kind) {
            return (_bits & 1 << (kind.getCode() - 1)) != 0;
        }
        return false;
    }

    private class ADKIterator implements Iterator<ArchiveDataKind> {

        int _remaining;

        ADKIterator() {
            _remaining = _bits;
        }

        @Override
        public boolean hasNext() {
            return _remaining != 0;
        }

        @Override
        public ArchiveDataKind next() {
            if (_remaining == 0) {
                throw new NoSuchElementException();
            }
            int index = _remaining & -_remaining;
            _remaining -= index;
	        return switch (index) {
		        case 1 -> ArchiveDataKind.ONLINE;
		        case 2 -> ArchiveDataKind.ONLINE_DELAYED;
		        case 4 -> ArchiveDataKind.REQUESTED;
		        case 8 -> ArchiveDataKind.REQUESTED_DELAYED;
		        default -> throw new AssertionError();
	        };
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }
}
