/*
 * Copyright 2005 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2006-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.main.config;

import de.bsvrz.dav.daf.communication.dataRepresentation.AttributeBaseValueDataFactory;
import de.bsvrz.dav.daf.communication.dataRepresentation.AttributeHelper;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.ResultData;
import java.util.Collection;

/**
 * Schnittstelle zum Zugriff auf Attribute und Eigenschaften einer Attributgruppe.
 *
 * @author Roland Schmitz (rs), Kappich Systemberatung
 * @author Stephan Homeyer (sth), Kappich Systemberatung
 * @version $Revision$ / $Date$ / ($Author$)
 */
public interface AttributeGroup extends AttributeSet {

    /**
     * Ermittelt, ob diese Attributgruppe mit dem Aspekt "asp.eigenschaften" für konfigurierende Datensätze benutzt werden kann.
     *
     * @return {@code true}, wenn die Attributgruppe mit dem Aspekt "asp.eigenschaften" für konfigurierende Datensätze benutzt werden kann; sonst
     *     {@code false}.
     *
     * @deprecated Wird durch die Methode {@link AttributeGroupUsage#isConfigurating} ersetzt.
     */
    @Deprecated
    boolean isConfigurating();

    /**
     * Gibt an, ob diese Attributgruppe als Parameter verwendet werden kann oder nicht.
     *
     * @return {@code true}, wenn die Attributgruppe als Parameter verwendet werden kann;<br> {@code false}, wenn die Attributgruppe nicht als
     *     Parameter verwendet werden kann.
     */
    boolean isParameter();

    /**
     * Bestimmt die möglichen Aspekte, unter denen die Attributgruppe verwendet werden kann.
     *
     * @return Liste von {@link Aspect Aspekten}
     */
    Collection<Aspect> getAspects();

    /**
     * Liefert alle Attributgruppenverwendungen dieser Attributgruppe zurück.
     *
     * @return alle Attributgruppenverwendungen dieser Attributgruppe
     */
    Collection<AttributeGroupUsage> getAttributeGroupUsages();

    /**
     * Gibt die Attributgruppenverwendung des angegebenen Aspekts zurück.
     *
     * @param aspect Aspekt der gewünschten Attributgruppenverwendung
     *
     * @return Attributgruppenverwendung für den angegebenen Aspekt oder {@code null}, falls zum angegebenen Aspekt keine Attributgruppenverwendung
     *     definiert ist.
     */
    AttributeGroupUsage getAttributeGroupUsage(Aspect aspect);

    /**
     * Gibt die Attributgruppenverwendung des angegebenen Aspekts zurück.
     *
     * @param aspectPid Aspekt der gewünschten Attributgruppenverwendung
     *
     * @return Attributgruppenverwendung für den angegebenen Aspekt oder {@code null}, falls zum angegebenen Aspekt keine Attributgruppenverwendung
     *     definiert ist.
     *
     * @since 3.14
     */
    default AttributeGroupUsage getAttributeGroupUsage(String aspectPid) {
        Aspect aspect = getDataModel().getAspect(aspectPid);
        if (aspect == null) {
            return null;
        }
        return getAttributeGroupUsage(aspect);
    }

    /**
     * Erzeugt ein neues {@link Data}-Objekt zu dieser Attributgruppe. Mit diesem Data-Objekt können Daten zu dieser Attributgruppe hinterlegt werden.
     * Das erzeugte Data-Objekt kann dann mit Werten befüllt werden (z. B. {@code data.getTextValue("Attribut").setText("Test");}) und dann später
     * entweder {@linkplain SystemObject#setConfigurationData(AttributeGroup, Data) konfigurierend gespeichert werden} oder {@linkplain
     * de.bsvrz.dav.daf.main.ClientDavInterface#sendData(ResultData) als Online-Datensatz versendet werden}. Für letzteres muss das Data-Objekt noch
     * in ein {@link ResultData} verpackt werden, um Meta-Informationen wie den Datenzeitstempel zu ergänzen.
     * <p>
     * <b>Diese Methode ist eine kompakte Alternative zu {@link de.bsvrz.dav.daf.main.ClientDavInterface#createData(AttributeGroup)} und sollte nur
     * benutzt werden, wenn Kompatibilität mit einer alten Kernsoftware nicht erforderlich ist.</b>
     *
     * @return neuer initialisierter Datensatz.
     *
     * @since 3.14
     */
    default Data createData() {
        // Klassen sollten diese Methode möglichst überschreiben, um einen Cache zu ergänzen.
        // Siehe z. B. DafAttributeGroup.

        Data adapter = AttributeBaseValueDataFactory.createAdapter(this, AttributeHelper.getAttributesValues(this));
        adapter.setToDefault();
        return adapter;
    }
}
