/*
 * Copyright 2005 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2006-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.dav.daf.main.impl.config;

import de.bsvrz.dav.daf.communication.dataRepresentation.AttributeBaseValueDataFactory;
import de.bsvrz.dav.daf.communication.dataRepresentation.AttributeHelper;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.AttributeGroupUsage;
import de.bsvrz.dav.daf.main.config.ObjectSet;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.util.Memoizer;
import de.bsvrz.sys.funclib.debug.Debug;

import java.util.*;
import java.util.function.Supplier;

/**
 * Klasse, die den Zugriff auf Attributgruppen seitens der Datenverteiler-Applikationsfunktionen ermöglicht.
 *
 * @author Kappich Systemberatung
 */
public class DafAttributeGroup extends DafAttributeSet implements AttributeGroup {

    /** DebugLogger für Debug-Ausgaben */
    private static final Debug _debug = Debug.getLogger();

    /**
     * Speichert alle Attributgruppenverwendungen. Als Schlüssel dient der Aspekt. Die Map wird erst initialisiert, wenn das erste mal auf eine
     * Attributgruppenverwendungen zugegriffen wird. {@code Memoizer.memoize} sorgt für die lazy-Initialisierung, sodass die Map nur einmal erzeugt
     * wird.
     */
    private final Supplier<Map<Aspect, AttributeGroupUsage>> _atgUsageMapSupplier = Memoizer.memoize(() -> {
	    Map<Aspect, AttributeGroupUsage> builder = new HashMap<>();
        final ObjectSet usageSet = getObjectSet("AttributgruppenVerwendungen");
        if (usageSet == null) {
            throw new AssertionError("Menge AttributgruppenVerwendungen fehlt");
        }
        Collection<SystemObject> atgUsage = usageSet.getElements();
        for (SystemObject systemObject : atgUsage) {
            final AttributeGroupUsage usage = (AttributeGroupUsage) systemObject;
            builder.put(usage.getAspect(), usage);
        }
	    return Map.copyOf(builder);
    });

    /**
     * Supplier für Data-Objekte. {@code Memoizer.memoize} sorgt für die lazy-Initialisierung, sodass das ursprüngliche Data-Objekt nur einmal
     * erzeugt wird.
     */
    private final Supplier<Data> _dataSupplier = Memoizer.memoize(() -> {
        Data adapter = AttributeBaseValueDataFactory.createAdapter(this, AttributeHelper.getAttributesValues(this));
        adapter.setToDefault();
        return adapter;
    });

    /**
     * Erzeugt ein neues Objekt, dessen Eigenschaften im Anschluss mit der read-Methode eingelesen werden sollten.
     *
     * @param dataModel DataModel Implementierung, der das neue Objekt zugeordnet ist.
     */
    public DafAttributeGroup(DafDataModel dataModel) {
        super(dataModel);
        _internType = ATTRIBUTE_GROUP;
    }

    /** Erzeugt ein neues Objekt mit den angegebenen Eigenschaften */
    public DafAttributeGroup(long id, String pid, String name, long typId, byte state, String error, DafDataModel dataModel,
                             short validFromVersionNumber, short validToVersionNumber, long responsibleObjectId, long[] setIds) {
        super(id, pid, name, typId, state, error, dataModel, validFromVersionNumber, validToVersionNumber, responsibleObjectId, setIds);
        _internType = ATTRIBUTE_GROUP;
    }

    public final String parseToString() {
        StringBuilder str = new StringBuilder("Attributgruppe: \n");
        str.append(super.parseToString());
	    Collection<Aspect> aspects = getAspects();
        str.append("Aspekte: \n");
        for (Aspect aspect : aspects) {
            str.append(((DafAspect) aspect).parseToString());
        }
        return str.toString();
    }

    @Override
    public final boolean isConfigurating() {
        final AttributeGroupUsage attributeGroupUsage = getAttributeGroupUsage("asp.eigenschaften");
        return attributeGroupUsage != null && attributeGroupUsage.isConfigurating();
    }

    @Override
    public boolean isParameter() {
        final Collection<Aspect> aspects = getAspects();
        final Aspect asp01 = getDataModel().getAspect("asp.parameterSoll");
        final Aspect asp02 = getDataModel().getAspect("asp.parameterVorgabe");
        return aspects.contains(asp01) && aspects.contains(asp02);
    }

    @Override
    public Collection<AttributeGroupUsage> getAttributeGroupUsages() {
        return _atgUsageMapSupplier.get().values();
    }

    @Override
    public AttributeGroupUsage getAttributeGroupUsage(Aspect asp) {
        AttributeGroupUsage attributeGroupUsage = _atgUsageMapSupplier.get().get(asp);
        if (attributeGroupUsage == null) {
            _debug.fine("Attributgruppenverwendung für " + this.getPidOrNameOrId() + " und " + asp.getPidOrNameOrId() + " nicht gefunden");
        }
        return attributeGroupUsage;
    }

    @Override
    public Data createData() {
        return _dataSupplier.get().createModifiableCopy();
    }

    @Override
    public final Collection<Aspect> getAspects() {
	    return _atgUsageMapSupplier.get().keySet();
    }
}
