/*
 * Copyright 2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.main.impl.notification;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * Hilfsklasse für den Thread, der im {@link de.bsvrz.dav.daf.main.impl.config.DafDataModel} Listener-Benachrichtigungen an die Client-Applikation
 * verschickt. Dies wird von einem eigenen Thread durchgeführt, damit interne Threads wie der
 * {@link de.bsvrz.dav.daf.communication.lowLevel.LowLevelThread}
 * nicht durch die Applikation blockiert werden können (Beispiel:  Anlegen eines dynamischen Objekts innerhalb einer Listener-Benachrichtigung)
 *
 * @author Kappich Systemberatung
 */
public class DafMainEventThread {

    /**
     * Executor-Service, der den EventThread bereitstellt. Events können mit {@link ExecutorService#submit(Runnable)} an den Thread übergeben werden.
     */
    private final ExecutorService _executorService;

    /**
     * Erstellt einen neuen DafMainEventThread.
     */
    public DafMainEventThread() {
        _executorService = Executors.newSingleThreadExecutor(r -> {
            Thread thread = new Thread(r, "DafMainEventThread");
            thread.setDaemon(true);
            return thread;
        });
    }

    /**
     * Führt ein Ereignis aus.
     *
     * @param runnable Auszuführendes Ereignis. Das Runnable (bzw. der Lambda-Ausdruck) wird im Ereignis-Thread ausgeführt.
     */
    public void invoke(Runnable runnable) {
        _executorService.submit(runnable);
    }

    /**
     * Fährt den Thread herunter.
     */
    public void shutdown() {
        _executorService.shutdown();
    }

    @Override
    public String toString() {
        return "DafMainEventThread{" + "_executorService=" + _executorService + '}';
    }
}
