/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.util;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;

/**
 * RandomAccessFile-ähnliche implementierung, die es erlaubt, die Datei zu schließen und bei Bedarf wieder von der selben Position zu öffnen. Diese
 * Implementierung kann verwendet werden, weil eine Applikation sonst viele Dateien gleichzeitig öffnet um Beschränkungen im Betriebssystem zu
 * umgehen.
 * <p>
 * Zusätzlich ist die Ein- und Ausgabe gepuffert, wodurch die Performance wesentlich erhöht wird.
 * <p>
 * Es sollte sichergestellt werden, dass kein anderer Prozess parallel die Datei bearbeitet, sonst entsteht möglicherweise unerwünschtes Verhalten.
 *
 * @author Kappich Systemberatung
 */
public class CloseableRandomAccessFile extends FileAccess {

    private final File _file;

    private OutputStream _rawOutputStream;

    private InputStream _rawInputStream;

    private FileChannel _channel;

    private boolean _closed;

    public CloseableRandomAccessFile(final File file) {
        super(defaultBufferSize);
        _file = file;
    }

    public CloseableRandomAccessFile(final File file, final int bufferSize) {
        super(bufferSize);
        _file = file;
    }

    @Override
    protected DataOutputStream getDataOutStream() throws IOException {
        flushInStream(); // Vor dem Schreiben Leseoperationen abschließen und Lesepuffer löschen (um FilePointer abschließend zu setzen)
        if (_dataOutStream == null) {
            _dataOutStream = new DataOutputStream(new BufferedOutputStream(getRawOutputStream(), _bufferSize));
        }
        return _dataOutStream;
    }

    private OutputStream getRawOutputStream() throws IOException {
        if (_rawOutputStream == null) {
            _rawOutputStream = Channels.newOutputStream(getChannel());
        }
        return _rawOutputStream;
    }

    @Override
    protected DataInputStream getDataInStream() throws IOException {
        flushOutStream(); // Vor dem Lesen Schreiboperationen abschließen und Schreibpuffer löschen (um FilePointer abschließend zu setzen)
        if (_dataInStream == null) {
            _dataInStream = new DataInputStream(new BufferedInputStream(getRawInputStream(), _bufferSize));
        }
        return _dataInStream;
    }

    private InputStream getRawInputStream() throws IOException {
        if (_rawInputStream == null) {
            _rawInputStream = Channels.newInputStream(getChannel());
        }
        return _rawInputStream;
    }

    @Override
    protected final FileChannel getChannel() throws IOException {
        if (_channel == null) {
            openFile();
        }
        return _channel;
    }

    protected void openFile() throws IOException {
        _channel = new RandomAccessFile(_file, "rw").getChannel();
        _channel.position(_position);
    }

    @Override
    public boolean isOpen() {
        return !_closed;
    }

    @Override
    public void close() throws IOException {
        closeFile();
        _closed = true;
    }

    /**
     * Schließt die Datei auf Betriebssystemebene, aber erlaubt weitere Lese- und Schreibzugriffe. Falls später wieder Daten gelesen oder geschrieben
     * werden sollen wird die Datei automatisch erneut geöffnet. Da nicht sichergestellt wird, dass die Datei nicht zwischenzeitlich manipuliert wird
     * muss der Anwender hier vorsichtig sein.
     */
    public void closeFile() throws IOException {
        if (_dataInStream != null) {
            _dataInStream.close();
        }
        if (_dataOutStream != null) {
            _dataOutStream.close();
        }
        if (_channel != null) {
            _channel.close();
        }
        _dataInStream = null;
        _dataOutStream = null;
        _rawInputStream = null;
        _rawOutputStream = null;
        _channel = null;
    }
}
