/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.dav.daf.util;

import java.io.DataInput;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.SeekableByteChannel;

/**
 * Abstrakte Basisklasse für Klassen, die RandomAccessFile-ähnliche Funktionalität bieten
 *
 * @author Kappich Systemberatung
 */
public abstract class FileAccess implements DataInput, DataOutput, SeekableByteChannel {

    /** Standardpuffergröße */
    protected static final int defaultBufferSize = 512;

    /**
     * Größe des Lese und Schreibpuffers
     */
    protected final int _bufferSize;

    /** Gepufferter EingabeStream, wird bei Bedarf initialisiert und gelöscht */
    protected DataInputStream _dataInStream;

    /** Gepufferter AusgabeStream, wird bei Bedarf initialisiert und gelöscht */
    protected DataOutputStream _dataOutStream;

    /**
     * Aktuelle Dateiposition aus Anwendersicht, muss hier gemerkt und selbst berechnet werden, weil die Position des FileChannels durch die Pufferung
     * beim Lesen und Schreiben nicht notwendigerweise der aktuellen logischen Position entspricht
     */
    protected long _position;

    /**
     * Aktueller Modus Schreiben? Zwischen Lesen und Schreiben muss jeweils der Schreibpuffer geleert werden oder die Schreibposition korrekt gesetzt
     * werden.
     */
    private boolean _isWriting;

    public FileAccess(final int bufferSize) {
        _bufferSize = bufferSize;
    }

    protected void flushInStream() {
        // Eingabestream verwerfen, flushen nicht möglich und nicht nötig
        _dataInStream = null;
    }

    protected void flushOutStream() throws IOException {
        // Ausgabestream verwerfen, nur flushen, nicht schließen, weil schließen das ganze RandomAccessFile schließen würde!
        if (_dataOutStream != null) {
            _dataOutStream.flush();
            _dataOutStream = null;
        }
    }

    /**
     * Gibt einen DataOutputStream zum Schreiben zurück
     *
     * @return DataOutputStream
     */
    protected abstract DataOutputStream getDataOutStream() throws IOException;

    /**
     * Gibt einen DataInputStream zum Lesen zurück
     *
     * @return DataInputStream
     */
    protected abstract DataInputStream getDataInStream() throws IOException;

    /**
     * Gibt einen gültigen FileChannel zurück, mit dem die Klasse die Datei manipulieren kann.
     *
     * @return einen gültigen FileChannel
     */
    protected abstract FileChannel getChannel() throws IOException;

    @Override
    public abstract boolean isOpen();

    @Override
    public void close() throws IOException {
        if (_dataInStream != null) {
            _dataInStream.close();
        }
        if (_dataOutStream != null) {
            _dataOutStream.close();
        }
        getChannel().close();
    }

    @Override
    public void readFully(final byte[] b) throws IOException {
        switchToReading();
        getDataInStream().readFully(b);
        _position += b.length;
    }

    @Override
    public void readFully(final byte[] b, final int off, final int len) throws IOException {
        switchToReading();
        getDataInStream().readFully(b, off, len);
        _position += len;
    }

    /**
     * Überspringt n genau Bytes. Anders als DataInput definiert wird immer genau die übergebene Zahl an bytes übersprungen, d.h. die Methode gibt
     * immer den Parameter n zurück. Daher entspricht diese Methode {@code position(position() + n); return n;}
     * <p>
     * Diese Methode kann über das Dateiende hinausspringen, vgl. {@link RandomAccessFile#seek(long)}.
     *
     * @param n Anzahl zu überspringender Bytes (kann negativ sein, dann wird rückwärts gesprungen)
     *
     * @return n
     *
     * @throws IOException Eingabe-/Ausgabefehler beim Lesen oder Schreiben der Datei
     */
    @Override
    public int skipBytes(final int n) throws IOException {
        return (int) skip(n);
    }

    /**
     * Überspringt genau n Bytes. Daher entspricht diese Methode {@code position(position() + n); return n;}
     * <p>
     * Diese Methode kann über das Dateiende hinausspringen, vgl. {@link RandomAccessFile#seek(long)}.
     *
     * @param n Anzahl zu überspringender Bytes (kann negativ sein, dann wird rückwärts gesprungen)
     *
     * @return Der Parameter n (zur Kompatibilität mit FileChannel)
     *
     * @throws IOException Eingabe-/Ausgabefehler beim Lesen oder Schreiben der Datei
     */
    public long skip(long n) throws IOException {
        if (n == 0) {
            return 0;
        }
        if (n > 0 && n < _bufferSize / 2 && _dataInStream != null) {
            // Es besteht eine gute Wahrscheinlichkeit, dass die Zielposition noch gepuffert ist
            int remaining = (int) n;
            while (remaining > 0) {
                int skipped = getDataInStream().skipBytes(remaining);
                if (skipped <= 0) {
                    break;
                }
                remaining -= skipped;
            }
            if (remaining == 0) {
                // Skip Erfolgreich
                _position += n;
                return n;
            }
            // Sonst als Fallback position() benutzen und den _dataInStream bei Bedarf neu initialisieren
        }
        position(position() + n);
        return n;
    }

    @Override
    public boolean readBoolean() throws IOException {
        switchToReading();
        boolean b = getDataInStream().readBoolean();
        _position++;
        return b;
    }

    @Override
    public byte readByte() throws IOException {
        switchToReading();
        byte b = getDataInStream().readByte();
        _position++;
        return b;
    }

    @Override
    public int readUnsignedByte() throws IOException {
        switchToReading();
        int unsignedByte = getDataInStream().readUnsignedByte();
        _position++;
        return unsignedByte;
    }

    @Override
    public short readShort() throws IOException {
        switchToReading();
        short readShort = getDataInStream().readShort();
        _position += 2;
        return readShort;
    }

    @Override
    public int readUnsignedShort() throws IOException {
        switchToReading();
        int unsignedShort = getDataInStream().readUnsignedShort();
        _position += 2;
        return unsignedShort;
    }

    @Override
    public char readChar() throws IOException {
        switchToReading();
        char readChar = getDataInStream().readChar();
        _position += 2;
        return readChar;
    }

    @Override
    public int readInt() throws IOException {
        switchToReading();
        int readInt = getDataInStream().readInt();
        _position += 4;
        return readInt;
    }

    @Override
    public long readLong() throws IOException {
        switchToReading();
        long readLong = getDataInStream().readLong();
        _position += 8;
        return readLong;
    }

    @Override
    public float readFloat() throws IOException {
        switchToReading();
        float readFloat = getDataInStream().readFloat();
        _position += 8;
        return readFloat;
    }

    @Override
    public double readDouble() throws IOException {
        switchToReading();
        double readDouble = getDataInStream().readDouble();
        _position += 16;
        return readDouble;
    }

    /**
     * Das Lesen einer einzelnen Zeile wird von dieser Klasse nicht unterstützt, da sie für binäre Daten gedacht ist.
     *
     * @throws UnsupportedOperationException immer
     */
    @Override
    @Deprecated
    public String readLine() {
        throw new UnsupportedOperationException();
    }

    @Override
    public String readUTF() throws IOException {
        switchToReading();
        DataInputStream inStream = getDataInStream();
        inStream.mark(2);
        int len = inStream.readUnsignedShort();
        _position += len + 2;
        inStream.reset();
        return inStream.readUTF();
    }

    @Override
    public void write(final int b) throws IOException {
        switchToWriting();
        getDataOutStream().write(b);
        _position += 1;
    }

    @Override
    public void write(final byte[] b) throws IOException {
        switchToWriting();
        getDataOutStream().write(b);
        _position += b.length;
    }

    @Override
    public void write(final byte[] b, final int off, final int len) throws IOException {
        switchToWriting();
        getDataOutStream().write(b, off, len);
        _position += len;
    }

    @Override
    public void writeBoolean(final boolean v) throws IOException {
        switchToWriting();
        getDataOutStream().writeBoolean(v);
        _position += 1;
    }

    @Override
    public void writeByte(final int v) throws IOException {
        switchToWriting();
        getDataOutStream().writeByte(v);
        _position += 1;
    }

    @Override
    public void writeShort(final int v) throws IOException {
        switchToWriting();
        getDataOutStream().writeShort(v);
        _position += 2;
    }

    @Override
    public void writeChar(final int v) throws IOException {
        switchToWriting();
        getDataOutStream().writeChar(v);
        _position += 2;
    }

    @Override
    public void writeInt(final int v) throws IOException {
        switchToWriting();
        getDataOutStream().writeInt(v);
        _position += 4;
    }

    @Override
    public void writeLong(final long v) throws IOException {
        switchToWriting();
        getDataOutStream().writeLong(v);
        _position += 8;
    }

    @Override
    public void writeFloat(final float v) throws IOException {
        switchToWriting();
        getDataOutStream().writeFloat(v);
        _position += 4;
    }

    @Override
    public void writeDouble(final double v) throws IOException {
        switchToWriting();
        getDataOutStream().writeDouble(v);
        _position += 8;
    }

    @Override
    public void writeBytes(final String s) throws IOException {
        switchToWriting();
        getDataOutStream().writeBytes(s);
        _position += s.length();
    }

    @Override
    public void writeChars(final String s) throws IOException {
        switchToWriting();
        getDataOutStream().writeChars(s);
        _position += s.length() * 2;
    }

    @Override
    public void writeUTF(final String s) throws IOException {
        switchToWriting();
        DataOutputStream outStream = getDataOutStream();
        outStream.writeUTF(s);

        // Die Position ausnahmsweise nicht selbst berechnen, sondern die Ausgabe flushen und den FilePointer des Channels nehmen
        // denn die Anzahl Bytes, die bei writeUTF geschrieben werden sind nicht trivial und performant zu ermitteln
        outStream.flush();
        _position = getChannel().position();
    }

    @Override
    public int read(final ByteBuffer dst) throws IOException {
        switchToReading();
        flushOutStream();
        flushInStream();
        int read = getChannel().read(dst);
        _position += read;
        return read;
    }

    /**
     * Methode analog zu {@link RandomAccessFile#read(byte[])}. Sollte nicht benutzt werden, sie fehleranfällig ist falls nicht der ganze Puffer
     * gelesen wird. Besser: {@link #readFully(byte[])}
     *
     * @param b Puffer
     *
     * @return Anzahl gelesener bytes
     *
     * @throws IOException Eingabe-/Ausgabefehler beim Lesen oder Schreiben der Datei
     */
    public int read(final byte[] b) throws IOException {
        switchToReading();
        return read(b, 0, b.length);
    }

    /**
     * Methode analog zu {@link RandomAccessFile#read(byte[], int, int)}. Sollte nicht benutzt werden, sie fehleranfällig ist falls nicht der ganze
     * Puffer gelesen wird. Besser: {@link #readFully(byte[], int, int)}
     *
     * @param b   Puffer
     * @param off Position im Puffer an die die Daten geschrieben werden
     * @param len Maximalanzahl zu lesender Bytes
     *
     * @return Anzahl gelesener bytes
     *
     * @throws IOException Eingabe-/Ausgabefehler beim Lesen oder Schreiben der Datei
     */
    public int read(final byte[] b, final int off, final int len) throws IOException {
        switchToReading();
        int numRead = getDataInStream().read(b, off, len);
        _position += numRead;
        return numRead;
    }

    /**
     * Methode analog zu {@link RandomAccessFile#read()}. Sollte nicht benutzt werden, da fehleranfällig bei Dateiende. Besser: {@link #readByte()}
     *
     * @return Gelesenes byte oder -1 falls am Dateiende.
     *
     * @throws IOException Eingabe-/Ausgabefehler beim Lesen oder Schreiben der Datei
     */
    @Deprecated
    public int read() throws IOException {
        switchToReading();
        int read = getDataInStream().read();
        if (read != -1) {
            _position++;
        }
        return read;
    }

    @Override
    public int write(final ByteBuffer src) throws IOException {
        switchToWriting();
        flushOutStream();
        flushInStream();
        int write = getChannel().write(src);
        _position += write;
        return write;
    }

    /**
     * @see java.nio.channels.FileChannel#position()
     */
    @Override
    public long position() {
        // NICHT "return _channel.position();" wegen Pufferung, vgl. Javadoc zu _position
        return _position;
    }

    /**
     * @see java.nio.channels.FileChannel#position(long)
     */
    @Override
    public FileAccess position(final long newPosition) throws IOException {
        if (newPosition == _position) {
            return this;
        }
        flushOutStream();
        flushInStream();
        getChannel().position(newPosition);
        _position = newPosition;
        return this;
    }

    /**
     * @see java.nio.channels.FileChannel#truncate(long)
     */
    @Override
    public FileAccess truncate(final long size) throws IOException {
        flushOutStream();
        flushInStream();
        getChannel().truncate(size);
        return this;
    }

    /**
     * Gibt die Dateilänge zurück
     *
     * @return Länge in Bytes
     *
     * @see java.nio.channels.FileChannel#size()
     */
    @Override
    public long size() throws IOException {
        flushOutStream();
        return getChannel().size();
    }

    /**
     * Für RandomAccessFile-Kompatibilität
     *
     * @param position Neue Position
     *
     * @see #position(long)
     * @see RandomAccessFile#seek(long)
     */
    public void seek(long position) throws IOException {
        position(position);
    }

    /**
     * Für RandomAccessFile-Kompatibilität
     *
     * @return Position
     *
     * @see #position()
     * @see RandomAccessFile#getFilePointer()
     */
    public long getFilePointer() {
        return position();
    }

    /**
     * Für RandomAccessFile-Kompatibilität
     *
     * @return Dateilänge
     *
     * @see #size()
     * @see RandomAccessFile#length()
     */
    public long length() throws IOException {
        return size();
    }

    /**
     * Für RandomAccessFile-Kompatibilität
     *
     * @param len neue Dateilänge
     *
     * @see RandomAccessFile#setLength(long)
     */
    public void setLength(final long len) throws IOException {
        flushOutStream();
        flushInStream();
        getChannel().truncate(len);
    }

    /**
     * Schreibt den Schreibpuffer auf die Festplatte
     *
     * @throws IOException Eingabe-/Ausgabefehler beim Lesen oder Schreiben der Datei
     */
    public void flush() throws IOException {
        flushOutStream();
    }

    @Override
    public String toString() {
        return "[pos=" + position() + "]";
    }

    private void switchToWriting() throws IOException {
        if (_isWriting) {
            return;
        }
        flushInStream();
        getChannel().position(_position);
        _isWriting = true;
    }

    private void switchToReading() throws IOException {
        if (!_isWriting) {
            return;
        }
        flushOutStream();
        _isWriting = false;
    }

}
