/*
 * Copyright 2004 by Kappich+Kniß Systemberatung, Aachen
 * Copyright 2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.sys.funclib.debug.
 *
 * de.bsvrz.sys.funclib.debug is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.sys.funclib.debug is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.sys.funclib.debug; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.sys.funclib.debug;

import java.io.PrintWriter;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.Optional;
import java.util.logging.Handler;

/**
 * Gibt die DebugMeldungen als Text aus.
 *
 * @author Hans Christian Kniß (HCK)
 * @version $Revision$ / $Date$
 */
public class DebugFormatterFileText extends DebugFormatterStdErrText {

    /**
     * Gibt am Anfang des Protokolls bzw. der Datei den Text "Ausgabedatei angelegt." aus
     *
     * @param h Handler der den Formatter aufgerufen hat. Wird nicht verwendet.
     *
     * @return String mit einleitendem Meldungstext
     */
    public String getHead(Handler h) {
        StringBuilder sb = new StringBuilder().append(NEWLINE);
        sb.append("------- ").append(_dateFormat.format(ZonedDateTime.now()));
        sb.append(" (TID:......)");
        sb.append(SINGLE_LINE).append(NEWLINE);
        sb.append("STATUS").append(NEWLINE);
        appendRuntimeInfo(sb);
        sb.append("Ausgabedatei angelegt.").append(NEWLINE);
        return sb.toString();
    }

    private void appendRuntimeInfo(StringBuilder sb) {
        try {
            Runtime runtime = Runtime.getRuntime();
            long reservedMemory = runtime.totalMemory();
            long usedMemory = reservedMemory - runtime.freeMemory();
            long maxMemory = runtime.maxMemory();
            sb.append("Speicher genutzt/reserviert/maximal: ")
                    .append(formatMemory(usedMemory)).append("/")
                    .append(formatMemory(reservedMemory)).append("/")
                    .append(formatMemory(maxMemory)).append(NEWLINE);

            // Zugriff mit Reflection, damit es in Java 8 kompilierbar bleibt
            Class<?> processHandleClass = Class.forName("java.lang.ProcessHandle");
            Class<?> infoClass = Class.forName("java.lang.ProcessHandle$Info");
            Object currentProcess = processHandleClass.getMethod("current").invoke(null);
            Object pid = processHandleClass.getMethod("pid").invoke(currentProcess);
            Object info = processHandleClass.getMethod("info").invoke(currentProcess);
            Optional<?> command = (Optional<?>) infoClass.getMethod("command").invoke(info);
            Optional<?> start = (Optional<?>) infoClass.getMethod("startInstant").invoke(info);
            
//            Äquivalenter Code ohne Reflection:
//            ProcessHandle currentProcess = ProcessHandle.current();
//            long pid = currentProcess.pid();
//            Optional<String> command = currentProcess.info().command();
//            Optional<Instant> start = currentProcess.info().startInstant();
            
            sb.append("Prozess-ID: ").append(pid).append(NEWLINE);
            command.ifPresent(it -> sb.append("Kommando  : ").append(it).append(NEWLINE));
            sb.append("Läuft seit: ").append(formatTime((Instant)start.orElse(null))).append(NEWLINE);
        }
        catch (ClassNotFoundException ignored) {
            sb.append("Prozess-ID: N/A (JDK veraltet)").append(NEWLINE);
        }
        catch (Throwable e) {
            e.printStackTrace(new PrintWriter(new StringBuilderWriter(sb)));
        }
    }

    private String formatTime(Instant start) {
        if(start != null) {
            ZonedDateTime startTime = ZonedDateTime.ofInstant(start, ZoneId.systemDefault());
            return _dateFormat.format(startTime);
        }
        else {
            return "N/A";
        }
    }

    /**
     * Formatiert eine Speichermenge
     * @param memory Anzahl Bytes
     * @return z. B. "235,2M" aber das Format ist nicht vorgeschrieben
     */
    private static String formatMemory(long memory) {
        return String.format("%.1fM", memory / (1048576.0));
    }
    
    /**
     * Gibt am Ende des Protokolls bzw. der Datei den Text "Ausgabedatei korrekt abgeschlossenen." aus. ACHTUNG: Wird nicht bei StdErr ausgegeben oder bei abnormaler
     * Beendigung der Debugausgabe!
     *
     * @param h Handler der den Formatter aufgerufen hat. Wird nicht verwendet.
     *
     * @return String mit abschliessendem Meldungstext
     */

    public String getTail(Handler h) {
        StringBuilder sb = new StringBuilder().append(NEWLINE);
        sb.append("-------").append(_dateFormat.format(ZonedDateTime.now()));
        sb.append("(TID:......)");
        sb.append(SINGLE_LINE);
        sb.append("STATUS").append(NEWLINE);
        sb.append("Ausgabedatei korrekt abgeschlossenen.").append(NEWLINE);
        return sb.toString();
    }

}
